/*	fsck 1.5 - File system consistency check.	Author: Kees J. Bot
 *
 * This is just a forerunner to fsck1, fsck1f, fsck2, and fsck2f.
 */
#define nil	0
#include <sys/types.h>
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <signal.h>
#include <sys/mnttab.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <minix/config.h>
#include <minix/const.h>
#include <minix/type.h>
#include <fs/const.h>
#include <fs/type.h>
#include <fs/super.h>

#undef printf

#define arraysize(a)		(sizeof(a) / sizeof((a)[0]))
#define arraylimit(a)		((a) + arraysize(a))
#define between(a, c, z)	((unsigned) ((c) - (a)) <= ((z) - (a)))

char FSTAB[]=	"/etc/fstab";
char MTAB[]=	"/etc/mtab";

int boot_time= 0;	/* True if rebooting. */
int preen= 0;		/* -p: Preen mode. */
int force= 0;		/* -f: Ignore clean flag. */
int wronly= 0;		/* -w: Check read-write file systems only. */
int affirmative= 0;	/* -y: Assume 'yes' on all questions. */
int negative= 0;	/* -n: Assume 'no' on all questions. */
int clean_toggle= 0;	/* -c: Enable/disable the clean flag. */

/* The things that can go wrong. */
#define DIS_REPAIRED	0x01	/* Repairs have been made. */
#define DIS_MOUNTED	0x02	/* The repaired FS is mounted. */
#define DIS_BAD		0x04	/* A check failed miserably. */
#define DIS_INTERRUPT	0x08	/* Interrupt typed, abandon ship. */
#define DIS_QUIT	0x10	/* Quit typed, single user. */

int disaster= 0;	/* No disasters yet. */

void report(const char *label)
{
	if (label != nil && *label != 0)
		fprintf(stderr, "fsck: %s: %s\n", label, strerror(errno));
	else
		fprintf(stderr, "fsck: %s\n", strerror(errno));
	disaster|= DIS_BAD;
}

void fatal(const char *label)
{
	report(label);
	exit(8);
}

char *copystr(const char *s)
{
	char *c;

	if ((c= malloc((strlen(s) + 1) * sizeof(*s))) == nil) fatal("");
	strcpy(c, s);
	return c;
}

int mounted(char *device, char **dir)
/* Check if device is mounted on dir by either comparing device numbers, or
 * by looking in /etc/mtab.  Fill in *dir if non-nil.
 */
{
	FILE *mtf;
	struct mnttab mte, look;
	int r;

	if (boot_time) {
		/* Reboot time, /etc/mtab is stale, check numbers. */
		struct stat devst, dirst;

		if (*dir == nil) { *dir="???"; return 1; } /* ??? */

		if (stat(device, &devst) < 0) return 0;
		if (stat(*dir, &dirst) < 0) return 0;

		return devst.st_rdev == dirst.st_dev;
	}

	/* Read /etc/mtab and look for device. */
	if ((mtf= fopen(MTAB, "r")) == nil) {
		if (errno != ENOENT) fatal(device);
		return 0;
	}

	look.mnt_special= device;
	look.mnt_mountp= nil;
	look.mnt_fstype= nil;
	look.mnt_mntopts= nil;

	r= getmntany(mtf, &mte, &look);
	(void) fclose(mtf);

	if (r >= 0) {
		*dir= mte.mnt_mountp;
		return 1;
	}
	return 0;
}

enum direction { READ, WRITE };
#define EMPTY	ECHILD		/* Overload ECHILD to mean "short read". */

int rw_super(enum direction dir, char *device, struct super_block *super)
/* Read or write the super block. */
{
	int err= 0;
	int fd;
	int n;
	int i;

	if ((fd= open(device, dir == READ ? O_RDONLY : O_WRONLY)) < 0)
		return -1;

	if (lseek(fd, (off_t) SUPER_BLOCK * BLOCK_SIZE, SEEK_SET) < 0)
		err= errno;

	if (dir == READ) {
		if (err == 0
			&& (n= read(fd, (void *) super, sizeof(*super))) < 0
		)
			err= errno;

		if (err == 0 && n < sizeof(*super)) err= EMPTY;
	} else {
		if (err == 0 && write(fd, (void *) super, sizeof(*super)) < 0)
			err= errno;
	}

	close(fd);

	if (err != 0) {
		errno= err;
		return -1;
	}
	return 0;
}

char *fstype(struct super_block *super)
{
	switch (super->s_magic) {
	case SUPER_MAGIC:
		return super->s_flags & S_FLEX ? "1f" : "1";
	case SUPER_V2:
		return super->s_flags & S_FLEX ? "2f" : "2";
	default:
		return nil;
	}
}

void (*icatch)(int sig);	/* Interrupt status at startup. */

void catch(int sig)
{
	signal(sig, catch);
	disaster|= DIS_QUIT;
}

int run_fsck(char *fsck, char *device, char *dir, int rdonly)
{
	char *fkv[20], **fkp= fkv;
	int pid, status, r;
	void (*istat)(int sig);

	*fkp++= fsck;

	if (rdonly || negative) {
		/* Default for the old fsck. */
	} else
	if (affirmative) {
		*fkp++= "-a";
	} else
	if (preen) {
		*fkp++= "-p";
	} else {
		*fkp++= "-r";
	}

	*fkp++= device;
	*fkp= nil;

	if (preen) signal(SIGQUIT, SIG_IGN);	/* Quit won't reach child. */

	fflush(nil);

	switch (pid= fork()) {
	case -1:	report("");	return DIS_BAD;
	case 0:
		signal(SIGINT, icatch);		/* Interrupt as it was. */
		execvp(fsck, fkv);
		fatal(fsck);
	}

	if (preen) signal(SIGQUIT, catch);	/* Quit will reach me. */

	while ((r= wait(&status)) != pid) {
		if (r < 0 && errno != EINTR) { report(""); return DIS_BAD; }
	}

	if (!WIFEXITED(status))		/* Signalled? */
		return WTERMSIG(status) == SIGINT ? DIS_INTERRUPT : DIS_BAD;

	switch (WEXITSTATUS(status)) {
	case 0:				/* No errors. */
		return 0;
	case 4:				/* Errors were repaired. */
		if (mounted(device, &dir)) return DIS_MOUNTED | DIS_REPAIRED;
		/*FALL THROUGH*/
	case 3:
		return DIS_REPAIRED;
	case 8:				/* Errors still exist. */
	default:			/* Some other unknown error. */
		return DIS_BAD;
	}
}

void fsck(char *device, char *dir)
/* Check the device "device", that is, or should be mounted on "dir" (if
 * non-nil).
 */
{
	struct super_block super;
	static char real_fsck[]= "fsck??";
	char *type;
	int r, rdonly, ismounted, isclean, clean_enabled, oldflags;

	/* Unless in preen mode or at boot time, we do not check mounted file
	 * systems.
	 */
	ismounted= mounted(device, &dir);

	if (ismounted && !negative && !preen && !boot_time && !clean_toggle) {
		fprintf(stderr, "fsck: %s is mounted on %s\n", device, dir);
		disaster|= DIS_BAD;
		return;
	}

	if (rw_super(READ, device, &super) < 0) {
		if (errno == EMPTY) {
			fprintf(stderr, "fsck: Unexpected EOF on %s\n", device);
			disaster|= DIS_BAD;
		} else
			report(device);
		return;
	}

	if ((type= fstype(&super)) == nil) {
		fprintf(stderr, "fsck: %s: bad magic number in superblock\n",
			device);
		disaster|= DIS_BAD;
		return;
	}

	/* The clean flag only works if especially enabled. */
	clean_enabled= super.s_fsck_magic[0] == FSCK_MAGIC0
			&& super.s_fsck_magic[1] == FSCK_MAGIC1;

	if (clean_toggle) {
		/* fsck -c: Just toggle the clean flag. */

		if (!clean_enabled) {
			/* Enable the clean flag. */
			super.s_fsck_magic[0]= FSCK_MAGIC0;
			super.s_fsck_magic[1]= FSCK_MAGIC1;
		} else {
			/* Disable the clean flag. */
			super.s_fsck_magic[0]= !FSCK_MAGIC0;
			super.s_fsck_magic[1]= !FSCK_MAGIC1;
		}
		super.s_flags&= ~S_CLEAN;

		if (rw_super(WRITE, device, &super) < 0)
			report(device);
		else {
			printf("%s: clean flag %sabled\n", device,
				!clean_enabled ? "en" : "dis");
		}
		return;
	}

	/* Do not check file systems marked clean unless forced. */
	isclean= super.s_flags & S_CLEAN && clean_enabled;

	if (!force && isclean) {
		printf("%s: is clean\n", device);
		return;
	} else {
		/* What are we checking? */
		printf("%s: V%c%s filesystem\n", device,
			type[0], type[1] == 'f' ? " flex" : "");

		/* Tell if we're going to check a mounted FS. */
		if (ismounted)
			printf("%s is mounted on %s\n", device, dir);

		/* Try to mark it dirty before checking, assume -n if we can't
		 * write.
		 */
		super.s_flags&= ~S_CLEAN;

		if (!(rdonly= negative) && rw_super(WRITE, device, &super) < 0)
		{
			if (errno != EACCES) {
				report(device);
				return;
			}
			rdonly= 1;
		}

		/* Time to call the real McCoy. */
		strcpy(real_fsck + 4, type);

		disaster|= r= run_fsck(real_fsck, device, dir, rdonly);

#if TRUST_FSCK_COMPLETELY
		isclean= (r & ~DIS_REPAIRED) == 0 && clean_enabled;
#else
		/* Fsck needs to be called twice sometimes to fully repair
		 * a device.  So don't mark it clean if repairs were made.
		 */
		isclean= r == 0 && clean_enabled;
#endif
	}

	/* The file system has been checked.  Set the clean flag? */
	oldflags= super.s_flags;

	if (isclean && (!ismounted || boot_time || preen)) {
		super.s_flags|= S_CLEAN;
	}

	/* Rewrite superblock? */
	if (!rdonly && super.s_flags != oldflags) {
		if (rw_super(WRITE, device, &super) < 0) report(device);

		printf("%s: marked clean\n", device);
		return;
	}

	/* Say something. */
	if (!(r & DIS_INTERRUPT)) {
		printf("%s: %s\n", device,
			(r & ~DIS_MOUNTED) == 0 ? "ok" :
			(r & DIS_REPAIRED) ? "repaired" : "bad");
	}
}

struct checklist {	/* List of fstab devices to check. */
	struct checklist *next;
	int	pass;
	char	*special;
	char	*mountp;
};

struct checklist *makelist(void)
/* Make a list of devices in fstab to check sorted by pass number. */
{
	FILE *fst;
	struct mnttab mte, look;
	int pass;
	struct checklist *cl= nil, **pcl, *new;

	/* Look for filesystem devices. */
	look.mnt_special= nil;
	look.mnt_mountp= nil;
	look.mnt_fstype= "dev";
	look.mnt_mntopts= nil;

	if ((fst= fopen(FSTAB, "r")) == nil) fatal(FSTAB);

	while (getmntany(fst, &mte, &look) == 0) {
		/* Pass number. */
		pass= atoi(mte.mnt_fsckpass);

		/* Zero pass numbers must be skipped. */
		if (pass <= 0) continue;

		/* Only do read-write mounted devices? */
		if (wronly && hasmntopt(&mte, "ro")) continue;

		/* Insert this device in the checklist in pass order. */
		pcl= &cl;
		while (*pcl != nil && (*pcl)->pass <= pass) pcl= &(*pcl)->next;

		if ((new= malloc(sizeof(*new))) == nil) fatal("");
		new->pass= pass;
		new->special= copystr(mte.mnt_special);
		new->mountp= copystr(mte.mnt_mountp);
		new->next= *pcl;
		*pcl= new;
	}
	(void) fclose(fst);

	return cl;
}

void abandon(void)
/* The root file system is not as described from /etc/fstab, probably a
 * boot from a backup.
 */
{
	printf("Root FS does not match /etc/fstab, skipping checks.\n");
	exit(0);
}

void usage(void)
{
	fprintf(stderr,
		"Usage: fsck -[pfwcyn] [-l number] [filesystem ...]\n");
	exit(8);
}

int main(int argc, char **argv)
{
	int i, fd;

	/* Process options. */
	for (i = 1; i < argc && argv[i][0] == '-'; i++) {
		char *p= argv[i]+1;

		while (*p != 0) {
			switch (*p++) {
			case 'p':	preen= 1;	break;
			case 'f':	force= 1;	break;
			case 'w':	wronly= 1;	break;
			case 'c':	clean_toggle= 1;break;
			case 'y':	affirmative= 1;	break;
			case 'n':	negative= 1;	break;
			case 'l':
				if (between('0', *p, '9')) {
					do p++; while (between('0', *p, '9'));
				} else {
					char *end;
					if (++i >= argc) usage();
					(void) strtoul(argv[i], &end, 10);
					if (*end != 0) usage();
				}
				break;
			default:
				usage();
			}
		}
	}

	if (affirmative && negative) {
		fprintf(stderr, "fsck -yn?  Make up your mind!\n");
		exit(8);
	}

	icatch= signal(SIGINT, SIG_IGN);    /* exit(12) if child interrupted. */
	if (preen) signal(SIGQUIT, SIG_IGN);/* exit(2) if Quit typed. */

	/* Try to open mtab read-write.  If it fails with EROFS then this
	 * must be reboot time and /etc/mtab is stale.
	 */
	if ((fd= open(MTAB, O_RDWR)) < 0) {
		if (errno == EROFS) boot_time= 1;
	} else
		close(fd);

	if (i == argc) {
		/* No arguments, process /etc/fstab in pass order. */
		struct checklist *cl;

		for (cl= makelist(); cl != nil; cl= cl->next) {
			/* Only preen the file systems if the root is
			 * mounted from the expected device.
			 */
			if (preen && strcmp(cl->mountp, "/") == 0
				&& !mounted(cl->special, &cl->mountp)
			) abandon();

			fsck(cl->special, cl->mountp);

			if (disaster & DIS_INTERRUPT) break;
		}
	} else {
		/* Process arguments. */
		FILE *fst;
		struct mnttab mte, look;
		struct stat st;
		int r;
		char *fsname, *dir;

		for (; i < argc; i++) {
			fsname= argv[i];
			dir= nil;

			r= stat(argv[i], &st);
			if (r < 0 && errno != ENOENT) fatal(argv[i]);

			if (r < 0 || !S_ISBLK(st.st_mode)) {
				/* Search fstab for the mount point. */

				if ((fst= fopen(FSTAB, "r")) == nil)
					fatal(FSTAB);

				look.mnt_special= nil;
				look.mnt_mountp= fsname;
				look.mnt_fstype= nil;
				look.mnt_mntopts= nil;

				if (getmntany(fst, &mte, &look) == 0) {
					dir= argv[i];
					fsname= copystr(mte.mnt_special);
				}
				(void) fclose(fst);
			}
			fsck(fsname, dir);

			if (disaster & DIS_INTERRUPT) break;
		}
	}
	if (disaster & DIS_INTERRUPT) exit(12);	/* Interrupt typed. */
	if (disaster & DIS_BAD) exit(8);	/* Really bad. */
	if (disaster & DIS_MOUNTED) exit(4);	/* Mounted FS fixed. */
	if (disaster & DIS_QUIT) exit(2);	/* Want to be single user. */
	exit(0);
}
/* Kees J. Bot 6-12-92. */

/*
 * $PchId: fsck.c,v 1.3 1995/11/27 22:12:18 philip Exp $
 */
