/* readclock - read the real time clock		Authors: T. Holm & E. Froese */

/************************************************************************/
/*									*/
/*   readclock.c							*/
/*									*/
/*		Read the clock value from the 64 byte CMOS RAM		*/
/*		area, then set system time.				*/
/*									*/
/*		If the machine ID byte is 0xFC or 0xF8, the device	*/
/*		/dev/mem exists and can be opened for reading,		*/
/*		and no errors in the CMOS RAM are reported by the	*/
/*		RTC, then the time is read from the clock RAM		*/
/*		area maintained by the RTC.				*/
/*									*/
/*		The clock RAM values are decoded and fed to mktime	*/
/*		to make a time_t value, then stime(2) is called.	*/
/*									*/
/*		This fails if:						*/
/*									*/
/*		If the machine ID does not match 0xFC or 0xF8 (no	*/
/*		error message.)						*/
/*									*/
/*		If the machine ID is 0xFC or 0xF8 and /dev/mem		*/
/*		is missing, or cannot be accessed.			*/
/*									*/
/*		If the RTC reports errors in the CMOS RAM.		*/
/*									*/
/************************************************************************/
/*    origination          1987-Dec-29              efth                */
/*    robustness	   1990-Oct-06		    C. Sylvain		*/
/* incorp. B. Evans ideas  1991-Jul-06		    C. Sylvain		*/
/*    set time & calibrate 1992-Dec-17		    Kees J. Bot		*/
/*    clock timezone	   1993-Oct-10		    Kees J. Bot		*/
/*    set CMOS clock	   1994-Jun-12		    Kees J. Bot		*/
/************************************************************************/


#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <errno.h>
#include <ibm/portio.h>
#include <sys/svrctl.h>

/* Calibration: Readclock will use three values from a line in /etc/timeinfo
 * to adjust a too slow or too fast clock.  The first value is the time the
 * clock was exactly on time, the second is the offset from the base to the
 * time according to the clock and the third is the offset to the correct
 * time at the moment of calibration. The current time is computed as:
 *	time = base + (clock - base) * (offtime / offclock))
 */
char CAL_INFO[] =	"/etc/timeinfo";
int cflag = 0;		/* Time has been set precisely, recalibrate. */
int bflag = 0;		/* Clock has been set precisely, rebase. */
int nflag = 0;		/* Tell what, but don't do it. */
int wflag = 0;		/* Set the CMOS clock. */
int Wflag = 0;		/* Also set the CMOS clock register bits. */
time_t base = 0;	/* Time when clock was precise. */
time_t offclock = 1;	/* Offset value of clock at calibration. */
time_t offtime = 1;	/* Offset value of correct time at calibration. */
enum caldir { READCAL, WRITECAL };

/* Wide multiply and divide. */
typedef struct {
	u32_t	low;
	u32_t	high;
} u64_t;

_PROTOTYPE(u64_t mul64, (u32_t x, u32_t y) );
_PROTOTYPE(u32_t div64, (u64_t m, u32_t x) );

char clocktz[128];	/* Timezone of the clock. */

#define MACH_ID_ADDR	0xFFFFE		/* BIOS Machine ID at FFFF:000E */

#define PC_AT		   0xFC		/* Machine ID byte for PC/AT,
					   PC/XT286, and PS/2 Models 50, 60 */
#define PS_386		   0xF8		/* Machine ID byte for PS/2 Model 80 */

/* Manufacturers usually use the ID value of the IBM model they emulate.
 * However some manufacturers, notably HP and COMPAQ, have had different
 * ideas in the past.
 *
 * Machine ID byte information source:
 *	_The Programmer's PC Sourcebook_ by Thom Hogan,
 *	published by Microsoft Press
 */

#define CLK_ELE		0x70	/* CMOS RAM address register port (write only)
				 * Bit 7 = 1  NMI disable
				 *	   0  NMI enable
				 * Bits 6-0 = RAM address
				 */

#define CLK_IO		0x71	/* CMOS RAM data register port (read/write) */

#define  YEAR             9	/* Clock register addresses in CMOS RAM	*/
#define  MONTH            8
#define  DAY              7
#define  HOUR             4
#define  MINUTE           2
#define  SECOND           0
#define  REGA          0x0A	/* Status register A: RTC configuration	*/
#define  REGB          0x0B	/* Status register B: RTC configuration	*/
#define  HEALTH	       0x0E	/* Diagnostic status: (should be set by Power
				 * On Self-Test [POST])
				 * Bit  7 = RTC lost power
				 *	6 = Checksum (for addr 0x10-0x2d) bad
				 *	5 = Config. Info. bad at POST
				 *	4 = Mem. size error at POST
				 *	3 = I/O board failed initialization
				 *	2 = CMOS time invalid
				 *    1-0 =    reserved
				 */
#define  DIAG_BADBATT       0x80
#define  DIAG_MEMDIRT       0x40
#define  DIAG_BADTIME       0x04

/* CMOS RAM and RTC information source:
 *	_System BIOS for PC/XT/AT Computers and Compatibles_
 *	by Phoenix Technologies Ltd., published by Addison-Wesley
 */

_PROTOTYPE(int main, (int argc, char **argv));
_PROTOTYPE(void errmsg, (char *s));
_PROTOTYPE(void get_time, (struct tm *t));
_PROTOTYPE(int read_register, (int reg_addr));
_PROTOTYPE(void set_time, (struct tm *t));
_PROTOTYPE(void write_register, (int reg_addr, int value));
_PROTOTYPE(int bcd_to_dec, (int n));
_PROTOTYPE(int dec_to_bcd, (int n));
_PROTOTYPE(void usage, (void));
_PROTOTYPE(int templ_eq, (int c1, int c2));
_PROTOTYPE(void rw_cal, (enum caldir));

int main(argc, argv)
int argc; char **argv;
{
  struct stat pdev;
  struct tm time1;
  struct tm time2;
  struct tm tmnow;
  time_t ct, now;
  char time_buf[32], *ascii_time;
  int i, j, mem;
  unsigned char mach_id, cmos_state;

  /* Request I/O privileges. */
  if (svrctl(SYSIOPRIV, (void *) NULL) < 0) {
	errmsg( "Can't obtain I/O privileges." );
	exit(1);
  }

  /* Open /dev/mem to get access to physical memory. */
  if ((mem = open("/dev/mem", O_RDONLY)) == -1) {
	errmsg( "Permission denied." );
	exit(1);
  }
  if (lseek(mem, (off_t) MACH_ID_ADDR, SEEK_SET) == -1
		|| read(mem, (void *) &mach_id, sizeof(mach_id)) < 0) {
	mach_id = -1;
  }
  if (mach_id != PS_386 && mach_id != PC_AT) {
	errmsg( "Machine ID unknown." );
	fprintf( stderr, "Machine ID byte = %02x\n", mach_id );

	exit(1);
  }
  cmos_state = read_register(HEALTH);
  if (cmos_state & (DIAG_BADBATT | DIAG_MEMDIRT | DIAG_BADTIME)) {
	errmsg( "CMOS RAM error(s) found..." );
	fprintf( stderr, "CMOS state = 0x%02x\n", cmos_state );

	if (cmos_state & DIAG_BADBATT)
	    errmsg( "RTC lost power. Reset CMOS RAM with SETUP." );
	if (cmos_state & DIAG_MEMDIRT)
	    errmsg( "CMOS RAM checksum is bad. Run SETUP." );
	if (cmos_state & DIAG_BADTIME)
	    errmsg( "Time invalid in CMOS RAM. Reset clock." );
	exit(1);
  }

  /* Process options. */
  while (argc > 1) {
	char *p = *++argv;

	if (*p++ != '-') usage();

	while (*p != 0) {
		switch (*p++) {
		case 'c':	cflag = 1;	break;
		case 'b':	bflag = 1;	break;
		case 'n':	nflag = 1;	break;
		case 'w':	wflag = 1;	break;
		case 'W':	Wflag = 1;	break;
		default:	usage();
		}
	}
	argc--;
  }
  if (Wflag) wflag = 1;		/* -W implies -w */

  if (cflag && wflag) {
	fprintf(stderr,
		"readclock: flags -w and -c do not make sense together.\n");
	exit(1);
  }

  /* The hardware clock may run in a different time zone, likely GMT or
   * winter time.  Select that time zone.
   */
  strcpy(clocktz, "TZ=");
  if (sysenv("TZ", clocktz+3, sizeof(clocktz)-3) >= 0) {
	putenv(clocktz);
	tzset();
  }

  /* Read the CMOS clock. */
  for (i = 0; i < 10; sleep(10), i++) {
	for (j = 0; j < 10; j++) {
		get_time(&time1);
		get_time(&time2);

		if (time1.tm_year == time2.tm_year &&
		    time1.tm_mon == time2.tm_mon &&
		    time1.tm_mday == time2.tm_mday &&
		    time1.tm_hour == time2.tm_hour &&
		    time1.tm_min == time2.tm_min &&
		    time1.tm_sec == time2.tm_sec
		) break;
	}
	if (j == 10) {
		errmsg( "Failed to get an accurate time." );
		continue;
	}
	time(&now);

	time1.tm_isdst= -1;	/* Do timezone calculations. */

	ct= mktime(&time1);	/* Transform to a time_t. */
	if (ct != -1) break;

	fprintf(stderr,
	"readclock: Invalid time read from CMOS: %d-%02d-%02d %02d:%02d:%02d\n",
		time2.tm_year+1900, time2.tm_mon+1, time2.tm_mday,
		time2.tm_hour, time2.tm_min, time2.tm_sec);
  }
  if (i == 10) exit(1);

  rw_cal(READCAL);

  if (bflag) {
	/* The clock is now precise, set the base. */
	base = now;
  }
  if (cflag) {
	/* The time is now precise, set the calibration times. */
	if (now - base <= 12 * 3600L) {
		/* A bit soon for a calibrate, use safe values. */
		offtime = offclock = 1;
		if (!bflag) {
			printf(
		  "To soon for a good calibrate, do it again in a few days.\n");
		}
	} else {
		offclock = ct - base;
		offtime = now - base;
	}
  }
  if (bflag || cflag) {
	rw_cal(WRITECAL);
  }
  if (!wflag && !bflag && !cflag) {
	/* Set system time to the (calibrated) CMOS clock time. */
	now = base + div64(mul64(ct - base, offtime), offclock);

	if (nflag) {
		printf("stime(%lu)\n", (unsigned long) now);
	} else {
		if (stime(&now) < 0) {
			errmsg( "Not allowed to set time." );
			exit(1);
		}
	}
  }
  if (wflag) {
	/* Set the CMOS clock to the system time. */
	if (nflag) {
		tmnow = *localtime(&now);
		printf("%04d-%02d-%02d %02d:%02d:%02d\n",
			tmnow.tm_year + 1900,
			tmnow.tm_mon + 1,
			tmnow.tm_mday,
			tmnow.tm_hour,
			tmnow.tm_min,
			tmnow.tm_sec);
	} else {
		struct timeval tvnow;

		/* Position half a second after the time we want to set,
		 * because the hardware clock counts only half a second on
		 * the first tick.
		 */
		now++;
		tmnow = *localtime(&now);
		tvnow.tv_sec= now;
		tvnow.tv_usec= 500000L;
		u_sleep(&tvnow);

		set_time(&tmnow);
	}
  }
  exit(0);
}

void errmsg(s) char *s;
{
  static char *prompt = "readclock: ";

  fprintf(stderr, "%s%s\n", prompt, s);
  prompt = "";
}


/***********************************************************************/
/*                                                                     */
/*    get_time( time )                                                 */
/*                                                                     */
/*    Update the structure pointed to by time with the current time    */
/*    as read from CMOS RAM of the RTC.				       */
/*    If necessary, the time is converted into a binary format before  */
/*    being stored in the structure.                                   */
/*                                                                     */
/***********************************************************************/

void get_time(t)
struct tm *t;
{
  t->tm_year = read_register(YEAR);
  t->tm_mon = read_register(MONTH);
  t->tm_mday = read_register(DAY);
  t->tm_hour = read_register(HOUR);
  t->tm_min = read_register(MINUTE);
  t->tm_sec = read_register(SECOND);

  if ((read_register(REGB) & 0x04) == 0) {
	/* Convert BCD to binary (default RTC mode) */
	t->tm_year = bcd_to_dec(t->tm_year);
	t->tm_mon = bcd_to_dec(t->tm_mon);
	t->tm_mday = bcd_to_dec(t->tm_mday);
	t->tm_hour = bcd_to_dec(t->tm_hour);
	t->tm_min = bcd_to_dec(t->tm_min);
	t->tm_sec = bcd_to_dec(t->tm_sec);
  }
  t->tm_mon--;	/* Counts from 0. */

  /* Correct the year, good until 2080. */
  if (t->tm_year < 80) t->tm_year += 100;
}


int read_register(reg_addr)
int reg_addr;
{
  outb(CLK_ELE, reg_addr);
  return inb(CLK_IO);
}



/***********************************************************************/
/*                                                                     */
/*    set_time( time )                                                 */
/*                                                                     */
/*    Set the CMOS RTC to the time found in the structure.             */
/*                                                                     */
/***********************************************************************/

void set_time(t)
struct tm *t;
{
  int regB;

  if (Wflag) {
	/* Set A and B registers to their proper values according to the AT
	 * reference manual.  (For if it gets messed up, but the BIOS doesn't
	 * repair it.)
	 */
	write_register(REGA, 0x26);
	write_register(REGB, 0x02);
  }

  /* Stop the CMOS clock. */
  regB= read_register(REGB);
  write_register(REGB, regB | 0x80);

  t->tm_mon++;	/* Counts from 1. */

  if ((regB & 0x04) == 0) {
	/* Convert binary to BCD (default RTC mode) */
	t->tm_year = dec_to_bcd(t->tm_year % 100);
	t->tm_mon = dec_to_bcd(t->tm_mon);
	t->tm_mday = dec_to_bcd(t->tm_mday);
	t->tm_hour = dec_to_bcd(t->tm_hour);
	t->tm_min = dec_to_bcd(t->tm_min);
	t->tm_sec = dec_to_bcd(t->tm_sec);
  }
  write_register(YEAR, t->tm_year);
  write_register(MONTH, t->tm_mon);
  write_register(DAY, t->tm_mday);
  write_register(HOUR, t->tm_hour);
  write_register(MINUTE, t->tm_min);
  write_register(SECOND, t->tm_sec);

  /* Restart the clock. */
  write_register(REGB, regB & ~0x80);
}


void write_register(reg_addr, value)
int reg_addr;
int value;
{
  outb(CLK_ELE, reg_addr);
  outb(CLK_IO, value);
}

int bcd_to_dec(n)
int n;
{
  return ((n >> 4) & 0x0F) * 10 + (n & 0x0F);
}

int dec_to_bcd(n)
int n;
{
  return ((n / 10) << 4) | (n % 10);
}

char CAL_TEMPLATE[] =	"#*** 0000000000 0000000000 0000000000\n";

void usage()
{
  fprintf(stderr, "Usage: readclock [-bcnwW]\n");
  exit(1);
}

#define isdigit(c)	((unsigned) ((c) - '0') < 10)

int templ_eq(c1, c2)
int c1, c2;
{
	return c1 == c2 || (isdigit(c1) && isdigit(c2));
}

void rw_cal(dir)
enum caldir dir;
{
  off_t pos = 0;
  FILE *cf;
  int c, i = 0;

  if ((cf = fopen(CAL_INFO, dir == READCAL ? "r" : "r+")) == NULL) {
	if (dir == READCAL && errno == ENOENT) return;
	errmsg("can't open calibration info file");
	exit(1);
  }

  while ((c = getc(cf)) != EOF) {
	if (templ_eq(c, CAL_TEMPLATE[i])) {
		CAL_TEMPLATE[i++] = c;
		if (c == '\n') break;
	} else {
		pos += i+1;
		i = 0;
	}
  }
  if (c == EOF) { pos += i; i = 0; }

  if (dir == READCAL && i > 0) {
	base = atol(CAL_TEMPLATE + 5);
	offclock = atol(CAL_TEMPLATE + 16);
	offtime = atol(CAL_TEMPLATE + 27);
  }
  if (dir == WRITECAL && !nflag) {
	fseek(cf, pos, SEEK_SET);
	if (i == 0) fprintf(cf, "\n# Clock calibration info.\n");
	fprintf(cf, "#*** %010lu %010lu %010lu\n",
		(u32_t) base, (u32_t) offclock, (u32_t) offtime);
  }
  fclose(cf);
  if (bflag || cflag) {
  	u32_t factor;

	factor = div64(mul64(offtime, 1000000000L), offclock);

	printf("%c  %10lu %10lu %10lu  (%lu.%09lu)\n",
		dir == READCAL ? '<' : '>',
		(u32_t) base, (u32_t) offclock, (u32_t) offtime,
		factor / 1000000000L, factor % 1000000000L);
  }
}

u64_t mul64(x, y) u32_t x, y;
/* Multiply two 32 bit numbers to form a 64 bit number.  (There are no doubt
 * very efficient ways to do this, but efficiency is not called for.)
 */
{
	u32_t x1 = x >> 16;
	u32_t x0 = x & 0xFFFF;		/* x == x1<<16 + x0 */
	u32_t y1 = y >> 16;
	u32_t y0 = y & 0xFFFF;		/* y == y1<<16 + y0 */
	u32_t x1y0, x0y1, middle;
	u64_t m;

	/* To compute: m = x1*y1<<32 + x1*y0<<16 + x0*y1<<16 + x0*y0 */
	m.low = x0 * y0;
	m.high = x1 * y1;
	x1y0 = x1 * y0;
	x0y1 = x0 * y1;

	m.high += (x1y0 >> 16) + (x0y1 >> 16);
	x1y0 &= 0xFFFF;
	x0y1 &= 0xFFFF;

	middle = x1y0 + x0y1 + (m.low >> 16);
	m.high += middle >> 16;
	m.low = (m.low & 0xFFFF) + (middle << 16);

	return m;
}

u32_t div64(m, x) u64_t m; u32_t x;
/* Divide a 64 bit number by a 32 bit number, no overflow expected.  (Like
 * above, no efficiency to be seen.)
 */
{
	u32_t q, b;
	u64_t t;

	/* Set each of the 32 bits in q and check if it's a good idea. */
	q = 0;
	for (b = 0x80000000; b != 0; b >>= 1) {
		q |= b;
		t = mul64(x, q);

		/* If t > m then bit b should not be set in q. */
		if (t.high > m.high || (t.high == m.high && t.low > m.low))
			q &= ~b;
	}
	return q;
}
