/*
 * W-NEWS	A simple NEWS processing package for MINIX.
 *		Unbatch is used by the NEWS system to unbatch the news
 *		batches that were prepared and queued by the 'rnews'
 *		program.  Its task is to (optionally) uncompress a batch,
 *		and then to unpack all articles from the news batch, and
 *		to put those articles in the article spool directories.
 *
 * Usage:	unbatch [-d] [ batch_file ]
 *
 * Version:	3.00	03/30/91
 *
 * Author:	Fred N. van Kempen, MicroWalt Corporation
 */
#include <sys/types.h>
#include <sys/wait.h>
#include <dirent.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <stdio.h>
#include "wnews.h"


static char *Version = "@(#) unbatch 3.00 (03/30/91)";


char *lgmonths[] = {			/* for debugging purposes	*/
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};
int debug = 0;				/* debugging flag		*/


extern int getopt(), optind, opterr;	/* from getopt(3) package	*/
extern char *optarg;


/* Write something to the log files. */
void logmsg(err, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int err;
char *fmt;
{
  char buff[1024];
  FILE *lfp, *efp;
  char lfn[128], efn[128];
  struct tm *mydate;
  time_t now;

  sprintf(lfn, "%s/%s", LIBDIR, LOGFILE);
  sprintf(efn, "%s/%s", LIBDIR, ERRLOG);

  if ((efp = fopen(efn, "a")) == (FILE *)NULL) {
	fprintf(stderr, "unbatch: cannot append(%s)\n", efn);
	return;
  }

  if ((lfp = fopen(lfn, "a")) == (FILE *)NULL) {
	fprintf(efp, "unbatch: cannot append(%s)\n", lfn);
	(void) fclose(efp);
	return;
  }

  time(&now);
  mydate = localtime(&now);
  sprintf(buff, "%s %02.2d %02.2d:%02.2d  ", lgmonths[mydate->tm_mon],
			mydate->tm_mday, mydate->tm_hour, mydate->tm_min);
  if (err == 1) {
	fprintf(efp, buff);
	fprintf(efp, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  } else {
	fprintf(lfp, buff);
	fprintf(lfp, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  }
  if (debug > 0) {
	fprintf(stderr, buff);
	fprintf(stderr, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  }
  (void) fclose(lfp);
  (void) fclose(efp);
}


/*
 * Unpack the current batch of articles contained in 'batch'.
 * A batch consists of a header, followed by the actual articles
 * in a format like:
 *
 *	#! rnews 1234\n
 *	article-text
 *	#! rnews 5523\n
 *	article-text
 *	...
 *
 * The number following the 'rnews' text is the size of the message
 * in bytes.
 *
 * Optionally, we may have a compressed batch file, which is essentially
 * a normal batch (see above), but now routed through the compress(1)
 * program, and prepended with a header like:
 *
 *	#! cunbatch\n
 *
 * So, we have to perform the following actions:
 *
 *	1) (optionally) decompress a compressed batch.
 *	2) Separate the messages.
 *	3) Write the separated messages into their dirs by calling the
 *         'inews' program for local posting.
 */
int unbatch(batch)
char *batch;
{
  char buff[1024], fname[128];
  char cmd[128], artnam[128];
  char *args[8];
  int pid, status;
  int pfds[2], bfd, fd;
  register FILE *fp, *ap;
  char *bp, *sp, endofbatch;

  logmsg(0, "unbatch: processing %s\n", batch);

  strcpy(fname, "/tmp/unb.XXXXXX");
  strcpy(artnam, "/tmp/unbart.XXXXXX");
  mktemp(fname); mktemp(artnam);

  /* First of all, open the batch file. */
  if ((fp = fopen(batch, "r")) == (FILE *)NULL) {
	logmsg(1, "unbatch: cannot open(%s)\n", batch);
	return(-1);
  }

  /* Now, check the type of this batch. */
  if (fgets(buff, 128, fp) == (char *)NULL) {
	logmsg(1, "unbatch: cannot read batch header\n");
	(void) fclose(fp);
	return(-1);
  }

  /* Is this a compressed batch? */
  if (strncmp(buff, COMPRHDR, strlen(COMPRHDR)) == 0) {
	/* Yes. Re-open the batch in file descriptor mode. */
	(void) fclose(fp);
	if ((fd = open(batch, O_RDONLY)) < 0) {
		logmsg(1, "unbatch: cannot re-open(%s)\n", batch);
		return(-1);
	}
	/* Create a temp file for decompression. */
	(void) unlink(fname);
	if ((bfd = creat(fname, 0600)) < 0) {
		logmsg(1, "unbatch: cannot create(%s)\n", fname);
		(void) close(fd);
		return(-1);
	}
	/* Skip the 'compressed' mark header. */
	if (read(fd, buff, strlen(COMPRHDR)) != strlen(COMPRHDR)) {
		logmsg(1, "unbatch: cannot skip compress header\n");
		(void) close(fd);
		(void) close(bfd);
		(void) unlink(fname);
		return(-1);
	}
	/* Create a pipeline and decompress the temp file. */
	pid = 0;
	args[pid++] = COMPRESS;
	args[pid++] = "-d";
	args[pid] = (char *)NULL;

	if ((pid = fork()) == 0) {
		(void) close(0);
		(void) close(1);
		(void) dup(fd);
		(void) dup(bfd);
		if (debug == 0) {
			(void) close(2);
			(void) open("/dev/null", O_RDWR);
		}
		status = execv(args[0], args);
		exit(status);
	} else {
		status = 0;
		if (pid > 0) while(wait(&status) != pid)
					;
		(void) close(fd);
		(void) close(bfd);

		if (pid < 0 || WEXITSTATUS(status) != 0) {
			logmsg(1, "unbatch: cannot EXEC %s\n", args[0]);
			(void) unlink(fname);
			return(-1);
		}
	}

	/* We now have a decompressed file in /tmp. Open it! */
	if ((fp = fopen(fname, "r")) == (FILE *)NULL) {
		logmsg(1, "unbatch: cannot open(%s)\n", fname);
		(void) unlink(fname);
		return(-1);
	}
  } else rewind(fp);

  /* Build a calling sequence for 'inews'. */
  sprintf(cmd, "%s/inews", LIBDIR);
  pid = 0;
  args[pid++] = cmd;
  if (debug == 1) args[pid++] = "-v";
  args[pid++] = "-p";
  args[pid++] = artnam;
  args[pid] = (char *)NULL;
  endofbatch = -1;

  /*
   * We are now ready to post articles.
   * If this is a batch of files, then the first line will start
   * with the "!# rnews" header (RNEWSHDR) text.  Otherwise, the
   * text might be a single article.  Start reading the text and
   * post the generated articles by calling inews(8).
   */
  if ((bp = fgets(buff, 80, fp)) != (char *)NULL) do {
	if (!strncmp(buff, RNEWSHDR, strlen(RNEWSHDR)) || endofbatch == -1) {
		endofbatch = 0;
		ap = fopen(artnam, "w");
		if (ap == (FILE *)NULL) {
			logmsg(1, "unbatch: cannot create(%s)\n", artnam);
			(void) fclose(fp);
			(void) unlink(fname);
			return(-1);
		}
		while(1) {
			bp = fgets(buff, 1024, fp);
			if (bp == (char *)NULL) {
				(void) fclose(fp);
				endofbatch = 1;
				break;
			}
			if (!strncmp(buff, RNEWSHDR, strlen(RNEWSHDR)))
								break;
			(void) fprintf(ap, "%s", buff);
		}
		(void) fprintf(ap, "\n");
		(void) fflush(ap);
		(void) fclose(ap);

		if ((pid = fork()) == 0) {
			status = execv(cmd, args);
			exit(status);
		} else {
			status = 0;
			if (pid > 0) while(wait(&status) != pid)
					;
		}
		(void) unlink(artnam);
	}
  } while(endofbatch == 0);

  /* Close batch file and remove (decompressed temporary) files. */
  (void) fclose(fp);
  (void) unlink(fname);
  (void) unlink(artnam);

  return(0);
}


void usage()
{
  fprintf(stderr, "Usage: unbatch [-d] [ batch_file ]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char temp[128], buff[128];
  char unblock[128];
  struct dirent *de;
  DIR *dp;
  int st;

  opterr = 0;
  while ((st = getopt(argc, argv, "d")) != EOF) switch(st) {
	case 'd':
		debug = 1;
		break;
	default:
		usage();
  }

  (void) umask(033);			/* set protection to rw-r--r-- */
  signal(SIGPIPE, SIG_IGN);

  /* Check if we are already (or still?) running. */
  sprintf(unblock, "%s/%s", LIBDIR, UNBLOCK);
  if (access(unblock, 0) == 0) exit(0);

  /* No, we are not yet busy. */
  (void) creat(unblock, 0);

  /* At most one more argument allowed. */
  if (optind == (argc - 1)) {
	st = unbatch(argv[optind]);
	if (st == 0) (void) unlink(argv[optind]);
  } else if (optind != argc) usage();

  (void) chdir(SPOOLDIR);
  if ((dp = opendir(INCOMING)) == (DIR *)NULL) {
	logmsg(1, "unbatch: cannot opendir(%s/%s)\n", SPOOLDIR, INCOMING);
	(void) unlink(unblock);
	exit(-1);
  }

  (void) readdir(dp);
  (void) readdir(dp);
  while ((de = readdir(dp)) != (struct dirent *)NULL) {
	sprintf(buff, "%s/%s/%s", SPOOLDIR, INCOMING, de->d_name);
	if ((st = unbatch(buff)) != 0) {
		sprintf(temp, "%s/%s", SPOOLDIR, de->d_name);
		logmsg(1, "unbatch: inbound news garbled; saved in %s\n",
									temp);
		(void) link(buff, temp);
	}
	(void) unlink(buff);
  }
  (void) closedir(dp);

  /* Remove the lockfile. */
  (void) unlink(unblock);

  exit(0);
}
