/*
 * W-NEWS	A simple NEWS processing package for MINIX.
 *		This program takes care of removing (and/or archiving)
 *		articles which have expired.  It also updates the counts
 *		in the "active" file.
 *
 * Usage:	expire [-d] [-n groups] -a [groups] [-e days] [-v level]
 *		[-f user] [-p] [-t archiver]
 *
 * Notes:	-d = -v3 and "do not act"
 *		-t archiver %f %a %s ...
 *	        as in "-t lharc a %f %a [ --s %s ]"
 *
 * Version:	3.00	03/30/91
 *
 * Author:	Miquel van Smoorenburg, miquels@drinkel.nl.mugnet.org
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <blocksize.h>
#include <ctype.h>
#include <dirent.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <time.h>
#include <stdio.h>
#include "wnews.h"


#define OLDNEWS "/usr/spool/oldnews"

#define EXPIRE  1
#define ARCHIVE 2
#define MAYPOST 4


struct group {
  struct group *next;
  char status;
  int first;
  int last;
  char group[1];
};

struct article {
  int stat_days;
  int art_days;
  char seq[8];
  char from[128];
  char subject[128];
};


static char *Version = "@(#) expire 3.00 (03/30/91)";


char *months[] = {			/* for date parser 		*/
  "", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec", (char *)NULL
};
int debug = 0;				/* debugging flag		*/
int errflg = 0;				/* an error has occurred	*/
int expiretime = 15;			/* expiration interval		*/
int usepostdate = 0;			/* use Date: field in article	*/
int verbose = 2;			/* be verbose about it all	*/
int needsubject = 0;			/* we need a Subject: def.	*/
char *archv[16];			/* archiver definition		*/
int archc = 0;				/* will we use an archiver?	*/
char *usefrom = (char *)NULL;		/* look at the From: line ?	*/
int tty_output = 0;			/* If this is NOT run from cron */
struct group groups;			/* the in-core "active" file	*/
struct group define;
struct group archdef;


extern int getopt(), optind, opterr;	/* from the getopt(3) package	*/
extern char *optarg;
extern long atol();			/* should be in stdlib.h	*/

/* Write something to the log files. */
void logmsg(err, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int err;
char *fmt;
{
  char buff[32];
  FILE *lfp, *efp;
  char lfn[128], efn[128];
  
  struct tm *mydate;
  time_t now;

  sprintf(lfn, "%s/%s", LIBDIR, LOGFILE);
  sprintf(efn, "%s/%s", LIBDIR, ERRLOG);

  if ((efp = fopen(efn, "a")) == (FILE *)NULL) {
	fprintf(stderr, "inews: cannot open %s\n", efn);
	return;
  }

  if ((lfp = fopen(lfn, "a")) == (FILE *)NULL) {
	fprintf(efp, "inews: cannot open logfile %s\n", lfn);
	(void) fclose(efp);
	return;
  }

  time(&now);
  mydate = localtime(&now);
  sprintf(buff, "%s %02.2d %02.2d:%02.2d  ",
	months[mydate->tm_mon + 1], mydate->tm_mday,
			mydate->tm_hour, mydate->tm_min);
  if (err == 1) {
	fprintf(efp, buff);
	fprintf(efp, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  } else {
	fprintf(lfp, buff);
	fprintf(lfp, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  }
  if (debug > 0 || tty_output)
	fprintf(stderr, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
  (void) fclose(lfp);
  (void) fclose(efp);
}


/* Convert newsgroup to directory. */
char *gtof(g)
char *g;
{
  static char dir[128];
  char gr[128];
  char *f;
  
  strcpy(gr, g);
  for(f = gr; *f; f++)
	*f = *f == '.' ? '/' : *f;
  sprintf(dir, "%s/%s", SPOOLDIR, gr);
  return(dir);
}	


/* Return next string in a line. */
char *nextstring(linep, last, message)
char **linep;
int last;
char *message;
{
  char *s;
  
  s = *linep;
  while(**linep && **linep != ' ' && **linep != '\t' && **linep != '\n')
								(*linep)++;
  if (**linep == '\n') **linep = '\0';	
  if (**linep == '\0' && !last) {
	if (message) logmsg(1, message);
	return("");
  }
  *(*linep)++ = '\0';
  for (; **linep == ' ' || **linep == '\t'; *(*linep)++)
		    ;
  for (; *s == ' ' || *s == '\t'; s++)
		    ;
  return(s);
}


/* Read all newsgroups in /usr/lib/news/active into structures. */
int act_rd() 
{
  char active[128];
  char line[128];
  char *linep;
  char *s;
  FILE *fp;
  struct group *sp;
  struct group *prev;
  char *warn = "expire: ACTIVE file format error\n";
  
  sprintf(active, "%s/%s", LIBDIR, ACTIVE);
  if ((fp = fopen(active, "r")) == (FILE *)NULL) {
	logmsg(1, "expire: cannot open (%s)\n", active);
	return(1);
  }
  prev = &groups;
  while(fgets(line, 128, fp) != (char *)NULL) {
	if (debug) fprintf(stderr, "expire: act_rd: reading %s", line);
	linep = line;
	if (*(s = nextstring(&linep, 0, warn)) == '\0') return(1);
	sp = (struct group *)malloc(sizeof(struct group) + strlen(s));
	if (sp == (struct group *)NULL) {
		logmsg(1, "expire: out of memory - too many newsgroups.\n");
		return(1);
	}
	strcpy(sp->group, s);
	if (*(s = nextstring(&linep, 0, warn)) == '\0') return(1);
	sp->last = (unsigned int) atol(s);
	if (*(s = nextstring(&linep, 0, warn)) == '\0') return(1);
	sp->first = (unsigned int) atol(s);
	if (*nextstring(&linep, 1, (char *)NULL) == 'n') sp->status = 0;
	  else sp->status = MAYPOST;
	sp->next = (struct group *)NULL;
	prev->next = sp;
	prev = sp;
  }
  (void) fclose(fp);
  return(0);
}
  

/*
 * Extension to getopt to give more than one argument to an option.
 * Arguments may be separated by comma's.
 */
char *nextarg(argc, argv)
int argc;
char **argv;
{
  static char *nxtarg;
  static char *nxt = (char *)NULL;
  
  if (nxt == (char *)NULL) {
	nxtarg = (char *)NULL;
	if (argv[optind] != (char *)NULL)
		if (argv[optind][0] != '-') nxtarg = argv[optind++];
		  else if (argv[optind][1] == '-')
				nxtarg = 1 + argv[optind++];	
  } else nxtarg = nxt;

  if ((nxtarg != (char *)NULL) &&
      (nxt = strchr(nxtarg, ',')) != (char *)NULL)
	*nxt++ = '\0';
  return(nxtarg);
}


/* Make a list of arguments for the -t option. */
void opt_t(argc, argv)
{
  static char buf[256];
  int fileseen = 0;
  int artseen = 0;
  char *s;
  char *bufp = buf;
  int f;
  
  for(f = 0; f < 15; f++) {
	if ((s = nextarg(argc, argv)) == (char *)NULL) break;
	if (strcmp(s, "%f") == 0) fileseen++;
	if (strcmp(s, "%a") == 0) artseen++;
	if (strcmp(s, "%s") == 0) needsubject = 1;	
	archv[f] = bufp;
	strcpy(bufp, s);
	bufp += (1 + strlen(s));
	archc++;
  }
  archv[f] = (char *)NULL;

  if (f == 0 || fileseen == 0 || artseen == 0) {
	errflg++;
	if (tty_output) fprintf(stderr, "expire: -t: wrong format\n");
  } else if (debug) {
	fprintf(stderr, "expire: archiver: ");

	for (f = 0; f < archc; f++)
		fprintf(stderr, "%s ", archv[f]);
	fprintf(stderr, "\n");
  }
}
 

/* Make a linked list of groups given by the -n and -a options. */
int initgroups(argc, argv, gp, option)
int argc;
char **argv;
struct group *gp;
char option;
{
  char *arg;
  struct group *m;
  struct group *prev;
  int needarg = 0;
  
  if (option == 'n') needarg = 1;
  prev = gp;

  while(arg = nextarg(argc, argv)) {
	if (debug) fprintf(stderr, "expire: option -%c: processing %s\n",
								option, arg);
	m = (struct group *)malloc(sizeof(struct group) + strlen(arg));
	if (m == (struct group *)NULL) {
		logmsg(1, "expire: initgroups: out of memory\n");
		return(1);
	}
	m->next = (struct group *)NULL;
	strcpy(m->group, arg);
	prev->next = m;
	prev = m;
	needarg = 0;
  }
  if (needarg && arg == (char *)NULL) {
	if (tty_output) fprintf(stderr, "expire: -n argument missing\n");
	return(1);
  } 
  return(0);
}


/*
 * Match a pattern using "all" as "*"
 * Returns 0 if a match was found.
 */
int groupcmp(s, p)
register char *s, *p;
{
  register int sc, pc;
  char *t;
  
  if (s == (char *)NULL || p == (char *)NULL)
  	return(1);
  while (*(t = p++) != '\0') {
  	pc = *t;
  	sc = *s++;
  	if (strncmp(t, "all", 3) == 0) {
  		s--;
  		p += 2;
  		do {
  			if (*p == '\0' || !groupcmp(s, p))
  				return(0);
  		} while (*s++ != '\0');
  		return(1);
  	}
  	if (sc != pc)
  		return(1);
  }
  return(*s != '\0');
}


/*
 * We have a linked list of groups starting in wgroup; match
 * these with the list starting in wpat (from the -a/-n option).
 */
void select(wgroup, wpat, status, caller)
struct group *wgroup;
struct group *wpat;
char status;
char *caller;
{
  struct group *g, *d;
  int keep, delete;
  
  for(g = wgroup->next; g; g = g->next) {
	keep = 0;
	delete = 0;
	if (wpat->next == (struct group *)NULL) keep = 1;
	for(d = wpat->next; d; d = d->next) {
		/* !group means, skip this group and subgroups */
		if (d->group[0] == '!' && !groupcmp(g->group, d->group + 1))
								delete = 1;
		/* Just group means keep group and its subgroups. */
		if (!groupcmp(g->group, d->group)) keep = 1;
	}
	/* Delete has precedence over keep. */
	if (!delete && keep) g->status |= status;
  }
  if (debug) {
	fprintf(stderr, "expire: %s: Going to act on groups:\n", caller);
	keep = 0;
	if (wgroup->next) {
		for(g = wgroup->next; g; g = g->next)
			if((g->status & EXPIRE) && (g->status & status)) {
				keep = 1;
				fprintf(stderr, "%s\n", g->group);
			}	
	}
	if (!keep) fprintf(stderr, "NO GROUPS TO ACT ON!\n");
  }
}


/* See if string is in table. */
int match(s, t)
char *s;
char **t;
{
  int f;
  for(f = 0; t[f]; f++)
	if (strncmp(t[f], s, 3) == 0) return(f);
  return(0);
}


/*
 * Turn date string into time.
 * There are 3 formats: 8 Dec 90 00:00... 
 *			Sat, 8 Dec 90 00:00..
 *			Sat Jan 12 00:00:00 1991
 * Include thourough checking to be sure no wrong date is calculated!
 */
int parsedate(s, t)
char *s;
time_t t;
{
  char *p;
  struct tm tm;
  time_t tim;
  int day, year, mon;
  
  if (*(p = nextstring(&s, 1, (char *)NULL)) == '\0') return(0);
	
  if (isdigit(*p)) {					/* First format */
	day = atoi(p);
	mon = match(nextstring(&s, 0, (char *)NULL), months);
	year = atoi(nextstring(&s, 0, (char *)NULL));
  } else
  if (isdigit(*(p = nextstring(&s, 0, (char *)NULL)))) { /* Second format */
	day = atoi(p);
	mon = match(nextstring(&s, 0, (char *)NULL), months);
	year = atoi(nextstring(&s, 0, (char *)NULL));
  } else {						/* Third format */
	mon = match(p, months);
	day = atoi(nextstring(&s, 0, (char *)NULL));
	nextstring(&s, 0, (char *)NULL);
	year = atoi(nextstring(&s, 0, (char *)NULL));
  }
  if (day == 0 || mon == 0 || year == 0) return(0);
	
  if (year > 1900) year -= 1900;	
  tm.tm_sec = 0;
  tm.tm_min = 0;
  tm.tm_hour = 0;
  tm.tm_mday = day;
  tm.tm_mon = mon - 1;
  tm.tm_year = year;
  tim = mktime(&tm);
  if (tim < 0) return(0);
  return((t - tim) / 86400);
}
  

/* Examine article, fill in stucture and maybe open it to read data. */
void examine(g, art, file, st, tim)
struct group *g;
struct article *art;
char *file;
struct stat *st;
time_t tim;
{
  char line[256];
  char dat[128];
  char *date = (char *)NULL;
  register char *s;
  register char *t;
  FILE *fp;
  int li = 0;
  
  art->stat_days = (tim - st->st_mtime) / 86400;
  art->art_days = art->stat_days;
  strcpy(art->subject, "(not read)");
  strcpy(art->from, "(not read)");
  strcpy(art->seq, file);
  
  if (!(needsubject && (g->status & ARCHIVE)) &&
      usefrom == (char *)NULL && usepostdate == 0) return;

  if ((fp = fopen(file, "r")) == (FILE *)NULL) {
	logmsg(1, "expire: cannot open(%s) in newsgroup %s", file, g->group);
	return;
  }
  
  while(li++ < 20 && fgets(line, 256, fp) != (char *)NULL) {
	if (*line == '\n') break;
	line[127] = '\0';
	if (strncmp(line, "From:", 5) == 0) {
		t = line + 6;
		s = art->from;
		while (*t && *t != ' ' && *t != '\t' && *t != '\n')
							*s++ = *t++;
		*s = '\0';
	}
	if (strncmp(line, "Subject:", 8) == 0) {
		strcpy(art->subject, line + 9);
		for (s = art->subject; *s; s++)
			if (*s == '\n') *s = '\0';
	}
	if (strncmp(line, "Date:", 5) == 0) {
		date = dat;
		strcpy(dat, line + 6);
	}
  }
  (void) fclose(fp);
  if (date) {
	strcpy(line, date);
	if ((art->art_days = parsedate(date, tim)) == 0) {
		logmsg(1, "expire: article %s: cannot parse date %s",
							file, line);
		art->art_days = art->stat_days;
	}
  }	
}


/* Store (archive) an article. */
char *store(art, gr)
struct article *art;
struct group *gr;
{
  struct stat st;
  int f, g;
  static char buf1[256];
  char buf2[256];
  char *args[16];
  char *s;
  int status, pid;
  
  if (stat(OLDNEWS, &st) < 0) {
	logmsg(1, "expire: can't archive: no directory %s\n", OLDNEWS);
	return((char *)NULL);
  }

  /* Now build path and make directories as we go. */
  for(f = 0; ; f++) {
	strcpy(buf1, gr->group);
	if (f) for (g = 0; g < f; g++) {
		if ((s = strchr(buf1, '.')) == (char *)NULL) g = f;
		  else *s = '/';
	}
	if ((s = strchr(buf1, '.')) != (char *)NULL) *s = '\0';
	sprintf(buf2, "%s/%s", OLDNEWS, buf1);

	if (stat(buf2, &st) < 0) {
		if (mkdir(buf2, 0755) < 0) {
			logmsg(1, "expire: can't create directory %s\n",
								buf2);
			return((char *)NULL);
		} else {
			if (verbose > 0)
				logmsg(0, "expire: created directory %s\n",
								buf2);
		}
	}
	if (s == (char *)NULL) break;
  }

  /* Allright we have it, now put article in it / update archive. */
  if (archc == 0) {
	sprintf(buf1, "%s/%s", buf2, art->seq);
	if (link(art->seq, buf1) < 0) {
		logmsg(1, "expire: can't link(%s, %s).\n", art->seq, buf1);
		return((char *)NULL);
	}
  } else { /* Use archive program */
	sprintf(buf1, "%s/archive", buf2);
	for(f = 0; f < archc; f++) {
		args[f] = archv[f];
		if (strcmp("%f", args[f]) == 0) args[f] = buf1;
		if (strcmp("%a", args[f]) == 0) args[f] = art->seq;
		if (strcmp("%s", args[f]) == 0) args[f] = art->subject;
	}
	args[f] = (char *)NULL;

	/* Fork and execute. */
	if ((pid = fork()) == 0) {
		execv(args[0], args);
		exit(1);
	} else if (pid > 0) {
		while (wait(&status) != pid)
			    ;
	}

	/* Some archivers leave an "archive.bak" behind. Throw it away. */
	sprintf(buf2, "%s.bak", buf1);
	(void) unlink(buf2);
	
	if (pid < 0 || WEXITSTATUS(status) != 0) {
		logmsg(1, "expire: error in executing %s\n", args[0]);
		return((char *)NULL);
	}
	
  }
  return(buf1);
}


/* Walk through newsgroups. */
void walk() 
{
  struct group *g;
  char *dirname;
  char *s;
  char *arch_name;
  char temp[256];
  DIR *dir;
  struct dirent *entry;
  struct stat st;
  struct article art;
  time_t t;
  int delete;
  int num;
  off_t org_vol, del_vol, tot_del_vol;
  off_t bytes;
  
  (void) time(&t);
  
  tot_del_vol = 0L;
  
  for(g = groups.next; g; g = g->next) {
	if (debug) {
		if (g->status & EXPIRE)
			fprintf(stderr, "expire: examining %s\n", g->group);
		  else
			fprintf(stderr, "expire: skipping %s\n", g->group);
	}
	if (g->status & EXPIRE) {
		dir = opendir(dirname = gtof(g->group));
		if (dir == (DIR *)NULL) {
	    	logmsg(1, "expire: %s does not have a directory!\n",
								g->group);
	 		continue;
	 	}
		(void) chdir(dirname);

		/* Walk through directory */
		org_vol = 0L;
		del_vol = 0L;
		while((entry = readdir(dir)) != (struct dirent *)NULL) {
			delete = 0;
			arch_name = (char *)NULL;
			if (stat(entry->d_name, &st) < 0 || 
			    st.st_mode & S_IFMT != S_IFREG) continue;
			bytes = (1 + (st.st_size / BLOCK_SIZE)) * BLOCK_SIZE;
			org_vol += bytes;
			examine(g, &art, entry->d_name, &st, t);	
			if (usefrom) {
	 			if (strchr(usefrom, '@') == (char *)NULL) {
					s = strchr(art.from, '@');
					if (s == (char *)NULL) 	s = art.from;
					  else s++;
				} else s = art.from;

				if (strcmp(s, usefrom) == 0) {
					delete++;
				}
			} else {
	 			if (usepostdate &&
				    art.art_days >= expiretime) {
	 				delete++;
				} else if (art.stat_days >= expiretime) {
	 				delete++;
				}
			}
	 		if (delete) {
	 			if ((g->status) & ARCHIVE)
	 				if ((arch_name = store(&art, g)) ==
	 					(char *) NULL)
	 						delete = 0;
 				if (delete && !debug) {
 				    if (unlink(entry->d_name) < 0)
 					logmsg(1,
					" WARNING!! %s/%s NOT DELETED\n",
						dirname, entry->d_name);
 				    else {
					del_vol += bytes;
					if (verbose & 2) {
					    if (arch_name)
					    	sprintf(temp, " A %s",
					    	   arch_name);
					    else sprintf(temp, "");
					    logmsg(0,
					      "expire: D %s/%s %s\n",
					      g->group, entry->d_name, temp);
					}      
				    }
 				}
 			}
		}
		/* Report cumutated sizes */
		tot_del_vol += del_vol;
		if (verbose & 1)
			logmsg(0, "expire: %s: volume was %ld and is now %ld\n",
				g->group, org_vol, org_vol - del_vol);
	/* Rescan directory to find lowest & highest article numbers. */
		rewinddir(dir);
		g->first = 0;
		g->last = 0;
		while ((entry = readdir(dir)) != (struct dirent *)NULL) {
 			if (stat(entry->d_name, &st) < 0 || 
		        	st.st_mode & S_IFMT != S_IFREG)
  	   			continue;
			if ((num = atoi(entry->d_name)) > 0) {
				if (num < g->first || g->first == 0)
					g->first = num;
				if (num > g->last || g->last == 0)
					g->last = num;
			}
		}
		(void) closedir(dir);
	}
  }
  if (verbose & 1) logmsg(0, "expire: deleted %ld bytes\n", tot_del_vol);
}


void usage()
{
  fprintf(stderr,
	"Usage: expire [-d] [-n groups] [-a [groups]] [-e days]\n");
  fprintf(stderr,
	"       [-t archiver] [-v level] [-f user] [-p]\n");
  verbose = debug = 0;
  logmsg(1, "expire: argument format error\n");
  exit(lock_active(0));
}


/*
 * Lock or unlock active file.
 * Return 0 on succes.
 */
int lock_active(what)
int what;
{
  char fname[256];
  char lckname[256];
  int try;
  
  /* Create file names */
  sprintf(fname, "%s/%s", LIBDIR, ACTIVE);
  sprintf(lckname, "%s.lock", fname);

  if (what == 0) {
  /* Try to unlock "active". */
  
  	if (unlink(lckname) < 0) {
  		logmsg(1, "expire: cannot unlink (%s)\n", lckname);
  		return(1);
  	}
  	return(0);
  }
  
  /* Try to lock "active".  We allow others one minute. */
  try = 0;
  while (try < 10) {
	if (link(fname, lckname) == 0) break;
	if (++try >= 10) {
		logmsg(1, "expire: cannot lock(%s)\n", fname);
		return(1);
	}
	sleep(6);
  }
  return(0);
}
  

/*
 * Write out new active file
 */
int upd_active()
{
  struct group *g;
  FILE *fp;
  char fname[256];
  
  if (debug)
  	fp = stderr;
  else {
  	sprintf(fname, "%s/%s", LIBDIR, ACTIVE);
  	if ((fp = fopen(fname, "w")) == (FILE *) NULL) {
  		logmsg(1, "expire: cannot open(%s)\n", fname);
  		return(1);
  	}
  }
  for(g = groups.next; g; g = g->next) {
  	fprintf(fp, "%s %05.5d %05.5d %c\n", g->group, g->last, g->first,
  		g->status & MAYPOST ? 'y' : 'n');
  }
  fclose(fp);
  return(0);
}


/*
 * Trap signals (SIGINT and SIGPIPE).
 * Useful for debugging purposes.
 */
void trap()
{
  signal(SIGINT, SIG_IGN);
  signal(SIGPIPE, SIG_IGN);
  
  logmsg(1, "expire: recieved signal.Goodbye!\n");
  exit(lock_active(0));
}


/*
 * Main part of the program.
 * Process options and call appropriate functions.
 */
int main(argc, argv)
int argc;
char **argv;
{
  int c;
  int do_arch = 0;
  
  groups.next  = (struct group *)NULL;
  define.next  = (struct group *)NULL;
  archdef.next = (struct group *)NULL;
  
  signal(SIGINT, trap);
  signal(SIGPIPE, trap);
  
  if (isatty(0)) {
  	verbose = 3;
  	tty_output = 1;
  }
  opterr = 0;
  while((c = getopt(argc, argv, "tdne:av:pf:")) != EOF) {
	switch(c) {
		case 't':  /* specify archiver */
			opt_t(argc, argv);
			break;
		case 'd':  /* debug: only report */
			verbose = 3;
			debug++;
			break;
		case 'n':  /* specify newsgroups */
			errflg += initgroups(argc, argv, &define, 'n');
			break;
		case 'e': /* specify expire time */
			if ((expiretime = atoi(optarg)) == 0) errflg++;
			break;
		case 'a': /* specify groups to arcive */
			do_arch = 1;
			errflg += initgroups(argc, argv, &archdef, 'a');
			break;
		case 'v': /* set verbose mode */
			if ((verbose = atoi(optarg)) == 0 &&
			     optarg[0] != '0') errflg++;
			if (verbose < 0) verbose = 0;
			if (verbose > 3) verbose = 3;     
			break;	
		case 'p': /* use posting date of article */
			usepostdate++;
			break;
		case 'f': /* remove messages send by specific user */
			usefrom = optarg;
			break;
		default:
			errflg++;
			break;
	}
	if (errflg) usage();
  }

  if (lock_active(1))
  	exit(1);

  if (act_rd())
  	exit(lock_active(0));

  select(&groups, &define, EXPIRE, "select groups ");

  if (do_arch)
  	select(&groups, &archdef, ARCHIVE, "archive groups ");
  
  walk();
  
  upd_active();

  exit( lock_active(0) );
}
