/* uuxqt - run remote commands.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

/* general constants */

#define LOCAL		 0
#define REMOTE		 1
#define FALSE		 0
#define TRUE	    ~FALSE
#define OK		 1
#define MAYBE		 0
#define FAILED		-1
#define SAME		 0
#define MAXLOCKS	10
#define MAXNEED		10
#define MAXARGS		30

/* log messages */

#define M_STD		0	/* standard messages */
#define M_NOLOG		1	/* no output to log file */
#define M_ACCESS	2	/* problems in access permissions */
#define M_PARSE		3	/* problems in parsing */

#ifndef lint
static char *Version = "@(#) uuxqt 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_a = FALSE;		/* Notify name, not user, on completion */
int opt_b = FALSE;		/* On error, mail standard input to the user */
int opt_c = TRUE;		/* Do not link or copy source file (default) */
int opt_C = FALSE;		/* Force copy of source file */
int opt_g = FALSE;		/* Set grade of workfile */
int opt_j = FALSE;		/* Output UUCP job number on stderr */
int opt_m = FALSE;		/* Write command status to file */
int opt_n = FALSE;		/* Do not notify user of command status */
int opt_p = FALSE;		/* Use standard input */
int opt_r = FALSE;		/* Do not call UUCICO, queue the job */
int opt_s = FALSE;		/* Do work for named site only */
int opt_x = FALSE;		/* Debugging flag */
int opt_z = FALSE;		/* Do not notify user if command succeeds */
int opt_Z = FALSE;		/* Reverse the Z line action */

int errno;			/* used by errno.h */

int location;			/* uux site location, LOCAL or REMOTE */
int cmdseq;			/* current command sequence number */
int pid;			/* current pid */
int needed;			/* number of files needed to run the command */
int outflg;			/* output file needed */
int usrfflg;			/* USERFILE read flag */
int dbglvl;			/* debugging level */

char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char spoolist[MAXLOCKS][LOCKLEN + 1];	/* spool file list */
char *xcmdlist[MAXXQTS];		/* uuxqt command file list */
char userpath[256];		/* USERFILE path list */
char spooldir[132];		/* spool directory */
char workdir[132];		/* initial working directory */
char cmdfile[132];		/* command file name */
char datafile[132];		/* data file name */
char uuxqtfile[132];		/* uuxqt command file name */
char scratch[132];		/* scratch buffer */
char uuxqtcmd[132];		/* uuxqt command line */
char fromfile[132];		/* source of copy */
char tofile[132];		/* destination of copy */
char infile[132];		/* command input file */
char outfile[132];		/* command output file */
char mailbox[132];		/* redefined mailbox file */
char cmdpath[132];		/* path to search for commands */
char cmdline[132];		/* uuxqt command string */
char fromsite[20];		/* from sitename */
char tosite[20];		/* to sitename */
char outsite[20];		/* output site */
char locsite[20];		/* local site name */
char rmtsite[20];		/* remote site */
char uuxsite[20];		/* uux site name */
char locuser[20];		/* local user name */
char rmtuser[20];		/* remote user name */
char uuxuser[20];		/* uux user name */
char options[20];		/* command option string */
char progname[20];		/* program name */
char grade = 'n';		/* job grade */

int fdlog;			/* system log file descriptor */

time_t now;			/* current time */

struct {			/* files needed */
	int pflg;		/* file has absolute path */
	int sflg;		/* file is in SPOOLDIR */
	int xflg;		/* file is in XQTDIR */
	char sfile[15];		/* spool file name */
	char cfile[132];	/* command file name */
} need[MAXNEED];

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Needed by exec */
extern char **environ;

/* Forward declarations */
int catchsignal();
int checkcmd();
int checklock();
int checkuser();
void cleanup();
char *expdtilde();
int genname();
int getargs();
int getdate();
int getname();
int getseq();
uid_t get_uid();
int getuser();
int parseuux();
int movefile();
void printmsg();
int scandir();
int sendmail();
int setsignal();
int sortlist();
int stripbang();
int unlock();
void usage();
int usercheck();
char *visib();
int writelog();
int writetask();


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT are ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining
 * ten or so signals are scarce enough, and rare enough, to be let run.
 * One problem is that the shell protects background commands from
 * SIGINT but not SIGHUP, so if SIGHUP is ignored only SIGKILL can stop a
 * background process, leaving pieces all over the floor.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, SIG_IGN) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k c m d
 *
 * Check that a command is permitted
 *
 * Return:	OK	Permitted
 *		FAILED	Not permitted
 *		cmdpath is updated as a side effect
 */
int checkcmd(cmd)
char *cmd;
{
  char *fields[MAXARGS], command[132], line[BUFSIZ];
  int j, flg, maxargs;
  FILE *fpcmd;

  /* we can do anything */
  if (strcmp(locsite, uuxsite) == SAME) return(OK);

  /* but other people are more limited */
  if ((fpcmd = fopen(LCMDS, "r")) == (FILE *)NULL) return(FAILED);

  flg = FAILED;
  cmdpath[0] = '\0';
  strncpy(command, cmd, 131);
  (void) getargs(command, fields);
  while (flg == FAILED && fgets(line, BUFSIZ, fpcmd) != (char *)NULL) {
	if (line[0] == '#' || line[0] == '\n') continue;
	/* lose any newline */
	j = strlen(line) - 1;
	if (line[j] == '\n') line[j] = '\0';
	/* we've got the PATH string */
	if (strncmp(line, "PATH", 4) == SAME) {
		strncpy(cmdpath, line, 131);
		continue;
	}
	/* make sure of whitespace in L-cmds entry */
	for (j = 0 ; j < strlen(line) ; j++)
		if (line[j] == ',') line[j] = ' ';
	if ((maxargs = getargs(line, fields)) > 0 &&
				strcmp(command, fields[0]) == SAME) {
		if (maxargs == 1)	/* default to all sites */
			flg = OK;
		else {			/* check for this site */
			for (j = 1 ; j < maxargs ; j++) {
				if (strcmp(fields[j], uuxsite) == SAME)
					flg = OK; 
			}
		}
	}
  }
  (void) fclose(fpcmd);

  return(flg);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sig;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT, the only un-ignored signal */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c h e c k u s e r
 *
 * Check if a given user/site has access to a given path
 *
 * Return:	OK		Yes
 *		FAILED		No
 *		usrfflg is updated as a side effect 
 */
int checkuser(site, user, path)
char *site, *user, *path;
{
  char *cp, *bp;
  int flg;

  if (usrfflg == MAYBE)
	usrfflg = usercheck(site, user);

  if (usrfflg == FAILED)
	return(FAILED);

  if (path == (char *)NULL)
	return(OK);
 
  printmsg(M_ACCESS, "checkuser: path |%s| userpath |%s|", path, userpath);
 
  cp = userpath;
  bp = path;
  flg = FAILED;
  while (TRUE) {
	/* successful match */
	if (*cp == '\0' || *cp == ' ') {
		flg =  OK;
		break;
	}
	/* ok so far */
	else if (*bp == *cp) {
		bp++;
		cp++;
	}
	else {
		/* reset bp */
		bp = path;
		/* use up the rest of this userpath entry */
		while (*cp && !isspace(*cp))
			cp++;
		while (isspace(*cp))
			cp++;
		/* no more possibilities */
		if (*cp == '\0')
			break;
	}
  }
 
  return(flg);
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files and exit
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  if (dbglvl > 0)		/* no harm in trying */
	(void) close(fdlog);

  /* remove all lockfiles */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0') {
		(void) unlink(locklist[j]);
		printmsg(M_ACCESS, "cleanup: lockfile %s removed", locklist[j]);
	}
	if (spoolist[j][0] != '\0') {
		(void) unlink(spoolist[j]);
		printmsg(M_ACCESS, "cleanup: spool file %s removed", spoolist[j]);
	}
	j++;
  }
  /* remove all the files in XQTDIR */
  for (j = 0 ; j < needed ; j++) {
	sprintf(scratch, "%s/%s", XQTDIR, need[j].cfile);
	(void) unlink(scratch);
  }
  /* remove any output files */
  if (outflg) {
	sprintf(scratch, "%s/%s", XQTDIR, outfile);
	(void) unlink(scratch);
	sprintf(scratch, "%s/%s", SPOOLDIR, datafile);
	(void) unlink(scratch);
  }
  /* remove the command file */
  sprintf(scratch, "%s/%s", SPOOLDIR, uuxqtfile);
  (void) unlink(scratch);

  exit(1);
}


/*
 * e x p d t i l d e
 *
 * Expand a filename using the usual "~" convention
 *
 * Return:	A valid pointer		Success
 *		A NULL pointer		Local filename not parsed
 *		scratch buffer is updated as a side effect
 *
 * The file "~user" expands to the home directory of "user". "~/" expands
 * to the home directory of the effective userid, which in this case
 * is usually the /usr/spool/uucppublic directory. Care is taken not
 * to overflow the (static) name buffer.
 */
char *expdtilde(filename)
char *filename;
{
  register char *pc, *pd;
  int j;
  struct passwd *pw;

  for (j = 0 ; j < 132 ; j++) scratch[j] = '\0';
  pc = scratch;
  pd = filename;

  /* if no leading ~, nothing to do */
  if (*pd++ != '~') {
	strncpy(scratch, filename, 131);
	return(scratch);
  }

  /* get the user name from the path */
  if (*pd != '/') {
	while ((*pc++ = *pd++) && *pd != '/')
		;
	if (*pd != '/') pd--;
  }
  /* pd now points to the end of the user name, for ~user,
   * or to the start of the path, for ~/.
   */

  /* use /usr/spool/uucppublic for ~/ */
  if (strlen(scratch) == 0)
	pw = getpwnam("uucp");
  else
	pw = getpwnam(scratch);

  /* check for valid username */
  if (pw == (struct passwd *)NULL)
	return((char *)NULL);

  /* check for valid path */
  if ((strlen(pw->pw_dir) + strlen(pd)) > 131)
	return((char *)NULL);

  strcpy(scratch, pw->pw_dir);
  strcat(scratch, pd);
  return(scratch);
}


/*
 * g e n n a m e
 *
 * Create a unique UUCP file name
 *
 * Return:	0		Failure (no sequence number)
 *		Number		Otherwise
 *		scratch buffer is updated as side effect
 *		SEQF file is updated as side effect
 *
 * The filename is a combination of prefix, grade, site name and a
 * sequential number taken from the SEQF file.
 */
int genname(prefix, grd, site)
char prefix, grd, *site;
{
  char *pc, *pd, sbuf[132];
  int seqnum;

  pc = sbuf;
  pd = site;
  while ((*pc++ = *pd++) && *pd != '!')	/* multiple bangs ? */
	;

  seqnum = getseq();
  sprintf(scratch, "%c.%s%c%04d", prefix, sbuf, grd, seqnum);

  printmsg(M_PARSE, "genname: prefix %c, site %s, grade %c, seq %d",
		prefix, site, grd, seqnum);

  return(seqnum);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t d a t e
 *
 * Get the date in a format suitable for the logfile
 *
 * Return	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(buff)
char *buff;
{
  int j;

  sprintf(buff, "%s", ctime(&now));
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[j] = '\0';			/* clear the newline */

  return(OK);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t s e q
 *
 * Get the next sequence number, update the sequence file
 *
 * Return:	1-9999	Success
 *		0	Failure
 */
int getseq()
{
  int fd, seq, flg;

  if (checklock(LCKSEQ, 2) == FAILED) {
	printmsg(M_ACCESS, "getseq: cannot create LCK.SEQ file");
	return(0);
  }

  /* if the SEQF file doesn't exist, create it */
  flg = TRUE;
  if (access(SEQF, 0) == -1) {
  	if ((fd = open(SEQF, O_CREAT | O_WRONLY | O_TRUNC, 0600)) == -1)
		flg = FALSE;
	else {
		if (write(fd, "0001", 4) != 4)
			flg = FALSE;
		if (close(fd) == -1)
			flg = FALSE;
	}	
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* read the seq file */
  if ((fd = open(SEQF, O_RDONLY, 0600)) == -1)
	flg = FALSE;
  else {
	if (read(fd, scratch, 4) != 4)
		flg = FALSE;
	else {
		scratch[4] = '\0';
		seq = atoi(scratch);
        }
	(void) close(fd);		/* about to overwrite */
	if (seq < 1 || seq > 9999)
		flg = FALSE;
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* write new seq number */
  if ((fd = open(SEQF, O_WRONLY | O_TRUNC, 0600)) == -1)
	flg = FALSE;
  else {
	sprintf(scratch, "%04d", seq < 9999 ? seq + 1 : 1);
	if (write(fd, scratch, 4) != 4)
		flg = FALSE;
	if (close(fd) == -1)
		flg = FALSE;
  }
  (void) unlock(LCKSEQ);
  printmsg(M_ACCESS, "getseq: got sequence number %d", seq);
  if (flg)
	return(seq);
  else
	return(0);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp, cmduser[30], message[256];
  int j, k, fd, cmdcount, cmdstatus, cmdflg, count, status;
  uid_t tuid;
#ifndef TESTDBG
  dbglvl = 0;
#else
  FILE *fp;
  if ((fp = fopen(DEFDBG, "r")) != (FILE *)NULL &&
		fgets(scratch, 132, fp) != (char *)NULL &&
		sscanf(scratch, "%d", &j) == 1)
	dbglvl = j;		/* default debug level */
  else
	dbglvl = 0;
  if (fp != (FILE *)NULL) (void) fclose(fp);
#endif

  /* set various defaults */
  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  uuxsite[0] = uuxuser[0] = '\0';
  for (j = 0 ; j < MAXLOCKS ; j++)
	locklist[j][0] = spoolist[j][0] = '\0';

  /* parse options */
  opterr = 0;
  while ((j = getopt(argc, argv, "s:x:Z")) != EOF)
	switch (j & 0377) {
	case 's':		/* site name */
		strncpy(uuxsite, optarg, SITELEN);
		opt_s = TRUE;
		break;
	case 'x':		/* set debug level */
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case 'Z':		/* reverse the Z line action */
		opt_Z = TRUE;
		break;
	case '?':		/* default arguments */
	default:
		usage();
		exit(1);
	}

#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid)
	dbglvl = 0;
#endif

  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  /* get local site name, local user, and pid */
  if (getname(locsite) == FAILED) {
	printmsg(M_NOLOG, "uuxqt: cannot find local uucpname");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	printmsg(M_NOLOG, "uuxqt: cannot find local username");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	printmsg(M_NOLOG, "uuxqt: cannot find own pid");
	exit(1);
  }
  if (getcwd(workdir, 131) == (char *)NULL) {
	printmsg(M_NOLOG, "uuxqt: cannot find own directory name");
	exit(1);
  }

  /* test both directories now to save trouble later */
  if (chdir(XQTDIR) == FAILED) {	/* switch to execute directory */
	printmsg(M_NOLOG, "uuxqt: cannot change to directory %s", XQTDIR);
	exit(1);
  }
  if (chdir(spooldir) == FAILED) {	/* switch to spool directory */
	printmsg(M_NOLOG, "uuxqt: cannot change to directory %s", spooldir);
	exit(1);
  }

  /* set defaults not already set by options */
  if (uuxsite[0] == '\0')		/* default location for uux */
	strncpy(uuxsite, locsite, SITELEN);
  if (uuxuser[0] == '\0')		/* default user for uux */
	strncpy(uuxuser, locuser, USERLEN);

  /* get the default command path and user */
  j = 0;
  strncpy(cmdpath, UUXPATH, 131);
  strncpy(cmduser, UUXUSER, 29);
  while (environ[j] != (char *)NULL) {
	if (strncmp(environ[j], "PATH", 4) == SAME) {
		strncpy(cmdpath, environ[j], 131);
		break;
	}
	if (strncmp(environ[j], "LOGNAME", 7) == SAME) {
		strncpy(cmduser, environ[j], 29);
		break;
	}
	j++;
  }

  /* set up to catch signals */
  if (catchsignal() == FAILED) {
	printmsg(M_NOLOG, "uuxqt: cannot catch signals");
	exit(1);
  }

  /* set up the log file; if debugging, lock it for the duration */
  if (dbglvl > 0) {
	if (checklock(LCKLOG, 2) == FAILED) {
		printmsg(M_NOLOG, "uuxqt: cannot get lock %s", LCKLOG);
		exit(1);
	}
	if ((fdlog = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1) {
		printmsg(M_NOLOG, "uuxqt: cannot open %s", LOGFILE);
		(void) unlock(LCKLOG);
		exit(1);
	}
	printmsg(dbglvl, "========== %-24.24s ==========", ctime(&now));
  }
  printmsg(M_ACCESS,
	"uuxqt: locsite %s locuser %s workdir %s\n       logfile %s spooldir %s",
	locsite, locuser, workdir, LOGFILE, spooldir);

  /* see if there's any work available */
  if ((cmdcount = scandir(uuxsite)) < 1) {
	printmsg(M_ACCESS, "uuxqt: no work for %s", uuxsite);
	if (dbglvl > 0) {
		(void) close(fdlog);
		(void) unlock(LCKLOG);
	}
	exit(0);
  }
  else
	printmsg(M_ACCESS, "uuxqt: site %s, tasks %d", uuxsite, cmdcount);

  /* try each uuxqt command file in turn */
  for (count = 0 ; count < cmdcount ; count++) {
	/* start from scratch on each new command file */
	usrfflg = MAYBE;
	userpath[0] = '\0';
	/* parse the uux command file and build a file list */
	strncpy(uuxqtfile, xcmdlist[count], 14);
	uuxqtfile[15] = '\0';
	if ((needed = parseuux(uuxqtfile)) == -1) {
		printmsg(M_ACCESS, "uuxqt: cannot parse %s", uuxqtfile);
		(void) unlink(uuxqtfile);
		continue;
	}

	/* check that all necessary data files exist */
	for (j = 0 ; j < needed ; j++) {
		fd = -1;
		if (need[j].sfile[0] == '/') {	/* full path given */
			if (checkuser(uuxsite, uuxuser, need[j].sfile) == FAILED
					|| (fd = open(need[j].sfile, 0)) == -1) {
				printmsg(M_ACCESS, "uuxqt: cannot access %s",
						need[j].sfile);
			}
			else
				need[j].pflg = TRUE;
		}
		else {				/* file in SPOOLDIR */
			if ((fd = open(need[j].sfile, 0)) == -1)
				printmsg(M_ACCESS, "uuxqt: cannot access %s",
						need[j].sfile);
			else
				need[j].sflg = TRUE;
		}
		if (fd != -1) (void) close(fd);
	}
	/* file missing failure */
	if (j != needed) continue;

	/* check command permissions */
	if (checkcmd(cmdline) == FAILED) {
		/* remove the command, and all associated files */
		(void) unlink(uuxqtfile);
		for (j = 0 ; j < needed ; j++)
			if (need[j].sflg)
				(void) unlink(need[j].sfile);
		printmsg(M_ACCESS, "uuxqt: illegal command |%s|", cmdline);
		sprintf(message, "uuxqt cmd (%s) status (DENIED)", cmdline);
		if (opt_m)
			(void) sendmail(rmtsite, rmtuser, message,
						(char *)NULL, TRUE);
		if (!opt_n)
			(void) sendmail(rmtsite, rmtuser, message,
						(char *)NULL, FALSE);
		continue;
	}

	/* try to lock .XQTDIR */
	if (checklock(LCKXQT, 2) == FAILED) {
		printmsg(M_ACCESS, "uuxqt: cannot lock .XQTDIR");
		break;
	}

	/* move to XQTDIR already checked */
	(void) chdir(XQTDIR);

	/* link/copy all necessary files to XQTDIR */
	for (j = 0 ; j < needed ; j++) {
		if (need[j].sflg) {	/* file in SPOOLDIR */
			sprintf(scratch, "%s/%s", SPOOLDIR, need[j].sfile);
			if (link(scratch, need[j].cfile) == -1) {
				printmsg(M_ACCESS, "cannot link file %s",
						need[j].sfile);
				break;
			}
		}
		else if (movefile(need[j].sfile, need[j].cfile) == FAILED) {
			printmsg(M_ACCESS, "cannot copy file %s",
						need[j].sfile);
			break;
		}
		need[j].xflg = TRUE;
	}
	/* remove files from XQTDIR on failure */
	if (j != needed) {
		needed = j;
		for (j = 0 ; j < needed ; j++)
			(void) unlink(need[j].cfile);
		(void) chdir(SPOOLDIR);
		(void) unlock(LCKXQT);
		continue;
	}

	/* run the command */
	(void) time(&now);
	sprintf(uuxqtcmd, "%s <%s >%s", cmdline, infile, datafile);

	switch (j = fork()) {
		case -1:		/* can't create new process */
			break;
		case 0:			/* forked, try to execute sh */
			(void) fflush(stderr);
			(void) freopen("/dev/null", "w", stderr);
			environ[0] = cmdpath;
			environ[1] = cmduser;
			environ[2] = (char *)NULL;
			(void) execlp(SHELL, "sh", "-c", uuxqtcmd, (char *)NULL);
			exit(1);	/* never happen */
			break;
		default:		/* command running, wait for status */
			while ((k = wait(&cmdstatus)) != j && k != -1)
				;
			break;
	}
	if (j < 1 || k < 1 || WEXITSTATUS(cmdstatus) != 0 ||
						WTERMSIG(cmdstatus) != 0) {
		printmsg(M_ACCESS, "uuxqt: command |%s| failed", uuxqtcmd);
		cmdflg = FAILED;
	}
	else if (outflg) {
		/* link output file to SPOOLDIR and remove from XQTDIR */
		sprintf(scratch, "%s/%s", spooldir, datafile);
		if (link(datafile, scratch) == -1)
			printmsg(M_ACCESS, "uuxqt: output %s failed", datafile);
		(void) unlink(datafile);
		cmdflg = OK;		/* not the command's fault */
	}
	else
		cmdflg = OK;

	/* remove all the files in XQTDIR */
	for (j = 0 ; j < needed ; j++)
		(void) unlink(need[j].cfile);

	/* move to SPOOLDIR already checked */
	(void) chdir(spooldir);
	(void) unlock(LCKXQT);

	/* generate the output status message */
	sprintf(message, "uuxqt cmd (%s) status (exit %d, signal %d)\n",
			cmdline, WEXITSTATUS(cmdstatus), WTERMSIG(cmdstatus));

	/* tell everyone the outcome */
	if (cmdflg == OK) {			/* deal with success */
		/* update the log file */
		printmsg(M_STD, "%s XQT (%s %s %s)",
				uuxuser, cmdpath, cmduser, uuxqtcmd);

		/* write the status to a file */
		if (opt_m)
			(void) sendmail(rmtsite, rmtuser, message,
							(char *)NULL, TRUE);
		/* send output to wherever */
		if (outflg) {
			sprintf(scratch, "%s!%s", outsite, outfile);
			printmsg(M_ACCESS, "uuxqt: uucp command |-rC %s %s|",
							datafile, scratch);
			switch (j = fork()) {
			case -1:	/* can't create new process */
				break;
			case 0:		/* forked, try to execute uucp */
				(void) fflush(stderr);
				(void) freopen("/dev/null", "w", stderr);
				(void) execlp(UUCP, "uucp", "-rC", datafile,
							scratch, (char *)NULL);
				exit(1);/* never happen */
				break;
			default:	/* command running, wait for status */
				while ((k = wait(&status)) != j && k != -1)
					;
				break;
			}
			printmsg(M_ACCESS,
				"uuxqt: uucp fork %d, wait %d, status %xx",
							j, k, status);
			(void) unlink(datafile);
		}
	}
	else {					/* deal with failure */
		/* update the log file */
		printmsg(M_STD, "%s XQT DENIED (%s)", uuxuser, uuxqtcmd);

		/* write the status to a file */
		if (opt_m)
			(void) sendmail(rmtsite, rmtuser, message,
							(char *)NULL, TRUE);

		/* send input to wherever */
		if (opt_b) {
			strcat(message, "Input data follows:\n");
			(void) sendmail(rmtsite, rmtuser, message, infile, FALSE);
		}
	}

	/* send the status message */
	if (opt_Z) opt_z = !opt_z;
#ifdef GOODREP
	if (!opt_n && (!opt_z || (!opt_b && cmdflg == FAILED)))
#else
	if (!opt_n && (opt_z || (!opt_b && cmdflg == FAILED)))
#endif
		(void) sendmail(rmtsite, rmtuser, message, (char *)NULL, FALSE);

	/* remove all the files in SPOOLDIR */
	(void) unlink(uuxqtfile);
	for (j = 0 ; j < needed ; j++) {
		if (need[j].sflg)
			(void) unlink(need[j].sfile);
	}
  }

  /* finished with the logfile */
  if (dbglvl > 0) {
	(void) close(fdlog);
	(void) unlock(LCKLOG);
  }

  exit(0);
  /* NOTREACHED */
}


/*
 * p a r s e u u x
 *
 * Parse a uux command file
 *
 * Return:	Number of files needed		Success
 *		-1				Otherwise
 *		cmdline, need[], infile, datafile, outfile, outsite
 *		outflg, rmtfile, rmtsite, uuxuser, uuxsite, mailbox
 *		are updated as side effects.
 */
int parseuux(filename)
char *filename;
{
  char *cp, *fields[MAXARGS], line[BUFSIZ];
  int j, cflg, uflg, iflg, oflg, flg, maxargs, needargs;
  FILE *fpcmd;

  cflg = uflg = iflg = oflg = FALSE;
  opt_a = opt_b = opt_m = opt_n = opt_z = FALSE;

  if ((fpcmd = fopen(filename, "r")) == (FILE *)NULL)
	return(-1);

  flg = OK;
  needargs = 0;
  while (flg == OK && fgets(line, BUFSIZ, fpcmd) != (char *)NULL) {
	j = strlen(line) - 1;
	if (line[j] == '\n') line[j] = '\0';
	cp = line;
	switch (*cp & 0177) {
	case '#':	/* HDB adds comment lines */
		break;
	case 'B':	/* On error, mail standard input to the user */
		opt_b = TRUE;
		break;
	case 'C':	/* Command line */
		cp++;
		while (*cp && (*cp == ' ' || *cp == '\t')) cp++;
		if (cflg || *cp == '\0')
			flg = FAILED;
		else {
			strncpy(cmdline, cp, 131);
			cflg = TRUE;
		}
		break;
	case 'F':
		maxargs = getargs(line, fields);
		if (maxargs == 0 || maxargs == 1)
			flg = FAILED;
		else if (maxargs == 2) {
			strncpy(need[needargs].sfile, fields[1], 14);
			strncpy(need[needargs].cfile, fields[1], 131);
			need[needargs].pflg = need[needargs].sflg
				= need[needargs].xflg = FALSE;
			needargs++;
		}
		else if (maxargs == 3) {
			strncpy(need[needargs].sfile, fields[1], 14);
			strncpy(need[needargs].cfile, fields[2], 131);
			need[needargs].pflg = need[needargs].sflg
				= need[needargs].xflg = FALSE;
			needargs++;
		}
		else
			flg = FAILED;
		break;
	case 'I':	/* Input file line */
		cp++;
		while (*cp && (*cp == ' ' || *cp == '\t')) cp++;
		if (iflg || *cp == '\0')
			flg = FAILED;
		else {
			strncpy(infile, cp, 131);
			iflg = TRUE;
		}
		break;
	case 'M':	/* Send transfer status to file */
		cp++;
		while (*cp && (*cp == ' ' || *cp == '\t')) cp++;
		if (*cp == '\0')
			flg = FAILED;
		else {
			if (*cp == '/')		/* absolute path */
				strncpy(mailbox, cp, 131);
			else {
				sprintf(mailbox, "%s/", SPOOLDIR);
				strncat(mailbox, cp, 131 - strlen(mailbox));
			}
			opt_m = TRUE;
		}
		break;
	case 'N':	/* Do not notify user of command status */
		opt_n = TRUE;
		break;
	case 'O':	/* Output file line */
		maxargs = getargs(line, fields);
		if (oflg || maxargs != 3)
			flg = FAILED;
		else {
			if ((cmdseq = genname('D', 'n', locsite)) == 0)
				flg = FAILED;
			else {
				strncpy(datafile, scratch, 131);
				strncpy(outfile, fields[1], 131);
				strncpy(outsite, fields[2], SITELEN);
				oflg = outflg = TRUE;
			}
		}
		break;
	case 'R':	/* Notify address, not initiator, on completion */
		cp++;
		while (*cp && (*cp == ' ' || *cp == '\t')) cp++;
		if (*cp == '\0')
			flg = FAILED;
		else {
			if (stripbang(cp, rmtsite, rmtuser) == FAILED) {
				strncpy(rmtuser, cp, USERLEN);
				strncpy(rmtsite, locsite, SITELEN);
			}
			opt_a = TRUE;
		}
		break;
	case 'U':	/* User line */
		maxargs = getargs(line, fields);
		if (uflg || maxargs != 3)
			flg = FAILED;
		else {
			strncpy(uuxuser, fields[1], USERLEN);
			strncpy(uuxsite, fields[2], SITELEN);
			uflg = TRUE;
		}
		break;
	case 'Z':	/* Do not notify user if command succeeds */
		opt_z = TRUE;
		break;
	default:	/* ### ignore unknown line type ? */
		flg = FAILED;
		break;
	}
  }
  (void) fclose(fpcmd);

  /* set location */
  if (strcmp(uuxsite, locsite) == SAME)
	location = LOCAL;
  else
	location = REMOTE;

  /* set mail defaults */
  if (!opt_a) {
	strncpy(rmtsite, uuxsite, SITELEN);
	strncpy(rmtuser, uuxuser, USERLEN);
  }

  /* set i/o defaults */
  if (!iflg)
	strcpy(infile, "/dev/null");
  if (!outflg)
	strcpy(datafile, "/dev/null");

  /* must have a Cline and a Uline */
  if (!cflg || !uflg) flg = FAILED;

  if (flg == OK)
	return(needargs);
  else
	return(-1);
}


/*
 * m o v e f i l e
 *
 * Move a file from a to b using link or copy
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * At present, this routine copies the file regardless, and doesn't link.
 */
int movefile(from, to)
char *from, *to;
{
  char buff[BUFSIZ];
  int j, infd, outfd;

  printmsg(M_ACCESS, "movefile: moving |%s| to |%s|", from, to);
#ifdef LNKFLG
  if (link(from, to) == FAILED) {	/* copy the file */
#endif
	if ((infd = open(from, 0)) == -1) {
		printmsg(M_ACCESS, "movefile: cannot open %s", from);
		return(FAILED);
	}
	if ((outfd = creat(to, 0644)) == -1) {
		(void) close(infd);
		printmsg(M_ACCESS, "movefile: cannot create %s", to);
		return(FAILED);
	}
	while ((j = read(infd, buff, sizeof(buff))) != 0) {
		if (j == -1 || write(outfd, buff, (size_t)j) != j) {
			(void) close(infd);
			(void) close(outfd);
			printmsg(M_ACCESS,
				"movefile: cannot copy |%s| to |%s|", from, to);
			return(FAILED);
		}
	}
	(void) close(infd);
	(void) close(outfd);
#ifdef LNKFLG
  }
#endif

  if (chmod(to, 0666) == -1)
	printmsg(M_ACCESS, "movefile: cannot chmod file %s", to);

  return(OK);
}


/*
 * p r i n t m s g
 *
 * Print an error or debugging message into the system error log file
 *
 * Return:	Nothing
 *
 * All messages at levels less than or equal to the current debug level
 * are printed, unless the M_NOLOG level is used. If debugging is on,
 * messages are also printed to standard error.
 */
/* VARARGS1 */
void printmsg(level, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int level; char *fmt, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8;
{
  char msg[BUFSIZ];

  (void) time(&now);
  if (level <= dbglvl) {
	sprintf(msg, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
	strcat(msg, "\n");
	if (level != M_NOLOG) {
		if (level == M_STD)
			(void) writelog(msg, TRUE);
		else
			(void) writelog(msg, FALSE);
	}
	if (dbglvl > 0) {
		fprintf(stderr, "%s", msg);
		(void) fflush(stderr);
	}
  }
}


/*
 * s c a n d i r
 *
 * Scan work dir for uux command files for the current site
 *
 * Return:	Number of files found		Success
 *		-1				Failure
 *		xcmdlist[] is updated as a side effect
 */
int scandir(site)
char *site;
{
  char name[15];
  int num;
  size_t len;
  DIR *dirp;
  struct dirent *dp;

  /* try to open work directory */
  if ((dirp = opendir(SPOOLDIR)) == (DIR *)NULL)
	return(-1);

  /* set up comparison */
  strcpy(name, "X.");
  if (opt_s) strncat(name, site, SITELEN);
  len = strlen(name);

  /* get a sorted list of any relevant workfiles */
  num = 0;
  xcmdlist[num] = (char *)NULL;
  while ((dp = readdir(dirp)) != (struct dirent *)NULL && num < MAXXQTS - 1) {
	if (strncmp(dp->d_name, name, len) == SAME) {
		if ((xcmdlist[num] = (char *) calloc(15, sizeof(char)))
							== (char *)NULL) {
			num = -1;
			break;
		}
		else {
			strncpy(xcmdlist[num++], dp->d_name, 14);
			xcmdlist[num] = (char *)NULL;
		}
	}
  }
  (void) closedir(dirp);
  if (num > 0)
	(void) sortlist(xcmdlist, num);
  
  return(num);
}


/*
 * s e n d m a i l
 *
 * Send messages or files to local or remote users
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int sendmail(site, name, message, file, flg)
char *site, *name, *message, *file; int flg;
{
  char address[32], curtime[64], header[256], buff[BUFSIZ];
  int j, k, status, err, fflg, mflg, infd, outfd, pfd[2];
  size_t len;

  printmsg(M_ACCESS, "sendmail: site |%s| user |%s| file |%s| message:\n|%s|",
		site, name, file, visib(message, strlen(message)));

  /* sort out what's to be done */
  err = fflg = mflg = FALSE;
  if (file != (char *)NULL)
	fflg = TRUE;
  if (message != (char *)NULL)
	mflg = TRUE;
  if (!fflg && !mflg) return(FAILED);

  /* write the message to a (local) file; mailbox is a full path */
  if (flg) {
	(void) getdate(curtime);
	sprintf(header, "From uucp (%s) ", curtime);
	if ((outfd = open(mailbox, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1)
		err = TRUE;
	(void) chmod(mailbox, 0666);
	len = strlen(header);
	if (!err && write(outfd, header, len) != len)
		err = TRUE;
	len = strlen(message);
	if (!err && mflg && write(outfd, message, len) != len)
		err = TRUE;
	if (!err && fflg) {
		if ((infd = open(file, 0)) == -1)
			err = TRUE;
		while (!err && (j = read(infd, buff, sizeof(buff))) != 0) {
			if (j == -1 || write(outfd, buff, (size_t)j) != j)
				err = TRUE;
		}
		(void) close(infd);
	}
	(void) close(outfd);
	if (err) (void) unlink(mailbox);
  }
  /* or mail it to the initiator */
  else {
	/* generate the message address and header (not needed with LMAIL) */
	sprintf(address, "%s!%s", site, name);
	sprintf(header, "From %s!%s %sTo: %s!%s\n\n", locsite, "uucp", ctime(&now),
							site, name);
	/* run a mailer, and write the text to its stdin */
	if (pipe(pfd) == -1)			/* generate the pipe */
		return(FAILED);
	switch (j = fork()) {			/* fork, dup and exec */
		case -1:			/* error */
			return(FAILED);
		case 0:				/* pregnant */
			if (close(0) == -1)
				exit(1);
			if (dup(pfd[0]) != 0)	/* child's stdin */
				exit(1);
			if (close(pfd[0]) == -1 || close(pfd[1]) == -1)
				exit(1);
			(void) fflush(stderr);
			(void) freopen("/dev/null", "w", stderr);
			(void) execlp(LMAIL, "lmail", address, (char *)NULL);
			exit(1);		/* never happen */
	}

	/* if we're here, we're a parent */
	if (close(pfd[0]) == -1) return(FAILED);
	len = strlen(header);
	if (write(pfd[1], header, len) != len)
		err = TRUE;
	if (!err && mflg) {
		len = strlen(message);
		if (write(pfd[1], message, len) != len)
			err = TRUE;
	}
	if (!err && fflg) {
		if ((infd = open(file, 0)) == -1) err = TRUE;
		while (!err && (k = read(infd, buff, sizeof(buff))) != 0) {
			if (k == -1 || write(pfd[1], buff, (size_t)k) != k)
				err = TRUE;
		}
		(void) close(infd);
	}
	if (close(pfd[1]) == -1) return(FAILED);
#ifdef MAILWAIT
	while ((k = wait(&status)) != j && k != -1)
		;
	if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 || WTERMSIG(status) != 0)
		err = FAILED;
	printmsg(M_ACCESS, "sendmail: mail fork %d, wait %d, status %d",
						j, k, status);
#endif
  }

  printmsg(M_ACCESS, "sendmail: mail %s", err ? "FAILED" : "OK");

  if (err)
	return(FAILED);
  else
	return(OK);
}

/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/* 
 * s o r t l i s t
 *
 * Sort a list of names
 *
 * Return:	OK	Always
 */
int sortlist(list, total)
char *list[]; int total;
{
  char tfil[15];
  int j, k, gap;

  /* shell sort */
  gap = 1;
  while (gap <= total) gap = 3 * gap + 1;

  for (gap /= 3 ; gap > 0 ; gap /= 3 ) {
 	for (j = gap ; j < total ; j++) { 
		strncpy(tfil, list[j], 14);
		for (k = j ;
		k >= gap && strncmp(list[k - gap], tfil, 14) > 0 ;
		k -= gap) {
			strncpy(list[k], list[k - gap], 14);
		}
		strncpy(list[k], tfil, 14);
	}
  }

  return(OK);
}


/*
 * s t r i p b a n g
 *
 * Strip the leading bang pathname from a string
 *
 * Return:	OK		Bang path found
 *		FAILED		Bang path not found
 *		path and site strings update as side effects
 */
int stripbang(arg, path, site)
char *arg, *path, *site;
{
  char *cp;
  int j;

  if ((cp = strchr(arg, '!')) == (char *)NULL) {
	*path = *site = '\0';
	return(FAILED);
  }

  j = 0;
  while (*arg && *arg != '!' && j++ < SITELEN)
	*site++ = *arg++;
  *site = '\0';
 
  strncpy(path, ++cp, 131);
  if (j == 0)
	strncpy(site, locsite, SITELEN);

  return(OK);
}


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/*
 * u s a g e 
 *
 * Usage message
 *
 * Return:	none
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-s sys] [-x #] [-Z]\n", progname);
}


/*
 * u s e r c h e c k
 *
 * Check the user file
 *
 * Return:	OK		Valid entry found
 *		FAILED  	No valid entry found
 *		cbflg, userpath are updated as side effects
 */
int usercheck(site, user)
char *site, *user;
{
  char *cp, *bp, tsite[20], tuser[20], tflag[20], buff[256];
  int j, level, flg;
  FILE *fp;

  printmsg(M_ACCESS, "usercheck: checking site |%s|, user |%s|", site, user);

  if ((fp = fopen(USERFILE, "r")) == (FILE *)NULL) {
	strncpy(userpath, "/", 255);
	return(OK);
  }

  level = 0;
  while (level < 4 && fgets(buff, 255, fp) != (char *)NULL) {
	tsite[0] = tuser[0] = tflag[0] = '\0';
	/* the trailing newline only confuses things */
	j = strlen(buff) - 1;
	if (buff[j] == '\n') buff[j] = '\0';
	if (buff[0] == '\0') continue;

	/* lose leading whitespace */
	cp = buff;
	while (isspace(*cp))
		cp++;

	/* we have a ',' or a username or junk */  
	if (*cp == ',')
		cp++;
	else if (isalpha(*cp)) {
		bp = tuser;
		while (isalpha(*cp) && (bp - tuser) < 20)
			*bp++ = *cp++;			
		*bp = '\0';
		while (isspace(*cp))
			cp++;
		if (*cp++ != ',')
			continue;
	}
	else
		continue;

	/* we have a sitename, a callback flag, a path, or junk */
	while (isspace(*cp))
		cp++;
	bp = tsite;
	while (isalpha(*cp) && (bp - tsite) < 20)
		*bp++ = *cp++;			
	*bp = '\0';
	while (isspace(*cp))
		cp++;
	bp = tflag;
	while (*cp != '/' && isalpha(*cp) && (bp - tflag) < 20)
		*bp++ = *cp++;			
	*bp = '\0';
	while (isspace(*cp))
		cp++;

	/* we now have a path, or junk */
	if (*cp != '/') continue;

	/* sort out site and callback entries */
	if (strcmp(tsite, "c") == SAME) {
		strcpy(tsite, "");
		strcpy(tflag, "c");
	}

	/* set callback flag */
	if (strcmp(tflag, "c") == SAME)
		flg = TRUE;
	else
		flg = FALSE;

	printmsg(M_ACCESS, "usercheck: found user |%s| site |%s| flag %s",
			tuser, tsite, flg ? "TRUE" : "FALSE");

	/* we have a valid entry; is it relevant ? */
	if (level < 1 && strcmp("", tsite) == SAME &&
						strcmp("", tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 1;
	}
	else if (level < 2 && strcmp("", tsite) == SAME &&
						strcmp(user, tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 2;
	}
	else if (level < 3 && strcmp(site, tsite) == SAME &&
						strcmp("", tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 3;
	}
	else if (level < 4 && strcmp(site, tsite) == SAME &&
						strcmp(user, tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 4;
	}
  }
  (void) fclose(fp);

  printmsg(M_ACCESS, "usercheck: exit level %d, userpath |%s|", level, userpath);

  if (level == 0)
	return(FAILED);
  else
	return(OK);
}


/*
 * v i s i b
 *
 * Print the buffer of character data in "visible" format
 *
 * Return:	A pointer to the data buffer
 *
 * Printable characters (except '\') are printed as themselves,
 * special control characters (CR, LF, BS, HT) are printed in the usual
 * escape format, and others are printed using 3-digit octal escapes.
 * The usual warning about a return value which points to a static buffer
 * which is overwritten on each call applies here.
 */
char *visib(data, len)
char *data; size_t len;
{
  static char buff[4 * 132];
  char c, *p;

  if (len > 132) len = 132;
  p = buff;
  while (len--) {
	c = *data++;
	if (isascii(c) && (isprint(c) || ' ' == c)) *p++ = c;
	  else if ('\n' == c) {
		*p++ = '\\';
		*p++ = 'n';
       	  } else if ('\t' == c) {
		*p++ = '\\';
		*p++ = 't';
          } else if ('\r' == c) {
		*p++ = '\\';
		*p++ = 'r';
         } else if ('\b' == c) {
		*p++ = '\\';
		*p++ = 'b';
         } else {
		sprintf(p, "\\%03o", c);
		p += 4;
        }
  }
  *p = '\0';
  return(buff);
}


/* 
 * w r i t e l o g
 *
 * Write a string to the log file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int writelog(msg, prefix)
char *msg; int prefix;
{
  char buff[BUFSIZ];
  char curtime[64];
  int flg;
  size_t sz;

  /* log format: site!user (date time) (id pid task) status (detail) */
  if (!prefix)
	strncpy(buff, msg, BUFSIZ - 1);
  else {
	(void) getdate(curtime);
	sprintf(buff, "%s!%s (%s) (Q %4d  0) ", rmtsite, locuser, curtime, pid);
	strncat(buff, msg, BUFSIZ - 50);
  }
  sz = strlen(buff);

  /* if debugging, the logfile is already locked */
  if (dbglvl > 0)
	flg = (write(fdlog, buff, sz) == sz);
  else {
	if (checklock(LCKLOG, 2) == FAILED) {
		printmsg(M_NOLOG, "uuxqt: cannot get lock %s", LCKLOG);
		return(FAILED);
	}
	if ((fdlog = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1) {
		printmsg(M_NOLOG, "uuxqt: cannot open %s", LOGFILE);
		(void) unlock(LCKLOG);
		return(FAILED);
	}
	flg = (write(fdlog, buff, sz) == sz);
	if (close(fdlog) == -1) flg = FALSE;
	(void) unlock(LCKLOG);
  }

  if (flg)
	return(OK);
  else
	return(FAILED);
}

