/* uux - unix to unix execute.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

/* general constants */

#define LOCAL		 0
#define REMOTE		 1
#define FALSE		 0
#define TRUE	    ~FALSE
#define OK		 1
#define FAILED		-1
#define SAME		 0
#define MAXLOCKS	10
#define MAXARGS		30
#define MAXSITE		10

/* log messages */

#define M_STD		0	/* standard messages */
#define M_NOLOG		1	/* no output to log file */
#define M_ACCESS	2	/* problems in access permissions */
#define M_PARSE		3	/* problems in parsing */

#ifndef lint
static char *Version = "@(#) uux 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_a = FALSE;		/* Notify name, not user, on completion */
int opt_b = FALSE;		/* On error, mail standard input to the user */
int opt_c = TRUE;		/* Do not link or copy source file (default) */
int opt_C = FALSE;		/* Force copy of source file */
int opt_g = FALSE;		/* Set grade of workfile */
int opt_j = FALSE;		/* Output UUCP job number on stderr */
int opt_n = FALSE;		/* Do not notify user of command status */
int opt_p = FALSE;		/* Use standard input */
int opt_r = FALSE;		/* Do not call UUCICO, queue the job */
int opt_s = FALSE;		/* Write transfer status to file */
int opt_x = FALSE;		/* Debugging flag */
int opt_z = FALSE;		/* Do not notify user if command succeeds */

int errno;			/* used by errno.h */

int location;			/* uuxqt location, LOCAL or REMOTE */
int cmdseq;			/* current command sequence number */
int pid;			/* current pid */
int jobstatus;			/* current job status */
int dbglvl;			/* debugging level */

char Uline[132];		/* uux command file U line */
char Fline[132];		/* uux command file F line */
char Cline[132];		/* uux command file C line */
char Iline[132];		/* uux command file I line */
char Oline[132];		/* uux command file O line */

char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char spoolist[MAXLOCKS][LOCKLEN + 1];	/* spool file list */
char spooldir[132];		/* spool directory */
char workdir[132];		/* intial working directory */
char cmdfile[132];		/* uucp command file name */
char datafile[132];		/* data file name */
char uuxqtfile[132];		/* uuxqt command file name */
char scratch[132];		/* scratch buffer */
char uuxqtcmd[132];		/* uuxqt command string */
char fromfile[132];		/* source of copy */
char tofile[132];		/* destination of copy */
char mailbox[132];		/* redefined mailbox file */
char fromsite[20];		/* from sitename */
char tosite[20];		/* to sitename */
char locsite[20];		/* local site name */
char rmtsite[20];		/* remote site name */
char uuxqtsite[20];		/* uuxqt site name */
char locuser[20];		/* local user name */
char uuxqtuser[20];		/* uuxqt user name */
char options[20];		/* command option string */
char progname[20];		/* program name */
char grade = 'A';		/* job grade */

int fdlog;			/* system log file descriptor */

FILE *fpuucp;			/* uucp command file pointer */
FILE *fpuuxqt;			/* uuxqt command file pointer */
time_t now;			/* current time */

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int addcmd();
int addfile();
int catchsignal();
int checkcmd();
int checklock();
int checkname();
void cleanup();
char *expdtilde();
int fbuild();
int genname();
int getargs();
int getdate();
int getname();
int getseq();
uid_t get_uid();
int getuser();
int lstatupdate();
int movefile();
void printmsg();
int rstatupdate();
int setsignal();
int stripbang();
int unlock();
void usage();
char *visib();
void wrapup();
int writelog();


/*
 * a d d c m d
 * 
 * Add a command to a uucp command file, generating the appropriate datafile
 * 
 * Return:	FAILED		Cannot link/copy datafile
 *		OK		Otherwise
 *		tofile is updated as a side effect
 */
int addcmd(fp)
FILE *fp;
{
  char buff[132];
  int j, fd;
  struct stat sb;

  /*
   * Workfile entries are S or R strings.
   *
   * Send from local to remote:
   * S full_source_file dest_file sender options data_file mode notify [mailbox]
   *
   * Request from remote to local:
   * R source_file full_dest_file sender options [mailbox]
   *
   * In this case only a restricted set of entries is needed.
   */

  printmsg(M_PARSE, "addcmd: fromsite %s tosite %s fromfile %s tofile %s",
		fromsite, tosite, fromfile, tofile);

  /* generate an S command for tosite, and copy data if necessary */
  if (strcmp(locsite, fromsite) == SAME) {
	if ((cmdseq = genname('D', 'S', tosite)) == 0) return(FAILED);
	strncpy(tofile, scratch, 131); 	/* tofile matches datafile */
	/* generate absolute path */
	if (fromfile[0] == '~') {
		if (expdtilde(fromfile, fromsite) == (char *)NULL)
			return(FAILED);
		strncpy(fromfile, scratch, 131);
	}
	else if (fromfile[0] != '/') {
		strncpy(scratch, workdir, 131);
		j = strlen(scratch);
		strncat(scratch, "/", (size_t)(131 - j));
		strncat(scratch, fromfile, (size_t)(130 - j));
		strncpy(fromfile, scratch, 131);
	}
	if (stat(fromfile, &sb) == FAILED) return(FAILED);
	if (opt_c && (sb.st_mode & S_IROTH))	/* placemarker, no file */
		strncpy(datafile, NODATAFN, 131);
	else {				/* generate data file name */
		strncpy(datafile, tofile, 131); 
		if (movefile(fromfile, datafile) == FAILED)
			return(FAILED);
		(void) chmod(datafile, 0600);
	}
	fprintf(fp, "S %s %s %s - %s 0666 %s\n",
		fromfile, tofile, locuser, datafile, uuxqtuser);
  }
  /* generate an R command for fromsite and send it to uuxqtsite */
  else {
	if((cmdseq = genname('D', 'R', fromsite)) == 0) return(FAILED);
	/* generate data and remote uucp command file names */
	strncpy(tofile, scratch, 131);
	tofile[0] = 'C';
	fprintf(fp, "S %s %s %s - %s 0666 %s\n",
		scratch, tofile, locuser, scratch, uuxqtuser);
	/* generate command string and write to data file */
	sprintf(tofile, "D.%sR%04d", uuxqtsite, cmdseq);
	sprintf(buff, "R %s %s %s -\n", fromfile, tofile, locuser);
	j = strlen(buff);
	if ((fd = creat(scratch, 0600)) == -1 || write(fd, buff, (size_t)j) != j)
		return(FAILED);
	(void) close(fd);

	/* add file to cleanup() list */
	(void) addfile(scratch);
  }

  return(OK);
}


/*
 * a d d f i l e
 *
 * Add a file name to the spool file list used by cleanup()
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int addfile(name)
char *name;
{
  char file[132];
  int j;

  /* build the file name */
  if (*name == '/') {
	strncpy(file, name, 131);
  }
  else {
	sprintf(file, "%s/", SPOOLDIR);
	strncat(file, name, 131 - strlen(file));
  }

  /* add it to the list */
  for (j = 0 ; j < MAXLOCKS ; j++) {
	if (spoolist[j][0] == '\0') {
		strncpy(spoolist[j], file, LOCKLEN);
		printmsg(M_ACCESS, "addfile: spool file %s listed", file);
		break;
	}
  }

  /* see if we succeeded */
  if (j == MAXLOCKS)
	return(FAILED);
  else
	return(OK);
}


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT are ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining
 * ten or so signals are scarce enough, and rare enough, to be let run.
 * One problem is that the shell protects background commands from
 * SIGINT but not SIGHUP, so if SIGHUP is ignored only SIGKILL can stop a
 * background process, leaving pieces all over the floor.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, SIG_IGN) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k c m d
 *
 * Check a uuxqt command string for validity
 *
 * Return:	OK	Valid
 *		FAILED	Invalid
 */
int checkcmd(cmd)
char *cmd;
{
  char *bp, *cp;
  int j, k, m, len, bflg, pflg, sflg;

  cp = cmd;
  len = strlen(cp);
  bflg = pflg = sflg = FALSE;

  while (*cp) {
	/* just in case */
	if (*cp == '\t')
		*cp = ' ';
	/* lose any spaces after > or < */
	if ((*cp == '<' || *cp == '>') && *(cp + 1) == ' ') {
		for (bp = cp + 1 ; *bp ; bp++)
			*bp = *(bp + 1);
		len--;
	}
	/* ensure a space after ; or | */
	else if ((*cp == ';' || *cp == '|') && *(cp + 1) != ' ') {
		for (bp = cp + len + 1 ; bp > cp ; bp--)
			*bp = *(bp - 1);
		*bp = ' ';
		len++;
	}
	else
		cp++;
  }
  cp = cmd;
  j = k = 0;
  while (*cp) {
	/* add a ! to any i/o argument without one */
	if (*cp == '<' || *cp == '>') {
		bp = cp;
		while (*bp && *bp != ' ' && *bp != '!')
			bp++;
		if (*bp != '!') {	/* bad path */
			for (bp = cmd + len + 1 ; bp > cp + 1 ; bp--)
				*bp = *(bp - 1);
			*bp = '!';
			len++;
		}
		/* check the number of inputs and outputs */
		if (*cp == '<')
			j++;
		else if (*cp == '>')
			k++;
	}
	/* check for balanced brackets */
	else if (*cp == '(') {
		bp = cp;
		while (*bp && *bp != ' ')
			bp++;
		if (*--bp != ')')	/* bad path */
			bflg = TRUE;
	}
	/* check for a pipeline */
	else if (*cp == '|')
		pflg = TRUE;
	/* check for a sequence */
	else if (*cp == ';')
		sflg = TRUE;
	cp++;
  }
  /* redirection symbols must be at the end of the command string */
  m = j + k;
  while (m > 0) {
	while (cp > cmd && *cp != ' ')
		cp--;
	if (*++cp != '>' && *cp != '<')
		break;
	m--;
  }
  /* we may be using a stdin flag */
  if (opt_p) j++;

  if (m > 0 || j > 1 || k > 1 || bflg || (pflg && sflg))
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sig;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT, the only un-ignored signal */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c h e c k n a m e
 *
 * Check if "name" is a known site
 *
 * Return:	FAILED	Cannot open L.sys file
 *		FAILED	Name not recognised
 *		OK	Name recognised
 */
int checkname(name)
char *name;
{
  char cont[132], line[BUFSIZ];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = strlen(name);
  if (j > SITELEN) j = SITELEN;
  while (fgets(line, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(line)) < (BUFSIZ - 132) &&
			line[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		line[k - 2] = '\0';
		strcat(line, cont);
	}
	for (k = 0 ; k < j ; k++) {
		if (line[k] != name[k])
			break;
	}
	if (k == j) {
		(void) fclose(fplsys);
       		return(OK);
	}
  }
  (void) fclose(fplsys);
  return(FAILED);
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files before exiting
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  if (dbglvl > 0)		/* no harm in trying */
	(void) close(fdlog);

  /* remove all our lockfiles, command files and data files */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0') {
		(void) unlink(locklist[j]);
		printmsg(M_ACCESS, "cleanup: lockfile %s removed", locklist[j]);
	}
	if (spoolist[j][0] != '\0') {
		(void) unlink(spoolist[j]);
		printmsg(M_ACCESS, "cleanup: spool file %s removed", spoolist[j]);
	}
	j++;
  }

  exit(1);
}


/*
 * e x p d t i l d e
 *
 * Expands a filename using the usual "~" convention.
 *
 * Return:	A valid pointer		SUCCESS
 *		A NULL pointer		Local filename not parsed
 *		scratch buffer is updated as a side effect
 *
 * The file "~user" expands to the home directory of "user". "~/" expands
 * to the home directory of the effective userid, which in this case
 * is usually the /usr/spool/uucppublic directory. Care is taken not
 * to overflow the (static) name buffer.
 *
 * "~user" cannot be parsed on a remote site, but "~/" is assumed still to
 * be /usr/spool/uucppublic".
 */
char *expdtilde(filename, site)
char *filename, *site;
{
  register char *pc, *pd;
  int j;
  struct passwd *pw;

  for (j = 0 ; j < 132 ; j++) scratch[j] = '\0';
  pc = scratch;
  pd = filename;

  /* if no leading ~, nothing to do */
  if (*pd++ != '~') {
	strncpy(scratch, filename, 131);
	return(scratch);
  }

  /* get the user name from the path */
  if (*pd != '/') {
	while ((*pc++ = *pd++) && *pd != '/')
		;
	if (*pd != '/') pd--;
  }
  /* pd now points to the end of the user name, for ~user,
   * or to the start of the path, for ~/.
   */

  /* use /usr/spool/uucppublic for ~/ */
  if (strlen(scratch) == 0)
	pw = getpwnam("uucp");
  else if (strcmp(site, locsite) == SAME)
	pw = getpwnam(scratch);
  else
	return((char *)NULL);

  /* check for valid username */
  if (pw == (struct passwd *)NULL)
	return((char *)NULL);

  /* check for valid path */
  if ((strlen(pw->pw_dir) + strlen(pd)) > 131)
	return((char *)NULL);
  strcpy(scratch, pw->pw_dir);
  strcat(scratch, pd);

  return(scratch);
}


/*
 * f b u i l d
 *
 * Build an F line
 *
 * Return	OK	Success
 *		FAILED	Otherwise
 */
int fbuild(cp, tempfile, tempsite, flg)
char *cp, *tempfile, *tempsite; int flg;
{
  int j, k, status;

  if (stripbang(cp, tempfile, tempsite) == FAILED) return(FAILED);
  if (tempfile[0] == '.') return(FAILED);

  /* file is already at uuxqt site */
  if (strcmp(uuxqtsite, tempsite) == SAME) {
	if (tempfile[0] == '~') {
		if (expdtilde(tempfile, uuxqtsite) == (char *)NULL)
			return(FAILED);
		strncpy(tempfile, scratch, 131);
	}
	else if (tempfile[0] != '/') {
		sprintf(scratch, "%s/",
			(location == REMOTE) ?  spooldir : workdir);
		strncat(scratch, tempfile, 131 - strlen(scratch));
		strncpy(tempfile, scratch, 131);
	}
  }
  else {
	strncpy(fromfile, tempfile, 131);
	strncpy(fromsite, tempsite, SITELEN);
 	strncpy(tosite, uuxqtsite, SITELEN);

	/* uuxqt is to run on the local site, file is remote */
	if (location == LOCAL)
		strncpy(tempsite, fromsite, SITELEN);
	/* uuxqt is to run on a remote site, file is local */
	else if (strcmp(locsite, tempsite) == SAME)
		strncpy(tempsite, tosite, SITELEN);
	/* uuxqt is to run on a remote site, file is elswhere */
	else
		strncpy(tempsite, tosite, SITELEN);

	/* see if we know where to go */
	if (checkname(tempsite) == FAILED) {
		printmsg(M_ACCESS, "fbuild: unknown site %s", tempsite);
		return(FAILED);
	}

	/* either run uucp to get the file we want */
	if (location == LOCAL) {
		if (dbglvl > 0) {
			(void) close(fdlog);	/* can't hog the logfile */
			(void) unlock(LCKLOG);
		}

		sprintf(scratch, "%s!%s", fromsite, fromfile);
		switch (j = fork()) {
		case -1:		/* can't create new process */
			break;
		case 0:			/* forked, try to execute mv */
			(void) fflush(stderr);
			(void) freopen("/dev/null", "w", stderr);
			if (opt_r)
				(void) execlp(UUCP, "uucp", scratch, tofile,
							(char *)NULL);
			else
				(void) execlp(UUCP, "uucp", "-r", scratch,
						tofile, (char *)NULL);
			exit(1);	/* never happen */
			break;
		default:		/* uucp running, wait for status */
			while ((k = wait(&status)) != j && k != -1)
				;
			break;
  		}
		if (dbglvl > 0 && (checklock(LCKLOG, 2) == FAILED ||
				(fdlog = open(LOGFILE, O_CREAT | O_WRONLY |
				O_APPEND, 0644)) == -1)) {
			printmsg(M_NOLOG, "uux: cannot reopen %s", LOGFILE);
			(void) unlock(LCKLOG);
			dbglvl = 0;
		}
		printmsg(M_ACCESS, "fbuild: uucp fork %d, wait %d, status %xx",
						j, k, status);

		if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 ||
						WTERMSIG(status) != 0) {
			printmsg(M_ACCESS, "fbuild: cannot run uucp");
			return(FAILED);
		}
	}
	/* or add the command to the current uucp command file */
	else {
		if (addcmd(fpuucp) == FAILED) {
			printmsg(M_ACCESS, "fbuild: cannot add command");
			return(FAILED);
		}
	}

	/* Write out the F line.
	 * SVR2 won't accept a local and remote name on an F line
	 * which refers to an I line, so we have to fake it.
	 */
	sprintf(Fline, "F %s %s", tofile, flg ? tempfile : "");
	fprintf(fpuuxqt, "F %s %s \n", tofile, flg ? tempfile : "");
	printmsg(M_PARSE, "fbuild: F line |F %s %s|", tofile,
						flg ? tempfile : "");
  }

  return(OK);
}


/*
 * g e n n a m e
 *
 * Create a unique UUCP file name
 *
 * Return:	0		Failure (no sequence number)
 *		Number		Otherwise
 *		scratch buffer is updated as side effect
 *		SEQF file is updated as side effect
 *
 * The filename is a combination of prefix, grade, site name and a
 * sequential number taken from the SEQF file.
 */
int genname(prefix, grd, site)
char prefix, grd, *site;
{
  char *pc, *pd, sbuf[132];
  int seqnum;

  pc = sbuf;
  pd = site;
  while ((*pc++ = *pd++) && *pd != '!')	/* multiple bangs ? */
	;

  seqnum = getseq();
  sprintf(scratch, "%c.%s%c%04d", prefix, sbuf, grd, seqnum);

  printmsg(M_PARSE, "genname: prefix %c, site %s, grade %c, seq %d",
		prefix, site, grd, seqnum);

  return(seqnum);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t d a t e
 *
 * Get the date in a format suitable for the logfile
 *
 * Return	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(buff)
char *buff;
{
  int j;

  sprintf(buff, "%s", ctime(&now));
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[j] = '\0';			/* clear the newline */

  return(OK);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t s e q
 *
 * Get the next sequence number, update the sequence file
 *
 * Return:	1-9999	Success
 *		0	Failure
 */
int getseq()
{
  int fd, seq, flg;

  if (checklock(LCKSEQ, 2) == FAILED) {
	printmsg(M_ACCESS, "getseq: cannot create LCK.SEQ file");
	return(0);
  }

  /* if the SEQF file doesn't exist, create it */
  flg = TRUE;
  if (access(SEQF, 0) == -1) {
  	if ((fd = open(SEQF, O_CREAT | O_WRONLY | O_TRUNC, 0600)) == -1)
		flg = FALSE;
	else {
		if (write(fd, "0001", 4) != 4)
			flg = FALSE;
		if (close(fd) == -1)
			flg = FALSE;
	}	
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* read the seq file */
  if ((fd = open(SEQF, O_RDONLY, 0600)) == -1)
	flg = FALSE;
  else {
	if (read(fd, scratch, 4) != 4)
		flg = FALSE;
	else {
		scratch[4] = '\0';
		seq = atoi(scratch);
        }
	(void) close(fd);		/* about to overwrite */
	if (seq < 1 || seq > 9999)
		flg = FALSE;
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* write new seq number */
  if ((fd = open(SEQF, O_WRONLY | O_TRUNC, 0600)) == -1)
	flg = FALSE;
  else {
	sprintf(scratch, "%04d", seq < 9999 ? seq + 1 : 1);
	if (write(fd, scratch, 4) != 4)
		flg = FALSE;
	if (close(fd) == -1)
		flg = FALSE;
  }
  (void) unlock(LCKSEQ);
  printmsg(M_ACCESS, "getseq: got sequence number %d", seq);
  if (flg)
	return(seq);
  else
	return(0);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * l s t a t u p d a t e
 *
 * Initialise an L_stat record if necessary
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int lstatupdate()
{
  char tfile[132], zbuff[sizeof(struct lstatrec)], buff[BUFSIZ];
  int j, fds, fdt, flg;
  sig_t sig;
  size_t len;
  struct lstatrec *lstat;

  /* set up to read L_stat records */
  len = sizeof(struct lstatrec);
  lstat = (struct lstatrec *)zbuff;
  strncpy(lstat->site, rmtsite, SITELEN);
  lstat->time_a = now;
  lstat->code = ET_QUEUED;
  lstat = (struct lstatrec *)buff;

  j = access(LSTAT, 0);
  printmsg(M_ACCESS, "lstatupdate: L_stat access call returns %d", j);
  if (j == -1) return (OK);	/* no file, don't bother */

  /* if the site is already listed in L_stat, fine */
  flg = FALSE;
  if ((fds = open(LSTAT, O_RDONLY, 0644)) == -1) {
	return(FAILED);
  }
  while (read(fds, buff, len) == len) {
	if (strncmp(lstat->site, rmtsite, SITELEN) == SAME) {
		flg = TRUE;
		break;
	}
  }
  (void) close(fds);
  if (flg) return(OK);

  /* we need to add a new site name */
  printmsg(M_ACCESS, "lstatupdate: new entry |%s|", visib(zbuff, len));

  if (checklock(LCKLSTAT, 2) == FAILED) return(FAILED); /* lock not possible */

  /* disable SIGINT while the R_stat and L_stat files are moved */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG) {
	(void) unlock(LCKLSTAT);
	return(FAILED);
  }

  /* move L_stat to a temporary file */
  sprintf(tfile, "%sL%04d", LTMPATH, pid);
  (void) unlink(tfile);		/* in case it already exists */
  if (link(LSTAT, tfile) == -1 || unlink(LSTAT) == -1) {
	(void) unlink(tfile);
	(void) unlock(LCKLSTAT);
	return(FAILED);
  }

  if ((fds = open(LSTAT, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(fdt = open(tfile, O_RDONLY, 0644)) == -1) {
	(void) link(tfile, LSTAT); /* recover if possible */
	(void) unlink(tfile);
	(void) unlock(LCKLSTAT);
	return(FAILED);
  }
  (void) unlink(tfile);	

  flg = FALSE;
  while (read(fdt, buff, len) == len) {
	if (!flg && strncmp(lstat->site, rmtsite, SITELEN) > 0) {
		if (write(fds, zbuff, len) != len) {
			(void) close(fds);	/* do not unlock - file corrupt */
			(void) close(fdt);
			return(FAILED);
		}
		flg = TRUE;
	}
	if (write(fds, buff, len) != len) {
		(void) close(fds);	/* do not unlock - file corrupt */
		(void) close(fdt);
		return(FAILED);
	}
  }
  if (!flg) {
	if (write(fds, zbuff, len) != len) {
		(void) close(fds);	/* do not unlock - file corrupt */
		(void) close(fdt);
		return(FAILED);
	}
  }
 
  (void) close(fds);
  (void) close(fdt);
  (void) unlock(LCKLSTAT);

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);
  else
	return(OK);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char  *bp, *cp, buff[BUFSIZ], tempfile[132], tempsite[132], *cmdargs[MAXARGS];
  int j, k, fd, jobnum, maxargs, cflg, uflg, iflg, oflg;
  uid_t tuid;
#ifndef TESTDBG
  dbglvl = 0;
#else
  FILE *fp;
  if ((fp = fopen(DEFDBG, "r")) != (FILE *)NULL &&
		fgets(scratch, 132, fp) != (char *)NULL &&
		sscanf(scratch, "%d", &j) == 1)
	dbglvl = j;		/* default debug level */
  else
	dbglvl = 0;
  if (fp != (FILE *)NULL) (void) fclose(fp);
#endif

  /* set various defaults */
  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  uuxqtsite[0] = uuxqtuser[0] = '\0';
  fpuucp = fpuuxqt = (FILE *)NULL;
  cflg = uflg = iflg = oflg = FALSE;
  for (j = 0 ; j < MAXLOCKS ; j++)
	locklist[j][0] = spoolist[j][0] = '\0';

  if (getenv("JOBNO") != (char *)NULL)
	opt_j = TRUE;

  /* parse options */
  for (j = 1 ; j < argc ; j++) {	/* getopt can't handle "-" */
	if (strcmp(argv[j], "-") == SAME) {
		for (k = j ; k < argc ; k++)
			argv[k] = argv[k + 1];
		argc--;
		opt_p = TRUE;
		break;
	}
  }
  opterr = 0;
  while ((j = getopt(argc, argv, "a:bcCg:jnprs:x:z")) != EOF)
	switch (j & 0377) {
	case 'a':		/* notify third party by mail o/c task */
		strncpy(uuxqtuser, optarg, USERLEN);
		opt_a = TRUE;
		break;
	case 'b':		/* send command input to user */
		opt_b = TRUE;
		break;
	case 'c':		/* do not link or copy source file */
		opt_c = TRUE;
		opt_C = FALSE;
		break;
	case 'C':		/* force copy of source file */
		opt_C = TRUE;
		opt_c = FALSE;
	        break;
	case 'g':		/* set grade of workfile */
		if (isalnum(optarg[0]) && optarg[1] == '\0')
			grade = optarg[0];
	    	else
			printmsg(M_NOLOG, "uux: unknown grade %s ignored", optarg);
		break;
	case 'j':		/* enable/suppress UUCP job number on stderr */
		opt_j = !opt_j;
		break;
	case 'n':		/* do not notify user of command status */
	  	opt_n = TRUE;
		break;
	case 'p':		/* use standard input */
	  	opt_p = TRUE;
		break;
	case 'r':		/* spool the job, but do not call uucico */
		opt_r = TRUE;
		opt_C = TRUE;	/* force copy of source file in this case */
		opt_c = FALSE;
		break;
	case 's':		/* write transfer status to file */
		strncpy(mailbox, optarg, 131);
		opt_s = TRUE;
		break;
	case 'x':		/* set debug level */
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case 'z':		/* do not notify user if command succeeds */
		opt_z = TRUE;
		break;
	case '?':		/* default arguments arrive here */
	default:
		usage();
		exit(1);
	}

  /* can't cope without filenames */
  if (optind >= argc) {
	usage();
	exit(1);
  }

#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid)
	dbglvl = 0;
#endif

  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  /* need space for file at end of path */
  if (strlen(spooldir) > 116 || strlen(workdir) > 116) {
	printmsg(M_NOLOG, "uux: directory path too long");
	exit(1);
  }

  /* get local site name, local user, and pid */
  if (getname(locsite) == FAILED) {
	printmsg(M_NOLOG, "uux: cannot find local uucpname");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	printmsg(M_NOLOG, "uux: cannot find local username");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	printmsg(M_NOLOG, "uux: cannot find own pid");
	exit(1);
  }
  if (getcwd(workdir, 131) == (char *)NULL) {
	printmsg(M_NOLOG, "uux: cannot find own directory name");
	exit(1);
  }
  if (chdir(spooldir) == FAILED) {	/* switch to spool directory */
	printmsg(M_NOLOG, "uux: cannot change to directory %s", spooldir);
	exit(1);
  }
  /* set up to catch signals */
  if (catchsignal() == FAILED) {
	printmsg(M_NOLOG, "uux: cannot catch signals");
	exit(1);
  }

  /* set defaults not already set by options */
  if (uuxqtsite[0] == '\0')		/* default uuxqt site */
	strncpy(uuxqtsite, locsite, SITELEN);
  if (uuxqtuser[0] == '\0')		/* default uuxqt user name */
	strncpy(uuxqtuser, locuser, USERLEN);

  /* set up the log file; if debugging, lock it for the duration */
  if (dbglvl > 0) {
	if (checklock(LCKLOG, 2) == FAILED) {
		printmsg(M_NOLOG, "uux: cannot get lock %s", LCKLOG);
		exit(1);
	}
	if ((fdlog = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1) {
		printmsg(M_NOLOG, "uux: cannot open %s", LOGFILE);
		(void) unlock(LCKLOG);
		exit(1);
	}
	printmsg(dbglvl, "========== %-24.24s ==========", ctime(&now));
  }

  /* build a single command string for uuxqt */
  uuxqtcmd[0] = '\0';
  k = 0;
  for (j = optind ; j < argc ; j++) {
	k = k + strlen(argv[j]);
	if (k > 129) {
		printmsg(M_PARSE, "uux: cannot parse command %s", argv[j]);
		wrapup("cannot parse %s", argv[j]);
	}
	strcat(uuxqtcmd, argv[j]);
	strcat(uuxqtcmd, " ");
  }
  uuxqtcmd[strlen(uuxqtcmd) - 1] = '\0';

  /* tidy up the uuxqt command string, and check for validity */
  if (checkcmd(uuxqtcmd) == FAILED) {
	printmsg(M_PARSE, "uux: cannot parse command %s", uuxqtcmd);
	wrapup("cannot parse %s", uuxqtcmd);
  }

  /* get the command site name and the first command */
  bp = scratch;
  cp = uuxqtcmd;
  while (*cp && *cp != ' ')
	*bp++ = *cp++;
  *bp = '\0';

  if (stripbang(scratch, tempfile, tempsite) == FAILED || strlen(tempsite) > SITELEN) {
	printmsg(M_PARSE, "uux: cannot parse command %s", uuxqtcmd);
	wrapup("cannot parse %s", uuxqtcmd);
  }

  /* initialise the C line with the command for uuxqt */
  strcpy(Cline, "C ");
  strncat(Cline, tempfile, 129);
  cflg = TRUE;

  /* check the uuxqt site name */
  strncpy(uuxqtsite, tempsite, SITELEN);
  if (strcmp(uuxqtsite, locsite) == SAME)
	location = LOCAL;
  else {
	location = REMOTE;
	if (checkname(uuxqtsite) == FAILED) {
		printmsg(M_ACCESS, "uux: unknown site %s", uuxqtsite);
		wrapup("unknown site %s", uuxqtsite);
	}
  }

  if ((cmdseq = genname((location == REMOTE) ? 'D' : 'X', 'X', locsite)) == 0) {
	printmsg(M_ACCESS, "uux: invalid sequence number");
	wrapup("invalid sequence", "");
  }
  jobnum = cmdseq;

  /* try to open the uux command file */
  strncpy(uuxqtfile, scratch, 131);
  if ((fpuuxqt = fopen(uuxqtfile, "w")) == (FILE *)NULL) {
	printmsg(M_ACCESS, "uux: cannot open %s", uuxqtfile);
	wrapup("cannot open %s", uuxqtfile);
  }
  (void) chmod(uuxqtfile, 0600);
  /* add file to cleanup() list */
  (void) addfile(uuxqtfile);

  /* try to open the uucp command file needed to move the uuxqt command */
  if (location == REMOTE) {
	sprintf(cmdfile, "C.%s%c%04d", uuxqtsite, grade, cmdseq);
	if ((fpuucp = fopen(cmdfile, "w")) == (FILE *)NULL) {
		printmsg(M_ACCESS, "uux: cannot open %s", cmdfile);
		wrapup("cannot open %s", cmdfile);
	}
	(void) chmod(cmdfile, 0600);
	/* add file to cleanup() list */
	(void) addfile(cmdfile);
	/* add file to R_stat, L_stat */
	strncpy(rmtsite, uuxqtsite, SITELEN);
	if (rstatupdate() == FAILED || lstatupdate() == FAILED) {
		printmsg(M_ACCESS, "uux: cannot update stat file(s)");
		wrapup("cannot update stats", "");
	}
  }

  /* build the U line */
  sprintf(Uline, "U %s %s", locuser, locsite);
  fprintf(fpuuxqt, "U %s %s\n", locuser, locsite);
  printmsg(M_PARSE, "uux: U line |U %s %s|", locuser, locsite);
  uflg = TRUE;

  /* If opt_p is TRUE, build the I line and an F line to match it.
   * Note that for some reason SVR2 won't accept a second argument
   * on the F line associated with an I line.  The standard fbuild()
   * cannot therefore be used.
   */
  if (opt_p) {
	if ((cmdseq = genname('D', 'S', uuxqtsite)) == 0) {
		printmsg(M_ACCESS, "uux: invalid sequence number");
		wrapup("invalid sequence", "");
	}
	if ((fd = creat(scratch, 0600)) == -1) {
		printmsg(M_ACCESS, "uux: cannot create file %s", scratch);
		wrapup("cannot create %s", scratch);
	}
	/* add file to cleanup() list */
	(void) addfile(scratch);
	while ((j = read(0, buff, sizeof(buff))) != 0) {
		if (j == -1 || write(fd, buff, (size_t)j) != j) {
			printmsg(M_ACCESS, "uux: cannot copy stdin");
			wrapup("cannot copy stdin", "");
		}
	}
	(void) close(fd);

	/* write out the F line */
	sprintf(Fline, "F %s", scratch);
	fprintf(fpuuxqt, "F %s\n", scratch);
	printmsg(M_PARSE, "fbuild: F line |F %s|", scratch);

	/* write out the I line */
	sprintf(Iline, "I %s", scratch);
	fprintf(fpuuxqt, "I %s\n", scratch);
	printmsg(M_PARSE, "uux: I line |I %s|", scratch);
	iflg = TRUE;

	/* add a command to the uucp command file */
	fprintf(fpuucp, "S %s %s %s - %s 0666 %s\n",
		scratch, scratch, locuser, scratch, uuxqtuser);
  }

  maxargs = getargs(uuxqtcmd, cmdargs);
  for (j = 1 ; j < maxargs ; j++) {
	cp = cmdargs[j];
	if (*cp == '<') {		/* we have an input string */
		if (iflg || fbuild(++cp, tempfile, tempsite, FALSE) == FAILED)
			wrapup("invalid input", "");

		/* write out the I line */
		sprintf(Iline, "I %s", tofile);
		fprintf(fpuuxqt, "I %s\n", tofile);
		printmsg(M_PARSE, "uux: I line |I %s|", tofile);
		iflg = TRUE;
	}
	else if (*cp == '>') {		/* we have an output string */
		/* parse the string */
		if (oflg || stripbang(++cp, tempfile, tempsite) == FAILED ||
		(location == LOCAL && checkname(tempsite) == FAILED &&
		strcmp(tempsite, locsite) != SAME) || tempfile[0] == '.') {
			printmsg(M_PARSE,
				"uux: invalid output string |%s|", cmdargs[j]);
			wrapup("invalid output %s", cmdargs[j]);
		}

		if (tempfile[0] == '~') {
			if (expdtilde(tempfile, tempsite) == (char *)NULL) {
				printmsg(M_PARSE,
					"uux: invalid output string |%s|",
						cmdargs[j]);
				wrapup("invalid output %s", cmdargs[j]);
			}
			strncpy(tempfile, scratch, 131);
		}
		else if (tempfile[0] != '/') {
			sprintf(scratch, "%s/", (strcmp(locsite, tempsite)
				== SAME) ? workdir : spooldir);
			strncat(scratch, tempfile, 131 - strlen(scratch));
			strncpy(tempfile, scratch, 131);
		}

		/* write out the O line */
		sprintf(Oline, "O %s %s", tempfile, tempsite);
		fprintf(fpuuxqt, "O %s %s\n", tempfile, tempsite);
		printmsg(M_PARSE, "uux: O line |O %s %s|", tempfile, tempsite);
		oflg = TRUE;
	}
	/* option or file argument */
	else {
		/* protected argument, strip and copy it */
		if (*cp == '(') {
			strncpy(tempfile, ++cp, 131);
			tempfile[strlen(tempfile) - 1] = '\0';
		}
		/* not a file name, copy it */
		else if (strchr(cp, '!') == (char *)NULL) {
			strncpy(tempfile, cp, 131);
			if (expdtilde(tempfile, uuxqtsite) == (char *)NULL) {
				printmsg(M_PARSE,
					"uux: cannot parse command |%s|", cp);
				wrapup("cannot parse %s", cp);
			}
			strncpy(tempfile, scratch, 131);
		}
		/* a file name, parse it */
		else if (fbuild(cp, tempfile, tempsite, TRUE) == FAILED) {
			printmsg(M_PARSE, "uux: cannot parse command |%s|", cp);
			wrapup("cannot parse %s", cp);
		}

		k = 131 - strlen(Cline) - strlen(tempfile) - 1;
		if (k < 0) {
			printmsg(M_PARSE, "uux: invalid Cline |%s %s|",
							Cline, tempfile);
			wrapup("cannot parse %s", Cline);
		}
		else {
			strcat(Cline, " ");
			strcat(Cline, tempfile);
		}
	}
  }
  /* add options to the uuxqt command file */
  if (opt_a)
	fprintf(fpuuxqt, "R %s\n", uuxqtuser);
  if (opt_b)
	fprintf(fpuuxqt, "B\n");
  if (opt_n)
	fprintf(fpuuxqt, "N\n");
  if (opt_s)
	fprintf(fpuuxqt, "M %s\n", mailbox);
  if (opt_z)
	fprintf(fpuuxqt, "Z\n");

  /* add the C line to the uuxqt command file */
  fprintf(fpuuxqt, "%s\n", Cline);
  printmsg(M_PARSE, "uux: C line |%s|", Cline);

  /* sending the uuxqt command takes last place in the uucp command file */
  if (location == REMOTE) {
	strncpy(scratch, uuxqtfile, 131);
	scratch[0] = 'X';
	printmsg(M_PARSE, "uux: from %s to %s", uuxqtfile, scratch);
	fprintf(fpuucp, "S %s %s %s - %s 0666 %s\n",
		uuxqtfile, scratch, locuser, uuxqtfile, uuxqtuser);
  }

  /* wrap up */
  if (fpuucp != (FILE *)NULL) (void) fclose(fpuucp);
  if (fpuuxqt != (FILE *)NULL) (void) fclose(fpuuxqt);
  printmsg(M_STD, "XQT QUE'D (%s)", Cline + 2);

  /* finished with the logfile */
  if (dbglvl > 0) {
	(void) close(fdlog);
	(void) unlock(LCKLOG);
  }
 
  /* print the job number */
  if (opt_j)
	fprintf(stderr, "uux job %d\n", jobnum);

  if (!opt_r && location == REMOTE) {		/* call the uux site */
	sprintf(scratch, "-s%s", uuxqtsite);
	sprintf(options, "-x%d", dbglvl);

	switch (j = fork()) {
	case -1:		/* can't create new process */
		break;
	case 0:			/* forked, try to execute uucico */
		(void) execlp(UUCICO, "uucico", "-r1", options, scratch,
							(char *)NULL);
		exit(1);	/* never happen */
		break;
	default:		/* uucico running, don't wait for status */
		break;
  	}
  }

  exit(0);
  /* NOTREACHED */
}


/*
 * m o v e f i l e
 *
 * Move a file from a to b using link or copy
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * At present, this routine copies the file regardless, and doesn't link.
 */
int movefile(from, to)
char *from, *to;
{
  char buff[BUFSIZ];
  int j, infd, outfd;

  printmsg(M_ACCESS, "movefile: moving |%s| to |%s|", from, to);
#ifdef LNKFLG
  if (link(from, to) == FAILED) {	/* copy the file */
#endif
	if ((infd = open(from, 0)) == -1) {
		printmsg(M_ACCESS, "movefile: cannot open %s", from);
		return(FAILED);
	}
	if ((outfd = creat(to, 0644)) == -1) {
		(void) close(infd);
		printmsg(M_ACCESS, "movefile: cannot create %s", to);
		return(FAILED);
	}
	while ((j = read(infd, buff, sizeof(buff))) != 0) {
		if (j == -1 || write(outfd, buff, (size_t)j) != j) {
			(void) close(infd);
			(void) close(outfd);
			printmsg(M_ACCESS,
				"movefile: cannot copy |%s| to |%s|", from, to);
			return(FAILED);
		}
	}
	(void) close(infd);
	(void) close(outfd);
#ifdef LNKFLG
  }
#endif

  if (chmod(to, 0666) == -1)
	printmsg(M_ACCESS, "movefile: cannot chmod file %s", to);

  return(OK);
}


/*
 * p r i n t m s g
 *
 * Print an error or debugging message into the system error log file.
 *
 * Return:	Nothing
 *
 * All messages at levels less than or equal to the current debug level
 * are printed, unless the M_NOLOG level is used. If debugging is on,
 * messages are also printed to standard error.
 */
/* VARARGS1 */
void printmsg(level, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int level; char *fmt, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8;
{
  char msg[BUFSIZ];

  (void) time(&now);
  if (level <= dbglvl) {
	sprintf(msg, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
	strcat(msg, "\n");
	if (level != M_NOLOG) {
		if (level == M_STD)
			(void) writelog(msg, TRUE);
		else
			(void) writelog(msg, FALSE);
	}
	if (dbglvl > 0) {
		fprintf(stderr, "%s", msg);
		(void) fflush(stderr);
	}
  }
}


/* 
 * r s t a t u p d a t e
 *
 * Initialise an R_stat record if the file exists 
 *
 * Return:	OK	Success (or no R_stat file)
 *		FAILED	Otherwise
 */
int rstatupdate()
{
  char buff[BUFSIZ];
  int j, fd, flg = TRUE;
  size_t len;
  struct rstatrec *rstat;

  j = access(RSTAT, 0);
  printmsg(M_ACCESS, "rstatupdate: R_stat access call returns %d", j);
  if (j == -1) return (OK);	/* no file, don't bother */

  /* Build an entry for R_stat
   * R_stat format: job_no user site command_time status_time status
   * Note that uucp puts zeroes in the second time entry as a place holder.
   */

  for (j = 0 ; j < BUFSIZ ; j++) buff[j] = '\0';
  rstat = (struct rstatrec *)buff;
  rstat->job = cmdseq;
  strncpy(rstat->user, locuser, USERLEN);
  strncpy(rstat->site, rmtsite, SITELEN);
  rstat->time_a = now;
  rstat->time_b = (time_t)0;
  rstat->code = ET_QUEUED;

  /* try to lock the R-stat file to append new data */
  if (checklock(LCKRSTAT, 2) == FAILED) return(FAILED);
  if ((fd = open(RSTAT, O_WRONLY | O_APPEND, 0600)) == -1) {
	(void) unlock(LCKRSTAT);
	return(FAILED);
  }

  /* append the data */
  len = sizeof(struct rstatrec);
  flg = (write(fd, buff, sizeof(struct rstatrec)) == sizeof(struct rstatrec));
  if (close(fd) == -1) flg = FALSE;
  (void) unlock(LCKRSTAT);

  printmsg(M_ACCESS, "rstatupdate: new entry |%s|", visib(buff, len));

  if (flg)
	return(OK);
  else
	return(FAILED);
}


/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/*
 * s t r i p b a n g
 *
 * Strip the leading bang pathname from a string
 *
 * Return:	OK		Bang path found
 *		FAILED		Bang path not found
 *		path and site strings update as side effects
 */
int stripbang(arg, path, site)
char *arg, *path, *site;
{
  char *cp;
  int j;

  if ((cp = strchr(arg, '!')) == (char *)NULL) {
	*path = *site = '\0';
	return(FAILED);
  }

  j = 0;
  while (*arg && *arg != '!' && j++ < SITELEN)
	*site++ = *arg++;
  *site = '\0';
 
  strncpy(path, ++cp, 131);
  if (j == 0)
	strncpy(site, locsite, SITELEN);

  return(OK);
}


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/*
 * u s a g e 
 *
 * Usage message
 *
 * Return:	none
 */
void usage()
{
  fprintf(stderr, "Usage: %s [bCcjnprz] [-a user] [-g grade] [-s spool]\n",
	progname);
  fprintf(stderr, "           [-x level] <command string>\n");
}


/*
 * v i s i b
 *
 * Prints the buffer of character data in "visible" format.
 *
 * Return:	A pointer to the data buffer
 *
 * Printable characters (except '\') are printed as themselves,
 * special control characters (CR, LF, BS, HT) are printed in the usual
 * escape format, and others are printed using 3-digit octal escapes.
 * The usual warning about a return value which points to a static buffer
 * which is overwritten on each call applies here.
 */
char *visib(data, len)
char *data; size_t len;
{
  static char buff[4 * 132];
  char c, *p;

  if (len > 132) len = 132;
  p = buff;
  while (len--) {
	c = *data++;
	if (isascii(c) && (isprint(c) || ' ' == c)) *p++ = c;
	  else if ('\n' == c) {
		*p++ = '\\';
		*p++ = 'n';
       	  } else if ('\t' == c) {
		*p++ = '\\';
		*p++ = 't';
          } else if ('\r' == c) {
		*p++ = '\\';
		*p++ = 'r';
         } else if ('\b' == c) {
		*p++ = '\\';
		*p++ = 'b';
         } else {
		sprintf(p, "\\%03o", c);
		p += 4;
        }
  }
  *p = '\0';
  return(buff);
}


/*
 * w r a p u p
 *
 * Write out an error message and call cleanup().
 *
 * Return:	Nothing
 */
void wrapup(format, message)
char *format, *message;
{
  fprintf(stderr, "uux failed: ");
  fprintf(stderr, format, message);
  fprintf(stderr, "\n");

  cleanup(SIGHUP);
}


/* 
 * w r i t e l o g
 *
 * Write a string to the log file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int writelog(msg, prefix)
char *msg; int prefix;
{
  char buff[BUFSIZ];
  char curtime[64];
  int flg;
  size_t sz;

  /* log format: site!user (date time) (id pid task) status (detail) */
  if (!prefix)
	strncpy(buff, msg, BUFSIZ - 1);
  else {
	(void) getdate(curtime);
	sprintf(buff, "%s!%s (%s) (X %4d  0) ", uuxqtsite, locuser, curtime, pid);
	strncat(buff, msg, BUFSIZ - 50);
  }
  sz = strlen(buff);

  /* if debugging, the logfile is already locked */
  if (dbglvl > 0)
	flg = (write(fdlog, buff, sz) == sz);
  else {
	if (checklock(LCKLOG, 2) == FAILED) {
		printmsg(M_NOLOG, "uux: cannot get lock %s", LCKLOG);
		return(FAILED);
	}
	if ((fdlog = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1) {
		printmsg(M_NOLOG, "uux: cannot open %s", LOGFILE);
		(void) unlock(LCKLOG);
		return(FAILED);
	}
	flg = (write(fdlog, buff, sz) == sz);
	if (close(fdlog) == -1) flg = FALSE;
	(void) unlock(LCKLOG);
  }

  if (flg)
	return(OK);
  else
	return(FAILED);
}

