/* uuto - send files for retrieval by uupick.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <pwd.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <unistd.h>
#include "uucp.h"

#define FALSE		0
#define TRUE	   ~FALSE
#define OK		1
#define FAILED	       -1
#define SAME		0

#ifndef lint
static char *Version = "@(#) uuto 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_m = FALSE;
int opt_p = FALSE;
int opt_P = FALSE;
int opt_x = FALSE;

int errno;			/* used by errno.h */

char srcname[132];		/* source file name */
char destname[132];		/* destination file name */
char locuser[20];		/* name of local user */
char rmtuser[20];		/* name of remote user */
char progname[20];		/* program name */

int dbglvl;

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int checkname();
int countfiles();
int getname();
int getuser();
char query();
int runuucp();
int senddir();
void usage();


/*
 * c h e c k n a m e
 *
 * Check if "name" is a known site
 *
 * Return:	FAILED	Cannot open L.sys file
 *		FAILED	Name not recognised
 *		OK	Name recognised
 */
int checkname(name)
char *name;
{
  char cont[132], line[BUFSIZ];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = strlen(name);
  if (j > SITELEN) j = SITELEN;
  while (fgets(line, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(line)) < (BUFSIZ - 132) &&
			line[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		line[k - 2] = '\0';
		strcat(line, cont);
	}
	for (k = 0 ; k < j ; k++) {
		if (line[k] != name[k])
			break;
	}
	if (k == j) {
		(void) fclose(fplsys);
       		return(OK);
	}
  }
  (void) fclose(fplsys);
  return(FAILED);
}


/*
 * c o u n t f i l e s
 *
 * Count the number of files in a directory
 *
 * Return:	Count	Success
 *		-1	Failure
 */
int countfiles(dirname, d, f)
char *dirname; int *d, *f;
{
  char name[132];
  int j, count;
  size_t len;
  DIR *dirp;
  struct dirent *dp;
  struct stat stat_buff;

  if ((dirp = opendir(dirname)) == (DIR *)NULL) {
	fprintf(stderr, "uuto: cannot open %s\n", dirname);
	return(-1);
  }

  /* build the full current path */
  strncpy(name, dirname, 130);
  strcat(name, "/");
  len = strlen(name);

  count = 0;
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	if (strcmp(dp->d_name, ".") == SAME ||
			strcmp(dp->d_name, "..") == SAME)
		continue;
	/* add the current path and name */
	for (j = 0 ; j < 15 ; j++) {
		if ((len + j) > 131) break;
		name[len + j] = dp->d_name[j];
	}
	name[len + j] = '\0';
	/* find out what we've got */
	if (stat(name, &stat_buff) == -1)
		continue;
	else if (stat_buff.st_mode & S_IFREG) {
		count++;
		(*f)++;
	}
	else if (stat_buff.st_mode & S_IFDIR) {
		count++;
		(*d)++;
	}
  }
  (void) closedir(dirp);

  return(count);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char locsite[20];			/* name of local site */
  char rmtsite[20];			/* name of remote site */
  char *cp;
  int j, k, status;
  struct stat stat_buff;

  /* set up defaults */
  if (getname(locsite) == FAILED) {	/* get local site name */
	fprintf(stderr, "uuto: cannot find local uucpname\n");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	fprintf(stderr, "uuto: cannot find own username\n");
	exit(1);
  }

  (void) umask(022);			/* set protection mask to rw-r--r-- */

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */

  dbglvl = 0;

  /* parse arguments */
  opterr = 0;
  while ((j = getopt(argc, argv, "mpPx:")) != EOF) {
	switch (j & 0177) {
	case 'm':
		opt_m = TRUE;
		break;
	case 'p':
		opt_p = TRUE;
		break;
	case 'P':
		opt_P = TRUE;
		break;
	case 'x':
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case '?':
	default:
		usage();
		exit(1);
		break;
	}
  }

  /* check validity of arguments */
  if (optind > argc - 2) {
	fprintf(stderr, "uuto: need at least two arguments\n");
	exit(1);
  }
  /* check that there is a path */
  rmtsite[0] = rmtuser[0] = '\0';
  if ((cp = strchr(argv[optind + 1], '!')) != (char *)NULL) {
	*cp++ = '\0';
	strncpy(rmtsite, argv[optind + 1], SITELEN);
	strncpy(rmtuser, cp--, USERLEN);
	*cp = '!';
  }
  if (strlen(rmtsite) == 0 || strlen(rmtuser) == 0) {
	fprintf(stderr, "uuto: cannot parse remote pathname %s\n",
							argv[optind + 1]);
	exit(1);
  }
  /* check that the site is known */
  if (checkname(rmtsite) == FAILED && strcmp(rmtsite, locsite) != SAME) {
	fprintf(stderr, "uuto: cannot find site %s\n", rmtsite);
	exit(1);
  }
  strncpy(srcname, argv[optind], 131);

  /* the destination directory is PUBDIR/receive/user/site or PUBDIR/user */
  if (opt_P)
	sprintf(destname, "%s!%s/%s", rmtsite, PUBDIR, rmtuser);
  else
	sprintf(destname, "%s!%s/%s/%s", rmtsite, PICKDIR, rmtuser, locsite);

  if (dbglvl > 0) {
	fprintf(stderr, "main: source |%s|\n", srcname);
	fprintf(stderr, "main: destination |%s|\n", destname);
  }

  /* try to open the input file. must be a wildcard, a file or a directory */
  status = 1;
  if (stat(srcname, &stat_buff) == -1 || (stat_buff.st_mode & S_IFREG)) {
	strcat(destname, "/");		/* force a directory as destination */
	if (runuucp(srcname, destname) == FAILED)
		fprintf(stderr, "uuto: cannot execute uucp\n");
	else
		status = 0;
  }
  else if (stat_buff.st_mode & S_IFDIR) {
	if ((j = countfiles(srcname, &k, &k)) == -1) {
	}
	else if (j == 0) {
		fprintf(stderr, "uuto: empty directory %s\n", srcname);
		status = 0;
	}
	else if (senddir(srcname, destname) == FAILED)
		fprintf(stderr, "uuto: cannot send directory %s\n", srcname);
	else
		status = 0;
  }
  else
	fprintf(stderr, "uuto: unusable input %s\n", srcname);

  exit(status);
  /* NOTREACHED */
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * q u e r y
 *
 * Get an answer to a question
 *
 * Return:	Letter response
 *		Answer buffer is updated as a side effect
 */
char query(question, valid, answer)
char *question, *valid, *answer;
{
  char ans[132], c = '\0';
  int j, k, m;

  while (c == '\0') {
	(void) fputs(question, stdout);
	(void) fflush(stdout);
	m = 0;
	while ((j = fgetc(stdin)) != EOF) {
		c = (char ) (j & 0177);
		if (c == '\n') break;
		if (c == '\b') m--;
		else ans[m++] = c;
		if (m < 0) m = 0;
		if (m > 130) break;
	}
	if (m == 0 && j == EOF) c = '\n';
	ans[m] = '\0';
	if (m != 0) c = ans[0];

	/* simple newline return gives default response */
	if (c == '\n') {
		break;
	}
	/* check for valid input */
	else if (*valid != '\0' && strchr(valid, c) == (char *)NULL) {
		c = '\0';
	}
	/* sort out any additional argument string */
	else {
		for (k = 1 ; k < m ; k++) {
			if (ans[k] != ' ')
				break;
		}
		m -= k;
		for (j = 0 ; j < m ; j++)
			ans[j] = ans[j + k];
		ans[j] = '\0';
		break;
	}
  }

  strncpy(answer, ans, 131);
  return(c);
}


/*
 * r u n u u c p
 *
 * Run the uucp program
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int runuucp(src, dest)
char *src, *dest;
{
  char tmp[20], debug[20], options[20], mailbox[20];
  int j, k, status;

  /* sort out the arguments; -drn is always used */
  sprintf(debug, "-x%d", dbglvl);
  if (opt_p)
	sprintf(options, "-drCn%s", rmtuser);
  else
	sprintf(options, "-drn%s", rmtuser);
  sprintf(mailbox, "-m%s", locuser);
  if (opt_m)
	sprintf(tmp, " %s|", mailbox);
  else
	strcpy(tmp, "|");

  if (dbglvl > 0)
	fprintf(stderr,
		"runuucp: opt |%s %s%s\n         src |%s|\n         dest |%s|\n",
		debug, options, tmp, src, dest);

  switch (j = fork()) {
	case -1:		/* can't create new process */
		break;
	case 0:			/* forked, try to execute uucp */
		(void) fflush(stderr);
		if (dbglvl == 0) (void) freopen("/dev/null", "w", stderr);
		if (opt_m)
			(void) execlp(UUCP, "uucp", debug, options, mailbox,
						src, dest, (char *)NULL);
		else
			(void) execlp(UUCP, "uucp", debug, options,
						src, dest, (char *)NULL);
		exit(1);	/* never happen */
		break;
	default:		/* uucp running, wait for status */
		while ((k = wait(&status)) != j && k != -1)
			;
		break;
  }

  if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 || WTERMSIG(status) != 0)
	return(FAILED);
  else
	return(OK);
}


/*
 * s e n d d i r
 *
 * Dispose of a directory and its contents
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int senddir(srcpath, destpath)
char *srcpath, *destpath;
{
  char c, source[132], dest[132], question[132], answer[132];
  int j, dirs, files;
  size_t slen, dlen;
  DIR *dirp;
  struct dirent *dp;
  struct stat stat_buff;

  if (dbglvl > 0)
	fprintf(stderr, "senddir: checking directory |%s|\n", srcpath);

  if ((j = countfiles(srcpath, &dirs, &files)) == -1)
	return(FAILED);
  else if (j == 0) {
	fprintf(stderr, "uuto: no files in %s\n", srcpath);
	return(OK);
  }

  /* get the full current source path */
  strncpy(source, srcpath, 129);
  strcat(source, "/*");
  slen = strlen(source) - 1;

  /* get the full current destination path */
  strncpy(dest, destpath, 130);
  strcat(dest, "/");
  dlen = strlen(dest);

  /* send all the files via uucp */
  if (files != 0 && runuucp(source, destpath) == FAILED)
	return(FAILED);
  else if (dirs == 0)
	return(OK);
  else if ((dirp = opendir(srcpath)) == (DIR *)NULL)
	return(FAILED);

  /* send all the subdirectories via uucp */
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	if (strcmp(dp->d_name, ".") == SAME ||
			strcmp(dp->d_name, "..") == SAME)
		continue;

	/* add the current path and source */
	for (j = 0 ; j < 15 ; j++) {
		if ((slen + j) > 131 || (dlen + j) > 131) break;
		source[slen + j] = dp->d_name[j];
		dest[dlen + j] = dp->d_name[j];
	}
	source[slen + j] = '\0';
	dest[dlen + j] = '\0';

	/* find out what we've got */
	(void) stat(source, &stat_buff);
	/* if it's a regular file, ignore it */
	if (stat_buff.st_mode & S_IFREG) {
		/* (void) runuucp(source, dest); */
	}
	/* if it's a subdirectory, check it out */
	else if (stat_buff.st_mode & S_IFDIR) {
		sprintf(question,
			"Send entire directory %s (y/N) ? ", source);
		answer[0] = '\0';
		c = query(question, "yYnN", answer);
		if (c == 'y' || c == 'Y')
			(void) senddir(source, dest);
	}
	else
		fprintf(stderr, "uuto: unusable input %s\n", source);
  }
  (void) closedir(dirp);

  return(OK);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-mpPx] files site!user\n", progname);
}

