/* uusub - monitor a UUCP network.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

#define FALSE		 0
#define TRUE	    ~FALSE
#define OK		 1
#define FAILED		-1
#define SAME		 0
#define MAXLOCKS	10
#define MAXARGS 	30
#define MAXSITE        100
#define MAXTASK        500

/* call status values */
#define S_OK  1 
#define S_DEV 2 
#define S_LOG 4 
#define S_NAK 8
#define S_OTH 16

/* call statistics update operations */
#define ADD 1
#define REM 2
#define CLR 4
#define UPD 8

#ifndef lint
static char *Version = "@(#) uusub 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_a = FALSE;
int opt_c = FALSE;
int opt_d = FALSE;
int opt_f = FALSE;
int opt_l = FALSE;
int opt_r = FALSE;
int opt_u = FALSE;
int opt_x = FALSE;
int c_flg = FALSE;

int errno;			/* used by errno.h */

char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char spooldir[132];		/* spool directory */
char logfile[132];		/* log file name */
char progname[20];		/* program name */
char locsite[10];		/* local sitename */
char locuser[10];		/* local user name */
char ga_site[10];		/* given site name to add */
char gr_site[10];		/* given site name to delete */
char gc_site[10];		/* given site name to call */
int totsites;			/* total number of sites in L.sys */
int pid;			/* own pid */
int g_age;			/* given age in hours */
int dbglvl;			/* debugging level */
time_t now;			/* current time */
char *cmdflds[MAXARGS];		/* pointer to parsed command fields */

struct taskrec {		/* collection of task data */
	int pid;
	char site[SITELEN + 1];
	int status;
	time_t log_time;
} *tasklist[MAXTASK];

struct siterec {		/* collection of site data */
	char site[SITELEN + 1];
	int sf_count;
	long int sb_count;
	int rf_count;
	long int rb_count;
	int all_count;
	int ok_count;
	time_t ok_time;
	int dev_count;
	int log_count;
	int nack_count;
	int rem_count;
	time_t last_time;
} *sitelist[MAXSITE];

struct msgrec { 		/* collection of (partial) log messages */
	int status;		/* and the type of error they indicate */
	unsigned int msglen;
	char *msg;
} msglist[] = {
  { 0, 0, "NULL MSG" }, /* Empty message */
  { 0, 0, "ASSERT ERROR" }, /* Assert error; full description in ERRLOG */
  { 0, 0, "QUEUED (" }, /* Command file _file_ written to spooldir */
  { S_OTH, 0, "NO CALL (RETRY TIME NOT REACHED)" }, /* Retry time not reached */
  { S_OTH, 0, "NO CALL (MAX RECALLS)" }, /* Reached maximum number of call attempts */
  { S_OTH, 0, "WRONG TIME TO CALL" }, /* Outside time limits in L.sys */
  { S_OTH, 0, "CANNOT CALL (SYSTEM STATUS)" }, /* An unexpired status file exists for the system */
  { S_OTH, 0, "LOCKED (call to" }, /* Already in contact with _sys_ */
  { S_DEV, 0, "FAILED AUTODIAL (call to" }, /* Modem on _dev_ in use */
  { S_DEV, 0, "FAILED DIRECT LINE (call to" }, /* Opening _tty#_ failed */
  { S_DEV, 0, "BAD READ" }, /* uucico cannot read/write to a device */
  { 0, 0, "OK (DIAL" }, /* Dial ok using _dev_, _pbone_number_, _dialling_time_ */
  { S_LOG, 0, "TIMEOUT (" }, /* _sys_ did not answer within set time */
  { S_LOG, 0, "BAD LOGIN/PASSWORD" }, /* Login to remote machine failed */
  { S_LOG, 0, "FAILED (call to" }, /* Call to _sys_ failed in the modem or the login script */
  { 0, 0, "SUCCEEDED (call to" }, /* Call to _sys_ succeeded */
  { S_NAK, 0, "HANDSHAKE FAILED (LCK)" }, /* A remote lock file exists for this system */
  { S_NAK, 0, "BAD SEQUENCE CHECK" }, /* SQFILE numbers do not match */
  { S_NAK, 0, "CALLBACK REQUIRED" }, /* Remote system requires callback */
  { 0, 0, "OK (startup)" }, /* Protocol matched ok */
  { 0, 0, "REQUESTED (" }, /* Work request _command_ in slave role */
  { 0, 0, "REQUEST (" }, /* Work request _command_ in master role */
  { 0, 0, "REQUEST (remote access to path/file denied)" }, /* Work request refused */
  { 0, 0, "REQUEST (cannot create TM file)" }, /* Work request refused */
  { 0, 0, "PERMISSION (DENIED)" }, /* Local access failure */
  { 0, 0, "COPY FAILED (reason not given by remote)" }, /* File transfer failed */
  { 0, 0, "COPY FAILED (cannot copy TM file)" }, /* File transfer failed */
  { 0, 0, "COPY (SUCCEEDED)" }, /* File transfer succeeded */
  { S_NAK, 0, "FAILED (conversation complete)" }, /* Call failed after successful startup */
  { S_OK, 0, "OK (conversation complete" }, /* Session ok using _dev_, _total_time_ */
  { 0, 0, "DONE (WORK HERE)" }, /* uucp local copy successful */
  { 0, 0, (char *)NULL } /* terminating value for searches - leave unchanged */
};

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

long int atoll();
int catchsignal();
int checklock();
int checksys();
void cleanup();
time_t convert();
int getargs();
int getdate();
int getname();
int getstatus();
time_t gettime();
uid_t get_uid();
int getuser();
int poll();
int readlist();
int readlog();
int readlsys();
int readsys();
int recclr();
int recinc();
int setsignal();
int syssort();
int unlock();
int update();
void usage();
int viewfile();


/*
 * a t o l l
 *
 * Convert string to positive long int
 *
 * Return:	a positive long number
 *
 * (1.5.10 Ansi/atol.c didn't work, and lint complained about portability)
 */
long int atoll(bp)
char *bp;
{
  int j;
  long int result = 0L;

  while (*bp && isspace(*bp))	/* skip leading white space */
	bp++;

  while (*bp) {
	j = 0;
	while (j < 10 && "0123456789"[j] != *bp) j++; 
	if (j == 10) break;
	result = (result << 1) + (result << 3) + j;
	bp++;
  }

  return(result);
}


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT are ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining
 * ten or so signals are scarce enough, and rare enough, to be let run.
 * One problem is that the shell protects background commands from
 * SIGINT but not SIGHUP, so if SIGHUP is ignored only SIGKILL can stop a
 * background process, leaving pieces all over the floor.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, SIG_IGN) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sig;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT, the only un-ignored signal */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c h e c k s y s
 *
 * Check if site is in L.sys, using the sorted in-memory table
 *
 * Return:	Index to siterec	If found
 *		-1		Otherwise
 *
 */
int checksys(site)
char *site;
{
  int comp, lower, middle, upper;

  lower = 0;
  upper = totsites - 1;

  while (TRUE) {
	if (lower > upper)
		return(-1);
	middle = (lower + upper) / 2;
	comp = strncmp(sitelist[middle]->site, site, SITELEN);
	if (comp < 0)
		lower = middle + 1;
	else if (comp > 0)
		upper = middle - 1;
	else
		return(middle);
  }
  /* NOTREACHED */
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files before exiting
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  /* remove all lockfiles */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0')
		(void) unlink(locklist[j]);
	j++;
  }

  exit(1);
}


/*
 * c o n v e r t
 *
 * Convert a ctime string to unixtime
 *
 * Return:	unixtime	Success
 *		0		Failure
 *
 * Warning - this is overly simplistic, ignoring localtime and DST
 */
time_t convert(string)
char *string;
{

#define SECS_PER_MIN    ((time_t)60)
#define SECS_PER_HOUR   (60*SECS_PER_MIN)
#define SECS_PER_DAY    (24*SECS_PER_HOUR)
#define SECS_PER_YEAR   (365*SECS_PER_DAY)

  static char *mths_per_year[] =
	{ "Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
  static int days_per_mth[12] =
	{ 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

  char buff[64], *timflds[MAXARGS];
  int j, year, month, date, hour, min, sec;
  time_t logtime;

  /* parse the time string */
  for (j = 0 ; j < strlen(string) ; j++) {
	if (string[j] == ':')
		buff[j] = ' ';
	else
		buff[j] = string[j];
  }
  if (getargs(buff, timflds) < 6) return((time_t)0);
  date = atoi(timflds[1]);
  hour = atoi(timflds[2]);
  min = atoi(timflds[3]);
  sec = atoi(timflds[4]);
  year = atoi(timflds[5]);

  /* convert the month name */
  for (j = 0 ; j < 12 ; j++) {
	if (strncmp(mths_per_year[j], timflds[0], 3) == SAME)
		break;
  }
  if (j == 12)
	return((time_t)0);	/* unknown month */
  else
	month = j;

  /* convert the year */
  if (year >= 70 && year <= 99)
	year -= 70;
  else if (year >= 1970 && year <= 1999)
	year -= 1970;
  else
	return((time_t)0);	/* unknown year */

  /* count seconds until the start of this year */
  logtime = (SECS_PER_YEAR * year) + (((year + 1) / 4) * SECS_PER_DAY);
  /* add a day if this year is a leap year */
  if ((year + 2) % 4)
	days_per_mth[1] = 28;
  else
	days_per_mth[1] = 29;
  /* count the seconds until the start of this month */
  for (j = 0; j < month; j++) {
	logtime += SECS_PER_DAY * days_per_mth[j];
  }
  logtime += (date - 1) * SECS_PER_DAY;
  logtime += hour * SECS_PER_HOUR;
  logtime += min * SECS_PER_MIN;
  logtime += sec;

  return(logtime);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t d a t e
 *
 * Get the date from the unixtime
 *
 * Return:	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(bp, tp) 
char *bp; time_t *tp;
{
  char buff[132];
  int j;

  sprintf(buff, "%s", ctime(tp));	/* truncate to fit into 80 cols */
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[12] = ' ';			/* lose the seconds */
  buff[13] = buff[18];			/* lose the decade */
  buff[14] = buff[19];
  buff[15] = '\0';
  strcpy(bp, buff);
  return(OK);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t s t a t u s
 *
 * Get the status of a call from a log message
 *
 * Return:	Status	Success
 *		0	Otherwise
 */
int getstatus(msg)
char *msg;
{
  int j, k;

  j = k = 0;
  while (msglist[j].msg != (char *)NULL) {
	if (strncmp(msg, msglist[j].msg, msglist[j].msglen) == SAME) {
		k =  msglist[j].status;
		break;
	}
	j++;
  }

  return(k);
}


/*
 * g e t t i m e
 *
 * Get the most recent time from the L_sub file.
 *
 * Return:	Time found	Success
 *		0		Otherwise
 *
 * File not locked.
 */
time_t gettime()
{
  char buff[BUFSIZ];
  int fd;
  time_t recent;
  struct lsubrec *lsub;

  /* if no file, then return earliest time possible */
  if (access(LSUB, 0) == -1) return((time_t)0);	/* no file, no name */

  /* unreadable file - same difference */
  if ((fd = open(LSUB, O_RDONLY, 0644)) == -1) return((time_t)0);

  /* set up to read the appropriate record layout */
  lsub = (struct lsubrec *)buff;
  recent = (time_t)0;
  while (read(fd, buff, sizeof(struct lsubrec)) == sizeof(struct lsubrec))
	if (lsub->time_a > recent)
		recent = lsub->time_a;

  (void) close(fd);

  return(recent);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp;
  int j, errflg;
  sig_t sig;
  uid_t tuid;
  time_t last;

  /* set up defaults */
  if (getname(locsite) == FAILED) {	/* get local site name */
	fprintf(stderr, "uusub: cannot find local uucpname\n");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {	/* get local user name */
	fprintf(stderr, "uusub: cannot find local username\n");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	fprintf(stderr, "uusub: cannot find own pid\n");
	exit(1);
  }
  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(logfile, LOGFILE, 131);	/* default log file name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  g_age = 1;				/* default collection time */
					/* easier to count with a computer ! */
  for (j = 0 ; msglist[j].msg != (char *)NULL ; j++)
	msglist[j].msglen = strlen(msglist[j].msg);
  dbglvl = 0;

  /* parse arguments */
  opterr = 0;
  if (argc == 1) {
	c_flg = TRUE;
  }
  else {
	while ((j = getopt(argc, argv, "a:c:d:flru:x:")) != EOF) {
		switch (j & 0177) {
		case 'a':
			strncpy(ga_site, optarg, SITELEN);
			opt_a = TRUE;
			break;
		case 'c':
			strncpy(gc_site, optarg, SITELEN);
			opt_c = TRUE;
			break;
		case 'd':
			strncpy(gr_site, optarg, SITELEN);
			opt_d = TRUE;
			break;
		case 'f':
			opt_f = TRUE;
			break;
		case 'l':
			opt_l = TRUE;
			break;
		case 'r':
			opt_r = TRUE;
			break;
		case 'u':
			g_age = atoi(optarg);
			opt_u = TRUE;
			break;
		case 'x':
			dbglvl = atoi(optarg);
			opt_x = TRUE;
			break;
		case '?':
		default:
			usage();
			exit(1);
			break;
		}
	}
  }

  /* we must have something to do */
  if (!(c_flg + opt_a + opt_c + opt_d + opt_f + opt_l + opt_r + opt_u)) {
	usage();
	exit(1);
  }
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if ((c_flg || opt_a || opt_c || opt_d || opt_f || opt_u) &&
				getuid() > MAXUID && getuid() != tuid) {
	fprintf(stderr, "uusub: option available only to administrators.\n");
	exit(1);
  }
#endif

  /* data validity */
  if (g_age < 1) {
	usage();
	exit(1);
  }

  /* get any necessary data from the files */
  if (c_flg || opt_a || opt_c || opt_u) {
	if ((totsites = readlsys()) == FAILED) {	/* get known site names */
		fprintf(stderr, "uusub: failure reading L.sys.\n");
		exit(1);
	}
  }
  if (c_flg || opt_u) {
	if (c_flg)
		last = gettime(); /* get the last time from L_sub */
	else
		last = now - (time_t)g_age * 3600L;
	last++;
	if (readsys(last) == FAILED) { /* collect from SYSLOG */
		fprintf(stderr, "uusub: failure reading SYSLOG.\n");
		exit(1);
	}
	if (readlog(last) == FAILED) { /* collect from LOGFILE */
		fprintf(stderr, "uusub: failure reading LOGFILE.\n");
		exit(1);
	}
  }

  /* set up to catch signals */
  if (catchsignal() == FAILED) {
	fprintf(stderr, "uusub: cannot catch signals\n");
	exit(1);
  }

  /* do the work */
  errflg = FALSE;

  /* disable SIGINT while the R_sub and L_sub files are moved */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	errflg = TRUE;

  if (!errflg && c_flg) {	/* update R_sub and L_sub */
	if (update(UPD) == FAILED)
		errflg = TRUE;
  }

  if (!errflg && (opt_a || opt_d)) {		/* add / delete a site */
	if (opt_a) {
		if (checksys(ga_site) != -1) {
			if (opt_d) {
				if (update(ADD | REM) == FAILED)
					errflg = TRUE;
			}
			else {
				if (update(ADD) == FAILED)
					errflg = TRUE;
			}
		}
		else
			errflg = TRUE;
	}
	else {				/* opt_d only */
		if (update(REM) == FAILED)
			errflg = TRUE;
	}
  }

  if (!errflg && (opt_f || opt_u)) {	/* update the call statistics */
	if (opt_f && !opt_u) {
		if (update(CLR) == FAILED)
			errflg = TRUE;
	}
	else  {
		if (update(CLR | UPD) == FAILED)
			errflg = TRUE;
	}
  }

  /* finished moving files, re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	errflg = TRUE;

  if (!errflg && opt_l) {		/* report the connection statistics */
	if (viewfile(LSUB) == FAILED)
		errflg = TRUE;
  }

  if (!errflg && opt_r) {		/* report the traffic statistics */
	if (viewfile(RSUB) == FAILED)
		errflg = TRUE;
  }

  if (!errflg && opt_c) {		/* poll site(s) */
	if (strncmp("all", gc_site, 3) != SAME) {
		if (checksys(gc_site) == -1 || poll(gc_site) == FAILED)
			errflg = TRUE;
	}
	else {
		/* clear out the L.sys site list */
		j = 0;
		while (j < MAXSITE && sitelist[j] != (struct siterec *)NULL) {
			(void) free((void *)sitelist[j]);
			sitelist[j] = (struct siterec *)NULL;
			j++;
		}
		/* build a new list of sites from R_sub */
		if ((totsites = readlist()) == -1)
			errflg = TRUE;
		j = 0;
		while (!errflg && 
			j < MAXSITE && sitelist[j] != (struct siterec *)NULL) {
			if (poll(sitelist[j]->site) == FAILED) {
				errflg = TRUE;
				break;
			}
			j++;
		}
	}
  }

  exit(0);
  /* NOTREACHED */
}


/*
 * p o l l
 *
 * Run uucico to poll a site, and wait for the response
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int poll(site)
char *site;
{
  int j, k, status;

  switch (j = fork()) {
	case -1:		/* can't create new process */
		break;
	case 0:			/* forked, try to execute uucico */
		(void) fflush(stderr);
		(void) freopen("/dev/null", "w", stderr);
		(void) execlp(UUCICO, "uucico", "-r1", "-s", site,
							(char *)NULL);
		exit(1);	/* never happen */
		break;
	default:		/* uucico running, wait for status */
		while ((k = wait(&status)) != j && k != -1)
			;
		break;
  }

  if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 || WTERMSIG(status) != 0)
	return(FAILED);
  else
	return(OK);
}


/*
 * r e a d l i s t
 *
 * Read a list of site names from R_sub
 *
 * Return:	Number of names		Success
 *		-1			Otherwise
 */
int readlist()
{
  char *bp;
  int j, fd;
  size_t len;
  struct stat statbuf;

  if (access(RSUB, 0) == -1) return (FAILED);	/* no file, don't bother */

  /* Open the file and check it for data */
  if ((fd = open(RSUB, O_RDONLY, 0600)) == -1) {
	return(FAILED);
  }
  if (fstat(fd, &statbuf) != 0 || statbuf.st_size == (off_t)0) {
	(void) close(fd);
	return(FAILED);
  }

  /* loop through the file reading the data into the sitelist array */
  j = 0;
  len = sizeof(struct siterec);
  bp = (char *) malloc(len);
  while (j < MAXSITE && read(fd, bp, len) == len) {
		sitelist[j++] = (struct siterec *)bp;
		bp = (char *) malloc(len);
  }
  (void) free((void *)bp);

  (void) close(fd);

  return(j);
}


/*
 * r e a d l o g
 *
 * Read the LOGFILE and collect data, starting at time start.
 *
 * Return:	OK	Success
 *		FAILED	Cannot open LOGFILE
 *
 * File not locked.
 */
int readlog(start)
time_t start;
{
  char buff[BUFSIZ], tmp[256], msg[132], tbuff[132], site[10];
  int j, k, taskpid, tasknum, taskstatus;
  time_t logtime;
  FILE *fplog;

  /* allocate storage for the data */
  if ((fplog = fopen(LOGFILE, "r")) == (FILE *)NULL) return(FAILED);
  
  /* log format: site!user (date time) (id pid task) status (detail) */

  /* scan through the log file */
  j = 0;
  while (fgets(buff, BUFSIZ, fplog) != (char *)NULL) {
	for (j = 0 ; j < 10 ; j++) {
		if (buff[j] == '!') buff[j] = ' ';
	}
 	strncpy(tmp, buff, 255);	/* getargs roaches the string */
	if (getargs(tmp, cmdflds) < 10) continue;	/* garbage ? */
	*(cmdflds[5] + strlen(cmdflds[5]) - 1) = ' ';	/* clear trailing ) */
	*(cmdflds[8] + strlen(cmdflds[8]) - 1) = ' ';
	strncpy(msg, buff + (cmdflds[9] - cmdflds[0]), 131);
	sprintf(tbuff, "%s %s %s %s", cmdflds[2] + 1, cmdflds[3],
				cmdflds[4], cmdflds[5]);
	logtime = convert(tbuff);
	if (logtime < start) continue;	/* not wanted on voyage */
	strncpy(site, cmdflds[0], SITELEN);
	taskpid = atoi(cmdflds[7]);
	tasknum = atoi(cmdflds[8]);
	taskstatus = getstatus(msg);

	/* look for this task */
	for (j = 0 ; j < MAXTASK && tasklist[j] != (struct taskrec *)NULL ; j++) {
		if (tasklist[j]->pid == taskpid) {
			if (taskstatus != 0) {
				tasklist[j]->status = taskstatus;
				tasklist[j]->log_time = logtime;
			}
			break;
		}
	}
	/* task pid not found, so add a new task */
	if (j < MAXTASK && tasklist[j] == (struct taskrec *)NULL) {
		tasklist[j] = (struct taskrec *) malloc(sizeof(struct taskrec));
		tasklist[j]->pid = taskpid;
		strncpy(tasklist[j]->site, site, SITELEN);
		tasklist[j]->site[SITELEN] = '\0';
		tasklist[j]->status = taskstatus;
		tasklist[j]->log_time = logtime;
	}
	/* task completed, so update the site list */
	if (j < MAXTASK && tasklist[j]->status == S_OK) {
		if ((k = checksys(site)) != -1) {
			sitelist[k]->all_count++;
			sitelist[k]->ok_count++;
			sitelist[k]->ok_time = tasklist[j]->log_time;
			sitelist[k]->last_time = tasklist[j]->log_time;
		}
		/* remove the completed task from the task list */
		/* ignore the sites that aren't in L.sys */
		free((void *)tasklist[j]);
		for (k = j ; k < (MAXTASK - 1) ; k++ ) {
			tasklist[k] = tasklist[k + 1];
			if (tasklist[k] == (struct taskrec *)NULL)
				break;
		}
		tasklist[MAXTASK - 1] = (struct taskrec *)NULL;
 	}
  }

  /* at EOF, some tasks will be uncompleted - add them to the list */
  for (j = 0 ; j < MAXTASK && tasklist[j] != (struct taskrec *)NULL ; j++) {
	if ((k = checksys(tasklist[j]->site)) != -1 && tasklist[j]->status != 0) {
		if (tasklist[j]->log_time > sitelist[k]->last_time)
			sitelist[k]->last_time = tasklist[j]->log_time;
		sitelist[k]->all_count++;
		switch (tasklist[j]->status) {
			case(S_DEV):
				sitelist[k]->dev_count++;
				break;
			case(S_LOG):
				sitelist[k]->log_count++;
				break;
			case(S_NAK):
				sitelist[k]->nack_count++;
				break;
			default:
				sitelist[k]->rem_count++;
				break;
		}
	}
	free((void *) tasklist[j]);
	tasklist[j] = (struct taskrec *)NULL;
  }

  (void) fclose(fplog);
  return(j);
}


/*
 * r e a d l s y s
 *
 * Read the L.sys file and store the names in memory.
 *
 * Return:	FAILED	Cannot open L.sys / cannot malloc memory
 *		#	Number of names found
 *
 * File not locked.
 */
int readlsys()
{
  char buff[BUFSIZ], cont[132], name[10];
  int j, k;
  FILE *fplsys;

  /* allocate storage for the data */
  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);
  j = 0;
  while (j < MAXSITE && fgets(buff, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(buff)) < (BUFSIZ - 132) &&
			buff[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		buff[k - 2] = '\0';
		strcat(buff, cont);
	}
	if (buff[0] == '#' || buff[0] == '\n') continue;
	for (k = 0 ; k < SITELEN ; k++) {
		name[k] = buff[k];
		if (name[k] == ' ' || name[k] == '\t') {
			name[k] = '\0';
			break;
		}
	}
	name[k] = '\0';

	sitelist[j] = (struct siterec *) malloc(sizeof(struct siterec));
	if (sitelist[j] == (struct siterec *)NULL) {
		(void) fclose(fplsys);
		return(FAILED);
	}
	strncpy(sitelist[j]->site, name, SITELEN);
	sitelist[j]->site[SITELEN] = '\0';
	sitelist[j]->sf_count = 0;
	sitelist[j]->sb_count = 0L;
	sitelist[j]->rf_count = 0;
	sitelist[j]->rb_count = 0L;
	sitelist[j]->all_count = 0;
	sitelist[j]->ok_count = 0;
	sitelist[j]->ok_time = (time_t)0;
	sitelist[j]->dev_count = 0;
	sitelist[j]->log_count = 0;
	sitelist[j]->nack_count = 0;
	sitelist[j]->rem_count = 0;
	sitelist[j]->last_time = (time_t)0;

	j++;
  }
  (void) fclose(fplsys);

  (void) syssort(j); /* helps to speed name lookup */

  return(j);
}


/*
 * r e a d s y s 
 *
 * Read the SYSLOG and collect data, starting at time start.
 *
 * Return:	OK	Success
 *		FAILED	Cannot open SYSLOG
 *
 * File not locked.
 */
int readsys(start)
time_t start;
{
  char buff[BUFSIZ], tbuff[132], dirn[16], site[16];
  int j, tasknum;
  long int bytes;
  time_t logtime;
  FILE *fpsys;

  /* allocate storage for the data */
  if ((fpsys = fopen(SYSLOG, "r")) == (FILE *)NULL) return(FAILED);
  
  /* log format: site!user status (date time) (id pid task) direction \
   * bytes / secs:ticks [proto] dev
   */
  j = 0;
  while (fgets(buff, BUFSIZ, fpsys) != (char *)NULL) {
	for (j = 0 ; j < 10 ; j++) {
		if (buff[j] == '!') buff[j] = ' ';
	}
	if (getargs(buff, cmdflds) < 15) continue;
	*(cmdflds[6] + strlen(cmdflds[6]) - 1) = ' ';	/* clear trailing ) */
	*(cmdflds[9] + strlen(cmdflds[9]) - 1) = ' ';
	sprintf(tbuff, "%s %s %s %s", cmdflds[3] + 1, cmdflds[4],
				cmdflds[5], cmdflds[6]);
	logtime = convert(tbuff);
	if (logtime < start) continue;
	strncpy(site, cmdflds[0], SITELEN);
	tasknum = atoi(cmdflds[9]);
	strncpy(dirn, cmdflds[10], 2);
	bytes = atoll(cmdflds[11]);

	if ((j = checksys(site)) != -1) {
		if (strncmp(dirn, "->", 2) == SAME) {
			sitelist[j]->sf_count++;
			sitelist[j]->sb_count += bytes;
		}
		else {
			sitelist[j]->rf_count++;
			sitelist[j]->rb_count += bytes;
		}
	}
  }
  (void) fclose(fpsys);

  return(j);
}


/*
 * r e c c l r
 *
 * Zero an R_sub and L_sub record pair
 *
 * Return:	OK	Always
 *
 */
int recclr(rsub, lsub)
struct rsubrec *rsub; struct lsubrec *lsub;
{
  rsub->sfile = 0;
  rsub->sbyte = 0L;
  rsub->rfile = 0;
  rsub->rbyte = 0L;
  lsub->all_calls = 0;
  lsub->ok_calls = 0;
  lsub->time_a = (time_t)0;
  lsub->time_b = (time_t)0;
  lsub->dev_calls = 0;
  lsub->log_calls = 0;
  lsub->nack_calls = 0;
  lsub->rem_calls = 0;

  return(OK);
}


/*
 * r e c i n c
 *
 * Add current data to an R_sub and L_sub record pair
 *
 * Return:	OK	Always
 *
 */
int recinc(rsub, lsub, site)
struct rsubrec *rsub; struct lsubrec *lsub; struct siterec *site;
{
  rsub->sfile += site->sf_count;
  rsub->sbyte += site->sb_count;
  rsub->rfile += site->rf_count;
  rsub->rbyte += site->rb_count;
  lsub->all_calls += site->all_count;
  lsub->ok_calls += site->ok_count;
  if (site->last_time > lsub->time_a)
	lsub->time_a = site->last_time;
  if (site->ok_time > lsub->time_b)
	lsub->time_b = site->ok_time;
  lsub->dev_calls += site->dev_count;
  lsub->log_calls += site->log_count;
  lsub->nack_calls += site->nack_count;
  lsub->rem_calls += site->rem_count;

  return(OK);
}


/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/* 
 * s y s s o r t
 *
 * Sort sitelist[] by site name
 *
 * Return:	OK	Always
 */
int syssort(total)
int total;
{
  int j, k, gap;
  struct siterec *tp;

  /* shell sort */
  gap = 1;
  while (gap <= total) gap = 3 * gap + 1;

  for (gap /= 3 ; gap > 0 ; gap /= 3 ) {
 	for (j = gap ; j < total ; j++) { 
		tp = sitelist[j];
		for (k = j ;
		k >= gap &&
		strncmp(sitelist[k - gap]->site, tp->site, SITELEN) > 0 ;
		k -= gap) {
			sitelist[k] = sitelist[k - gap];
		}
		sitelist[k] = tp;
	}
  }

  return(OK);
}


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/* 
 * u p d a t e
 *
 * Update the R_sub and L_sub files
 * 
 * Return:	OK	Success
 *		FAILED	Otherwise 
 */
int update(task)
int task;
{
  char name[10], trfile[132], tlfile[132], tbuff[132], rbuff[BUFSIZ], lbuff[BUFSIZ];
  int j, k, errflg, done, rfd, trfd, lfd, tlfd;
  size_t rlen, llen;
  struct stat rstatbuf;
  struct stat lstatbuf;
  struct rsubrec *rsub;
  struct lsubrec *lsub;

  rsub = (struct rsubrec *)rbuff;
  rlen = sizeof(struct rsubrec);
  lsub = (struct lsubrec *)lbuff;
  llen = sizeof(struct lsubrec);

  /* easier to stop now rather than later */
  if (!(task & ADD) && !(task & REM) && !(task & CLR) && !(task & UPD))
	return(FAILED);

  /* if the files don't exist, we can only add to them */
  if ((access(RSUB, 0) == -1 || access(LSUB, 0) == -1) &&
  ((task & ADD) == 0 || creat(RSUB, 0644) == -1 || creat(LSUB, 0644) == -1))
	return(FAILED);

  /* some sort of open failure - nothing to be done */
  if ((rfd = open(RSUB, O_RDONLY, 0644)) == -1  ||
		(lfd = open(LSUB, O_RDONLY, 0644)) == -1 ||
		fstat(rfd, &rstatbuf) != 0 ||
		fstat(lfd, &lstatbuf) != 0) {
	(void) close(rfd);
	(void) close(lfd);
	return(FAILED);
 }
 (void) close(rfd);
 (void) close(lfd);

 /* if the file is zero length, we can only add names */
 if (task != ADD && (rstatbuf.st_size == (off_t)0 || lstatbuf.st_size == (off_t)0))
	return(FAILED);

  /* try to lock the files */
  if (checklock(LCKRSUB, 2) == FAILED)
	return(FAILED);
  if (checklock(LCKLSUB, 2) == FAILED) {
	(void) unlock(LCKRSUB);
	return(FAILED);
  }

  errflg = FALSE;
  /* move R_sub and L_sub to temporary files */
  sprintf(trfile, "%sR%04d", LTMPATH, pid);
  (void) unlink(trfile);	/* in case it already exists */
  if (link(RSUB, trfile) == -1 || unlink(RSUB) == -1) {
	(void) unlink(trfile);
	(void) unlock(LCKRSUB);
	(void) unlock(LCKLSUB);
	return(FAILED);
  }
  sprintf(tlfile, "%sL%04d", LTMPATH, pid);
  (void) unlink(tlfile);
  if (link(LSUB, tlfile) == -1 || unlink(LSUB) == -1) {
	(void) unlink(tlfile);
	(void) link(trfile, RSUB);	/* recover if possible */
	(void) unlink(trfile);
	(void) unlock(LCKRSUB);
	(void) unlock(LCKLSUB);
	return(FAILED);
  }

  /* open the two output files and the two temporary files */
  if ((rfd = open(RSUB, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(trfd = open(trfile, O_RDONLY, 0644)) == -1 ||
		(lfd = open(LSUB, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(tlfd = open(tlfile, O_RDONLY, 0644)) == -1) {
	errflg = TRUE;
  }

  /* read the two files in parallel */
  done = FALSE;
  while (!errflg && (j = read(trfd, rbuff, rlen)) != -1 &&
				(k = read(tlfd, lbuff, llen)) != -1) {
	if (j == 0 && k == 0) {		/* new stuff may need adding at EOF */
		if ((task & ADD) && !done) {
			for (j = 0 ; j < 132 ; j++) tbuff[j] = '\0';
			/* write out the zeroed record */
			strncpy(tbuff, ga_site, SITELEN);
			if (write(rfd, tbuff, rlen) != rlen ||
					write(lfd, tbuff, llen) != llen) {
				errflg = TRUE;
			}
		}
		break;
	}
	if (strncmp(rsub->site, lsub->site, SITELEN) != SAME) { /* corrupt R/L_sub */
		errflg = TRUE;
		break;
	}
	strncpy(name, rsub->site, SITELEN);
	if ((task & ADD) || (task & REM)) {	/* change a single record */
		if ((task & REM) && strncmp(name, gr_site, SITELEN) == SAME) {
			continue;		/* ignore this record */
		}
		if (!done && (task & ADD) && (j = strncmp(name, ga_site, SITELEN)) >= 0) {
			if (j == SAME) {	/* entry already in file */
				errflg = TRUE;
				break;
			}
			for (j = 0 ; j < 132 ; j++) tbuff[j] = '\0';
			strncpy(tbuff, ga_site, SITELEN);
			if (write(rfd, tbuff, rlen) != rlen ||
					write(lfd, tbuff, llen) != llen) {
				errflg = TRUE;
				break;
			}
			done = TRUE;
		}
	}
	else if ((task & CLR) || (task & UPD)) {/* change all records */
		if (task & CLR) {
			(void) recclr(rsub, lsub);
		}
		if (task & UPD) {
			if ((k = checksys(name)) != -1) {
				(void) recinc(rsub, lsub, sitelist[k]);
			}
			/* ignore sites found in LOGFILE but not in L.sys */
		}
	}
	else {					/* never happen */
		errflg = TRUE;
		break;
	}

	/* write out the changed record */
	if (!errflg && write(rfd, rbuff, rlen) != rlen ||
				write(lfd, lbuff, llen) != llen) {
		errflg = TRUE;
		break;
	}
	
  }

  /* wrap up */
  (void) close(rfd);
  (void) close(trfd);
  (void) close(lfd);
  (void) close(tlfd);
  if (errflg) {			/* recover if possible */
	(void) unlink(RSUB);
	(void) link(trfile,RSUB);
	(void) unlink(LSUB);
	(void) link(tlfile,LSUB);
  }
  (void) unlink(trfile);
  (void) unlink(tlfile);
  (void) unlock(LCKRSUB);
  (void) unlock(LCKLSUB);
  if (errflg)
	return(FAILED);
  else
	return(OK);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-a sys] [-c sys] [-d sys] [-flr] [-u hrs]\n",
	progname);
}


/*
 * v i e w f i l e
 *
 * View a sub file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int viewfile(file)
char *file;
{
  char buff[BUFSIZ], time_b[64], site[10];
  int j, fd, choice;
  size_t len;
  struct stat statbuf;
  struct rsubrec *rsub;
  struct lsubrec *lsub;

  if (access(file, 0) == -1) return (FAILED);	/* no file, don't bother */

  /* Open  the file and check it for data */
  if ((fd = open(file, O_RDONLY, 0600)) == -1) {
	return(FAILED);
  }

  if (fstat(fd, &statbuf) != 0 || statbuf.st_size == (off_t)0) {
	(void) close(fd);
	return(FAILED);
  }

  /* set up to read the appropriate record layout */
  if (strcmp(file, RSUB) == SAME) {
	choice = 3;
	rsub = (struct rsubrec *)buff;
	len = sizeof(struct rsubrec);
	printf("%s", "site     sfile  sbyte      rfile  rbyte\n");
  }
  else if (strcmp(file, LSUB) == SAME) {
	choice = 4;
	lsub = (struct lsubrec *)buff;
	len = sizeof(struct lsubrec);
	printf("%s", "site     #calls #ok   ok_time          ");
	printf("#dev  #login #nack #other\n");
  }
  else
	return(FAILED);	/* unknown file */

  /* R_sub format: site files_sent bytes_sent files_received bytes_received
   * L_sub format: site all_count ok_count ok_time 4*failure_modes
   */

  /* loop through the file printing the data */
  while ((j = read(fd, buff, len)) != 0 && j != -1) {
	switch (choice) {
	case 3:		/* R_sub file */
		strncpy(site, rsub->site, SITELEN);
		printf("%-8s %-6d %-10ld %-6d %-10ld\n",
			site, rsub->sfile, rsub->sbyte,
			rsub->rfile, rsub->rbyte);
		break;
	case 4:		/* L_sub file */
		strncpy(site, lsub->site, SITELEN);
		(void) getdate(time_b, &lsub->time_b);
		printf("%-8s %-4d   %-4d  %s  %-4d  %-4d   %-4d  %-4d\n",
			site, lsub->all_calls,
			lsub->ok_calls, time_b,
			lsub->dev_calls, lsub->log_calls,
			lsub->nack_calls, lsub->rem_calls);
		break;
	default:	/* never happen */
		break;
	}
  }

  (void) close(fd);

  return(OK);
}

