/* uustat - UUCP status enquiry and job control.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#ifdef _MINIX
#include <utime.h>
#endif
#include "uucp.h"

#define FALSE		0
#define TRUE		~FALSE
#define OK		1
#define FAILED		-1
#define SAME		0
#define TOUCHED		0
#define KILLED 		1
#define MAXLOCKS	10
#define MAXARGS		30
#define MAXSYS  	100

#ifndef lint
static char *Version = "@(#) uustat 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_c = FALSE;
int opt_j = FALSE;
int opt_k = FALSE;
int opt_m = FALSE;
int opt_M = FALSE;
int opt_o = FALSE;
int opt_p = FALSE;
int opt_s = FALSE;
int opt_q = FALSE;
int opt_r = FALSE;
int opt_u = FALSE;
int opt_x = FALSE;
int opt_y = FALSE;

int errno;			/* used by errno.h */

char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char spooldir[132];		/* spool directory */
char logfile[132];		/* log file name */
char progname[20];		/* program name */
char locsite[10];		/* local sitename */
char locuser[10];		/* local user name */
char g_user[10];		/* given user name */
char g_site[10];		/* given site name */
char gs_site[10];		/* given site for site status */
int totsites;			/* total number of sites */
int jobnum;			/* current job number */
int pid;			/* own pid */
int g_job;			/* given job number */
int gk_job;			/* given job number to kill */
int gr_job;			/* given job number to rejuvenate */
int g_age;			/* given age in hours */
int gc_age;			/* given clearing age in hours */
int dbglvl;			/* debugging level */
time_t now;			/* current time */

/* Standard call and task status messages */

char *errlist[] = {
	"NULL MESSAGE",
	"STATUS UNKNOWN: SYSTEM ERROR",
	"LOCAL CAN'T CREATE TEMP FILE",
	"CAN'T COPY TO LOCAL DIRECTORY",
	"LOCAL ACCESS TO FILE DENIED",
	"REMOTE CAN'T CREATE TEMP FILE",
	"CAN'T COPY TO REMOTE DIRECTORY",
	"REMOTE ACCESS TO FILE DENIED",
	"BAD UUCP COMMAND GENERATED",
	"CAN'T EXECUTE UUCP",
	"JOB IS QUEUED",
	"COPY FAILED",
	"COPY (PARTIALLY) SUCCEEDED",
	"COPY FINISHED, JOB DELETED",
	"DONE (WORK HERE)",
	"JOB KILLED (INCOMPLETE)",
	"JOB KILLED (COMPLETE)"
};

struct siterec {			/* collection of site data */
	char site[SITELEN + 1];
	int job_count;
	int data_count;
	time_t first_time;
	time_t last_time;
	time_t lock_time;
} *sitelist[MAXSYS];

int (*comp)();			/* pointer to comparison function */

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

long int atoll();
int catchsignal();
int checklock();
int checksys();
void cleanup();
int clearfile();
int getargs();
int getdate();
int getmsg();
int getname();
uid_t get_uid();
int getuser();
long int htol();
int killdata();
int readlsys();
int runps();
int scandir();
int setsignal();
int ston();
int syssort();
int touch();
int unlock();
int update();
void usage();
int viewdir();
int viewfile();

/* Match functions */
int lok_match();
int lsys_match();
int rjob_match();
int rpost_match();
int rpre_match();
int rok_match();
int rsys_match();
int ruser_match();

/* ********** comparison routines for selecting stat file entries ********** */

/*
 * l o k _ m a t c h 
 *
 * Match everything in L_stat
 *
 * Return:      TRUE    always
 */
int lok_match(lstat)
struct lstatrec *lstat;
{
  return(TRUE);
}


/*
 * l s y s _ m a t c h
 *
 * Match the given site name with that in L_stat
 *
 * Return:      TRUE    on match
 *              FALSE   otherwise
 */
int lsys_match(lstat)
struct lstatrec *lstat;
{
  if (strncmp(gs_site, lstat->site, SITELEN) == SAME)
	return(TRUE);
  else
	return(FALSE);
}


/*
 * r j o b _ m a t c h
 *
 * Match the given job number with that in R_stat
 *
 * Return:      TRUE    on match
 *              FALSE   otherwise
 */
int rjob_match(rstat)
struct rstatrec *rstat;
{
  if (g_job == rstat->job)
	return(TRUE);
  else
	return(FALSE);
}


/*
 * r o k _ m a t c h 
 *
 * Match everything in R_stat
 *
 * Return:      TRUE    always
 */
int rok_match(rstat)
struct rstatrec *rstat;
{
  return(TRUE);
}


/*
 * r p o s t _ m a t c h
 *
 * Match the given time with all times later in R_stat
 *
 * Return:      TRUE    on match
 *              FALSE   otherwise
 */
int rpost_match(rstat)
struct rstatrec *rstat;
{
  if (rstat->time_a >= (time_t)(now - (long)g_age * 3600L))
	return(TRUE);
  else
	return(FALSE);
}


/*
 * r p r e _ m a t c h
 *
 * Match the given time with all times earlier in R_stat
 *
 * Return:      TRUE    on match
 *              FALSE   otherwise
 */
int rpre_match(rstat)
struct rstatrec *rstat;
{
  if (rstat->time_a < (time_t)(now - (long)g_age * 3600L))
	return(TRUE);
  else
	return(FALSE);
}


/*
 * r s y s _ m a t c h
 *
 * Match the given site name with that in R_stat
 *
 * Return:      TRUE    on match
 *              FALSE   otherwise
 */
int rsys_match(rstat)
struct rstatrec *rstat;
{
  if (strncmp(g_site, rstat->site, SITELEN) == SAME)
	return(TRUE);
  else
	return(FALSE);
}


/*
 * r u s e r _ m a t c h
 *
 * Match the given user name with that in R_stat
 *
 * Return:      TRUE    on match
 *              FALSE   otherwise
 */
int ruser_match(rstat)
struct rstatrec *rstat;
{
  if (strncmp(g_user, rstat->user, SITELEN) == SAME)
	return(TRUE);
  else
	return(FALSE);
}

/* *********************** end of comparison routines *********************** */


/*
 * a t o l l
 *
 * Convert string to positive long int
 *
 * Return:	a positive long number
 *
 * (1.5.10 Ansi/atol.c didn't work, and lint complained about portability)
 */
long int atoll(bp)
char *bp;
{
  int j;
  long int result = 0L;

  while (*bp && isspace(*bp))	/* skip leading white space */
	bp++;

  while (*bp) {
	j = 0;
	while (j < 10 && "0123456789"[j] != *bp) j++; 
	if (j == 10) break;
	result = (result << 1) + (result << 3) + j;
	bp++;
  }

  return(result);
}


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT are ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining
 * ten or so signals are scarce enough, and rare enough, to be let run.
 * One problem is that the shell protects background commands from
 * SIGINT but not SIGHUP, so if SIGHUP is ignored only SIGKILL can stop a
 * background process, leaving pieces all over the floor.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, SIG_IGN) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sig;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT, the only un-ignored signal */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c h e c k s y s
 *
 * Check if site is in L.sys, using the sorted in-memory table.
 *
 * Return:	Sysrec index	If found
 *		-1		Otherwise
 *
 */
int checksys(site)
char *site;
{
  int cmp, lower, middle, upper;

  lower = 0; upper = totsites - 1;

  while (TRUE) {
	if (lower > upper)
		return(-1);
	middle = (lower + upper)/2;
	cmp = strncmp(sitelist[middle]->site, site, SITELEN);
	if (cmp < 0)
		lower = middle + 1;
	else if (cmp > 0)
		upper = middle - 1;
	else
		return(middle);
  }
  /* NOTREACHED */
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files before exiting
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  /* remove all lockfiles */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0')
		(void) unlink(locklist[j]);
	j++;
  }

  exit(1);
}


/*
 * c l e a r f i l e
 *
 * Remove status entries older than # hours
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int clearfile(filename)
char *filename;
{
  char tfile[132], lockfile[132], buff[BUFSIZ];
  int fds, fdt, choice;
  size_t len;
  time_t then;
  struct rstatrec *rstat;
  struct lstatrec *lstat;

  /* set up to read the appropriate record layout */
  if (strcmp(filename, RSTAT) == SAME) {
	choice = 1;
	rstat = (struct rstatrec *)buff;
	len = sizeof(struct rstatrec);
	strncpy(lockfile, LCKRSTAT, 131);
  }
  else if (strcmp(filename, LSTAT) == SAME) {
	choice = 2;
	lstat = (struct lstatrec *)buff;
	len = sizeof(struct lstatrec);
	strncpy(lockfile, LCKLSTAT, 131);
  }
  else
	return(FAILED);					/* unknown file */

  if (access(filename, 0) == -1) return(FAILED);	/* no file, no name */
  if (checklock(lockfile, 2) == FAILED) return(FAILED); /* lock not possible */

  /* move R_stat or L_stat to a temporary file */
  sprintf(tfile, "%s%c%04d", LTMPATH, (choice == 1) ? 'R' : 'L', pid);
  (void) unlink(tfile);		/* in case it already exists */
  if (link(filename, tfile) == -1 || unlink(filename) == -1) {
	(void) unlink(tfile);
	(void) unlock(lockfile);
	return(FAILED);
  }

  fds = fdt = 0;
  if ((fds = open(filename, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(fdt = open(tfile, O_RDONLY, 0644)) == -1) {
	(void) link(tfile, filename); /* recover if possible */
	(void) unlink(tfile);
	if (fds != -1) (void) close(fds);
	if (fdt != -1) (void) close(fdt);
	(void) unlock(lockfile);
	return(FAILED);
  }
  (void) unlink(tfile);	

  then = now - (long)gc_age * 3600L;
  while (read(fdt, buff, len) == len) {
	if  ((choice == 1 && rstat->time_a > then) ||
			(choice == 2  && lstat->time_a > then)) {
		/* write only the recent records */
		if (write(fds, buff, len) != len) {
			(void) close(fds);	/* do not unlock - file corrupt */
			(void) close(fdt);
			return(FAILED);
		}
	}
  }

  (void) close(fds);
  (void) close(fdt);
  (void) unlock(lockfile);
  return(OK);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t d a t e
 *
 * Get the date from the unixtime
 *
 * Return:	OK	Always
 */
int getdate(bp, tp) 
char *bp; time_t *tp;
{
  char buff[132];
  int j;

  sprintf(buff, "%s", ctime(tp));	/* truncate to fit into 80 cols */
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[12] = ' ';			/* lose the seconds */
  buff[13] = buff[18];			/* lose the decade */
  buff[14] = buff[19];
  buff[15] = '\0';
  strcpy(bp, buff);
  return(OK);
}


/*
 * g e t m s g
 *
 * Get the appropriate message given the status number
 *
 * Return:	An index to errlist[]
 */
int getmsg(status)
int status;
{
  int j, k;

  if (status == 0) return(OK_MSG);

  j = 1;
  k = (sizeof(errlist) / sizeof(char *));
  while (j < k && (status & 1) == 0) {
	status >>= 1;
	j++;
  }
  if (j == k)
	j = ERR_MSG;

  return(j);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * h t o l
 *
 * Convert hex string to positive long int
 *
 * Return:	a positive long number
 */
long int htol(bp)
char *bp;
{
  int j;
  long int result = 0L;

  while (*bp && isspace(*bp))	/* skip leading white space */
	bp++;

  while (*bp) {
	j = 0;
	while (j < 22 && "0123456789ABCDEFabcdef"[j] != *bp) j++; 
	if (j == 22) break;
	if (j > 15) j -= 6;
	result = (result << 4) + j;
	bp++;
  }

  return(result);
}


/*
 * k i l l d a t a
 *
 * Remove the datafiles associated with a command file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int killdata(cmdname)
char *cmdname;
{
  char dataname[132], buff[132], scratch[132];
  FILE *fp;

  /*
   * Command file format:
   * S /tmp/uutest/test04 ~/cwr cwr -cd D.0 0666 cwr  
   */

  if ((fp = fopen(cmdname, "r")) == (FILE *)NULL)
	return(FAILED);

  while (fgets(buff, 132, fp) != (char *)NULL) {
	(void) sscanf(buff, "%*s%*s%*s%*s%*s%s", scratch);
	if (strcmp(scratch, NODATAFN) != SAME) {
		strncpy(dataname, SPOOLDIR, 131);
		strcat(dataname, "/");
		strncat(dataname, scratch, 131 - strlen(SPOOLDIR));
		(void) unlink(dataname);
	}
  }
  (void) fclose(fp);
  return(OK);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp;
  int j, errflg;
  uid_t tuid;
  sig_t sig;

  /* set up defaults */
  if (getname(locsite) == FAILED) {	/* get local site name */
	fprintf(stderr, "uustat: cannot find local uucpname\n");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {	/* get local user name */
	fprintf(stderr, "uustat: cannot find local username\n");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	fprintf(stderr, "uustat: cannot find own pid\n");
	exit(1);
  }
  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(logfile, LOGFILE, 131);	/* default log file name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  (void) getuser(getuid(), g_user);	/* default user name */
  strcpy(g_site, "all");		/* default site name */
  g_job = gk_job = gr_job = 1;
  g_age = gc_age = 1;
  dbglvl = 0;

  /* parse arguments */
  opterr = 0;
  if (argc == 1) {	/* with no options, report on current user */
	opt_u = TRUE;
	comp = ruser_match;
  }
  else
	while ((j = getopt(argc, argv, "c:j:k:m:M:o:ps:qr:u:x:y:")) != EOF)
		switch (j & 0177) {
		case 'c':
			gc_age = atoi(optarg);
			opt_c = TRUE;
			break;
		case 'k':
			gk_job = atoi(optarg);
			opt_k = TRUE;
			break;
		case 'j':
			if (strcmp(optarg, "all") == SAME) {
				comp = rok_match;
			}				
			else {
				g_job = atoi(optarg);
				comp = rjob_match;
			}
			opt_j = TRUE;
			break;
		case 'm':
			if (strcmp(optarg, "all") == SAME) {
				comp = lok_match;
			}				
			else {
				strncpy(gs_site, optarg, SITELEN);
				comp = lsys_match;
			}
			opt_m = TRUE;
			break;
		case 'M':
			if (strcmp(optarg, "all") == SAME) {
				comp = lok_match;
			}				
			else {
				strncpy(gs_site, optarg, SITELEN);
				comp = lsys_match;
			}
			opt_M = TRUE;
			break;
		case 'o':
			g_age = atoi(optarg);
			comp = rpre_match;
			opt_o = TRUE;
			break;
		case 'p':
			opt_p = TRUE;
			break;
		case 's':
			strncpy(g_site, optarg, SITELEN);
			comp = rsys_match;
			opt_s = TRUE;
			break;
		case 'q':
			opt_q = TRUE;
			break;
		case 'r':
			gr_job = atoi(optarg);
			opt_r = TRUE;
			break;
		case 'u':
			strncpy(g_user, optarg, USERLEN);
			comp = ruser_match;
			opt_u = TRUE;
			break;
		case 'x':
			dbglvl = atoi(optarg);
			opt_x = TRUE;
			break;
		case 'y':
			g_age = atoi(optarg);
			comp = rpost_match;
			opt_y = TRUE;
			break;
		case '?':
			if ((optopt & 0177) == 'j') { /* default to current user */
				opt_u = TRUE;
				comp = ruser_match;
				break;
			}
			/* else fall through */
		default:
			usage();
			exit(1);
			break;
		}

  /* validity checks */
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (opt_c && getuid() > MAXUID && getuid() != tuid) {
	fprintf(stderr, "uustat: option available only to administrators.\n");
	exit(1);
  }
#endif
  /* at most one of these options must be true */
  if ((j = (opt_j + opt_o + opt_s + opt_u + opt_y)) != TRUE && j != FALSE) {
	usage();
	exit(1);
  }
  if (opt_m && opt_M) {
	usage();
	exit(1);
  }
  if (g_job < 1 || g_job > 10000 ||
		gk_job < 1 || gk_job > 10000 ||
		gr_job < 1 || gr_job > 10000) {
	usage();
	exit(1);
  }
  if (g_age < 1 || gc_age < 0) {
	usage();
	exit(1);
  }

  /* set up to catch signals */
  if (catchsignal() == FAILED)
	fprintf(stderr, "uustat: cannot catch signals\n");

  /* do the work */
  errflg = FALSE;

  if (opt_c) {
	/* disable SIGINT while the R_stat and L_stat files are moved */
	if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
		errflg = TRUE;
	else {
		if (clearfile(RSTAT) == FAILED || clearfile(LSTAT) == FAILED)
			errflg = TRUE;
		/* re-enable SIGINT once moving is finished */
		if (signal(SIGINT, sig) == BADSIG)
			errflg = TRUE;
	}
  }

  if (!errflg && opt_p)
	errflg = (runps() == FAILED);
  if (!errflg && (opt_k || opt_r || opt_q))
	errflg = (viewdir() == FAILED);
  if (!errflg && (opt_j || opt_u || opt_s || opt_o || opt_y))
	errflg = (viewfile(RSTAT) == FAILED);
  if (!errflg && (opt_m || opt_M))
	errflg = (viewfile(LSTAT) == FAILED);

  exit(0);
  /* NOTREACHED */
}


/*
 * r e a d l s y s
 *
 * Read the L.sys file and store the names in memory.
 *
 * Return:	FAILED	Cannot open L.sys/ cannot malloc memory
 *		#	Number of names found
 *
 * File not locked.
 */
int readlsys()
{
  char buff[BUFSIZ], cont[132], name[10];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = 0;
  while (j < MAXSYS && fgets(buff, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(buff)) < (BUFSIZ - 132) &&
			buff[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		buff[k - 2] = '\0';
		strcat(buff, cont);
	}
	if (buff[0] == '#' || buff[0] == '\n') continue;
	for (k = 0 ; k < SITELEN ; k++) {
		name[k] = buff[k];
		if (name[k] == ' ' || name[k] == '\t') {
			name[k] = '\0';
			break;
		}
	}
	name[k] = '\0';

	sitelist[j] = (struct siterec *) malloc(sizeof(struct siterec));
	if (sitelist[j] == (struct siterec *)NULL) {
		(void) fclose(fplsys);
		return(FAILED);
	}
	strncpy(sitelist[j]->site, name, SITELEN);
	sitelist[j]->site[SITELEN] = '\0';
	sitelist[j]->job_count = 0;
	sitelist[j]->data_count = 0;
	sitelist[j]->first_time = (time_t)0;
	sitelist[j]->last_time = (time_t)0;
	sitelist[j]->lock_time = (time_t)0;
	
	j++;
  }
  (void) fclose(fplsys);

  (void) syssort(j);	/* helps to speed name lookup */

  return(j);
}


/*
 * r u n p s
 *
 * Run ps(1) and show the result for any processes owning STST files
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int runps()
{
  char name[20], scratch[132], header[132], buff[BUFSIZ], *fields[MAXARGS];
  int j, k, m, fd, total, pfd[2];
  FILE *psfp;
  DIR *dirp;
  struct dirent *dp;

  /* pid information structure */
  struct {
	int pid;
	char site[20];
	char ps[132];
  } pidbuff[MAXLOCKS], *pidlist[MAXLOCKS], *tpidp;

  /* initialise the structure */
  for (j = 0 ; j < MAXLOCKS ; j++) {
	pidlist[j] = &pidbuff[j];
	pidlist[j]->pid = 0;
	pidlist[j]->site[0] = '\0';
	pidlist[j]->ps[0] = '\0';
  }

  /* try to open work directory */
  if ((dirp = opendir(SPOOLDIR)) == (DIR *)NULL)
	return(FAILED);

  /* get a list of pids in STST files */
  total = 0;
  name[15] = '\0';
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	if (strncmp(dp->d_name, "STST.", 4) == SAME) {
		/* read the file */
		strncpy(name, dp->d_name, 14);
		strncpy(scratch, SPOOLDIR, 130);
		strcat(scratch, "/");
		strncat(scratch, name, 131 - strlen(scratch));
		if ((fd = open(scratch, 0)) == -1)
			continue;
		for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
		if (read(fd, buff, 131) == -1) {
			(void) close(fd);
			continue;
		}
		(void) close(fd);

		/* get the pid */
		for (j = 0 ; j < 6 ; j++)
			if (!isdigit(buff[j])) break;
		buff[j] = '\0';
		j = atoi(buff);
		if (j == 0) continue;

		/* add it to the sorted list */
		for (k = 0 ; k < total ; k++) {
			if (j < pidlist[k]->pid) {
				tpidp = pidlist[total];
				for (m = total ; m > k ; m--)
					pidlist[m] = pidlist[m - 1];
				pidlist[k] = tpidp;
				break;
			}
		}
		pidlist[k]->pid = j;
		strncpy(pidlist[k]->site, dp->d_name + 5, 19);
		/* mark duplicates */
		if (k != 0 && pidlist[k]->pid == pidlist[k - 1]->pid) {
			sprintf(pidlist[k]->ps,
				"          %5d *** duplicate pid, site %s\n",
				pidlist[k]->pid, pidlist[k]->site);
		}
		total++;
	}
  }
  (void) closedir(dirp);

  /* we may have nothing to do anyway */
  if (total == 0) {
	printf("No STST files found.\n");
	return(OK);
  }

  /* run ps(1) and collect output */
  if (pipe(pfd) == -1)			/* generate the pipe */
	return(FAILED);
  switch (fork()) {			/* fork, dup and exec */
	case -1:			/* error */
		return(FAILED);
	case 0:				/* pregnant */
		if (close(1) == -1)
			exit(1);
		if (dup(pfd[1]) != 1)	/* child's stdout */
			exit(1);
		if (close(pfd[0]) == -1 || close(pfd[1]) == -1)
			exit(1);
		(void) fflush(stderr);
		(void) freopen("/dev/null", "w", stderr);
		(void) execlp(PS, "ps", "-axl", (char *)NULL);
		exit(1);		/* never happen */
  }

  /* if we're here, we're a parent */
  if (close(pfd[1]) == -1) return(FAILED);
  if ((psfp = fdopen(pfd[0], "r")) == (FILE *)NULL)
	return(FAILED);

  /* get the ps header line */
  if (fgets(buff, BUFSIZ, psfp) == (char *)NULL) {
	(void) fclose(psfp);
	(void) close(pfd[0]);
	return(FAILED);
  }
  else
	strncpy(header, buff, 131);

  /* add relevant pid lines to the list */
  while (fgets(buff, BUFSIZ, psfp) != (char *)NULL) {
	/* get a pid */
	strncpy(scratch, buff, 131);	/* getargs roaches the string */
	if (getargs(scratch, fields) < (PSPID + 1)) continue;
	pid = atoi(fields[PSPID]);
	if (pid == 0) continue;
	/* check it against the pidlist */
	for (j = 0 ; j < total ; j++) {
		if (pid == pidlist[j]->pid && pidlist[j]->ps[0] == '\0') {
			strncpy(pidlist[j]->ps, buff, 131);
			break;
		}
	}
  }
  (void) fclose(psfp);
  (void) close(pfd[0]);

  /* display the list */
  printf("%s", header);
  for (j = 0 ; j < total ; j++) {
	if (pidlist[j]->ps[0] == '\0')
		printf("          %5d *** unused pid, site %s\n",
					pidlist[j]->pid, pidlist[j]->site);
	else
		printf("%s", pidlist[j]->ps);
  }

  return(OK);
}


/*
 * s c a n d i r
 *
 * Scan work dir for files
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		jobnum and sitelist[] are updated as side effects
 */
int scandir()
{
  char buff[132], cmdname[132], lckname[132], site[16];
  int j, k;
  DIR *dirp;
  struct dirent *dp;
  struct stat stat_buff;

  /* try to open work directory */
  if ((dirp = opendir(SPOOLDIR)) == (DIR *)NULL)
	return(FAILED);

  /* look for the required files */
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	if (opt_k || opt_r) {		/* kill or rejuvenate uucp job */
		if (strncmp(dp->d_name, "C.", 2) == SAME) {
			strncpy(buff, dp->d_name + strlen(dp->d_name) - 4, 4);
			jobnum = ston(buff);
			if (gk_job == jobnum || gr_job == jobnum) {
				strncpy(cmdname, SPOOLDIR, 131);
				strcat(cmdname, "/");
				strncat(cmdname, dp->d_name, 14);
				if (opt_k && gk_job == jobnum) {
					/* remove the datafiles */
					(void) killdata(cmdname);
					/* remove the command file */
					(void) unlink(cmdname);
					/* update R_stat file */
					if (update(KILLED) == FAILED) {
						(void) closedir(dirp);
						return(FAILED);
					}
					/* nothing left for anyone else */
					continue;
				}
				if (opt_r && gr_job == jobnum) {
					/* touch the file */
					(void) touch(cmdname);
					/* update R_stat file */
					if (update(TOUCHED) == FAILED) {
						(void) closedir(dirp);
						return(FAILED);
					}
				}
			}
		}
	}
	if (opt_q) {
		/* get data for lock file */
		if (strncmp(dp->d_name, "LCK..", 5) == SAME) {
			/* get the site name */
			strncpy(site, dp->d_name + 5, 14 - 5);
			if ((j = checksys(site)) != -1) {
				strncpy(lckname, SPOOLDIR, 131);
				strcat(lckname, "/");
				strncat(lckname, dp->d_name, 14);
				(void) stat(lckname, &stat_buff);
				sitelist[j]->lock_time = stat_buff.st_mtime;
			}
		}
		/* get data for command file */
		else if (strncmp(dp->d_name, "C.", 2) == SAME) {
			/* get the site name */
			for (j = 0 ; j < 12 ; j++)
				site[j] = dp->d_name[j + 2];
			site[12] = site[13] = '\0';
			k = strlen(site);
			for (j = k - 5 ; j < k ; j++)	/* lose grade as well */
				site[j] = '\0';
			if ((j = checksys(site)) != -1) {
				strncpy(buff, SPOOLDIR, 131);
				strcat(buff, "/");
				strncat(buff, dp->d_name, 14);
				(void) stat(buff, &stat_buff);
				if (stat_buff.st_mtime < sitelist[j]->first_time ||
						sitelist[j]->first_time == (time_t)0)
					sitelist[j]->first_time = stat_buff.st_mtime;	
				if (stat_buff.st_mtime > sitelist[j]->last_time)
					sitelist[j]->last_time = stat_buff.st_mtime;	
				sitelist[j]->job_count++;
			}
		}
		/* get data for data file */
		else if (strncmp(dp->d_name, "D.", 2) == SAME) {
			/* get the site name */
			for (j = 0 ; j < 12 ; j++)
				site[j] = dp->d_name[j + 2];
			site[12] = site[13] = '\0';
			k = strlen(site);
			for (j = k - 5 ; j < k ; j++)
				site[j] = '\0';
			if ((j = checksys(site)) != -1)
				sitelist[j]->data_count++;
		}
	}
  }
  (void) closedir(dirp);

  return(OK);
}


/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/*
 * s t o n
 *
 * Converts a string to an integer
 *
 * Returns:	A number
 *
 * Some machines use hexadecimal job number in file names.
 * This routine extracts some sort of int from the string.
 */
int ston(str)
char *str;
{
  char *cp = str;
  int j, aflg, dflg, hflg;
  long int lj;
  aflg = dflg = hflg = FALSE;
 
  while (*cp) {
	if (isdigit(*cp)) dflg = TRUE;
	else if (isxdigit(*cp)) hflg = TRUE;
	else aflg = TRUE;
	cp++;
  }

  if (aflg)			/* give up */
	lj = 0;
  else if (dflg)
	lj = atoll(str);	/* local version of atol */
  else if (hflg)
	lj = htol(str);

  j = lj % 0xffff;		/* job numbers are signed ints here */

  return(j);
}


/* 
 * s y s s o r t
 *
 * Sort sitelist[] by site name
 *
 * Return:	OK	Always
 */
int syssort(total)
int total;
{
  int j, k, gap;
  struct siterec *tp;

  /* shell sort */
  gap = 1;
  while (gap <= total) gap = 3 * gap + 1;

  for (gap /= 3 ; gap > 0 ; gap /= 3 ) {
 	for (j = gap ; j < total ; j++) { 
		tp = sitelist[j];
		for (k = j ;
		k >= gap &&
		strncmp(sitelist[k - gap]->site, tp->site, SITELEN) > 0 ;
		k -= gap) {
			sitelist[k] = sitelist[k - gap];
		}
		sitelist[k] = tp;
	}
  }

  return(OK);
}


/*
 * t o u c h
 *
 * Update a file creation time
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int touch(filename)
char *filename;
{
  struct utimbuf utimes;
  struct stat stat_buff;

  if (access(filename, 0) != 0) return(FAILED);
  if (stat(filename, &stat_buff) != 0) return(FAILED);
  if ((stat_buff.st_mode & S_IFREG) == 0) return(FAILED);
  (void) time(&now);
  utimes.actime = utimes.modtime = now;
  if (utime(filename, &utimes) !=  0)
	return(FAILED);
   else
	return(OK);
} 


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/*
 * u p d a t e
 *
 * Update the R_stat file for killed or rejuvenated tasks
 *
 * Returns:	OK	Success
 *		FAILED	Otherwise
 */
int update(action)
int action;
{
  char buff[BUFSIZ];
  int fd, flg;
  size_t len;
  struct rstatrec *rstat;

  /* set up to read an R_stat record */
  rstat = (struct rstatrec *)buff;
  len = sizeof(struct rstatrec);

  if (access(RSTAT, 0) == -1) return(FAILED);	/* no file, no name */
  if (checklock(LCKRSTAT, 2) == FAILED) return(FAILED); /* lock not possible */
  if ((fd = open(RSTAT, O_RDWR, 0644)) == -1) return(FAILED);

  flg = FALSE;
  while (read(fd, buff, len) == len) {
	if  (action == KILLED && gk_job == rstat->job) {
		rstat->time_b = now;
		rstat->code = ET_KINC;
		flg = TRUE;
	}
	if  (action == TOUCHED && gr_job == rstat->job) {
		rstat->time_b = now;
		flg = TRUE;
	}
	if (flg) {
		if (lseek(fd, -((off_t)len), 1) == -1 ||
				write(fd, buff, len) != len) {
			(void) close(fd); /* do not unlock - file corrupt */
			return(FAILED);
		}
		flg = FALSE;
	}
  }

  (void) close(fd);
  (void) unlock(LCKRSTAT);
  return(OK);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-coy hrs] [-jkr job] [-mMs sys] [-pq] [-u usr]\n",
		progname);
}


/*
 * v i e w d i r
 *
 * View the spool directory
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int viewdir()
{
  char *cp, first[40], last[40], lock[40];
  int j, olaflg, olkflg, fiflg, laflg, lkflg;
  cp = "       -       ";	/* placeholder */

  /* find the names of known sites */
  if (opt_q && ((totsites = readlsys()) == FAILED))
	return(FAILED);

  /* get data, and update SPOOLDIR files and R_stat if required */
  if (scandir() == FAILED)
	return(FAILED);

  if (opt_q) {				/* print the data if necessary */
	printf(
  "site     #jobs #files     earliest            latest              lock\n");
	for (j = 0 ; sitelist[j] != (struct siterec *)NULL && j < MAXSYS ; j++) {
		fiflg = laflg = lkflg = olaflg = olkflg = FALSE;
		if (sitelist[j]->first_time != (time_t)0) {
			fiflg = TRUE;
			(void) getdate(first, &sitelist[j]->first_time);
		}
		if (sitelist[j]->last_time != (time_t)0) {
			laflg = TRUE;
			olaflg = sitelist[j]->last_time < (time_t)(now - 86400L);
			(void) getdate(last, &sitelist[j]->last_time);
		}
		if (sitelist[j]->lock_time != (time_t)0) {
			lkflg = TRUE;
			olkflg = sitelist[j]->lock_time < (time_t)(now - 86400L);
			(void) getdate(lock, &sitelist[j]->lock_time);
		}
		printf("%-8s  %3d   %3d   %s  %s%s  %s%s\n",
			sitelist[j]->site,
			sitelist[j]->job_count,
			sitelist[j]->data_count,
			fiflg ? first : cp,
			olaflg ? "**" : "  ",
			laflg ? last : cp,
			olkflg ? "**" : "  ",
			lkflg ? lock : cp
			);
	  }
  }
  return(OK);
}


/*
 * v i e w f i l e
 *
 * View a stat file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int viewfile(file)
char *file;
{
  char buff[BUFSIZ], time_a[64], time_b[64], user[10], site[10];
  int j, fd, choice;
  size_t len;
  struct stat statbuf;
  struct rstatrec *rstat;
  struct lstatrec *lstat;

  if (access(file, 0) == -1) return (FAILED);	/* no file, don't bother */

  /* Open  the file and check it for data */
  if ((fd = open(file, O_RDONLY, 0600)) == -1) {
	return(FAILED);
  }
  if (fstat(fd, &statbuf) != 0 || statbuf.st_size == (off_t)0) {
	(void) close(fd);
	return(FAILED);
  }

  /* set up to read the appropriate record layout */
  if (strcmp(file, RSTAT) == SAME) {
	choice = 1;
	rstat = (struct rstatrec *)buff;
	len = sizeof(struct rstatrec);
	printf("%s",
	"job  user     site     command_time    status_time     status\n");
  }
  else if (strcmp(file, LSTAT) == SAME) {
	choice = 2;
	lstat = (struct lstatrec *)buff;
	len = sizeof(struct lstatrec);
	if (opt_m)
		printf("%s", "site     status_time     status\n");
	else
		printf("%s",
		"site     status_time     success_time    status\n");
  }
  else
	return(FAILED);	/* unknown file */

  /* R_stat format: job_no user site command_time status_time status_code
   * L_stat format: site status_time success_time status_code
   */

  /* loop through the file printing the data */
  while ((j = read(fd, buff, len)) != 0 && j != -1) {
	switch (choice) {
	case 1:		/* R_stat file */
		if ((*comp)(rstat) == TRUE) {
			strncpy(user, rstat->user, USERLEN);
			strncpy(site, rstat->site, SITELEN);
			(void) getdate(time_a, &rstat->time_a);
			(void) getdate(time_b, &rstat->time_b);
			printf("%4d %-8s %-8s %s %s %s\n",
				rstat->job, user, site, time_a, time_b,
				errlist[getmsg(rstat->code)]);
		}
		break;
	case 2:		/* L_stat file */
		if ((*comp)(lstat) == TRUE) {
			strncpy(site, lstat->site, SITELEN);
			(void) getdate(time_a, &lstat->time_a);
			(void) getdate(time_b, &lstat->time_b);
			if (opt_m)
				printf("%-8s %s %s\n",
					site, time_a,
					errlist[getmsg(lstat->code)]);
			else
				printf("%-8s %s %s %s\n",
					site, time_a, time_b,
					errlist[getmsg(lstat->code)]);
		}
		break;
	default:	/* never happen */
		break;
	}
  }

  (void) close(fd);

  return(OK);
}

