/* uuschk - check the UUCP _stat and _sub files.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */
 
#include <sys/stat.h>
#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

#undef SUB_LCK		/* lock the sub files before altering them */

#define FALSE		 0
#define TRUE	    ~FALSE
#define OK		 1
#define FAILED		-1
#define SAME		 0
#define MAXLOCKS	10
#define MAXSYS	       200

#ifndef lint
static char *Version = "@(#) uuschk 1.2 (31 October 1991)";
#endif

/* Global variables */

char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char progname[20];		/* current program name */
char locuser[10];		/* local username */
char g_site[SITELEN + 1];	/* given site name */
int g_job;			/* given job number */
int pid;			/* current pid */

int opt_a = FALSE;
int opt_d = FALSE;
int opt_f = FALSE;
int opt_s = FALSE;

struct rstatrec *rstatlst[MAXSYS];
struct lstatrec *lstatlst[MAXSYS];
struct rsubrec *rsublst[MAXSYS];
struct lsubrec *lsublst[MAXSYS];

/* Standard call and task status messages */

char *errlist[] = {
	"NULL MESSAGE",
	"STATUS UNKNOWN: SYSTEM ERROR",
	"LOCAL CAN'T CREATE TEMP FILE",
	"CAN'T COPY TO LOCAL DIRECTORY",
	"LOCAL ACCESS TO FILE DENIED",
	"REMOTE CAN'T CREATE TEMP FILE",
	"CAN'T COPY TO REMOTE DIRECTORY",
	"REMOTE ACCESS TO FILE DENIED",
	"BAD UUCP COMMAND GENERATED",
	"CAN'T EXECUTE UUCP",
	"JOB IS QUEUED",
	"COPY FAILED",
	"COPY (PARTIALLY) SUCCEEDED",
	"COPY FINISHED, JOB DELETED",
	"DONE (WORK HERE)",
	"JOB KILLED (INCOMPLETE)",
	"JOB KILLED (COMPLETE)"
};

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int catchsignal();
int checklock();
void cleanup();
int getdate();
int getmsg();
uid_t get_uid();
int getuser();
int setsignal();
int statupdate();
int subupdate();
int sysrem();
int syssort();
int unlock();
void usage();
int viewfile();


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT are ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining
 * ten or so signals are scarce enough, and rare enough, to be let run.
 * One problem is that the shell protects background commands from
 * SIGINT but not SIGHUP, so if SIGHUP is ignored only SIGKILL can stop a
 * background process, leaving pieces all over the floor.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, SIG_IGN) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sig;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT, the only un-ignored signal */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files before exiting
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  /* remove all lockfiles */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0')
		(void) unlink(locklist[j]);
	j++;
  }

  exit(1);
}


/*
 * g e t d a t e
 *
 * Get the date from the unixtime
 *
 * Return:	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(bp, tp) 
char *bp; time_t *tp;
{
  char buff[132];
  int j;

  sprintf(buff, "%s", ctime(tp));	/* truncate to fit into 80 cols */
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[12] = ' ';			/* lose the seconds */
  buff[13] = buff[18];			/* lose the decade */
  buff[14] = buff[19];
  buff[15] = '\0';
  strcpy(bp, buff);
  return(OK);
}


/*
 * g e t m s g
 *
 * Get the appropriate message given the status number
 *
 * Return:	An index to errlist[]
 */
int getmsg(status)
int status;
{
  int j, k;

  if (status == 0) return(OK_MSG);

  j = 1;
  k = (sizeof(errlist) / sizeof(char *));
  while (j < k && (status & 1) == 0) {
	status >>= 1;
	j++;
  }
  if (j == k)
	j = ERR_MSG;

  return(j);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp, g_file[10];
  int j, errflg;
  uid_t tuid;

  if (getuser(getuid(), locuser) == FAILED) {	/* get local user name */
	fprintf(stderr, "uuschk: cannot find local username\n");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	fprintf(stderr, "uuschk: cannot find own pid\n");
	exit(1);
  }

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */

  /* parse arguments */
  opterr = 0;
  if (argc == 1) {
	opt_a = TRUE;
  }
  else {
	while ((j = getopt(argc, argv, "d:f:s")) != EOF) {
		switch (j & 0177) {
		case 'd':
			if (isdigit(optarg[0]))  {
				g_job = atoi(optarg);
				g_site[0] = '\0';
			}
			else {
				strncpy(g_site, optarg, SITELEN);
				g_job = 0;
			}
			opt_d = TRUE;
			break;
		case 'f':
			strncpy(g_file, optarg, USERLEN);
			opt_f = TRUE;
			break;
		case 's':
			opt_s = TRUE;
			break;
		case '?':
		default:
			usage();
			exit(1);
		}
	}
  }

  /* validity checks */
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if ((opt_d || opt_s) && getuid() > MAXUID && getuid() != tuid) {
	fprintf(stderr, "uuschk: option available only to administrators.\n");
	exit(1);
  }
#endif
  if (opt_f && !(strcmp("R_stat", g_file) == SAME ||
		strcmp("L_stat", g_file) == SAME ||
		strcmp("R_sub", g_file) == SAME ||
		strcmp("L_sub", g_file) == SAME)) {
	usage();
	exit(1);
  }

  /* set up to catch signals */
  if (catchsignal() == FAILED) {
	fprintf(stderr, "uuschk: cannot catch signals\n");
	exit(1);
  }

  /* do work */
  errflg = FALSE;

  if (!errflg && opt_d) {		/* update files */
	if (g_job == 0) {
		errflg = (subupdate(g_site, TRUE) == FAILED);
		if (!errflg) errflg = (statupdate(LSTAT) == FAILED);
	}
	else
		errflg = (statupdate(RSTAT) == FAILED);
  }
  else if (!errflg && opt_s) {
	errflg = (subupdate("all", FALSE) == FAILED);
  }
  if (!errflg && opt_a) {		/* view files */
	(void) puts("R_stat file contents:");
	(void) viewfile(RSTAT);
	(void) puts("\nL_stat file contents:");
	(void) viewfile(LSTAT);
	(void) puts("\nR_sub file contents:");
	(void) viewfile(RSUB);
	(void) puts("\nL_sub file contents:");
	(void) viewfile(LSUB);
  }
  else if (!errflg && opt_f) {
	if (strncmp("R_stat", g_file, 3) == SAME) {
		(void) puts("R_stat file contents:");
		(void) viewfile(RSTAT);
	}
	else if (strncmp("L_stat", g_file, 3) == SAME) {
		(void) puts("L_stat file contents:");
		(void) viewfile(LSTAT);
	}
	else if (strncmp("R_sub", g_file, 3) == SAME) {
		(void) puts("R_sub file contents:");
		(void) viewfile(RSUB);
	}
	else if (strncmp("L_sub", g_file, 3) == SAME) {
		(void) puts("L_sub file contents:");
		(void) viewfile(LSUB);
	}
  }

  exit(0);
  /* NOTREACHED */
}


/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/*
 * s t a t u p d a t e
 *
 * Sort or purge the L_stat file, or purge the R_stat file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int statupdate(filename)
char *filename;
{
  char tfile[132], lockfile[132], buff[BUFSIZ];
  int fds, fdt, choice, total;
  size_t len;
  struct stat statbuf;
  struct rstatrec *rstat;
  struct lstatrec *lstat;

  /* set up to read the appropriate record layout */
  if (strcmp(filename, RSTAT) == SAME) {
	choice = 1;
	rstat = (struct rstatrec *)buff;
	len = sizeof(struct rstatrec);
	strncpy(lockfile, LCKRSTAT, 131);
  }
  else if (strcmp(filename, LSTAT) == SAME) {
	choice = 2;
	lstat = (struct lstatrec *)buff;
	len = sizeof(struct lstatrec);
	strncpy(lockfile, LCKLSTAT, 131);
  }
  else
	return(FAILED);					/* unknown file */

  if (access(filename, 0) == -1) return(FAILED);	/* no file, no name */

  /* if the file is zero length, or bigger than FILEBUFF, we can do nothing */
  if ((fds = open(filename, O_RDONLY, 0644)) == -1) {
	return(FAILED);
  }
  else if (fstat(fds, &statbuf) != 0 ||
		statbuf.st_size == (off_t)0 ||
		statbuf.st_size > (off_t)(MAXSYS * len)) {
	(void) close(fds);
	return(FAILED);
  }
  total = statbuf.st_size / len;
  printf("%s contains %d record%s\n", choice == 1 ? "R_stat" : "L_stat",
				total, total > 1 ? "s" : " ");
  (void) close(fds);

#ifdef LCK_FILE
  if (checklock(lockfile, 2) == FAILED) return(FAILED); /* lock not possible */
#endif

  /* move R_stat or L_stat to a temporary file */
  sprintf(tfile, "%s%c%04d", LTMPATH, (choice == 1) ? 'R' : 'L', pid);
  (void) unlink(tfile);
  if (link(filename, tfile) == -1 || unlink(filename) == -1) {
	(void) unlink(tfile);
#ifdef LCK_FILE
	(void) unlock(lockfile);
#endif
	return(FAILED);
  }

  if ((fds = open(filename, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(fdt = open(tfile, O_RDONLY, 0644)) == -1) {
	(void) link(tfile, filename); /* recover if possible */
	(void) unlink(tfile);
#ifdef LCK_FILE
	(void) unlock(lockfile);
#endif
	return(FAILED);
  }
  (void) unlink(tfile);	

  while (read(fdt, buff, len) == len) {
	if  (choice == 1 && rstat->job == g_job)
		rstat->code = ET_KINC;
	else if (choice == 2 && strncmp(lstat->site, g_site, SITELEN) == SAME)
		continue;
	if (write(fds, buff, len) != len) {
		(void) close(fds);	/* do not unlock - file corrupt */
		(void) close(fdt);
		return(FAILED);
	}
  }

  (void) close(fds);
  (void) close(fdt);
#ifdef LCK_FILE
  (void) unlock(lockfile);
#endif

  return(OK);
}


/*
 * s u b u p d a t e
 *
 * Update the R_sub and L_sub files
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int subupdate(site, delflg)
char *site ; int delflg ;
{
  char trfile[132], tlfile[132], rbuff[BUFSIZ], lbuff[BUFSIZ];
  int j, k, total, errflg, rfd, trfd, lfd, tlfd;
  size_t rlen, llen;
  struct stat rstatbuf;
  struct stat lstatbuf;
  struct rsubrec *rsub;
  struct lsubrec *lsub;

  rsub = (struct rsubrec *)rbuff;
  rlen = sizeof(struct rsubrec);
  lsub = (struct lsubrec *)lbuff;
  llen = sizeof(struct lsubrec);

  /* no file, no name */
  if (access(RSUB, 0) == -1 || access(LSUB, 0) == -1) return(FAILED);
  /* some sort of open failure */
  if ((rfd = open(RSUB, O_RDONLY, 0644)) == -1  ||
		(lfd = open(LSUB, O_RDONLY, 0644)) == -1 ||
		fstat(rfd, &rstatbuf) != 0 ||
		fstat(lfd, &lstatbuf) != 0) {
	(void) close(rfd);
	(void) close(lfd);
	return(FAILED);
 }
 (void) close(rfd);
 (void) close(lfd);

 /* if the file is zero length, or bigger than FILEBUFF, we can do nothing */
 if (rstatbuf.st_size == (off_t)0 || rstatbuf.st_size > (off_t)(MAXSYS * rlen) ||
	lstatbuf.st_size == (off_t)0 || lstatbuf.st_size > (off_t)(MAXSYS * llen)) {
		return(FAILED);
  }

#ifdef LCK_FILE
  /* try to lock the files */
  if (checklock(LCKRSUB, 2) == FAILED)
	return(FAILED);
  if (checklock(LCKLSUB, 2) == FAILED) {
	(void) unlock(LCKRSUB);
	return(FAILED);
  }
#endif

  /* move R_sub and L_sub to temporary files */
  sprintf(trfile, "%sR%04d", TMPATH, pid);
  (void) unlink(trfile);	/* in case it already exists */
  if (link(RSUB, trfile) == -1 || unlink(RSUB) == -1) {
	(void) unlink(trfile);
#ifdef LCK_FILE
	(void) unlock(LCKRSUB);
	(void) unlock(LCKLSUB);
#endif
	return(FAILED);
  }
  sprintf(tlfile, "%sL%04d", TMPATH, pid);
  (void) unlink(tlfile);
  if (link(LSUB, tlfile) == -1 || unlink(LSUB) == -1) {
	(void) unlink(tlfile);
	(void) link(trfile, RSUB);	/* recover if possible */
	(void) unlink(trfile);
#ifdef LCK_FILE
	(void) unlock(LCKRSUB);
	(void) unlock(LCKLSUB);
#endif
	return(FAILED);
  }
  errflg = FALSE;

  /* open the two output files and the two temporary files */
  if ((rfd = open(RSUB, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(trfd = open(trfile, O_RDONLY, 0644)) == -1 ||
		(lfd = open(LSUB, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(tlfd = open(tlfile, O_RDONLY, 0644)) == -1) {
	errflg = TRUE;
  }

  /* read the two files in parallel, and initialise the structure arrays */
  total = 0;
  rsub = (struct rsubrec *) malloc(rlen);
  lsub = (struct lsubrec *) malloc(llen);
  while ((j = read(trfd, (char *) rsub, rlen)) != 0 && j != -1 &&
		(k = read(tlfd, (char*) lsub, llen)) != 0 && k != -1) {
	rsublst[total] = rsub;
	lsublst[total] = lsub;
	total++;
	rsub = (struct rsubrec *) malloc(rlen);
	lsub = (struct lsubrec *) malloc(llen);
  }
  (void) free(rsub);
  (void) free(lsub);

  if (j == -1 || k == -1)
	errflg = TRUE;
  else {
	printf("L_sub and R_sub contain %d record%s\n", total,
				total > 1 ? "s" : " ");
	(void) syssort(total);
	for (j = 0 ; j < total ; j++) {
		if (strncmp(rsublst[j]->site, lsublst[j]->site, SITELEN) != SAME) {
			printf("Unmatched names found in record %d\n", j);
			errflg = TRUE;
			break;
		}
		if (delflg && strncmp(rsublst[j]->site, site, SITELEN) == SAME) {
			for (k = j ; k < total ; k++) {
				rsublst[k] = rsublst[k + 1];
				lsublst[k] = lsublst[k + 1];
			}
			total--;
			(void) free(rsublst[total]);
			(void) free(lsublst[total]);
			rsublst[total] = (struct rsubrec *)NULL;
			lsublst[total] = (struct lsubrec *)NULL;
			printf("Entry %d for %s deleted\n", j, site);
		}
	}
  }

  /* write out the changed records */
  for (j = 0 ; !errflg && j < total ; j++) {
	if (write(rfd, (char *)rsublst[j], rlen) == -1) errflg = TRUE;
	if (write(lfd, (char *)lsublst[j], llen) == -1) errflg = TRUE;
	(void) free(rsublst[j]);
	(void) free(lsublst[j]);
  }

  /* wrap up */
  (void) close(rfd);
  (void) close(trfd);
  (void) close(lfd);
  (void) close(tlfd);
  if (errflg) {			/* recover if possible */
	(void) unlink(RSUB);
	(void) link(trfile,RSUB);
	(void) unlink(LSUB);
	(void) link(tlfile,LSUB);
  }
  (void) unlink(trfile);
  (void) unlink(tlfile);
#ifdef LCK_FILE
  (void) unlock(LCKRSUB);
  (void) unlock(LCKLSUB);
#endif
  if (errflg)
	return(FAILED);
  else
	return(OK);
}


/* 
 * s y s s o r t
 *
 * Sort L.sub and R_sub by site name, using in-memory buffers
 *
 * Return:	OK	Always
 */
int syssort(total)
int total ;
{
  int j, k, gap;
  struct rsubrec *rsub;
  struct lsubrec *lsub;

  /* shell sort */
  gap = 1;
  while (gap <= total) gap = 3 * gap + 1;

  for (gap /= 3 ; gap > 0 ; gap /= 3 ) {
 	for (j = gap ; j < total ; j++) { 
		rsub = rsublst[j];
		lsub = lsublst[j];
		for (k = j ;
		k >= gap &&
		strncmp(rsublst[k - gap]->site, rsub->site, SITELEN) > 0 ;
		k -= gap) {
			rsublst[k] = rsublst[k - gap];
			lsublst[k] = lsublst[k - gap];
		}
		rsublst[k] = rsub;
		lsublst[k] = lsub;
	}
  }

  return(OK);
}


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-d sys] [-f file] [-s]\n", progname); 
}


/*
 * v i e w f i l e
 *
 * View a site statistics file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * File is not locked
 */
int viewfile(file)
char *file;
{
  char buff[BUFSIZ], time_a[64], time_b[64], site[20], user[20];
  int j, fd, choice;
  size_t len;

  struct stat statbuf;
  struct rstatrec *rstat;
  struct lstatrec *lstat;
  struct rsubrec *rsub;
  struct lsubrec *lsub;

  if (access(file, 0) == -1) return (FAILED);	/* no file, don't bother */

  /* try to open the file for reading only */
  if ((fd = open(file, O_RDONLY, 0600)) == -1) {
	return(FAILED);
  }

  /* see if the file is zero length */
  if (fstat(fd, &statbuf) != 0) {
	(void) close(fd);
	return(FAILED);
  }
  if (statbuf.st_size == (off_t)0)
	return(FAILED);

  /* set up to read the appropriate record layout */
  if (strcmp(file, RSTAT) == SAME) {
	choice = 1;
	rstat = (struct rstatrec *)buff;
	len =  sizeof(struct rstatrec);
	printf("%s",
	"job  user     site     command_time    status_time     status_code\n");
  }
  else if (strcmp(file, LSTAT) == SAME) {
	choice = 2;
	lstat = (struct lstatrec *)buff;
	len =  sizeof(struct lstatrec);
	printf("%s", "site     status_time     success_time    status_code\n");
  }
  else if (strcmp(file, RSUB) == SAME) {
	choice = 3;
	rsub = (struct rsubrec *)buff;
	len = sizeof(struct rsubrec);
	printf("%s", "site     sfile  sbyte      rfile  rbyte\n");
  }
  else if (strcmp(file, LSUB) == SAME) {
	choice = 4;
	lsub = (struct lsubrec *)buff;
	len = sizeof(struct lsubrec);
	printf("%s", "site     #calls #ok   last_time        ok_time          ");
	printf("%s", "#dev  #login #nack #other\n");
  }
  else
	return(FAILED);	/* unknown file */

  /* R_stat format: job_no user site command_time status_time status_code
   * L_stat format: site status_time success_time status_code
   * R_sub format: site files_sent bytes_sent files_received bytes_received
   * L_sub format: site call_count ok_count last_time ok_time 4*failure_modes
   */

  /* loop through the file printing the data */
  while ((j = read(fd, buff, len)) != 0 && j != -1) {
	switch (choice) {
	case 1:		/* R_stat file */
		strncpy(user, rstat->user, USERLEN);
		strncpy(site, rstat->site, SITELEN);
		(void) getdate(time_a, &rstat->time_a);
		(void) getdate(time_b, &rstat->time_b);
		printf("%4d %-8s %-8s %s %s %s\n",
			rstat->job, user, site, time_a, time_b,
			errlist[getmsg(rstat->code)]);
		break;
	case 2:		/* L_stat file */
		strncpy(site, lstat->site, SITELEN);
		(void) getdate(time_a, &lstat->time_a);
		(void) getdate(time_b, &lstat->time_b);
		printf("%-8s %s %s %s\n",
			site, time_a, time_b, errlist[getmsg(lstat->code)]);
		break;
	case 3:		/* R_sub file */
		strncpy(site, rsub->site, SITELEN);
		printf("%-8s %-6d %-10ld %-6d %-10ld\n",
			site, rsub->sfile, rsub->sbyte, rsub->rfile, rsub->rbyte);
		break;
	case 4:		/* L_sub file */
		strncpy(site, lsub->site, SITELEN);
		(void) getdate(time_a, &lsub->time_a);
		(void) getdate(time_b, &lsub->time_b);
		printf("%-8s %-4d   %-4d  %s  %s  %-4d  %-4d   %-4d  %-4d\n",
			site, lsub->all_calls, lsub->ok_calls,
			time_a, time_b, lsub->dev_calls, lsub->log_calls,
			lsub->nack_calls, lsub->rem_calls);
		break;
	default:	/* never happen */
		break;
	}
  }

  (void) close(fd);

  return(OK);
}

