/* uusched - invoke uucico for waiting tasks.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"


/* general constants */

#define FALSE		 0
#define TRUE	    ~FALSE
#define OK		 1
#define FAILED		-1
#define SAME		 0
#define MAXLOCKS	10
#define MAXSITES	30

/* log messages */

#define M_STD		0	/* standard messages */
#define M_NOLOG		1	/* no output to log file */
#define M_ACCESS	2	/* problems in access permissions */

#ifndef lint
static char *Version = "@(#) uusched 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_s = FALSE;		/* Send transfer status to file */
int opt_u = FALSE;		/* Set uucico debugging level */
int opt_x = FALSE;		/* Set debugging level */

int errno;			/* used by errno.h */

int cmdseq;			/* current command sequence number */
int pid;			/* current pid */
int dbglvl;			/* debugging flag */

char knownlist[MAXSITES][SITELEN + 1];	/* known site list */
char sitelist[MAXSITES][SITELEN + 1];	/* work site list */
char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char spooldir[132];		/* spool directory */
char scratch[132];		/* scratch buffer */
char locsite[20];		/* local site name */
char rmtsite[20];		/* remote site */
char locuser[20];		/* local user name */
char rmtuser[20];		/* remote user name */
char progname[20];		/* program name */

int fdlog;			/* system log file descriptor */

time_t now;			/* current time */

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */
int catchsignal();
int checklock();
int checkname();
void cleanup();
int getdate();
int getname();
uid_t get_uid();
int getuser();
void printmsg();
int readlsys();
int scandir();
int setsignal();
int sortlist();
int unlock();
void usage();
int writelog();


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT are ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining
 * ten or so signals are scarce enough, and rare enough, to be let run.
 * One problem is that the shell protects background commands from
 * SIGINT but not SIGHUP, so if SIGHUP is ignored only SIGKILL can stop a
 * background process, leaving pieces all over the floor.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, SIG_IGN) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sig;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT, the only un-ignored signal */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c h e c k n a m e
 *
 * Check if "name" is a known site
 *
 * Return:	OK	Name recognised
 *		FAILED	Name not recognised
 */
int checkname(name)
char *name;
{
  int j = 0;

  if (strcmp(locsite,name) == SAME)
	return(OK);

  while (knownlist[j][0] != '\0') {
	if (strcmp(knownlist[j++], name) == SAME)
		return(OK);
  }

  printmsg(M_ACCESS, "checkname: unknown site %s\n", name);
  return(FAILED);
}

 
/*
 * c l e a n u p
 *
 * Remove any outstanding lock files before exiting
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  /* remove all lockfiles */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0') {
		(void) unlink(locklist[j]);
		printmsg(M_ACCESS, "cleanup: lockfile %s removed", locklist[j]);
	}
	j++;
  }

  exit(1);
}


/*
 * g e t d a t e
 *
 * Get the date in a format suitable for the logfile
 *
 * Return	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(buff)
char *buff;
{
  int j;

  sprintf(buff, "%s", ctime(&now));
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[j] = '\0';			/* clear the newline */

  return(OK);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];
 
  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp, debug[8];
  int j, k, m, status, sitecount, count;
  uid_t tuid;
#ifndef TESTDBG
  dbglvl = 0;
#else
  FILE *fp;
  if ((fp = fopen(DEFDBG, "r")) != (FILE *)NULL &&
		fgets(scratch, 132, fp) != (char *)NULL &&
		sscanf(scratch, "%d", &j) == 1)
	dbglvl = j;		/* default debug level */
  else
	dbglvl = 0;
  if (fp != (FILE *)NULL) (void) fclose(fp);
#endif

  /* set various defaults */
  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  strcpy(debug, "-x");
  rmtsite[0] = '\0';
  for (j = 0 ; j < MAXLOCKS ; j++) locklist[j][0] = '\0';
  for (j = 0 ; j < MAXSITES ; j++) *knownlist[j] = *sitelist[j] = '\0';

  /* parse options */
  opterr = 0;
  while ((j = getopt(argc, argv, "s:u:x:")) != EOF)
	switch (j & 0377) {
	case 's':		/* site name */
		strncpy(rmtsite, optarg, SITELEN);
		opt_s = TRUE;
		break;
	case 'u':		/* set uucico debug level */
		strncat(debug, optarg, 6);
		opt_u = TRUE;
		break;
	case 'x':		/* set uusched debug level */
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case '?':		/* default arguments */
	default:
		usage();
		exit(1);
	}

#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid)
	dbglvl = 0;
#endif

  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  /* get local site name, local user, and pid */
  if (getname(locsite) == FAILED) {
	printmsg(M_NOLOG, "uusched: cannot find local uucpname");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	printmsg(M_NOLOG, "uusched: cannot find local username");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	printmsg(M_NOLOG, "uusched: cannot find own pid");
	exit(1);
  }
  if (chdir(spooldir) == FAILED) {	/* switch to spool directory */
	printmsg(M_NOLOG, "uusched: cannot change to directory %s", spooldir);
	exit(1);
  }
  /* set up to catch signals */
  if (catchsignal() == FAILED) {
	printmsg(M_NOLOG, "uusched: cannot catch signals");
	exit(1);
  }

  /* get a list of known sites */
  if (readlsys() == FAILED) {
	printmsg(M_NOLOG, "uusched: cannot read %s", LSYS);
	exit(1);
  }
  /* check the given site */
  if (opt_s && checkname(rmtsite) == FAILED) {
	printmsg(M_NOLOG, "uusched: unknown site %s", rmtsite);
	exit(1);
  }

  /* set defaults not already set by options */
  if (rmtsite[0] == '\0')
	strncpy(rmtsite, locsite, SITELEN);	/* default remote site */

  /* set up the log file; even if debugging, do not lock it continuously */
  if (dbglvl > 0) {
	printmsg(dbglvl, "========== %-24.24s ==========", ctime(&now));
  }

  /* see if there's any work available */
  if ((sitecount = scandir(rmtsite)) < 1) {
	if (!opt_s) strcpy(rmtsite, "anyone");
	printmsg(M_ACCESS, "uusched: no work for %s", rmtsite);
	exit(0);
  }
  else
	printmsg(M_ACCESS, "uusched: sites found (%d)", sitecount);

  /* try each site in turn, randomising the start point */
  for (m = pid % sitecount, count = 0 ; count < sitecount ; m++, count++) {
	m %= sitecount;
	strncpy(rmtsite, sitelist[m], SITELEN);
	printmsg(M_ACCESS, "uusched: running uucico %d, site %s", m, rmtsite);

	switch (j = fork()) {
	case -1:		/* can't create new process */
		break;
	case 0:			/* forked, try to execute uucp */
		(void) fflush(stderr);
		(void) freopen("/dev/null", "w", stderr);
		(void) execlp(UUCICO, "uucico", "-r1", "-s", rmtsite,
			opt_u ? debug : (char *)NULL, (char *)NULL);
		break;
	default:		/* command running, wait for status */
		while ((k = wait(&status)) != j && k != -1)
			;
		break;
	}

	/* report outcome */
	printmsg(M_ACCESS, "uusched: uucico fork %d, wait %d, status %xx",
						j, k, status);
  }

  exit(0);
  /* NOTREACHED */
}


/*
 * p r i n t m s g
 *
 * Print an error or debugging message into the system error log file.
 *
 * Return:	Nothing
 *
 * All messages at levels less than or equal to the current debug level
 * are printed, unless the M_NOLOG level is used. If debugging is on,
 * messages are also printed to standard error.
 */
/* VARARGS1 */
void printmsg(level, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int level; char *fmt, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8;
{
  char msg[BUFSIZ];
  (void) time(&now);
  if (level <= dbglvl) {
	sprintf(msg, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
	strcat(msg, "\n");
	if (level != M_NOLOG) {
		if (level == M_STD)
			(void) writelog(msg, TRUE);
		else
			(void) writelog(msg, FALSE);
	}
	if (dbglvl > 0) {
		fprintf(stderr, "%s", msg);
		(void) fflush(stderr);
	}
  }
}


/*
 * r e a d l s y s
 *
 * Read the L.sys file and store the names in memory.
 *
 * Return:	FAILED	Cannot open L.sys/ cannot malloc memory
 *		#	Number of names found
 */
int readlsys()
{
  char name[15], cont[132], buff[BUFSIZ];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = 0;
  while (j < MAXSITES && fgets(buff, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(buff)) < (BUFSIZ - 132) &&
			buff[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		buff[k - 2] = '\0';
		strcat(buff, cont);
	}
	if (buff[0] == '#' || buff[0] == '\n') continue;
	for (k = 0 ; k < SITELEN ; k++) {
		name[k] = buff[k];
		if (name[k] == ' ' || name[k] == '\t') {
			name[k] = '\0';
			break;
		}
	}
	name[k] = '\0';
	strncpy(knownlist[j++], name, SITELEN);
  }
  (void) fclose(fplsys);

  return(j);
}


/*
 * s c a n d i r
 *
 * Scan spool directory for sites with workfiles outstanding
 *
 * Return:	Number of sites found		Always
 *		sitelist[] is updated as a side effect
 *
 */
int scandir(site)
char *site;
{
  char name[15], *filelist[MAXCMDS];
  int j, k, m, n, num;
  size_t len;
  DIR *dirp;
  struct dirent *dp;

  /* try to open work directory */
  if ((dirp = opendir(SPOOLDIR)) == (DIR *)NULL) {
	printmsg(M_ACCESS, "scandir: cannot open SPOOLDIR");
	return(0);
  }

  /* set up comparison */
  strncpy(name, "C.", 14);
  if (opt_s) strncat(name, site, SITELEN);
  len = strlen(name);

  /* get a sorted list of any relevant workfiles */
  num = 0;
  filelist[num] = (char *)NULL;
  while ((dp = readdir(dirp)) != (struct dirent *)NULL && num < MAXCMDS) {
	if (strncmp(dp->d_name, name, len) == SAME) {
		if ((filelist[num] = (char *) calloc(15, sizeof(char)))
							!= (char *)NULL) {
			strncpy(filelist[num++], dp->d_name, 14);
			filelist[num] = (char *)NULL;
		}
		else
			break;
	}
  }
  (void) closedir(dirp);

  if (num == 0)		/* we have no command files */
	n = 0;
  else if (opt_s) {	/* we have command files for a specific site */
	n = 1;
	strncpy(sitelist[0], site, SITELEN);
	printmsg(M_ACCESS, "scandir: work for %s", site);
  }
  else {		/* we have command files for one or more sites */

	/* sort the files */
	(void) sortlist(filelist, num);

	/* extract the site names */
	n = 0;
	for (j = 0 ; j < num ; j++) {
		for (k = 0 ; k < 12 ; k++)	/* get the site name */	
			name[k] = filelist[j][k + 2];
		name[12] = name[13] = '\0';
		m = strlen(name);
		for (k = m - 5 ; k < m ; k++)	/* lose grade as well */
			name[k] = '\0';

		/* add a site to the list */
		if (checkname(name) == OK && strcmp(sitelist[n], name) != SAME
							&& n < MAXSITES) {
			strncpy(sitelist[n++], name, SITELEN);
			printmsg(M_ACCESS, "scandir: work for %s", name);
		}
	}
  }

  /* release memory */
  for (j = 0 ; j < num ; j++)
	free((void *)filelist[j]);

  return(n);
}


/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/* 
 * s o r t l i s t
 *
 * Sort a list of names
 *
 * Return:	OK	Always
 */
int sortlist(list, total)
char *list[]; int total;
{
  char tfil[15];
  int j, k, gap;

  /* shell sort */
  gap = 1;
  while (gap <= total) gap = 3 * gap + 1;

  for (gap /= 3 ; gap > 0 ; gap /= 3 ) {
 	for (j = gap ; j < total ; j++) { 
		strncpy(tfil, list[j], 14);
		for (k = j ;
		k >= gap && strncmp(list[k - gap], tfil, 14) > 0 ;
		k -= gap) {
			strncpy(list[k], list[k - gap], 14);
		}
		strncpy(list[k], tfil, 14);
	}
  }

  return(OK);
}


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/*
 * u s a g e 
 *
 * Usage message
 *
 * Return:	none
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-s sys] [-u #] [-x #]\n", progname);
}


/* 
 * w r i t e l o g
 *
 * Write a string to the log file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int writelog(msg, prefix)
char *msg; int prefix;
{
  char buff[BUFSIZ];
  char curtime[64];
  int flg;
  size_t sz;

  /* log format: site!user (date time) (id pid task) status (detail) */
  if (!prefix)
	strncpy(buff, msg, BUFSIZ - 1);
  else {
	(void) getdate(curtime);
	sprintf(buff, "%s!%s (%s) (S %4d  0) ", rmtsite, locuser, curtime, pid);
	strncat(buff, msg, BUFSIZ - 50);
  }
  sz = strlen(buff);

  /* even when debugging, the logfile is not kept locked */
  if (checklock(LCKLOG, 2) == FAILED) {
	printmsg(M_NOLOG, "uusched: cannot get lock %s", LCKLOG);
	return(FAILED);
  }
  if ((fdlog = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1) {
	printmsg(M_NOLOG, "uusched: cannot open %s", LOGFILE);
	(void) unlock(LCKLOG);
	return(FAILED);
  }
  flg = (write(fdlog, buff, sz) == sz);
  if (close(fdlog) == -1) flg = FALSE;
  (void) unlock(LCKLOG);

  if (flg)
	return(OK);
  else
	return(FAILED);
}
