#!/bin/sh

##
##  name:       uupoll
##
##  version:    1.2
##
##  author:     paul@ukpoit.co.uk, paul@max.uk.mugnet.org
##              Copyright 1991 by Paul Wood, Sheffield, UK
##
##  purpose:    check the Poll file and queue any poll requests,
##              then poll any systems specified in the parameter list
##              which has outstanding poll requests
##
##                  uupoll <sysinfo> ...
##
##              where
##                  <sysinfo> is <system>[-<baud>[-<bits>[-<parity>]]]
##                  <system> is the name of the system
##                  <baud> is the baud rate, default 2400 baud
##                  <bits> is the number of bits, default 8 bits
##                  <parity> is the parity, O=odd E=even N=none, default is none
##
##              eg
##                  uupoll scud-2400-8-N patriot-9600
##
##              normally called each hour by cron
##

##
##  default settings
##
DEBUG="1"                               ##  debug level
BAUD="2400"                             ##  modem baudrate
BITS="8"                                ##  number of bits
PARITY="N"                              ##  parity

##
##  constants
##
UUCP="/usr/lib/uucp"                    ##  name of the UUCP home directory
SPOOL="/usr/spool/uucp"                 ##  name of the UUCP spool directory
UUCLEAN="${UUCP}/uuclean"		##  path to the UUCP uuclean program
UUCICO="${UUCP}/uucico"                 ##  path to the UUCP uucico program
UUXQT="${UUCP}/uuxqt"                   ##  path to the UUCP uuxqt program
POLL="${UUCP}/Poll"                     ##  name of the poll file
NEWS="/usr/lib/news"                    ##  name of the news home directory
UNGETTY="/etc/ungetty"                  ##  path to the ungetty program
LOCKS="/usr/spool/locks"                ##  name of the uugetty locks directory
SYSADM=root                             ##  system administrator
MODEM="tty1"                            ##  modem port

##
##  variables
##
SYSTEM=""                               ##  system name to be called
CALL=""                                 ##  indicates the attempt number
CURRHOUR=""                             ##  current hour, derived from date
HOURLIST=""                             ##  list of hours system is to be polled
HOUR=""                                 ##  one of the hours in HOURLIST

##
##  find out the current hour
##
CURRHOUR=`date | cut -c12-13`

##
##  process the Poll file, creating poll requests as necessary
##
if [ -r "${POLL}" ]
then
    cat ${POLL} | while read SYSTEM HOURLIST
    do
        case ${SYSTEM} in
        "" | "#"* )
            ;;
        * )
            for HOUR in ${HOURLIST}
            do
                if [ "${HOUR}" -eq "${CURRHOUR}" ]
                then
                    if [ ! -f "${SPOOL}/C.${SYSTEM}n0000" ]
                    then
                        touch ${SPOOL}/C.${SYSTEM}n0000
                        chown uucp.mail ${SPOOL}/C.${SYSTEM}n0000
                    fi
                    break
                fi
            done
            ;;
        esac
    done
fi

##
##  process each parameter in turn
##
while [ $# -gt 0 ]
do
    ##
    ##  extract system info
    ##
    case $1 in
        *-*-*-* )
            SYSTEM=`echo $1 | cut -d"-" -f1`
            BAUD=`echo $1 | cut -d"-" -f2`
            BITS=`echo $1 | cut -d"-" -f3`
            PARITY=`echo $1 | cut -d"-" -f4`
            ;;
        *-*-* )
            SYSTEM=`echo $1 | cut -d"-" -f1`
            BAUD=`echo $1 | cut -d"-" -f2`
            BITS=`echo $1 | cut -d"-" -f3`
            ;;
        *-* )
            SYSTEM=`echo $1 | cut -d"-" -f1`
            BAUD=`echo $1 | cut -d"-" -f2`
            ;;
        * )
            SYSTEM=$1
            ;;
    esac

    ##
    ##  if the ungetty feature is available, suspend the uugetty and
    ##  get control of the tty
    ##       
    if [ -f "${UNGETTY}" ]
    then
        ${UNGETTY} -o ${MODEM} >/dev/null 2>&1
        if [ $? -eq 0 ]
        then
            ##
            ##  uugetty has suspended successfully,
            ##  remove any old lock files for the modem port
            ##
            rm -f ${LOCKS}/*${MODEM}
        else
            ##
            ##  if ungetty fails to get control of the tty, abort
            ##
            mail -s "uupoll error" ${SYSADM} <<!TEXT!
uupoll: ${UNGETTY} failed to get control of ${MODEM}
!TEXT!
            exit 1
        fi
    fi

    ##
    ##  set up the modem port
    ##
    case ${PARITY} in
        O)
            stty default ${BAUD} ${BITS} -even odd </dev/${MODEM}
            ;;
        E)
            stty default ${BAUD} ${BITS} even -odd </dev/${MODEM}
            ;;
        N)
            stty default ${BAUD} ${BITS} -even -odd </dev/${MODEM}
            ;;
    esac

    ##
    ##  remove unused lock files
    ##
    ${UUCLEAN} >/dev/null 2>&1

    ##
    ##  start up the uucico program
    ##
    ${UUCICO} -s${SYSTEM}

    ##
    ##  if uucico failed, say so
    ##
    if [ -f "${SPOOL}/C.${SYSTEM}n0000" ]
    then
        mail -s "uupoll error" ${SYSADM} <<!TEXT!
uupoll: uucico failed to connect to ${SYSTEM}
!TEXT!
    fi

    ##
    ##  resume the suspended uugetty
    ##
    if [ -f "${UNGETTY}" ]
    then
        ${UNGETTY} -i ${MODEM}
    fi

    ##
    ##  shift to next parameter
    ##
    shift

done

##
##  run the uuxqt program, in case uucico didn't
##
${UUXQT}

##
##  unbatch any news received
##
if [ -f ${NEWS}/rnews ]
then
    ${NEWS}/rnews -U
fi    

##
##  end of uupoll
##

