/* uupick - retrieve files sent by uuto.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

#define FALSE		 0
#define TRUE	    ~FALSE
#define OK		 1
#define FAILED		-1
#define SAME		 0

#ifndef lint
static char *Version = "@(#) uupick 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_P = FALSE;
int opt_s = FALSE;
int opt_u = FALSE;
int opt_x = FALSE;

int errno;			/* used by errno.h */

char spooldir[132];		/* receive directory of current user */
char workdir[132];		/* working directory of current user */
char progname[20];		/* program name */
int dbglvl;

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int checkname();
int copydir();
int copyfile();
int countfiles();
int getname();
uid_t get_uid();
int getuser();
void help();
int movefile();
char query();
void usage();
int view();


/*
 * c h e c k n a m e
 *
 * Check if "name" is a known site
 *
 * Return:	FAILED	Cannot open L.sys file
 *		FAILED	Name not recognised
 *		OK	Name recognised
 */
int checkname(name)
char *name;
{
  char cont[132], line[BUFSIZ];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = strlen(name);
  if (j > SITELEN) j = SITELEN;
  while (fgets(line, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(line)) < (BUFSIZ - 132) &&
			line[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		line[k - 2] = '\0';
		strcat(line, cont);
	}
	for (k = 0 ; k < j ; k++) {
		if (line[k] != name[k])
			break;
	}
	if (k == j) {
		(void) fclose(fplsys);
       		return(OK);
	}
  }
  (void) fclose(fplsys);
  return(FAILED);
}


/*
 * c o p y d i r
 *
 * Dispose of a directory and its contents
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int copydir(srcpath, destpath, queryflg)
char *srcpath, *destpath; int queryflg;
{
  char c, *cp, command[150], source[132], dest[132], question[132], answer[132];
  int j, dirs, files;
  size_t slen, dlen;
  DIR *dirp;
  struct dirent *dp;
  struct stat stat_buff;

  if (opt_P)
	cp = "PUBDIR";
  else
	cp = "PICKDIR";

  if (dbglvl > 0)
	fprintf(stderr, "copydir: checking directory |%s|\n", srcpath);

  if ((j = countfiles(srcpath, &dirs, &files)) == -1)
	return(FAILED);
  else if (j == 0) {
	fprintf(stderr, "No files in %s%s\n", cp, srcpath + strlen(spooldir));
	return(OK);
  }
  else if ((dirp = opendir(srcpath)) == (DIR *)NULL)
	return(FAILED);

  /* get the full current source path */
  strncpy(source, srcpath, 130);
  strcat(source, "/");
  slen = strlen(source);

  /* get the full current destination path */
  strncpy(dest, destpath, 130);
  strcat(dest, "/");
  dlen = strlen(dest);

  /* make the destination directory */
  if ((j = access(destpath, 0)) != 0) {
	if (dbglvl > 0)
		fprintf(stderr, "copydir: making directory |%s|\n", destpath);
#ifdef _MINIX
	if (mkdir(destpath, 0666) == -1)
		return(FAILED);
#endif
#ifdef _SVR2
	sprintf(command, "mkdir %s", destpath);
	if (system(command) != 0)
		return(FAILED);
#endif
	(void) chmod(destpath, 0777);
  }

  /* get all the files and subdirectories */
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	if (strcmp(dp->d_name, ".") == SAME ||
			strcmp(dp->d_name, "..") == SAME)
		continue;

	/* add the current path and source */
	for (j = 0 ; j < 15 ; j++) {
		if ((slen + j) > 131 || (dlen + j) > 131) break;
		source[slen + j] = dp->d_name[j];
		dest[dlen + j] = dp->d_name[j];
	}
	source[slen + j] = '\0';
	dest[dlen + j] = '\0';

	/* find out what we've got */
	(void) stat(source, &stat_buff);
	/* if it's a regular file, copy it */
	if (stat_buff.st_mode & S_IFREG)
		queryflg = copyfile(source, stat_buff.st_size, dest, queryflg);
	/* if it's a subdirectory, check it out */
	else if (stat_buff.st_mode & S_IFDIR) {
		sprintf(question, "Move entire directory %s%s (y/N) ? ",
				cp, source + strlen(spooldir));
		answer[0] = '\0';
		c = query(question, "yYnN", answer);
		if (c == 'y' || c == 'Y') {
			/* build the destination path */
			if (answer[0] != '\0')
				strncpy(dest, answer, 131);
			(void) copydir(source, dest, FALSE);
		}
		else
			(void) copydir(source, dest, TRUE);
	}
	else
		fprintf(stderr, "uupick: unusable input %s\n", source);
  }
  (void) closedir(dirp);

  /* remove an empty destination directory */
  if (strcmp(workdir, destpath) != SAME) {
	if (dbglvl > 0)
		fprintf(stderr, "copydir: removing directory |%s|\n", destpath);
#ifdef _MINIX
	(void) rmdir(destpath);
#endif
#ifdef _SVR2
	sprintf(command, "rmdir %s", destpath);
	(void) system(command);
#endif
  }

  return(OK);
}


/*
 * c o p y f i l e
 *
 * Dispose of a single file
 *
 * Return:	Mode flag	Always
 */
int copyfile(srcpath, srcsize, destpath, queryflg)
char *srcpath, *destpath; off_t srcsize; int queryflg;
{
  char c, *cp, site[20], source[132], dest[132], question[132], answer[132];
  int j, k;

  /* get the site name */
  if (opt_P)
	k = strlen(PUBDIR);
  else
	k = strlen(PICKDIR);
  k++;
  while (srcpath[k] != '/') k++;	/* lose the user subdirectory */
  k++;
  for (j = 0 ; j < 19 ; j++) {
	site[j] = srcpath[k + j];
	if (site[j] == '\0' || site[j] == '/')
		break; 
  }
  if (site[j] == '\0')			/* no site subdirectory found */
	strcpy(site, "Unknown");
  else
	site[j] = '\0';

  /* get a local copy of the paths */
  strncpy(source, srcpath, 131);
  strncpy(dest, destpath, 131);

  /* get the path alone */
  cp = strrchr(dest, '/');
  *cp = '\0';

  /* get the filename alone */
  cp = strrchr(source, '/') + 1;

  /* if queryflg is set, ask what to do */
  if (queryflg) {
	sprintf(question, "File %s, %ld bytes, from site %s [*admvq] ? ",
			cp, srcsize, site);
	do {
		answer[0] = '\0';
		c = query(question, "*aAdDmMvVqQ", answer);
		c &= 0177;
		switch (c) {
		case '*':
			/* list commands */
			help();
			break;
		case 'a':
		case 'A':
			/* move all files */
			if (answer[0] == '\0') {
				if (movefile(srcpath, destpath) == OK)
					printf("Files moved to %s\n", destpath);
			}
			else {
				if (movefile(srcpath, answer) == OK)
					printf("Files moved to %s\n", answer);
			}
			queryflg = FALSE;
			break;
		case 'd':
		case 'D':
			/* delete file */
			if (unlink(srcpath) == 0)
				printf("Deleted %s\n", srcpath);
			else
				printf("Cannot delete %s\n", srcpath);
			break;
		case 'm':
		case 'M':
			/* move file */
			if (answer[0] == '\0') {
				if (movefile(srcpath, destpath) == OK)
					printf("File moved to %s\n", destpath);
			}
			else {
				if (movefile(srcpath, answer) == OK)
					printf("File moved to %s\n", answer);
			}
			break;
		case 'v':
		case 'V':
			/* view file */
			(void) view(srcpath);
			c = '*';
			break;
		case 'q':
		case 'Q':
			/* quit */
			exit(0);
			break;
		default:
			/* ignore */
			c = '*';
			break;
		}
	} while (c == '*');
  }
  /* otherwise just move the file */
  else
	(void) movefile(srcpath, destpath);

  return(queryflg);
}


/*
 * c o u n t f i l e s
 *
 * Count the number of files in a directory
 *
 * Return:	Count	Success
 *		-1	Failure
 */
int countfiles(dirname, d, f)
char *dirname; int *d, *f;
{
  char name[132];
  int j, count;
  size_t len;
  DIR *dirp;
  struct dirent *dp;
  struct stat stat_buff;

  if ((dirp = opendir(dirname)) == (DIR *)NULL) {
	fprintf(stderr, "uupick: cannot open %s\n", dirname);
	return(-1);
  }

  /* build the full current path */
  strncpy(name, dirname, 130);
  strcat(name, "/");
  len = strlen(name);

  count = 0;
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	if (strcmp(dp->d_name, ".") == SAME ||
			strcmp(dp->d_name, "..") == SAME)
		continue;
	/* add the current path and name */
	for (j = 0 ; j < 15 ; j++) {
		if ((len + j) > 131) break;
		name[len + j] = dp->d_name[j];
	}
	name[len + j] = '\0';
	/* find out what we've got */
	if (stat(name, &stat_buff) == -1)
		continue;
	else if (stat_buff.st_mode & S_IFREG) {
		count++;
		(*f)++;
	}
	else if (stat_buff.st_mode & S_IFDIR) {
		count++;
		(*d)++;
	}
  }
  (void) closedir(dirp);

  return(count);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the current uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * h e l p
 *
 * Print a help message on the console
 *
 */
void help()
{

  printf(" *       : Help\n");
  printf(" a [dir] : Move directory\n");  
  printf(" d       : Delete file\n");
  printf(" m [dir] : Move file\n");
  printf(" v       : View file\n");
  printf(" q       : Quit\n");

}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp, tmp[20];
  char locsite[20];			/* name of local site */
  char rmtsite[20];			/* name of remote site */
  char locuser[20];			/* name of local user */
  int j, k;
  uid_t tuid;

  /* set up defaults */
  if (getname(locsite) == FAILED) {	/* get local site name */
	fprintf(stderr, "uupick: cannot find local uucpname\n");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	fprintf(stderr, "uupick: cannot find own username\n");
	exit(1);
  }
  if (getcwd(workdir, 131) == (char *)NULL) {
	fprintf(stderr, "uupick: cannot find own directory name\n");
	exit(1);
  }

  (void) umask(0111);			/* set protection mask to rw-rw-rw- */

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */

  dbglvl = 0;

  /* parse arguments */
  opterr = 0;
  while ((j = getopt(argc, argv, "Ps:u:x:")) != EOF) {
	switch (j & 0177) {
	case 'P':
		opt_P = TRUE;
		break;
	case 's':
		strncpy(rmtsite, optarg, SITELEN);
		opt_s = TRUE;
		break;
	case 'u':
		strncpy(tmp, optarg, 19);
		opt_u = TRUE;
		break;
	case 'x':
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case '?':
		/* else fall through */
	default:
		usage();
		exit(1);
		break;
	}
  }

  /* validity checks */
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid) {
	opt_u = FALSE;
	dbglvl = 0;
  }
#endif

  /* don't check that the user is known */
  if (opt_u) strncpy(locuser, tmp, USERLEN);

  /* check that the site is known */
  if (opt_s && checkname(rmtsite) == FAILED && strcmp(locsite, rmtsite) != SAME) {
	fprintf(stderr, "uupick: cannot find site %s\n", rmtsite);
	exit(1);
  }
  /* the spooling destination directory is PUBDIR/receive/user[/site]
   * or PUBDIR/user
   */
  sprintf(spooldir, "%s/%s", opt_P ? PUBDIR : PICKDIR, locuser);
  /* we can add a specific sitename if necessary */
  if (opt_s) {
	strcat(spooldir, "/");
	if ((strlen(spooldir) + strlen(rmtsite)) > 131) {
		fprintf(stderr, "uupick: cannot understand %s%s\n",
							spooldir, rmtsite);
		exit(1);
	}
	strcat(spooldir, rmtsite);
  }

  if (dbglvl > 0)
	fprintf(stderr, "uupick: checking directory |%s|\n", spooldir);

  /* scan the receive directory for work */
  if ((j = countfiles(spooldir, &k, &k)) == -1)
	exit(1);
  else if (j == 0)
	fprintf(stderr, "No files in %s/%s\n", opt_P ? "PUBDIR" : "PICKDIR",
								locuser);
  else
	(void) copydir(spooldir, workdir, TRUE);

  exit(0);
  /* NOTREACHED */
}


/*
 * m o v e f i l e
 *
 * Move a file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 *  The shell is needed to handle wildcard arguments
 */
int movefile(src, dest)
char *src, *dest;
{
  char command[512];
  int j, k, status;

  sprintf(command, "%s %s %s", MOVE, src, dest);

  if (dbglvl > 0)
	fprintf(stderr, "movefile: cmd |%s|\n     src: |%s|\n    dest: |%s|\n",
							MOVE, src, dest);

  switch (j = fork()) {
	case -1:		/* can't create new process */
		break;
	case 0:			/* forked, try to execute mv */
		/* become who we really are */
		(void) setgid(getgid());
		(void) setuid(getuid());
		(void) execlp(SHELL, "sh", "-c", command, (char *)NULL);
		exit(1);	/* never happen */
		break;
	default:		/* mv running, wait for status */
		while ((k = wait(&status)) != j && k != -1)
			;
		break;
  }
  if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 || WTERMSIG(status) != 0)
	fprintf(stderr, "uupick: move failed\n");

  return(OK);
}


/*
 * q u e r y
 *
 * Get an answer to a question
 *
 * Return:	Letter response
 *		Answer buffer is updated as a side effect
 */
char query(question, valid, answer)
char *question, *valid, *answer;
{
  char ans[132], c = '\0';
  int j, k, m;

  while (c == '\0') {
	(void) fputs(question, stdout);
	(void) fflush(stdout);
	m = 0;
	while ((j = fgetc(stdin)) != EOF) {
		c = (char ) (j & 0177);
		if (c == '\n') break;
		if (c == '\b') m--;
		else ans[m++] = c;
		if (m < 0) m = 0;
		if (m > 130) break;
	}
	if (m == 0 && j == EOF) c = '\n';
	ans[m] = '\0';
	if (m != 0) c = ans[0];

	/* simple newline return gives default response */
	if (c == '\n') {
		break;
	}
	/* check for valid input */
	else if (*valid != '\0' && strchr(valid, c) == (char *)NULL) {
		c = '\0';
	}
	/* sort out any additional argument string */
	else {
		for (k = 1 ; k < m ; k++) {
			if (ans[k] != ' ')
				break;
		}
		m -= k;
		for (j = 0 ; j < m ; j++)
			ans[j] = ans[j + k];
		ans[j] = '\0';
		break;
	}
  }

  strncpy(answer, ans, 131);
  return(c);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-P] [-ssys] [-uusr] [-x#]\n", progname);
}


/*
 * v i e w
 *
 * View the contents of a file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int view(path)
char *path;
{
  char buff[512];
  int j, fd;

  if ((fd = open(path, 0)) == -1)
	return(FAILED);

  while ((j = read(fd, buff, 512)) != 0) {
	if (j == -1)
		break;
	if (write(1, buff, (unsigned int) j) != j) {
		j = -1;
		break;
	}
  }
  (void) close(fd);

  if (j == -1)
	return(FAILED);
  else
	return(OK);
  /* NOTREACHED */
}

