/* uulog - print information from UUCP logfile.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <pwd.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

#define FALSE		0
#define TRUE	   ~FALSE
#define OK		1
#define FAILED	       -1
#define SAME		0
#define MAXARGS	       30
#define PAGE	       22

#define EVERYTHING FALSE /* if true, match for either usr or site or both */

#ifndef lint
static char *Version = "@(#) uulog 1.2 (31 October 1991)";
#endif

/* Global variables */

int errno;		/* used by errno.h */
int dbglvl;
int opt_m;
int count;
char progname[20];

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int all_match();
int checktime();
time_t convert();
int getargs();
int getuser();
int no_match();
int more();
int putline();
int site_match();
void usage();
int user_match();


/*
 * a l l _ m a t c h
 *
 * Match everything
 *
 * Return:	TRUE	Always
 *
 */
int all_match(buff, pattern)
char *buff, *pattern;
{
  return(TRUE);
}


/*
 * c h e c k t i m e
 *
 * See if the logfile entry is later than start time
 *
 * Return:	TRUE	Later
 *		FALSE	Earlier
 */
int checktime(buff, start)
char *buff; time_t start;
{
  char tmp[256], tbuff[256], *cmdflds[MAXARGS];
  int j;
  time_t logtime;

  /* log format: site!user (date time) (id pid task) status (detail) */

  strncpy(tmp, buff, 255);	/* getargs roaches the string */
  for (j = 0 ; j < 10 ; j++) {
	if (tmp[j] == '!') tmp[j] = ' ';
  }
  if (getargs(tmp, cmdflds) < 10) return(FALSE);	/* garbage ? */
  *(cmdflds[5] + strlen(cmdflds[5]) - 1) = ' ';		/* clear trailing ) */
  sprintf(tbuff, "%s %s %s %s", cmdflds[2] + 1, cmdflds[3],
					cmdflds[4], cmdflds[5]);
  logtime = convert(tbuff);

  if (logtime < start)
	return(FALSE);
  else
	return(TRUE);
}


/*
 * c o n v e r t
 *
 * Convert a ctime string to unixtime
 *
 * Return:	unixtime	Success
 *		0		Failure
 *
 * Warning - this is overly simplistic, ignoring localtime and DST
 */
time_t convert(string)
char *string;
{

  /* typical ctime() output is "Wed Jul 08 18:43:07 1987\n\0" */

#define SECS_PER_MIN    ((time_t)60)
#define SECS_PER_HOUR   (60*SECS_PER_MIN)
#define SECS_PER_DAY    (24*SECS_PER_HOUR)
#define SECS_PER_YEAR   (365*SECS_PER_DAY)

  static char *mths_per_year[] =
	{ "Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };
  static int days_per_mth[12] =
	{ 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31 };

  char buff[64], *timflds[MAXARGS];
  int j, year, month, date, hour, min, sec;
  time_t logtime;

  /* parse the time string */
  for (j = 0 ; j < strlen(string) ; j++) {
	if (string[j] == ':')
		buff[j] = ' ';
	else
		buff[j] = string[j];
  }
  if (getargs(buff, timflds) < 6) return((time_t)0);
  date = atoi(timflds[1]);
  hour = atoi(timflds[2]);
  min = atoi(timflds[3]);
  sec = atoi(timflds[4]);
  year = atoi(timflds[5]);

  /* convert the month name */
  for (j = 0 ; j < 12 ; j++) {
	if (strncmp(mths_per_year[j], timflds[0], 3) == SAME)
		break;
  }
  if (j == 12)
	return((time_t)0);	/* unknown month */
  else
	month = j;

  /* convert the year */
  if (year >= 70 && year <= 99)
	year -= 70;
  else if (year >= 1970 && year <= 1999)
	year -= 1970;
  else
	return((time_t)0);	/* unknown year */

  /* count seconds until the start of this year */
  logtime = (SECS_PER_YEAR * year) + (((year + 1) / 4) * SECS_PER_DAY);
  /* add a day if this year is a leap year */
  if ((year + 2) % 4)
	days_per_mth[1] = 28;
  else
	days_per_mth[1] = 29;
  /* count the seconds until the start of this month */
  for (j = 0; j < month; j++) {
	logtime += SECS_PER_DAY * days_per_mth[j];
  }
  logtime += (date - 1) * SECS_PER_DAY;
  logtime += hour * SECS_PER_HOUR;
  logtime += min * SECS_PER_MIN;
  logtime += sec;

  return(logtime);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp, buff[BUFSIZ];
  char logfile[132], locuser[10], s_pattern[10], u_pattern[10];
  int j, g_age, (*s_comp)(), (*u_comp)();
  time_t now, first;
  FILE *fplog;

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  if (getuser(getuid(), locuser) == FAILED) {	/* get local user name */
	fprintf(stderr, "uulog: cannot find local username\n");
	exit(1);
  }
  strncpy(logfile, LOGFILE, 131);		/* default log file */
  (void) time(&now);
  opt_m = FALSE;
  g_age = 0;
  dbglvl= 0;
  count = 0;

#if EVERYTHING
  s_comp = u_comp = no_match;		/* match nothing to start with */
#else
  s_comp = u_comp = all_match;		/* match everything to start with */
#endif

  /* parse arguments */
  opterr = 0;
  if (argc < 2)
	s_comp = u_comp = all_match;
  else {
	while ((j = getopt(argc, argv, "mn:s:u:x:y:")) != EOF) {
		switch (j & 0177) {
		case 'm':
			opt_m = TRUE;
			break;
		case 'n':
			strncpy(logfile, optarg, 131);
			break;
		case 's':
			s_comp = site_match;
			strncpy(s_pattern, optarg, SITELEN);
			break;
		case 'u':
			u_comp = user_match;
			strncpy(u_pattern, optarg, USERLEN);
			break;
		case 'x':
			dbglvl = atoi(optarg);
			break;
		case 'y':
			g_age = atoi(optarg);
			break;
		case '?':
			if ((optopt & 0177) == 'n') {
				strncpy(logfile, NEWSLOG, 131);
				break;
			}
			if ((optopt & 0177) == 'u') {
				u_comp = user_match;
				strncpy(u_pattern, locuser, USERLEN);
				break;
			}
			/* else fall through */
		default:
			usage();
			exit(1);
		}
	}
  }
  /* check options for validity */
  if (g_age < 0) {
	usage();
	exit(1);
  }

  /* sort out the start time */
  first = now - (time_t) g_age * 3600L;

  /* scan throught the logfile */
  if ((fplog = fopen(logfile, "r")) != (FILE *)NULL) {
	while (fgets(buff, BUFSIZ, fplog) != (char *)NULL) {
		if ((g_age == 0 || checktime(buff, first) == TRUE) &&
			((*s_comp)(buff, s_pattern) == TRUE
#if EVERYTHING
				||
#else
				&&
#endif 
				(*u_comp)(buff, u_pattern) == TRUE))
			if (putline(buff) == FAILED) break;
	}
	(void) fclose(fplog);
  }

  exit(0);
  /* NOTREACHED */
}


/*
 * n o _ m a t c h
 *
 * Match nothing
 *
 * Return:	FALSE	Always
 *
 */
int no_match(buff, pattern)
char *buff, *pattern;
{
  return(FALSE);
}


/*
 * m o r e
 *
 * Wait after each screen of data
 *
 * Return:	OK	Keep going
 *		FAILED	Quit
 */
int more()
{
  int j;

  if (opt_m) {
	count++;
	if (count > PAGE) {
		(void) fflush(stdout);
		do {
			j = getchar();
		} while (j != EOF && j != '\n');
		count = 0;
	}
  }

  if (j == EOF)
	return(FAILED);
  else
	return(OK);
}


/*
 * p u t l i n e
 *
 * Output a line to stdout, with wrap
 *
 * Return:	OK	Always
 */
int putline(line)
char *line;
{
  char *bp, *cp;

  bp = cp = line;

  while (TRUE) {
	if (strlen(bp) < 79) {
		(void) fputs(bp, stdout);
		if (more() != OK) return(FAILED);
		break;
	}
	else {
		/* find a convenient breakpoint */
		for (cp = bp + 75 ; *cp != ' ' && *cp != '\t' ; cp--)
			;
		if (cp > bp) {
			*cp = '\0';
			(void) fputs(bp, stdout);
			(void) fputs("\n", stdout);
			bp = cp + 1;
			if (more() != OK) return(FAILED);
			(void) fputs(" ", stdout);
		}
		else {
			(void) fputs(bp, stdout);
			if (more() != OK) return(FAILED);
			break;
		}
	}
  }

  return(OK);
}


/*
 * s i t e _ m a t c h
 *
 * Match a line containing the given site name
 *
 * Return:	TRUE	Success
 *		FALSE	Otherwise
 *
 */
int site_match(buff, pattern)
char *buff, *pattern;
{
  /* log format: site!user (date time) (id pid task) status (detail) */

  char *bp, *cp;

  bp = buff;
  cp = pattern;
  while (*bp++ == *cp++)
	if (*bp == '\0') break;
  bp--;
  if (*bp == '!' || *bp == ' ')		/* some logs use space separator */
	return(TRUE);
  else
	return(FALSE);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-m] [-n [log]] [-s sys] [-u usr] [-y hrs]\n", progname); 
}


/*
 * u s e r _ m a t c h
 *
 * Match a line containing the given user name
 *
 * Return:	TRUE	Success
 *		FALSE	Otherwise
 *
 */
int user_match(buff, pattern)
char *buff, *pattern;
{
  /* log format: site!user (date time) (id pid task) status (detail) */

  char *bp, *cp;

  bp = buff;
  cp = pattern;
  while (*bp != '!' && *bp != ' ')	/* some logs use space separator */
	if (*bp++ == '\0') return(FALSE);
  bp++;
  while (*bp++ == *cp++)
	if (*bp == '\0') break;
  if (*--bp == ' ')
	return(TRUE);
  else
	return(FALSE);
}

