/*
 * UUCP - Perform Unix to Unix File CoPy Requests.
 *
 * Version:	2.2	10/31/91
 *		2.1	08/31/91
 *		2.0	07/31/91
 *		1.9	03/30/91
 *	 	1.8	01/30/91
 *		1.7	12/06/90   
 *		1.6	02/17/90
 *
 * Author:	Fred van Kempen, MINIX User Group Holland
 *			waltje@minixug.hobbynet.nl
 *
 * History:	Added functionality, versions 1.7 to 2.2.  Will Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

/* general constants */

#define LOCAL		 0
#define REMOTE		 1
#define TO		 2 
#define FROM		 3
#define FALSE		 0
#define TRUE	    ~FALSE
#define OK		 1
#define MAYBE		 0
#define FAILED		-1
#define SAME		 0
#define MAXLOCKS	10
#define MAXARGS		30

/* log messages */

#define M_STD		0	/* standard messages */
#define M_NOLOG		1	/* no output to log file */
#define M_ACCESS	2	/* problems in access permissions */
#define M_PARSE		3	/* problems in parsing */
#define M_MATCH		4	/* problems in pattern matching */

#ifndef lint
static char *Version = "@(#) uucp 2.2 (31 October 1991)";
#endif

/* Global variables */

int opt_c = TRUE;		/* Do not link or copy source file (default) */
int opt_C = FALSE;		/* Force copy of source file */
int opt_d = TRUE;		/* Make all necessary directories (default) */
int opt_f = FALSE;		/* Do not make intermediate directories */
int opt_g = FALSE;		/* Set grade of workfile */
int opt_j = FALSE;		/* Output UUCP job number on stderr */
int opt_m = FALSE;		/* Notify sender by mail o/c copy */
int opt_n = FALSE;		/* Notify recipient by mail o/c copy */
int opt_o = FALSE;		/* Report status to specific file */
int opt_r = FALSE;		/* Queue the job, but do not call UUCICO */
int opt_s = FALSE;		/* Use filename as spool directory */
int opt_x = FALSE;		/* Debugging flag */

int errno;			/* used by errno.h */

int mode;			/* FROM, TO, LOCAL or REMOTE */
int cmdseq;			/* command sequence number */
int pid;			/* current pid */
int jobstatus;			/* current job status */
int usrfflg;			/* USERFILE read flag */
int dbglvl;			/* debugging level */

char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char spoolist[MAXLOCKS][LOCKLEN + 1];	/* spool file list */
char userpath[256];		/* USERFILE path list */
char spooldir[132];		/* spool directory */
char workdir[132];		/* intial working directory */
char cmdfile[132];		/* command file name */
char datafile[132];		/* data file name */
char scratch[132];		/* scratch buffer */
char fromfile[132];		/* source of copy */
char tofile[132];		/* destination of copy */
char mailbox[132];		/* redefined mailbox file */
char fromsite[20];		/* from sitename */
char tosite[20];		/* to sitename */
char locsite[20];		/* local site name */
char rmtsite[20];		/* remote site */
char locuser[20];		/* local user name */
char rmtuser[20];		/* remote user name */
char options[20];		/* command option string */
char progname[20];		/* program name */
char grade = 'n';		/* job grade */

int errno;			/* used by errno.h */
int fdlog;			/* system log file descriptor */

FILE *fpcmd;			/* command file pointer */
FILE *fpdata;			/* data file pointer */
time_t now;			/* current time */

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int addcmd();
int addfile();
int catchsignal();
int checklock();
int checkname();
int checkuser();
void cleanup();
int copyfile();
char *expdtilde();
int genname();
int genopt();
int getargs();
int getdate();
int getfiles();
int getname();
int getseq();
uid_t get_uid();
int getuser();
int lstatupdate();
int movefile();
void printmsg();
int rstatupdate();
int setsignal();
int spoolcmd();
int spooluux();
int stripbang();
int unlock();
void usage();
int usercheck();
char *visib();
void wrapup();
int writelog();

/* General pattern matching from Software tools - version 0.2 */

/*
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <dirent.h>
#include <stdio.h>
*/

#define CLOSURE '*'
#define BOL '%'
#define EOL '$'
#define ANY '?'
#define CCL '['
#define CCLEND ']'
#define NEGATE '^'
#define NCCL '!'
#define DASH '-'
#define LITCHAR 'c'
#define ESCAPE '@'
#define NEWLINE '\n'
#define TAB '\t'

#define ENDLINE '\0'
#define ENDSTR '\0'
#define MAXPAT 64
#define MAXSTR 64


/* check for valid alphanumeric: return TRUE or FALSE */
int isvalch(c)
char c;
{
    char *str1 = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
    char *str2 = "1234567890";

    if (strchr(str1, c) == (char *)0 && strchr(str2, c) == (char *)0)
	return(FALSE);
    return(TRUE);
}


/* map s[j] into escaped characters: return the character */
char esc(s, pj)
char *s; int *pj;
{
    char escc;

    if (s[*pj] != ESCAPE)
	escc = s[*pj];
    else if (s[*pj + 1] == ENDSTR)
	escc = ESCAPE;
    else {
	(*pj)++;		/* skip the escape */
	if (s[*pj] == 'n')
		escc = NEWLINE;
	else if (s[*pj] == 't')
		escc = TAB;
	else
		escc = s[*pj];
    }
    return(escc);		/* return the character */
}

/* expand set at src[k] to dest[j], stop at delim: return TRUE or FALSE */
int dodash(delim, src, pk, dest, pj)
char delim, *src, *dest; int *pj, *pk;
{
    char c;

    while (src[*pk] != delim && src[*pk] != ENDSTR) {
	if (src[*pk] == ESCAPE)
		dest[(*pj)++] = esc(src, pk);
	else if (src[*pk] != DASH)
		dest[(*pj)++] = src[*pk];
	else if ((*pj < 2) || src[*pk + 1] == ENDSTR)
		dest[(*pj)++] = DASH;
	else if (isvalch(src[*pk - 1]) &&
			isvalch(src[*pk + 1]) &&
			src[*pk - 1] <= src[*pk + 1]) {
		c = src[*pk - 1] + 1;		/* already got one char */
		while (c != src[*pk + 1]) {
			dest[(*pj)++] = c++;
			if (*pj > MAXSTR - 1)	/* out of space */
				return(FALSE);
		}
	}
	else
		dest[(*pj)++] = DASH;

	if (*pj > MAXSTR - 1) /* out of space */
		return(FALSE);
	(*pk)++;	
    }
    return TRUE;	
} 


/* expand char class at arg[k] into pat[j]: return TRUE or FALSE */
int getccl(arg, pk, pat, pj)
char *arg, *pat; int *pk, *pj;
{
    char buff[8];
    int jstart;

    (*pk)++;		/* skip '[' */
    if (arg[*pk] == NEGATE) {
	pat[(*pj)++] = NCCL;
	(*pk)++;
    }
    else
	pat[(*pj)++] = CCL;

    if (*pj > MAXPAT - 5) {	/* not enough room */
	pat[*pj] = ENDSTR; 
	return(FALSE);
    }	
 
    jstart = *pj;	/* the count goes here */ 
    *pj += 3;		/* space for count */
    if (!dodash(CCLEND, arg, pk, pat, pj))
	return(FALSE);
    sprintf(buff, "%3d ", *pj - jstart);
    strncpy(pat + jstart, buff, 3);
    pat[(*pj)++] = CCLEND;

    return(arg[*pk] == CCLEND);
}


/* build a pattern to handle general expressions
 * return 1 for expansion, 0 for no expansion, -1 for failure
 */
int makepat(pat, arg, start, delim)
char *pat, *arg, delim; int start;
{
    int j, k, m, n, lastj; 

    j = 0;		/* pat index */
    k = start;		/* arg index */
    lastj = 0;
    
    /* start at position start, and end when delim is seen */
    while (arg[k] != delim && arg[k] != ENDSTR && j < MAXPAT - 1) {
	m = j;
	if (arg[k] == ANY)
		pat[j++] = ANY;
	else if (arg[k] == BOL && k == start)	/* valid only at start */
		pat[j++] = BOL;
	else if (arg[k] == EOL && arg[k + 1] == delim) /* and only at end */
		pat[j++] = EOL;
	else if (arg[k] == CCL) {
		if (!getccl(arg, &k, pat, &j)) {
			return(-1);
		}
	}
	else if (arg[k] == CLOSURE && k > start) {
		m = lastj;
		if (pat[m] == BOL || pat[m] == EOL || pat[m] == CLOSURE)
			return(-1);		/* can't replicate these */
		else {				/* move the pattern along */
			for (n = j++ ; n > lastj ; n--)
				pat[n] = pat[n - 1];
			pat[n] = CLOSURE;
		}
	}
	else {
		pat[j++] = LITCHAR;
		pat[j++] = esc(arg, &k);
	}
	lastj = m;
	k++;
    }
    pat[j] = ENDSTR;

    /* check for any expansion */
    for (m = 0 ; m < j ; m += 2) {
	if (pat[m] != LITCHAR)
		break;
    }

    if (arg[k] == delim)	/* ok */
	return((pat[m] == EOL) ? 0 : 1);
    else
	return(-1);
}
		

/* look for c in character class at pat[offset]: return TRUE or FALSE */
int locate(c, pat, offset)
char c, *pat; int offset;
{
    char buff[4];
    int j, k;

    /* size of class is at pat[offset], characters follow */ 
    strncpy(buff, pat + offset, 3);
    k = offset + atoi(buff);

    for (j = offset + 3 ; j < k ; j++)
	if (c == pat[j]) return(TRUE);

    return(FALSE);
}


/* match one pattern entry at pat[j]: return the advance made in the line */
int omatch(line, m, pat, j)
char *line, *pat; int m, j;
{
    int advance = FAILED;

    if (line[m] == ENDLINE && pat[j] == EOL)	/* ENDLINE may be ENDSTR */
	advance = 0;
    else if (line[m] != ENDSTR) {
	switch (pat[j]) {
		case LITCHAR:
			if (line[m] == pat[j + 1])
				advance = 1;
			break;
		case BOL:
			if (m == 0)
				advance = 0;
			break;
		case ANY:
			if (line[m] != ENDLINE)
				advance = 1;
			break;
		case EOL:
			if (line[m] == ENDLINE)
				advance = 0;
			break;
		case CCL:
			if (locate(line[m], pat, j + 1))
				advance = 1;
			break;
		case NCCL:
			if (line[m] != ENDLINE && !locate(line[m], pat, j + 1))
				advance = 1;
			break;
		default:
			printmsg(M_MATCH, "omatch: never happen");
			break;
	}
    }
    return(advance);
}	


/* find size of pattern entry at pat[n]: return size */
int patsize(pat, n)
char *pat; int n;
{
    char buff[4];
    int j;

    switch (pat[n]) {
	case LITCHAR:	j = 2;
			break;

	case BOL:
	case EOL:
	case ANY:	j = 1;
			break;

	case CCL:
	case NCCL:	strncpy(buff, pat + n + 1, 3);
			j = atoi(buff) + 2;
			break;

	case CLOSURE:	j = 1;
			break;

	default:	printmsg(M_MATCH, "patsize: never happen");
			break;
    }
    return(j);
}


/* match from line[j], pat[k]: return the new relative position in the line */
int amatch(line, j, pat, k)
char *line, *pat; int j, k;
{
    int n, start, offset;

    start = j;
    while (pat[k] != ENDSTR) {
	if (pat[k] == CLOSURE) {
		offset = j;
		k += patsize(pat, k);	/* step over closure */
		while (line[offset]) {
			if ((n = omatch(line, offset, pat, k)) == FAILED)
				break;		/* failure on this pass */ 
			else
				offset += n;	/* match the longest pattern */
		}

		/* offset points to character that caused failure */
		/* match rest of pattern against rest of line */
		/* shrink closure by 1 after each failure */ 

		if ((n = patsize(pat, k)) == FAILED)
			break;
		else
			k += n;
		while (offset >= j) {
			if ((n = amatch(line, offset, pat, k)) == FAILED)
				offset--;	/* no match so far */
			else
				break;
		}
		j += offset;
		break;
	}
	else if ((n = omatch(line, j, pat, k)) == FAILED) {
		j = start;
		break;
	}
	else {
		j += n;			/* increment the line position */
		k += patsize(pat, k);	/* increment the pattern position */
	}
    }
    if (n == FAILED) 
	return(FAILED);
    else
	return(j - start);	
} 

/* End of Software Tools pattern matching code */


/*
 * a d d c m d
 * 
 * Add a command to a command file, generating the appropriate datafile
 * 
 * Return:	FAILED	Cannot link/copy datafile
 *		OK	Otherwise
 */
int addcmd(file, flg)
char *file; int flg;
{
  char name[132];

  /* if the file is not world-readable, we must spool it regardless */
  if (flg && opt_c) {
	strncpy(datafile, NODATAFN, 131);	/* placemarker, no file */
  }
  else {					/* generate data file name */
	if (genname('D', 'n', tosite) == 0) return(FAILED);
	strncpy(datafile, scratch, 131); 
	/* Make "from" an absolute pathname; '~' must have been removed */
	if (*file == '/')
		strncpy(name, file, 131);
	else {
		sprintf(name, "%s/", workdir);
		strncat(name, file, 131 - strlen(name));
	}
	if (movefile(name, datafile) == FAILED) { /* link/copy data file */
		return(FAILED);
	}
	(void) chmod(datafile, 0600);
  }

  printmsg(M_ACCESS, "addcmd: file %s dname %s", file, datafile);
  fprintf(fpcmd, "S %s %s %s %s %s 0666 %s %s\n",
	fromfile, tofile, locuser, options, datafile, rmtuser,
	(opt_m && opt_o) ? mailbox : " ");

  return(OK);
}


/*
 * a d d f i l e
 *
 * Add a file name to the spool file list used by cleanup()
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int addfile(name)
char *name;
{
  char file[132];
  int j;

  /* build the file name */
  if (*name == '/') {
	strncpy(file, name, 131);
  }
  else {
	sprintf(file, "%s/", SPOOLDIR);
	strncat(file, name, 131 - strlen(file));
  }

  /* add it to the list */
  for (j = 0 ; j < MAXLOCKS ; j++) {
	if (spoolist[j][0] == '\0') {
		strncpy(spoolist[j], file, LOCKLEN);
		printmsg(M_ACCESS, "addfile: spool file %s listed", file);
		break;
	}
  }

  /* see if we succeeded */
  if (j == MAXLOCKS)
	return(FAILED);
  else
	return(OK);
}


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT are ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining
 * ten or so signals are scarce enough, and rare enough, to be let run.
 * One problem is that the shell protects background commands from
 * SIGINT but not SIGHUP, so if SIGHUP is ignored only SIGKILL can stop a
 * background process, leaving pieces all over the floor.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, SIG_IGN) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sig;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT, the only un-ignored signal */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c h e c k n a m e
 *
 * Check if "name" is a known site
 *
 * Return:	FAILED	Cannot open L.sys file
 *		FAILED	Name not recognised
 *		OK	Name recognised
 */
int checkname(name)
char *name;
{
  char cont[132], line[BUFSIZ];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = strlen(name);
  if (j > SITELEN) j = SITELEN;
  while (fgets(line, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(line)) < (BUFSIZ - 132) &&
			line[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		line[k - 2] = '\0';
		strcat(line, cont);
	}
	for (k = 0 ; k < j ; k++) {
		if (line[k] != name[k])
			break;
	}
	if (k == j) {
		(void) fclose(fplsys);
       		return(OK);
	}
  }
  (void) fclose(fplsys);
  return(FAILED);
}


/*
 * c h e c k u s e r
 *
 * Check if a given user/site has access to a given path
 *
 * Return:	OK		Yes
 *		FAILED		No
 *		usrfflg is updated as a side effect 
 */
int checkuser(site, user, path)
char *site, *user, *path;
{
  char *cp, *bp;
  int flg;

  if (usrfflg == MAYBE)
	usrfflg = usercheck(site, user);

  if (usrfflg == FAILED)
	return(FAILED);

  if (path == (char *)NULL)
	return(OK);
 
  printmsg(M_ACCESS, "checkuser: path |%s| userpath |%s|", path, userpath);
 
  cp = userpath;
  bp = path;
  flg = FAILED;
  while (TRUE) {
	/* successful match */
	if (*cp == '\0' || *cp == ' ') {
		flg =  OK;
		break;
	}
	/* ok so far */
	else if (*bp == *cp) {
		bp++;
		cp++;
	}
	else {
		/* reset bp */
		bp = path;
		/* use up the rest of this userpath entry */
		while (*cp && !isspace(*cp))
			cp++;
		while (isspace(*cp))
			cp++;
		/* no more possibilities */
		if (*cp == '\0')
			break;
	}
  }
 
  return(flg);
}


/*
 * c o p y f i l e
 *
 * Copy a file, making directories as necessary 
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int copyfile(from, to)
char *from, *to;
{
  char *dirp[16], *cp;
  int j, k, flg, status, level, depth;

  flg = FAILED;		/* pessimistic, or realistic */
  if (to[0] != '/') {	/* the path is wrong */
	printmsg(M_ACCESS, "copyfile: path %s invalid", to);
	return(flg);
  }
  if (access(to, 0) == 0) { /* the file already exists */
	printmsg(M_ACCESS, "copyfile: file %s exists", to);
	return(flg);
  }
  if (mode == LOCAL && access(from, 4) == -1) { /* no read permission */
	printmsg(M_ACCESS, "copyfile: file %s inaccessible", from);
	return(flg);
  }

  /* check the parent directory */
  strncpy(scratch, to, 131);
  cp = strrchr(scratch, '/');
  *cp = '\0';

  printmsg(M_ACCESS, "copyfile: checking directory %s", scratch);
  if (cp == scratch) {			/* root directory */
	if (access("/", 0) == 0)
		flg = movefile(from, to);
  }
  else if (access(scratch, 0) == 0) {	/* parent directory */
	flg = movefile(from, to);
  }
  else if (!opt_d) {		/* we cannot make directories */
	printmsg(M_ACCESS, "copyfile: need directory %s", scratch);
  }
  else if (opt_d) {		/* we can try and make directories */
	/* parse the path string */
	depth = 0;
	for (cp = scratch ; *cp ; cp++) {
		if (*cp == '/')
			dirp[depth++] = cp;
 	}
	dirp[depth] = cp;

	for (level = depth - 1 ; level > 0 ; level-- ) {
		*dirp[level] = '\0';
		if (access(scratch, 0) == 0) break; /* directory exists */
	}

	/* check for real user's write permission in existing parent */
	if (mode == LOCAL && strncmp(scratch, PUBDIR, strlen(PUBDIR)) != SAME
				&& access(scratch, 2) == -1)
		return(flg);

	/* try to make the required directories */
	while (level < depth) {
		*dirp[level] = '/';

		switch (j = fork()) {	/* run mkdir */
		case -1:		/* can't create new process */
			break;
		case 0:			/* forked, try to execute mkdir */
			(void) fflush(stderr);
			(void) freopen("/dev/null", "w", stderr);
			(void) execlp(MKDIR, "mkdir", scratch, (char *)NULL);
			exit(1);	/* never happen */
			break;
		default:		/* mkdir running, wait for status */
			while ((k = wait(&status)) != j && k != -1)
				;
			break;
		} 
		if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 ||
						WTERMSIG(status) != 0) {
			printmsg(M_ACCESS, "copyfile: cannot make directory %s",
							scratch);
			break;
		}
		else
			(void) chmod(scratch, 0777);
		level++;
	}
	if (j > 1 && k > 1 && WEXITSTATUS(status) == 0 && WTERMSIG(status) == 0)
		flg = movefile(from, to);
  }

  return(flg);
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files and exit
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  if (dbglvl > 0)		/* no harm in trying */
	(void) close(fdlog);

  /* remove all our lockfiles, command files and data files */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0') {
		(void) unlink(locklist[j]);
		printmsg(M_ACCESS, "cleanup: lockfile %s removed", locklist[j]);
	}
	if (spoolist[j][0] != '\0') {
		(void) unlink(spoolist[j]);
		printmsg(M_ACCESS, "cleanup: spool file %s removed", spoolist[j]);
	}
	j++;
  }

  exit(1);
}


/*
 * e x p d t i l d e
 *
 * Expand a filename using the usual "~" convention
 *
 * Return:	A valid pointer		SUCCESS
 *		A NULL pointer		Local filename not parsed
 *		scratch buffer is updated as a side effect
 *
 * The file "~user" expands to the home directory of "user". "~/" expands
 * to the home directory of the effective userid, which in this case
 * is usually the /usr/spool/uucppublic directory. Care is taken not
 * to overflow the (static) name buffer.
 */
char *expdtilde(filename)
char *filename;
{
  register char *pc, *pd;
  int j;
  struct passwd *pw;

  for (j = 0 ; j < 132 ; j++) scratch[j] = '\0';
  pc = scratch;
  pd = filename;

  /* if no leading ~, nothing to do */
  if (*pd++ != '~') {
	strncpy(scratch, filename, 131);
	return(scratch);
  }

  /* get the user name from the path */
  if (*pd != '/') {
	while ((*pc++ = *pd++) && *pd != '/')
		;
	if (*pd != '/') pd--;
  }
  /* pd now points to the end of the user name, for ~user,
   * or to the start of the path, for ~/.
   */

  /* use /usr/spool/uucppublic for ~/ */
  if (strlen(scratch) == 0)
	pw = getpwnam("uucp");
  else
	pw = getpwnam(scratch);

  /* check for valid username */
  if (pw == (struct passwd *)NULL)
	return((char *)NULL);

  /* check for valid path */
  if ((strlen(pw->pw_dir) + strlen(pd)) > 131)
	return((char *)NULL);

  strcpy(scratch, pw->pw_dir);
  strcat(scratch, pd);
  return(scratch);
}


/*
 * g e n n a m e
 *
 * Create a unique UUCP file name
 *
 * Return:	0		Failure (no sequence number)
 *		Number		Otherwise
 *		scratch buffer is updated as side effect
 *		SEQF file is updated as side effect
 *
 * The filename is a combination of prefix, grade, site name and a
 * sequential number taken from the SEQF file.
 */
int genname(prefix, grd, site)
char prefix, grd, *site;
{
  char *pc, *pd, sbuf[132];
  int seqnum;

  pc = sbuf;
  pd = site;
  while ((*pc++ = *pd++) && *pd != '!')	/* multiple bangs ? */
	;

  seqnum = getseq();
  sprintf(scratch, "%c.%s%c%04d", prefix, sbuf, grd, seqnum);

  printmsg(M_PARSE, "genname: prefix %c, site %s, grade %c, seq %d",
		prefix, site, grd, seqnum);

  return(seqnum);
}


/* g e n o p t
 *
 * Generate option string for command file
 *
 * Return:	OK	Always
 */
int genopt(cp)
char *cp;
{
  *cp++ = '-';
  if (opt_c) *cp++ = 'c';	/* do not link or copy source file */
  if (opt_C) *cp++ = 'C';	/* force copy of source file */
  if (opt_d) *cp++ = 'd';	/* make all necessary directories */
  if (opt_f) *cp++ = 'f';	/* do not make intermediate directories */
  if (opt_m) *cp++ = 'm';	/* notify sender by mail o/c transfer */
  if (opt_n) *cp++ = 'n';	/* notify recipient by mail o/c transfer */
  if (opt_m && opt_o) *cp++ = 'o'; /* report to specific mailbox file */
  *cp = '\0';
  return(OK);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t d a t e
 *
 * Get the date in a format suitable for the logfile
 *
 * Return	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(buff)
char *buff;
{
  int j;

  sprintf(buff, "%s", ctime(&now));
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[j] = '\0';			/* clear the newline */

  return(OK);
}


/*
 * g e t f i l e s
 * 
 * Find matching filenames in a directory, and add a command to a workfile
 * or copy the files directly.
 *
 * Return:	No. of files added	Success
 *		-1			Otherwise
 *		Fromfile, tofile are pdated as a side effects
 */
int getfiles(path)
char *path;
{
  char *cp, *bp, buff[132], dirname[132], fname[64], pat[64];
  int j, k, wflg, flg;
  DIR *dirp;
  struct dirent *dp;
  struct stat sb;

  /* split the path into a directory and afn */
  if ((cp = strrchr(path, '/')) == (char *)NULL) {
	printmsg(M_PARSE, "getfiles: cannot parse file path %s", path);
	return(-1);
  }
  j = 0; 
  bp = path;
  while (*bp) {
  	if (bp < cp) dirname[j++] = *bp;
  	else if (bp == cp) {
		dirname[j] = '\0';
		j = 0;
	}
  	else if (cp < bp) fname[j++] = *bp;
	bp++;
  }
  fname[j] = '\0';
  printmsg(M_PARSE, "getfiles: directory |%s| file |%s|", dirname, fname);

  if (strlen(dirname) > (131 - 14)) {
	printmsg(M_PARSE, "getfiles: directory name |%s| too long", dirname);
	return(-1);
  }
  if (strlen(tofile) > (131 - 14)) {
	printmsg(M_PARSE, "getfiles: tofile name |%s| too long", tofile);
	return(-1);
  }
  printmsg(M_PARSE, "getfiles: tofile |%s|", tofile);

  /* translate standard filename metacharacters to regexp metacharacters */
  for (j = 0, k = 0 ; fname[j] != '\0' ; j++) {
	if (fname[j] == '*')
		buff[k++] = '?';
	buff[k++] = fname[j];
  }
  buff[k++] = '$';
  buff[k] = '\0'; 		

  /* try to form regular expression from argument */
  if ((j = makepat(pat, buff, 0, ENDSTR)) == -1) {
	printmsg(M_MATCH, "getfiles: cannot understand pattern |%s|", buff);
	return(-1);
  }
  wflg = (j == 1) ? TRUE : FALSE;	/* for convenience */
  printmsg(M_MATCH, "getfiles: buf |%s| pat |%s| ret %d", buff, pat, j);

  /* try to open work directory */
  if ((dirp = opendir(dirname)) == (DIR *)NULL) {
	printmsg(M_ACCESS, "getfiles: cannot open directory %s", dirname);
	return(-1);
  }

  /* get pointer to start of input file name */
  bp = fromfile + strlen(dirname) + 1;

  /* The tofile is a directory if the fromfile contains wildcards, or if
   * the tofile has a terminating '/'.  In the first case, wflg is already
   * set.  In the second case, it needs setting.  In either case, tofile
   * must be terminated with '/'.
   */
  cp = tofile + strlen(tofile);
  if (wflg && *(cp - 1) != '/') {
	*cp++ = '/';
	*cp = '\0';
  }
  else if (*(cp - 1) == '/')
	wflg = TRUE;

  /* look for matching files and add commands to work file, or copy */
  j = 0;
  fname[15] = '\0';
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	strncpy(fname, dp->d_name, 14);
	sprintf(buff, "%s/%s", dirname, fname);
	if (strcmp(".", fname) == 0 || strcmp("..", fname) ==  0) continue;
	if (stat(buff, &sb) == FAILED) continue;
	if ((sb.st_mode & S_IFMT) == S_IFDIR) continue;
	/* if match found, do the right thing */
	if (amatch(fname, 0, pat, 0) != FAILED) {
		if (sb.st_mode & S_IROTH) flg = TRUE;
		else flg = FALSE;
		strcpy(bp, fname);
		if (wflg) strcpy(cp, fname);
		if ((mode == LOCAL && copyfile(fromfile, tofile) == FAILED) ||
			(mode != LOCAL && addcmd(fname, flg) == FAILED)) {
			j = -1;
			break;
		}
		j++;
	}
  }
  (void) closedir(dirp);

  /* clear any trailing '/' */
  if (wflg) *(--cp) = '\0';

  return(j);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t s e q
 *
 * Get the next sequence number, update the sequence file
 *
 * Return:	1-9999	Success
 *		0	Failure
 */
int getseq()
{
  int fd, seq, flg;

  if (checklock(LCKSEQ, 2) == FAILED) {
	printmsg(M_ACCESS, "getseq: cannot create LCK.SEQ file");
	return(0);
  }

  /* if the SEQF file doesn't exist, create it */
  flg = TRUE;
  if (access(SEQF, 0) == -1) {
  	if ((fd = open(SEQF, O_CREAT | O_WRONLY | O_TRUNC, 0600)) == -1)
		flg = FALSE;
	else {
		if (write(fd, "0001", 4) != 4)
			flg = FALSE;
		if (close(fd) == -1)
			flg = FALSE;
	}	
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* read the seq file */
  if ((fd = open(SEQF, O_RDONLY, 0600)) == -1)
	flg = FALSE;
  else {
	if (read(fd, scratch, 4) != 4)
		flg = FALSE;
	else {
		scratch[4] = '\0';
		seq = atoi(scratch);
        }
	(void) close(fd);		/* about to overwrite */
	if (seq < 1 || seq > 9999)
		flg = FALSE;
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* write new seq number */
  if ((fd = open(SEQF, O_WRONLY | O_TRUNC, 0600)) == -1)
	flg = FALSE;
  else {
	sprintf(scratch, "%04d", seq < 9999 ? seq + 1 : 1);
	if (write(fd, scratch, 4) != 4)
		flg = FALSE;
	if (close(fd) == -1)
		flg = FALSE;
  }
  (void) unlock(LCKSEQ);
  printmsg(M_ACCESS, "getseq: got sequence number %d", seq);
  if (flg)
	return(seq);
  else
	return(0);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * l s t a t u p d a t e
 *
 * Initialise an L_stat record if necessary
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int lstatupdate()
{
  char tfile[132], zbuff[sizeof(struct lstatrec)], buff[BUFSIZ];
  int j, fds, fdt, flg;
  sig_t sig;
  size_t len;
  struct lstatrec *lstat;

  /* set up to read L_stat records */
  len = sizeof(struct lstatrec);
  lstat = (struct lstatrec *)zbuff;
  strncpy(lstat->site, rmtsite, SITELEN);
  lstat->time_a = now;
  lstat->code = ET_QUEUED;
  lstat = (struct lstatrec *)buff;

  j = access(LSTAT, 0);
  printmsg(M_ACCESS, "lstatupdate: L_stat access call returns %d", j);
  if (j == -1) return (OK);	/* no file, don't bother */

  /* if the site is already listed in L_stat, fine */
  flg = FALSE;
  if ((fds = open(LSTAT, O_RDONLY, 0644)) == -1) {
	return(FAILED);
  }
  while (read(fds, buff, len) == len) {
	if (strncmp(lstat->site, rmtsite, SITELEN) == SAME) {
		flg = TRUE;
		break;
	}
  }
  (void) close(fds);
  if (flg) return(OK);

  /* we need to add a new site name */
  printmsg(M_ACCESS, "lstatupdate: new entry |%s|", visib(zbuff, len));

  if (checklock(LCKLSTAT, 2) == FAILED) return(FAILED); /* lock not possible */

  /* disable SIGINT while the R_stat and L_stat files are moved */
  if ((sig = signal(SIGINT, SIG_IGN)) == BADSIG) {
	(void) unlock(LCKLSTAT);
	return(FAILED);
  }

  /* move L_stat to a temporary file */
  sprintf(tfile, "%sL%04d", LTMPATH, pid);
  (void) unlink(tfile);		/* in case it already exists */
  if (link(LSTAT, tfile) == -1 || unlink(LSTAT) == -1) {
	(void) unlink(tfile);
	(void) unlock(LCKLSTAT);
	return(FAILED);
  }

  if ((fds = open(LSTAT, O_CREAT | O_WRONLY, 0644)) == -1 ||
		(fdt = open(tfile, O_RDONLY, 0644)) == -1) {
	(void) link(tfile, LSTAT); /* recover if possible */
	(void) unlink(tfile);
	(void) unlock(LCKLSTAT);
	return(FAILED);
  }
  (void) unlink(tfile);	

  flg = FALSE;
  while (read(fdt, buff, len) == len) {
	if (!flg && strncmp(lstat->site, rmtsite, SITELEN) > 0) {
		if (write(fds, zbuff, len) != len) {
			(void) close(fds);	/* do not unlock - file corrupt */
			(void) close(fdt);
			return(FAILED);
		}
		flg = TRUE;
	}
	if (write(fds, buff, len) != len) {
		(void) close(fds);	/* do not unlock - file corrupt */
		(void) close(fdt);
		return(FAILED);
	}
  }
  if (!flg) {
	if (write(fds, zbuff, len) != len) {
		(void) close(fds);	/* do not unlock - file corrupt */
		(void) close(fdt);
		return(FAILED);
	}
  }
 
  (void) close(fds);
  (void) close(fdt);
  (void) unlock(LCKLSTAT);

  /* re-enable SIGINT */
  if (signal(SIGINT, sig) == BADSIG)
	return(FAILED);
  else
	return(OK);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp;
  int j, from, to, frombang, tobang;
  uid_t tuid;
#ifndef TESTDBG
  dbglvl = 0;
#else
  FILE *fp;
  if ((fp = fopen(DEFDBG, "r")) != (FILE *)NULL &&
		fgets(scratch, 132, fp) != (char *)NULL &&
		sscanf(scratch, "%d", &j) == 1)
	dbglvl = j;		/* default debug level */
  else
	dbglvl = 0;
  if (fp != (FILE *)NULL) (void) fclose(fp);
#endif

  /* set various defaults */
  fpdata = (FILE *)NULL;
  fpcmd = (FILE *)NULL;
  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  mailbox[0] = rmtsite[0] = rmtuser[0] = userpath[0] = '\0';
  usrfflg = MAYBE;
  if (getenv("JOBNO") != (char *)NULL)
	opt_j = TRUE;

  /* parse options */
  opterr = 0;
  while ((j = getopt(argc, argv, "cCdfg:jm:n:rs:x:")) != EOF)
	switch (j & 0377) {
	case 'c':		/* do not link or copy source file */
		opt_c = TRUE;
		opt_C = FALSE;
		break;
	case 'C':		/* force copy of source file */
		opt_C = TRUE;
		opt_c = FALSE;
	        break;
	case 'd':		/* make all necessary directories */ 
		opt_d = TRUE;
		opt_f = FALSE;
		break;
	case 'f':		/* do not make intermediate directories */
		opt_f = TRUE;
		opt_d = FALSE;
		break;
	case 'g':		/* set grade of workfile */
		if (isalnum(optarg[0]) && optarg[1] == '\0')
			grade = optarg[0];
	    	else
			printmsg(M_NOLOG, "uucp: unknown grade %s ignored", optarg);
		break;
	case 'j':		/* enable/suppress UUCP job number on stderr */
		opt_j = !opt_j;
		break;
	case 'm':		/* notify sender by mail o/c copy */
		/* if there's a leading '/', we've defined a mailbox */
		strncpy(mailbox, optarg, 131);
		if (mailbox[0] == '/')
			opt_o = TRUE;
		opt_m = TRUE;
		break;
	case 'n':		/* notify recipient by mail o/c copy */
		/* if there's an argument, we've defined an addressee */
		strncpy(rmtuser, optarg, USERLEN);
	  	opt_n = TRUE;
		break;
	case 'r':		/* do not call uucico, spool the job */
		opt_r = TRUE;
		break;
	case 's':		/* use file as spool directory */
		strncpy(spooldir, optarg, 131);
		opt_s = TRUE;
		break;
	case 'x':		/* set debug level */
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case '?':
		if ((optopt & 0177) == 'm') { /* mailbox defaults to owner */
 			opt_m = TRUE;
 			break;
		}
 		else if ((optopt & 0177) == 'n') { /* addressee defaults to recipient */
 			opt_n = TRUE;
			break;
 		}
 		/* else fall through */
	default:
		usage();
		exit(1);
	}

  from = optind++ ;			/* from file argument */
  to = optind;				/* to file argument */

  /* can't cope without filenames, or with leading '.' or '!' */
  if (optind >= argc || argv[to][0] == '.' || argv[to][0] == '!' ||
		argv[from][0] == '.' || argv[from][0] == '!') {
	usage();
	exit(1);
  }

#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid)
	dbglvl = 0;
#endif

  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  /* get local site name, local user, and pid */
  if (getname(locsite) == FAILED) {
	printmsg(M_NOLOG, "uucp: cannot find local uucpname");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	printmsg(M_NOLOG, "uucp: cannot find local username");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	printmsg(M_NOLOG, "uucp: cannot find own pid");
	exit(1);
  }
  if (getcwd(workdir, 131) == (char *)NULL) {
	printmsg(M_NOLOG, "uucp: cannot find own directory name");
	exit(1);
  }
  if (chdir(spooldir) == FAILED) {	/* switch to spool directory */
	printmsg(M_NOLOG, "uucp: cannot change to directory %s", spooldir);
	exit(1);
  }
  /* set up to catch signals */
  if (catchsignal() == FAILED) {
	printmsg(M_NOLOG, "uucp: cannot catch signals");
	exit(1);
  }

  /* set defaults not already set by options */
  if (rmtsite[0] == '\0')		/* default remote site */
	strncpy(rmtsite, locsite, SITELEN);
  if (rmtuser[0] == '\0')		/* default remote user name */
	strncpy(rmtuser, locuser, USERLEN);
  if (mailbox[0] == '\0')		/* default local user name */
	strncpy(mailbox, locuser, USERLEN);

  /* set up the log file; if debugging, lock it for the duration */
  if (dbglvl > 0) {
	if (checklock(LCKLOG, 2) == FAILED) {
		printmsg(M_NOLOG, "uucp: cannot get lock %s", LCKLOG);
		exit(1);
	}
	if ((fdlog = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1) {
		printmsg(M_NOLOG, "uucp: cannot open %s", LOGFILE);
		(void) unlock(LCKLOG);
		exit(1);
	}
	printmsg(dbglvl, "========== %-24.24s ==========", ctime(&now));
  }

  printmsg(M_ACCESS,
	"uucp: locsite %s locuser %s workdir %s\n      logfile %s spooldir %s",
	locsite, locuser, workdir, LOGFILE, spooldir);

  /* strip leading bang from path and parse arg */
  frombang = (stripbang(argv[from], fromfile, fromsite) == OK);
  tobang = (stripbang(argv[to], tofile, tosite) == OK);

  printmsg(M_PARSE, "uucp: fromarg %s, fromfile %s, fromsite %s",
	argv[from], fromfile, fromsite);
  printmsg(M_PARSE, "uucp: toarg %s, tofile %s, tosite %s",
	argv[to], tofile, tosite);

  /* check for use of local site in bang path */
  if (frombang && strncmp(fromsite, locsite, SITELEN) == SAME &&
		(cp = strchr(argv[from], '!')) != (char *) NULL) {
	argv[from] = ++cp;
	frombang = FALSE;
  }
  if (tobang && strncmp(tosite, locsite, SITELEN) == SAME &&
		(cp = strchr(argv[to], '!')) != (char *) NULL) {
	argv[to] = ++cp;
	tobang = FALSE;
  }

  /* check for known sites, and if unknown give up now */
  if (frombang && checkname(fromsite) == FAILED) {
	printmsg(M_ACCESS, "uucp: unknown site %s", fromsite);
	wrapup("unknown site %s", fromsite);
  }
  /* accept ignorance of the tosite, if we already have a valid fromsite */
  if (!frombang && tobang && checkname(tosite) == FAILED) {
	printmsg(M_ACCESS, "uucp: unknown site %s", tosite);
	wrapup("unknown site %s", tosite);
  }

  /* prepare the command and data files */
  if (!frombang && !tobang) {		/* local copy */
	mode = LOCAL;

	if (expdtilde(argv[from]) == (char *)NULL) {
		printmsg(M_ACCESS, "uucp: illegal filename %s", argv[from]);
		wrapup("cannot parse %s", argv[from]);
	}
	if (scratch[0] != '/') {
		strcpy(fromfile, workdir);
		strcat(fromfile, "/");
		if ((strlen(fromfile) + strlen(scratch)) > 131) {
			printmsg(M_ACCESS, "uucp: illegal filename %s%s",
				fromfile, scratch);
			wrapup("cannot parse %s", scratch);
		}
		strcat(fromfile, scratch);
	}
	else
		strncpy(fromfile, scratch, 131);

	if (expdtilde(argv[to]) == (char *)NULL) {
		printmsg(M_ACCESS, "uucp: illegal filename %s", argv[to]);
		wrapup("cannot parse %s", argv[to]);
	}
	if (scratch[0] != '/') {
		strcpy(tofile, workdir);
		strcat(tofile, "/");
		if ((strlen(tofile) + strlen(scratch)) > 131) {
			printmsg(M_ACCESS, "uucp: illegal filename %s%s",
				tofile, scratch);
			wrapup("cannot parse %s", scratch);
		}
		strcat(tofile, scratch);
	}
	else
		strncpy(tofile, scratch, 131);
  }
  else if (frombang && tobang) {	/* remote operation */
	mode = REMOTE;

	/* clear the remote username set by the -n option */
	opt_n = FALSE;
	strncpy(rmtuser, locuser, USERLEN);	/* default remote user name */
	strncpy(rmtsite, fromsite, SITELEN);	/* default site for operation */

	/* generate an X command string, and send it to fromsite */
	sprintf(scratch, "%s!%s", locsite, argv[to]); /* undo parsing */
	strncpy(tofile, scratch, 131);
	if (spoolcmd() == FAILED) {
		printmsg(M_ACCESS, "uucp: failed to spool remote workfile");
		wrapup("spool failure", "");
	}
  }
  else if (frombang) {			/* from file is remote */
	mode = FROM;

	/* clear the remote username set by the -n option */
	opt_n = FALSE;
	strncpy(rmtuser, locuser, USERLEN);	/* default remote user name */
	strncpy(rmtsite, fromsite, SITELEN);

	/* expand to file tilde */
	if (expdtilde(argv[to]) == (char *)NULL) {
		printmsg(M_ACCESS, "uucp: illegal filename %s\n", argv[to]);
		wrapup("cannot parse %s", argv[to]);
	}

	if (scratch[0] != '/') {
		strcpy(tofile, workdir);
		strcat(tofile, "/");
		if ((strlen(tofile) + strlen(scratch)) > 131) {
			printmsg(M_ACCESS, "uucp: illegal filename %s%s",
				tofile, scratch);
			wrapup("cannot parse %s", scratch);
		}
		strcat(tofile, scratch);
	}
	else
		strncpy(tofile, scratch, 131);

	if (spoolcmd() == FAILED) {
		printmsg(M_ACCESS, "uucp: failed to spool workfile");
		wrapup("spool failure", "");
	}
  }
  else if (tobang) {			/* to file is remote */
	mode = TO;
	strcpy(rmtsite, tosite);

	/* expand from file tilde */
	if (expdtilde(argv[from]) == (char *)NULL) {
		printmsg(M_ACCESS, "uucp: illegal filename %s\n", argv[from]);
		wrapup("cannot parse %s", argv[from]);
	}

	if (scratch[0] != '/') {
		strcpy(fromfile, workdir);
		strcat(fromfile, "/");
		if ((strlen(fromfile) + strlen(scratch)) > 131) {
			printmsg(M_ACCESS, "uucp: illegal filename %s%s",
				fromfile, scratch);
			wrapup("cannot parse %s", scratch);
		}
		strcat(fromfile, scratch);
	}
	else
		strncpy(fromfile, scratch, 131);

	if (spoolcmd() == FAILED) {
		printmsg(M_ACCESS, "uucp: failed to spool workfile");
		wrapup("spool failure", "");
	}
  }

  /* print the job number */
  if (opt_j && (frombang || tobang))
	fprintf(stderr, "uucp job %d\n", cmdseq);

  printmsg(M_ACCESS, "uucp: fromfile %s, tofile %s", fromfile, tofile);

  /* run the required task */
  if (mode == LOCAL) {			/* local copy */
	if (checkuser(locsite, locuser, tofile) == FAILED) {
		printmsg(M_ACCESS, "uucp: no access to file %s", tofile);
		wrapup("cannot access %s", tofile);
	}
	else if (getfiles(fromfile) == -1) {
		printmsg(M_ACCESS, "uucp: cannot copy %s", fromfile);
		wrapup("cannot copy %s", fromfile);
	}
	printmsg(M_STD, "DONE (WORK HERE)");
	if (dbglvl > 0) {
		(void) close(fdlog);	/* finished with the logfile */
		(void) unlock(LCKLOG);
	}

	exit(0);
  }
  else {
	printmsg(M_STD, "QUE'D (%s)", cmdfile);
	if (dbglvl > 0) {
		(void) close(fdlog);	/* finished with the logfile */
		(void) unlock(LCKLOG);
	}

	if (!opt_r) {			/* call the site */
		sprintf(scratch, "-s%s", rmtsite);
		sprintf(options, "-x%d", dbglvl);
		(void) execlp(UUCICO, "uucico", "-r1", options, scratch, (char *)NULL);
		exit(1);		/* never happen */
	}
	else
		exit(0);
  }
  /* NOTREACHED */
}


/*
 * m o v e f i l e
 *
 * Move a file from a to b using link or copy
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * At present, this routine copies the file regardless, and doesn't link.
 */
int movefile(from, to)
char *from, *to;
{
  char *cp, path[132], buff[BUFSIZ];
  int j, infd, outfd;

  printmsg(M_ACCESS, "movefile: moving |%s| to |%s|", from, to);

  /* in local mode, check for real user's access rights */
  strncpy(path, to, 131);
  if (mode == LOCAL && strncmp(path, PUBDIR, strlen(PUBDIR)) != SAME) {
	cp = strrchr(path, '/');
	*cp = '\0';
	if ((cp == path && access("/", 2) == -1) ||
			(cp != path && access(path, 2) == -1)) {
		printmsg(M_ACCESS, "movefile: cannot access %s", to);
		return(FAILED);
	}
  }

#ifdef LNKFLG
  if (link(from, to) == FAILED) {	/* copy the file */
#endif
	if ((infd = open(from, 0)) == -1) {
		printmsg(M_ACCESS, "movefile: cannot open %s", from);
		return(FAILED);
	}
	if ((outfd = creat(to, 0644)) == -1) {
		(void) close(infd);
		printmsg(M_ACCESS, "movefile: cannot create %s", to);
		return(FAILED);
	}
	while ((j = read(infd, buff, sizeof(buff))) != 0) {
		if (j == -1 || write(outfd, buff, (size_t)j) != j) {
			(void) close(infd);
			(void) close(outfd);
			printmsg(M_ACCESS,
				"movefile: cannot copy |%s| to |%s|", from, to);
			return(FAILED);
		}
	}
	(void) close(infd);
	(void) close(outfd);
#ifdef LNKFLG
  }
#endif

  if (chmod(to, 0666) == -1)
	printmsg(M_ACCESS, "movefile: cannot chmod file %s", to);

  return(OK);
}


/*
 * p r i n t m s g
 *
 * Print an error or debugging message into the system error log file
 *
 * Return:	Nothing
 *
 * All messages at levels less than or equal to the current debug level
 * are printed, unless the M_NOLOG level is used. If debugging is on,
 * messages are also printed to standard error.
 */
/* VARARGS1 */
void printmsg(level, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int level; char *fmt, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8;
{
  char msg[BUFSIZ];

  (void) time(&now);
  if (level <= dbglvl) {
	sprintf(msg, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
	strcat(msg, "\n");
	if (level != M_NOLOG) {
		if (level == M_STD)
			(void) writelog(msg, TRUE);
		else
			(void) writelog(msg, FALSE);
	}
	if (dbglvl > 0) {
		fprintf(stderr, "%s", msg);
		(void) fflush(stderr);
	}
  }
}


/* 
 * r s t a t u p d a t e
 *
 * Initialise an R_stat record if the file exists 
 *
 * Return:	OK	Success (or no R_stat file)
 *		FAILED	Otherwise
 */
int rstatupdate()
{
  char buff[BUFSIZ];
  int j, fd, flg = TRUE;
  size_t len;
  struct rstatrec *rstat;

  j = access(RSTAT, 0);
  printmsg(M_ACCESS, "rstatupdate: R_stat access call returns %d", j);
  if (j == -1) return (OK);	/* no file, don't bother */

  /* Build an entry for R_stat
   * R_stat format: job_no user site command_time status_time status
   * Note that uucp puts zeroes in the second time entry as a place holder.
   */

  for (j = 0 ; j < BUFSIZ ; j++) buff[j] = '\0';
  rstat = (struct rstatrec *)buff;
  rstat->job = cmdseq;
  strncpy(rstat->user, locuser, USERLEN);
  strncpy(rstat->site, rmtsite, SITELEN);
  rstat->time_a = now;
  rstat->time_b = (time_t)0;
  rstat->code = jobstatus;

  /* try to lock the R-stat file to append new data */
  if (checklock(LCKRSTAT, 2) == FAILED) return(FAILED);
  if ((fd = open(RSTAT, O_WRONLY | O_APPEND, 0600)) == -1) {
	(void) unlock(LCKRSTAT);
	return(FAILED);
  }

  /* append the data */
  len = sizeof(struct rstatrec);
  flg = (write(fd, buff, sizeof(struct rstatrec)) == sizeof(struct rstatrec));
  if (close(fd) == -1) flg = FALSE;
  (void) unlock(LCKRSTAT);

  printmsg(M_ACCESS, "rstatupdate: new entry |%s|", visib(buff, len));

  if (flg)
	return(OK);
  else
	return(FAILED);
}


/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/*
 * s p o o l c m d
 *
 * Create the various spool-file entries
 *
 * Return:	FAILED	Unknown mode
 *		FAILED	File access failure
 *		OK	Otherwise
 */
int spoolcmd()
{
  /* generate the right command file name */
  if (mode == FROM && (cmdseq = genname('C', grade, fromsite)) == 0)
	return(FAILED);
  else if (mode == TO && (cmdseq = genname('C', grade, tosite)) == 0)
	return(FAILED);
  else if (mode == REMOTE && (cmdseq = genname('C', grade, fromsite)) == 0)
	return(FAILED);
  else if (mode != FROM && mode != TO && mode != REMOTE)
	return(FAILED);
  strcpy(cmdfile, scratch);

  printmsg(M_ACCESS, "spoolcmd: command file name %s", cmdfile);

  /* try to open the command file */
  if ((fpcmd = fopen(cmdfile, "w")) == (FILE *)NULL) {
	printmsg(M_ACCESS, "spoolcmd: cannot open %s", cmdfile);
	return(FAILED);
  }
  (void) chmod(cmdfile, 0600);
  /* add file to cleanup() list */
  (void) addfile(cmdfile);

  /*
   * Add a workfile entry to the command file.
   * Workfile entries are S or R or X strings.
   *
   * Send from local to remote:
   * S full_source_file dest_file sender options data_file mode notify [mailbox]
   *
   * Request from remote to local:
   * R source_file full_dest_file sender options [mailbox]
   * 
   * Request multiple files:
   * X source_afn full_dest_path sender options [mailbox]
   */

  (void) genopt(options);		/* generate option string */

  jobstatus = ET_QUEUED;		/* reset if necessary */
  if (mode == TO) {			/* send file to remote */
	if (getfiles(fromfile) == -1) {
		printmsg(M_ACCESS, "spoolcmd: cannot open %s", fromfile);
		(void) unlink(cmdfile);
		jobstatus = ET_LACCESS;
	}
  }
  else if (mode == FROM) {		/* receive file from remote */
	fprintf(fpcmd, "R %s %s %s %s %s\n", fromfile, tofile, locuser, options,
			(opt_m && opt_o) ? mailbox : " ");
  }
  else if (mode == REMOTE) {		/* send remote uucp command */
	fprintf(fpcmd, "X %s %s %s %s\n", fromfile, tofile, locuser, options,
			(opt_m && opt_o) ? mailbox : " ");
  }
  else {
	(void) unlink(cmdfile);
	jobstatus = ET_BADCMD;
  }
  (void) fclose(fpcmd);

  if (jobstatus == ET_QUEUED) {
	/* initialise an R_stat record, and an L_stat record if necessary */
	if (rstatupdate() == FAILED || lstatupdate() == FAILED)
		return(FAILED);
	else
		return(OK);
  }
  else
	return(FAILED);
}


/*
 * s p o o l u u x 
 *
 * Spool a uux command file
 *
 * Return:	FAILED	File access failure
 *		OK	Otherwise
 *
 * (Used only by -e option)
 */
int spooluux()
{
  /* generate the various spool file names */
  if (genname('C', 'A', rmtsite) == 0) return(FAILED);	/* command file name */
  strcpy(cmdfile, scratch);
  if (genname('D', 'X', locsite) == 0) return(FAILED);	/* datafile name */
  strcpy(datafile, scratch);
  scratch[0] = 'X';		/* remote uux file name */

  printmsg(M_PARSE, "spooluux: rmtsite %s locsite %s", rmtsite, locsite);

  /* try to open the command file */
  if ((fpcmd = fopen(cmdfile, "w")) == (FILE *)NULL) {
	printmsg(M_ACCESS, "spooluux: cannot open %s", cmdfile);
	return(FAILED);
  } 
  (void) chmod(cmdfile, 0600);
  fprintf(fpcmd, "S %s %s %s - %s 0666 %s\n",
	datafile, scratch, locuser, datafile, locuser);
  (void) fclose(fpcmd);

  /* try to open the data file */
  if ((fpdata = fopen(datafile, "w")) == (FILE *)NULL) {
	printmsg(M_ACCESS, "spooluux: cannot open %s", datafile);
	return(FAILED);
  } 
  (void) chmod(datafile, 0600);
  fprintf(fpdata, "U %s %s\n", locuser, locsite);
  fprintf(fpdata, "C uucp -C %s!%s %s!%s\n",
			fromsite, fromfile, tosite, tofile);
  (void) fclose(fpdata);

  return(OK);
}


/*
 * s t r i p b a n g
 *
 * Strip the leading bang pathname from a string
 *
 * Return:	OK		Bang path found
 *		FAILED		Bang path not found
 *		path and site strings update as side effects
 */
int stripbang(arg, path, site)
char *arg, *path, *site;
{
  char *cp;
  int j;

  if ((cp = strchr(arg, '!')) == (char *)NULL) {
	*path = *site = '\0';
	return(FAILED);
  }

  j = 0;
  while (*arg && *arg != '!' && j++ < SITELEN)
	*site++ = *arg++;
  *site = '\0';
 
  strncpy(path, ++cp, 131);
  if (j == 0)
	strncpy(site, locsite, SITELEN);

  return(OK);
}


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/*
 * u s a g e 
 *
 * Usage message
 *
 * Return:	none
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-Ccdfjr] [-g grade] [-m user] [-n user]\n",
	progname);
  fprintf(stderr, "            [-s spool] [-x level] <from> <to>\n");
}


/*
 * u s e r c h e c k
 *
 * Check the user file
 *
 * Return:	OK		Valid entry found
 *		FAILED  	No valid entry found
 *		cbflg, userpath are updated as side effects
 */
int usercheck(site, user)
char *site, *user;
{
  char *cp, *bp, tsite[20], tuser[20], tflag[20], buff[256];
  int j, level, flg;
  FILE *fp;

  printmsg(M_ACCESS, "usercheck: checking site |%s|, user |%s|", site, user);

  if ((fp = fopen(USERFILE, "r")) == (FILE *)NULL) {
	strncpy(userpath, "/", 255);
	return(OK);
  }

  level = 0;
  while (level < 4 && fgets(buff, 255, fp) != (char *)NULL) {
	tsite[0] = tuser[0] = tflag[0] = '\0';
	/* the trailing newline only confuses things */
	j = strlen(buff) - 1;
	if (buff[j] == '\n') buff[j] = '\0';
	if (buff[0] == '\0') continue;

	/* lose leading whitespace */
	cp = buff;
	while (isspace(*cp))
		cp++;

	/* we have a ',' or a username or junk */  
	if (*cp == ',')
		cp++;
	else if (isalpha(*cp)) {
		bp = tuser;
		while (isalpha(*cp) && (bp - tuser) < 20)
			*bp++ = *cp++;			
		*bp = '\0';
		while (isspace(*cp))
			cp++;
		if (*cp++ != ',')
			continue;
	}
	else
		continue;

	/* we have a sitename, a callback flag, a path, or junk */
	while (isspace(*cp))
		cp++;
	bp = tsite;
	while (isalpha(*cp) && (bp - tsite) < 20)
		*bp++ = *cp++;			
	*bp = '\0';
	while (isspace(*cp))
		cp++;
	bp = tflag;
	while (*cp != '/' && isalpha(*cp) && (bp - tflag) < 20)
		*bp++ = *cp++;			
	*bp = '\0';
	while (isspace(*cp))
		cp++;

	/* we now have a path, or junk */
	if (*cp != '/') continue;

	/* sort out site and callback entries */
	if (strcmp(tsite, "c") == SAME) {
		strcpy(tsite, "");
		strcpy(tflag, "c");
	}

	/* set callback flag */
	if (strcmp(tflag, "c") == SAME)
		flg = TRUE;
	else
		flg = FALSE;

	printmsg(M_ACCESS, "usercheck: found user |%s| site |%s| flag %s",
			tuser, tsite, flg ? "TRUE" : "FALSE");

	/* we have a valid entry; is it relevant ? */
	if (level < 1 && strcmp("", tsite) == SAME &&
						strcmp("", tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 1;
	}
	else if (level < 2 && strcmp("", tsite) == SAME &&
						strcmp(user, tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 2;
	}
	else if (level < 3 && strcmp(site, tsite) == SAME &&
						strcmp("", tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 3;
	}
	else if (level < 4 && strcmp(site, tsite) == SAME &&
						strcmp(user, tuser) == SAME) {
		strncpy(userpath, cp, 255);
		level = 4;
	}
  }
  (void) fclose(fp);

  printmsg(M_ACCESS, "usercheck: exit level %d, userpath |%s|", level, userpath);

  if (level == 0)
	return(FAILED);
  else
	return(OK);
}


/*
 * v i s i b
 *
 * Print the buffer of character data in "visible" format
 *
 * Return:	A pointer to the data buffer
 *
 * Printable characters (except '\') are printed as themselves,
 * special control characters (CR, LF, BS, HT) are printed in the usual
 * escape format, and others are printed using 3-digit octal escapes.
 * The usual warning about a return value which points to a static buffer
 * which is overwritten on each call applies here.
 */
char *visib(data, len)
char *data; size_t len;
{
  static char buff[4 * 132];
  char c, *p;

  if (len > 132) len = 132;
  p = buff;
  while (len--) {
	c = *data++;
	if (isascii(c) && (isprint(c) || ' ' == c)) *p++ = c;
	  else if ('\n' == c) {
		*p++ = '\\';
		*p++ = 'n';
       	  } else if ('\t' == c) {
		*p++ = '\\';
		*p++ = 't';
          } else if ('\r' == c) {
		*p++ = '\\';
		*p++ = 'r';
         } else if ('\b' == c) {
		*p++ = '\\';
		*p++ = 'b';
         } else {
		sprintf(p, "\\%03o", c);
		p += 4;
        }
  }
  *p = '\0';
  return(buff);
}


/*
 * w r a p u p
 *
 * Write out an error message and call cleanup()
 *
 * Return:	Nothing
 */
void wrapup(format, message)
char *format, *message;
{
  fprintf(stderr, "uucp failed: ");
  fprintf(stderr, format, message);
  fprintf(stderr, "\n");

  cleanup(SIGHUP);
}


/* 
 * w r i t e l o g
 *
 * Write a string to the log file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int writelog(msg, prefix)
char *msg; int prefix;
{
  char buff[BUFSIZ];
  char curtime[64];
  int flg;
  size_t sz;

  /* log format: site!user (date time) (id pid task) status (detail) */
  if (!prefix)
	strncpy(buff, msg, BUFSIZ - 1);
  else {
	(void) getdate(curtime);
	sprintf(buff, "%s!%s (%s) (U %4d  0) ", rmtsite, locuser, curtime, pid);
	strncat(buff, msg, BUFSIZ - 50);
  }
  sz = strlen(buff);

  /* if debugging, the logfile is already locked */
  if (dbglvl > 0)
	flg = (write(fdlog, buff, sz) == sz);
  else {
	if (checklock(LCKLOG, 2) == FAILED) {
		printmsg(M_NOLOG, "uucp: cannot get lock %s", LCKLOG);
		return(FAILED);
	}
	if ((fdlog = open(LOGFILE, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1) {
		printmsg(M_NOLOG, "uucp: cannot open %s", LOGFILE);
		(void) unlock(LCKLOG);
		return(FAILED);
	}
	flg = (write(fdlog, buff, sz) == sz);
	if (close(fdlog) == -1) flg = FALSE;
	(void) unlock(LCKLOG);
  }

  if (flg)
	return(OK);
  else
	return(FAILED);
}

