/* uuclean - clean up UUCP spool directory.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

#define FALSE		0
#define TRUE		~FALSE
#define OK		1
#define FAILED		-1
#define SAME		0
#define MAXAGE		72	/* default maximum age in hours */
#define MAXPIDS		50	/* maximum number of pids */

#ifndef lint
static char *Version = "@(#) uuclean 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_C = FALSE;
int opt_d = FALSE;
int opt_m = FALSE;
int opt_n = FALSE;
int opt_p = FALSE;
int opt_s = FALSE;
int opt_w = FALSE;
int opt_x = FALSE;

int errno;			/* used by errno.h */

char pref_p[MAXARGS][15];	/* file prefixes */
char site_p[MAXARGS][15];	/* site names */
char mailbox[132];		/* redefined mailbox file */
char mail[20];			/* redefined mail address */
char locsite[20];		/* name of local site */
char progname[20];		/* program name */
int dbglvl;			/* debugging level */
int pref_tot;			/* total number of prefixes */
int site_tot;			/* total number of sites */
int pidlist[MAXPIDS];		/* list of current pids */
time_t now;			/* current unix time */
FILE *fpwarn;			/* warning file pointer */

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int getargs();
int getdate();
int getname();
int getsite();
uid_t get_uid();
int getuser();
int pref_match();
int runps();
int sendmail();
int site_match();
void usage();
int warn();


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t d a t e
 *
 * Get the date in a format suitable for the logfile
 *
 * Return	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(buff)
char *buff;
{
  int j;

  sprintf(buff, "%s", ctime(&now));
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[j] = '\0';			/* clear the newline */

  return(OK);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t s i t e
 *
 * Extract a site name from a file name  
 *
 * Return:	TRUE		Name found
 *		FALSE		Name not found
 *		site is updated as a side-effect
 */
int getsite(file, site)
char *file, *site;
{
  char buff[20];
  int j, k;

  for (j = 0 ; j < SITELEN ; j++) site[j] = '\0';

  /* look at command or data or execute files only */
  if (strncmp(file, "C.", 2) == SAME || strncmp(file, "D.", 2) == SAME
			|| strncmp(file, "X.", 2) == SAME) {
	/* get the site name */
	for (j = 0 ; j < 12 ; j++)
		buff[j] = file[j + 2];
	buff[12] = buff[13] = '\0';
	k = strlen(buff);
	/* uux command files use a different naming convention */
	if (strncmp(file, "D.", 2) == SAME && buff[k - 5] == 'X')
		return(FAILED);
	for (j = k - 5 ; j < k ; j++)
		buff[j] = '\0';
	strncpy(site, buff, SITELEN);
	return(TRUE);
  }

  return(FALSE);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp, buff[132];
  char owner[20];			/* name of file's owner */
  char site[20];			/* name of file's site */
  char spooldir[132];			/* spool directory */
  char warning[132];			/* warning file */
  char pathname[132];			/* full path name */
  char filename[20];			/* file name */
  char message[256];
  int j, k, fd;
  int age;				/* maximum age of file in hours */
  uid_t tuid;
  time_t then;
  DIR *dirp;
  struct dirent *dp;
  struct stat stat_buff;

  /* get local site name */
  if (getname(locsite) == FAILED) {
	fprintf(stderr, "uuclean: cannot find local uucpname\n");
	exit(1);
  }

  /* set up defaults */
  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  age = MAXAGE;				/* default age in hours */
  for (j = 0 ; j < MAXPIDS ; j++) pidlist[j] = 0;
  for (j = 0 ; j < MAXARGS ; j++) pref_p[j][0] = site_p[j][0] = '\0';
  dbglvl = pref_tot = site_tot = 0;
  warning[0] = mailbox[0] = mail[0] = '\0';
  fpwarn = (FILE *)NULL;

  /* parse arguments */
  opterr = 0;
  while ((j = getopt(argc, argv, "d:m:n:p:s:w:x:")) != EOF) {
	switch (j & 0177) {
	case 'd':
		strcat(spooldir, "/");
		strncpy(spooldir, optarg, 130 - strlen(SPOOLDIR));
		opt_d = TRUE;
		break;
	case 'm':
		if (optarg[0] == '/')
			strncpy(mailbox, optarg, 131);
		else
			strncpy(mail, optarg, 19);
		opt_m = TRUE;
		break;
	case 'n':
		age = atoi(optarg);
		opt_n = TRUE;
		break;
	case 'p':
		if (pref_tot < MAXARGS) {
			strcpy(pref_p[pref_tot++], optarg);
			opt_p = TRUE;
		}
		break;
	case 's':
		if (site_tot < MAXARGS) {
			strcpy(site_p[site_tot++], optarg);
			opt_s = TRUE;
		}
		break;
	case 'w':
		strncpy(warning, optarg, 131);
		opt_w = TRUE;
		break;
	case 'x':
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case '?':
 		if ((optopt & 0177) == 'w') { /* warnings default to stdout */
 			warning[0] = '\0';
 			opt_w = TRUE;
			break;
		}
		else if ((optopt & 0177) == 'm') { /* mail defaults to owner */
 			mailbox[0] = mail[0] = '\0';
 			opt_m = TRUE;
 			break;
 		}
 		/* else fall through */
	default:
		usage();
		exit(1);
		break;
	}
  }

  opt_C = (opt_d || opt_m || opt_n || opt_p || opt_s || opt_w);

  /* validity checks */
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid && opt_C) {
	fprintf(stderr, "uuclean: option available only to administrators.\n");
	exit(1);
  }
#endif
  if (age < 0) {
	usage();
	exit(1);
  }
  /* default behaviour is to remove unused lock files */
  if (!opt_C) {
	opt_n = opt_p = TRUE;
	strcpy(pref_p[pref_tot++], "LCK");
	age = 0;
  }

  /* try to open the work directory */
  if ((dirp = opendir(spooldir)) == (DIR *)NULL) {
	fprintf(stderr, "uuclean: cannot open %s\n", spooldir);
	exit(1);
  }
  /* try to open the warning file, if any */
  if (warning[0] != '\0') {
	if ((fpwarn = fopen(warning, "a")) == (FILE *)NULL) {
		fprintf(stderr, "uuclean: cannot open %s\n", warning);
		exit(1);
	}
  }
  else
	fpwarn = stdout;

  /* get the pids of any running processes */
  if (runps() == FAILED) {
	fprintf(stderr, "uuclean: cannot run ps(1)\n");
	exit(1);
  }

  then = now - (time_t)age * 3600L;

  /* look for the required files */
  while ((dp = readdir(dirp)) != (struct dirent *)NULL) {
	strncpy(filename, dp->d_name, 14);
	strncpy(pathname, spooldir, 131);
	strcat(pathname, "/");
	strncat(pathname, dp->d_name, 14);
	(void) stat(pathname, &stat_buff);

	/* no tickee, no laundree */
	if (opt_m) {
		if (mail[0] != '\0')
			strncpy(owner, mail, 19);
		else if (getuser(stat_buff.st_uid, owner) == FAILED)
			opt_m = FALSE;
	}

	/* ignore all irrelevant pathnames, and those not to be removed */
	if (strncmp(filename, ".", 1) == SAME ||
			strncmp(filename, "o.", 2) == SAME ||
			strcmp(pathname, LOGFILE) == SAME ||
			strcmp(pathname, SYSLOG) == SAME ||
			strcmp(pathname, ERRLOG) == SAME ||
			strcmp(pathname, AUDLOG) == SAME ||
			strcmp(pathname, FOREIGN) == SAME)
		continue;
	else if ((stat_buff.st_mode & S_IFMT) != S_IFREG)
		continue;
	else if (stat_buff.st_mtime > then)
		continue;
	else if (opt_p && pref_match(filename) == FALSE)
		continue;
	else if (opt_s && site_match(filename) == FALSE)
		continue;
	else {
		/* if we have an STST or LCK file, check for running process */
		if (strncmp(filename, "STST.", 4) == SAME ||
				strncmp(filename, "LCK.", 3) == SAME) {

			/* read the file */
			if ((fd = open(pathname, 0)) == -1)
				continue;
			for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
			if (read(fd, buff, 131) == -1) {
				(void) close(fd);
				continue;
			}
			(void) close(fd);

			/* get the pid */
			for (j = 0 ; j < 6 ; j++)
				if (!isdigit(buff[j])) break;
			buff[j] = '\0';
			j = atoi(buff);
			if (j == 0) continue;

			/* see if there is a associated process */
			k = 0;
			while (k < MAXPIDS && pidlist[k] != 0 && pidlist[k] != j)
				k++;
			if (pidlist[k] != 0)
				continue;
		}

		/* if we get here, we are free to act */
		if (opt_w)
			(void) warn(filename, stat_buff.st_mtime);
		else {
			if (unlink(pathname) != 0)
				fprintf(stderr,
					"uuclean: cannot remove %s\n", pathname);
		}
		if (opt_m) {
			if (!getsite(filename, site))
				strncpy(site, locsite, SITELEN);
			sprintf(message,
				"%s!uuclean %s file %s over %d hour%sold\n",
				locsite, opt_w ? "found" : "deleted",
				pathname, age, age == 1 ? " " : "s ");
			if (sendmail(site, owner, message, (char *)NULL,
					(mailbox[0] != '\0')) == FAILED)
				opt_m = FALSE;
		}
	}
  }
  (void) closedir(dirp);
  if (warning[0] != '\0') (void) fclose(fpwarn);

  exit(0);
  /* NOTREACHED */
}


/*
 * p r e _ m a t c h
 *
 * Match a given prefix with a given file name
 *
 * Return:	TRUE	Matched
 *		FALSE	Not matched
 */
int pref_match(filename)
char *filename;
{
  int j;

  /* look at all files */
  for (j = 0 ; j < pref_tot && pref_p[j][0] != '\0'; j++) {
	if (strncmp(filename, pref_p[j], strlen(pref_p[j])) == SAME)
		return(TRUE);
  }

  return(FALSE);
}


/*
 * r u n p s
 *
 * Run ps(1) and collect a list of pids
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int runps()
{
  char *fields[MAXARGS], scratch[132], buff[BUFSIZ];
  int j, k, pfd[2];
  FILE *psfp;

  /* run ps(1) and collect output */
  if (pipe(pfd) == -1)			/* generate the pipe */
	return(FAILED);
  switch (fork()) {			/* fork, dup and exec */
	case -1:			/* error */
		return(FAILED);
	case 0:				/* pregnant */
		if (close(1) == -1)
			exit(1);
		if (dup(pfd[1]) != 1)	/* child's stdout */
			exit(1);
		if (close(pfd[0]) == -1 || close(pfd[1]) == -1)
			exit(1);
		(void) fflush(stderr);
		(void) freopen("/dev/null", "w", stderr);
		(void) execlp(PS, "ps", "-axl", (char *)NULL);
		exit(1);		/* never happen */
  }

  /* if we're here, we're a parent */
  if (close(pfd[1]) == -1) return(FAILED);
  if ((psfp = fdopen(pfd[0], "r")) == (FILE *)NULL)
	return(FAILED);

  /* get the ps header line */
  if (fgets(buff, BUFSIZ, psfp) == (char *)NULL) {
	(void) fclose(psfp);
	(void) close(pfd[0]);
	return(FAILED);
  }

  /* add relevant pids to the list */
  k = 0;
  while (fgets(buff, BUFSIZ, psfp) != (char *)NULL) {
	/* get a pid */
	strncpy(scratch, buff, 131);	/* getargs roaches the string */
	if (getargs(scratch, fields) < (PSPID + 1)) continue;
	j = atoi(fields[PSPID]);
	if (j == 0) continue;
	pidlist[k++] = j;
  }
  (void) fclose(psfp);
  (void) close(pfd[0]);

  return(OK);
}


/*
 * s e n d m a i l
 *
 * Send messages or files to local or remote users
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int sendmail(site, name, message, file, flg)
char *site, *name, *message, *file; int flg;
{
  char address[32], curtime[64], header[256], buff[BUFSIZ];
  int j, k, status, err, fflg, mflg, infd, outfd, pfd[2];
  size_t len;

  /* sort out what's to be done */
  err = fflg = mflg = FALSE;
  if (file != (char *)NULL)
	fflg = TRUE;
  if (message != (char *)NULL)
	mflg = TRUE;
  if (!fflg && !mflg) return(FAILED);

  /* write the message to a (local) file; mailbox is a full path */
  if (flg) {
	(void) getdate(curtime);
	sprintf(header, "From uucp (%s) ", curtime);
	if ((outfd = open(mailbox, O_CREAT | O_WRONLY | O_APPEND, 0644)) == -1)
		err = TRUE;
	(void) chmod(mailbox, 0666);
	len = strlen(header);
	if (!err && write(outfd, header, len) != len)
		err = TRUE;
	len = strlen(message);
	if (!err && mflg && write(outfd, message, len) != len)
		err = TRUE;
	if (!err && fflg) {
		if ((infd = open(file, 0)) == -1)
			err = TRUE;
		while (!err && (j = read(infd, buff, sizeof(buff))) != 0) {
			if (j == -1 || write(outfd, buff, (size_t)j) != j)
				err = TRUE;
		}
		(void) close(infd);
	}
	(void) close(outfd);
	if (err) (void) unlink(mailbox);
  }
  /* or mail it to the initiator */
  else {
	/* generate the message address and header (not needed with LMAIL) */
	sprintf(address, "%s!%s", site, name);
	sprintf(header, "From %s!%s %sTo: %s!%s\n\n", locsite, "uucp", ctime(&now),
							site, name);
	/* run a mailer, and write the text to its stdin */
	if (pipe(pfd) == -1)			/* generate the pipe */
		return(FAILED);
	switch (j = fork()) {			/* fork, dup and exec */
		case -1:			/* error */
			return(FAILED);
		case 0:				/* pregnant */
			if (close(0) == -1)
				exit(1);
			if (dup(pfd[0]) != 0)	/* child's stdin */
				exit(1);
			if (close(pfd[0]) == -1 || close(pfd[1]) == -1)
				exit(1);
			(void) fflush(stderr);
			(void) freopen("/dev/null", "w", stderr);
			(void) execlp(LMAIL, "lmail", address, (char *)NULL);
			exit(1);		/* never happen */
	}

	/* if we're here, we're a parent */
	if (close(pfd[0]) == -1) return(FAILED);
	len = strlen(header);
	if (write(pfd[1], header, len) != len)
		err = TRUE;
	if (!err && mflg) {
		len = strlen(message);
		if (write(pfd[1], message, len) != len)
			err = TRUE;
	}
	if (!err && fflg) {
		if ((infd = open(file, 0)) == -1) err = TRUE;
		while (!err && (k = read(infd, buff, sizeof(buff))) != 0) {
			if (k == -1 || write(pfd[1], buff, (size_t)k) != k)
				err = TRUE;
		}
		(void) close(infd);
	}
	if (close(pfd[1]) == -1) return(FAILED);
#ifdef MAILWAIT
	while ((k = wait(&status)) != j && k != -1)
		;
	if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 || WTERMSIG(status) != 0)
		err = FAILED;
#endif
  }

  if (err)
	return(FAILED);
  else
	return(OK);
}


/*
 * s i t e _ m a t c h
 *
 * Match a given site name with the current filename
 *
 * Return:	TRUE	Matched
 *		FALSE	Not matched
 */
int site_match(filename)
char *filename;
{
  char sitename[20];
  int j;

  if (getsite(filename, sitename))  {
	for (j = 0 ; j < site_tot && site_p[j][0] != '\0'; j++) {
		if (strcmp(sitename, site_p[j]) == SAME)
			return(TRUE);
	}
  }

  return(FALSE);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr,
	"Usage: %s [-d dir] [-m [file]] [-n age] [-p pre] [-s sys] [-w [file]]\n",
		progname);
}


/*
 * w a r n
 *
 * Generate a warning message
 */
int warn(name, age)
char *name; time_t age;
{
  char date[80];
  long int days, hours, minutes, seconds;
  time_t overdue;

  /* find out how long the file has been around */
  overdue = now - age;
  days = overdue / 86400L;
  overdue %= 86400L;
  hours = overdue / 3600L; 
  overdue %= 3600L;
  minutes = overdue / 60L;
  seconds = overdue % 60L;

  (void) getdate(date);

  fprintf(fpwarn, "uuclean: (%s) %s overdue (%02ld:%02ld:%02ld:%02ld)\n",
		date, name, days, hours, minutes, seconds);

  return(OK);
}
