/* uucico2.c
 *
 * Version 4.2 of 31 October 1991.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <time.h>
#include <unistd.h>
#ifdef _SVR2
#include <ustat.h>
#endif
#include "dial.h"
#include "uucp.h"
#include "uucico.h"


/*
 * h t o l
 *
 * Convert hex string to positive long int
 *
 * Return:	a positive long number
 */
long int htol(bp)
char *bp;
{
  int j;
  long int result = 0L;

  while (*bp && isspace(*bp))	/* skip leading white space */
	bp++;

  while (*bp) {
	j = 0;
	while (j < 22 && "0123456789ABCDEFabcdef"[j] != *bp) j++; 
	if (j == 22) break;
	if (j > 15) j -= 6;
	result = (result << 4) + j;
	bp++;
  }

  return(result);
}


/*
 * i o i n i t
 *
 * Initialise i/o
 * 
 * Return:	G	Success
 *		X	Failure
 */
char ioinit()
{
  struct TTY tty;

  if (role == SLAVE) {
	fdin = 0;	/* SLAVE standard input, MASTER set by DIAL() */
	fdout = 1;	/* SLAVE standard output, MASTER set by DIAL() */
  }

  /* park the current settings */
  if (ioctl(fdin, TTYGET, &stdtty) == -1) {
	if (errno != EINTR || ioctl(fdin, TTYGET, &stdtty) == -1) {
		printmsg(M_STD, "ASSERT ERROR");
		sprintf(errmsg, "STAT FAILED (0)");
		return('X');
	}
  }

  /* set raw mode: 8N1/RAW */
  tty = stdtty;

#ifdef _MINIX
  tty.sg_flags &= ~(XTABS|EVENP|ODDP|CRMOD|ECHO|CBREAK|BITS5|BITS6|BITS7);
  tty.sg_flags |= (RAW | BITS8);
#endif
#ifdef _SVR2
  tty.c_iflag &= ~(IGNBRK | IGNCR | INLCR | ICRNL | IUCLC |
			IXANY | IXON | IXOFF | INPCK | ISTRIP);
  tty.c_iflag |= (BRKINT | IGNPAR);
  tty.c_oflag &= ~OPOST;
  tty.c_lflag &= ~(ICANON | ISIG | ECHO);
  tty.c_cflag &= ~(CSIZE | PARENB);
  tty.c_cflag |= (CS8 | CREAD);
  tty.c_cc[4] = 1;
  tty.c_cc[5] = 5;
#endif

  if (ioctl(fdin, TTYSET, &tty) == -1) {
	if (errno != EINTR || ioctl(fdin, TTYSET, &tty) == -1) { 
		printmsg(M_STD, "ASSERT ERROR");
		sprintf(errmsg, "STAT FAILED (0)");
		return('X');
	}
  }

  return('G');
}
 

/*
 * i o e n d
 *
 * Close i/o
 *
 * Return:	Y	Always
 */
char ioend()
{
  /* we'll shut down in our own good time, thank you */
  (void) signal(SIGHUP, SIG_IGN);

  /* Clear the RS232-buffer from any HANGUP-garbage. */
  if (ioctl(fdin, TTYFLUSH, 0) == -1 && errno == EINTR)
	(void) ioctl(fdin, TTYFLUSH, 0);

  /* restore the original settings */
  if (ioctl(fdin, TTYSET, &stdtty) == -1) {
	if (errno != EINTR || ioctl(fdin, TTYSET, &stdtty) == -1) { 
		printmsg(M_STD, "ASSERT ERROR");
		sprintf(errmsg, "STAT FAILED (0)");
		return('Y');		/* for what it's worth */
	}
  }

  return('Y');
}


/*
 * l o g o n
 *
 * Try to log-on to a site
 * 
 * Return:	Y	Login script failure
 *		F	Success
 */
char logon()
{
  char *cp, *flds[MAXARGS], buff[132];
  int j, k, max, flg;

  printmsg(M_CALL, "logon: attempting to log on to host %s", rmtsite);

  /* The script has the form expect[-subsend-expect...] send [expect send...]
   * The whole string has already been put in tempbuff and parsed by getsys.
   */
  for (j = F_EXPECT; j < maxfld; j += 2)  {
	printmsg(M_CALL, "logon: expecting %d of %d, string %s",
		j, maxfld, cmdflds[j]);

	/* parse the current expect string, looking for substrings.
	 * substrings are separated by '-', but '--' means send CR
	 * (or LF on some systems).
	 */
	strncpy(buff, cmdflds[j], 131);
	for (k = 0 ; k < MAXARGS ; k++) flds[k] = (char *)NULL;
	max = 0;
	cp = flds[max++] = buff;
	while (*cp && cp < (buff + 131) && max < MAXARGS) {
		if (*cp == '-') {
			*cp++ = '\0';
			flds[max++] = cp;
			if (*cp == '-') {	/* ie. "--" */
				*cp++ = '\0';
				flds[max++] = cp;
			}
		}
		else
			cp++;
	}
	/* max now contains the number of valid field pointers available,
	 * some of which may point to '\0'.  Note that there is an odd
	 * number of fields.
	 */
	if (dbglvl >= M_CALLMSG)
		for (k = 0 ; k < max ; k += 2) {
			sprintf(scratch,
			"logon: expect [%02d]: %-30s    send [%02d]: %s",
	             	k, flds[k], k + 1,
			flds[k + 1] == (char *)NULL ? "*EMPTY*" : flds[k + 1]);
			printmsg(M_CALLMSG, scratch);
		}
	k = 0;
	flg = expectstr(flds[k++], MSGTOUT);
	printmsg(M_CALLMSG, "\nlogon: got %s",
		(flg == TRUE) ? "that" : "timeout");
	while (!flg && k < max) {
		printmsg(M_CALLMSG, "logon: sending %s",
			(*flds[k] == '\0') ? "\\r" : flds[k]);
		if (*flds[k] == '\0')
			(void) sendstr("\"\"");
		else
			(void) sendstr(flds[k]);
		flg = expectstr(flds[k + 1], MSGTOUT);
		printmsg(M_CALLMSG, "\nlogon: got %s",
			(flg == TRUE) ? "that" : "timeout");
		k += 2;
	}

	if (flg) {		/* success */
		printmsg(M_CALL, "logon: sending %d of %d, %s",
					j + 1, maxfld, cmdflds[j + 1]);
		(void) sleep(1);
		(void) sendstr(cmdflds[j + 1]);
	}
	else
		break;		/* failure */
  }

  /* update the status file and quit */
  if (flg) {
	(void) writestatus("TALKING");
	(void) fclose(fpstat);
	fpstat = (FILE *)NULL;
	callstatus |= EC_CPROG;	/* clear this in sysend() */
	sterflg = FALSE;
	return('F');
  }
  else {
	/* If we've been on line, tidy up */
	if (fdout != -1) {
		(void) clearbuff();
		(void) (*devclose)(fdout);
	}
	printmsg(M_STD, "BAD LOGIN/PASSWORD");
	(void) writestatus("LOGIN FAILED");
	callstatus |= EC_LFAIL;
	sterflg = TRUE;
  	return('B');
  }
}


/*
 * m a i n
 *
 * Main program
 */ 
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp, lckuu[132];
  int j, k, status;
  uid_t tuid;
  struct stat statbuff;
  struct Proto *pr;
#ifndef TESTDBG 
  dbglvl = 0;			/* default debug level */
#else
  FILE *fp;
  if ((fp = fopen(DEFDBG, "r")) != (FILE *)NULL &&
		fgets(scratch, 132, fp) != (char *)NULL &&
		sscanf(scratch, "%d", &j) == 1)
	dbglvl = j;		/* default debug level */
  else
	dbglvl = 0;
  if (fp != (FILE *)NULL) (void) fclose(fp);
#endif

  /* set various defaults */
  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  strncpy(spooldir, SPOOLDIR, 131);	/* default spool directory */
  fpdata = (FILE *)NULL;
  fpcmd = (FILE *)NULL;
  fpstat = (FILE *)NULL;
  location = REMOTE;
  role = SLAVE;
  slckflg = FALSE;
  sterflg = FALSE;
  usrfflg = MAYBE;
  cbflg = FALSE;
  status = FALSE;
  opt_S = FALSE;
#ifdef RUNUUXQT
  opt_U = TRUE;
#else
  opt_U = FALSE;
#endif
  framefirst = BFRAME;
  framelast = EFRAME;
  userpath[0] = '\0';
  errmsg[0] = '\0';
  retrynum = 0;
  jobnum = 0;
  tasknum = 0;
  taskstatus = 0;
  callstatus = 0;
  lsysindex = 0;
  fdin = fdout = -1;
  j = 0;
  pr = Protolst;
  while (protocols[j++] = (pr++)->Pname)/* default protocols */
	;

  /* parse options */
  opterr = 0;
  while ((j = getopt(argc, argv, "r:S:s:TUx:")) != EOF) {
	switch (j & 0377) {
	case 'r':
		role = atoi(optarg);
		break;
	case 'S':
		opt_S = TRUE;
		/* fall through */
	case 's':
		role = MASTER;
		status = TRUE;
		strncpy(rmtsite, optarg, SITELEN);
		break;
	case 'T':
		if (framelast == '\0')
			framelast = '\020';
		else
			framelast = '\0';
		break;
	case 'U':
		opt_U = !opt_U;
		break;
	case 'x':
		dbglvl = atoi(optarg);
		break;
	case '?':
	default:
		usage();
		exit(1);
		break;
	}
  }
  /* check that options are valid */
  if (role == MASTER && !status) {
	printmsg(M_NOLOG, "uucico: need site to call when master");
	exit(1);
  }
  if (role == SLAVE && status) {
	printmsg(M_NOLOG, "uucico: cannot dial out when slave");
	exit(1);
  }
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid) {
	dbglvl = 0;
	opt_S = FALSE;
  }
#endif

  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  /* get local site name, local user, and pid */
  if (getname(locsite) == FAILED) {
	printmsg(M_NOLOG, "uucico: cannot find local uucpname");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	printmsg(M_NOLOG, "uucico: cannot find local username");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	printmsg(M_NOLOG, "uucico: cannot find own pid");
	exit(1);
  }
  if (chdir(spooldir) == FAILED) {	/* switch to spool directory */
	printmsg(M_NOLOG, "uucico: cannot change to directory %s", spooldir);
	exit(1);
  }
  if (stat(spooldir, &statbuff) == -1) {
	printmsg(M_NOLOG, "uucico: cannot stat() directory %s", spooldir);
	exit(1);
  }
  devnum = statbuff.st_dev;
  
  /* set up to catch signals */
  if (catchsignal() == FAILED) {
	printmsg(M_NOLOG, "uucico: cannot catch signals");
	exit(1);
  }

  /* get a process lock file if one is needed */
#ifdef UUMAX
  for (j = 0 ; j < UUMAX ; j++) {
	sprintf(lckuu, "%s%d", LCKUU, j);
	if (checklock(lckuu, 1) == OK)
		break;
  }
  if (j == UUMAX) {
	printmsg(M_NOLOG, "uucico: too many processes");
	exit(1);
  }
#endif

  /* set up the log file; if debugging, lock it for the duration */
  if (dbglvl > 0) {
	if (role == MASTER)
		cp = LCKLOG;
	else
		cp = LCKAUD;	/* use the audit file */
	if (checklock(cp, 2) == FAILED) {
		printmsg(M_NOLOG, "uucico: cannot get lock %s", cp);
		exit(1);
	}
	if (role == MASTER) {
		cp = LOGFILE;
		j = 0644;
	}
	else {
		cp = AUDLOG;
		j = 0600;
	}
	if ((fdlog = open(cp, O_CREAT | O_WRONLY | O_APPEND, j)) == -1) {
		printmsg(M_NOLOG, "uucico: cannot open %s", cp);
		exit(1);
	}
	printmsg(dbglvl, "========== %-24.24s ===========", ctime(&now));
  }

  /* set up as MASTER */
  if (role == MASTER) {
	location = LOCAL;
	/* check that access is available */
	if (checkuser(locsite, locuser, (char *)NULL) == FAILED) {
		printmsg(M_CALL, "uucico: no access for site %s, user %s",
							locsite, locuser);
		cleanup(SIGHUP);
		exit(1);
	}
  }

  /* set up the message protocol drivers */
  (void) setproto('\0');

  state = 'A';
  while (role != NOROLE) {

	printmsg(M_CALL, "uucico: ***  TOP  ***  state %c role %s", state,
			role == MASTER ? "Master" : "Slave");

	/* main state table switcher */
	switch (state) {
		case 'A':	/* initialisation - not used */
			state = 'B';
			break;
		case 'B':	/* get details of site to call */
			if (role == MASTER)
				state = getsystem();
			else
				state = 'C';
			break;
		case 'C':	/* check site details are valid */
			if (role == MASTER)
				state = checksystem();
			else
				state = 'D';
			break;
		case 'D':	/* call the site */
			if (role == MASTER)
				state = connect();
			else
				state = 'E';
			break;
		case 'E':	/* attempt to log on */
			if (role == MASTER)
				state = logon();
			else
				state = 'F';
			break;
		case 'F':	/* initialise i/o */
			state = ioinit();
			break;
		case 'G':	/* negotiate access */
			if (role == MASTER)
				state = sysminit();
			else
				state = syssinit();
			break;
		case 'H':	/* start up protocol */
			state = pktinit();
			break;
		case 'J':	/* get work from spool dir or remote */
			if (role == MASTER)
				state = getmtask(); /* get job from C.file */
			else
				state = getstask(); /* get job from remote */
			break;
		case 'Q':		/* negotiate hangup */
			state = sysbreak();
			break;
		case 'R':		/* receive a file */
			state = recvf();
			break;
		case 'S':		/* send all files */
			state = sendf();
			break;
		case 'U':		/* execute uucp */
			state = runuucp();
			break;
		case 'V':		/* shut down protocol */
			state = pktend();
			break;
		case 'W':		/* end access */
			state = sysend();
			break;
		case 'X':		/* revert i/o */
			state = ioend();
			break;
		case 'Y':		/* end call */
			state = callbreak();
			break;
		case 'Z':		/* quit state machine */
		        printmsg(M_CALL, "uucico: quitting, state: %c", state);
			if (errmsg[0] != '\0') (void) writerr(errmsg);
			role = NOROLE;
			break;
		default:
		        printmsg(M_CALL, "uucico: default state: %02.2xx", state);
			sterflg = TRUE;
			state = 'Y';
			break;
	}
  }

  /* run uuxqt unless told not to */
  if (opt_U) {
	/* run uuxqt */
	switch (j = fork()) {
		case -1:		/* can't create new process */
			break;
		case 0:			/* forked, try to execute uuxqt */
			(void) execlp(UUXQT, "uuxqt", (char *)NULL);
			exit(1);	/* never happen */
			break;
		default:		/* uuxqt running, wait for status */
			while ((k = wait(&status)) != j && k != -1)
				;
			break;
	}

	printmsg(M_CALL, "uucico: uuxqt fork %d, wait %d, status %xx", j, k, status);
	if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 || WTERMSIG(status) != 0)
		printmsg(M_CALL, "uucico: cannot run uuxqt");
  }

  /* close log file */
  if (dbglvl > 0) {
	(void) close(fdlog);
	if (strcmp(cp, LOGFILE) == SAME)
		(void) unlock(LCKLOG);
	else
		(void) unlock(LCKAUD);
  }
  (void) unlock(lckuu);

#ifdef RUNUUCICO
  /* call back if necessary */
  if (cbflg) {
	sprintf(scratch, "-s %s", rmtsite);
	switch (j = fork()) {
		case -1:		/* can't create new process */
			break;
		case 0:			/* forked, try to execute uuxqt */
			(void) execlp(UUCICO, "uucico", scratch, (char *)NULL);
			exit(1);	/* never happen */
			break;
		default:		/* don't wait for status */
			break;
	}
  }
#endif

  exit(0);
  /* NOTREACHED */
}


/*
 * m o v e f i l e
 *
 * Move a file from a to b using link or copy
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 */
int movefile(from, to)
char *from, *to;
{
  char buff[BUFSIZ];
  int j, infd, outfd;

  printmsg(M_SPOOL, "movefile: moving |%s| to |%s|", from, to);

  if (link(from, to) == FAILED) {	/* copy the file */
	if ((infd = open(from, 0)) == -1) {
		printmsg(M_TRANSFER, "movefile: cannot open %s", from);
		printmsg(M_STD, "CAN'T OPEN (%s)", from);
		return(FAILED);
	}
	if ((outfd = creat(to, 0644)) == -1) {
		(void) close(infd);
		printmsg(M_TRANSFER, "movefile: cannot create %s", to);
		printmsg(M_STD, "CAN'T CREATE (%s)", to);
		return(FAILED);
	}
	while ((j = read(infd, buff, sizeof(buff))) != 0) {
		if (j == -1 || write(outfd, buff, (size_t)j) != j) {
			(void) close(infd);
			(void) close(outfd);
			printmsg(M_TRANSFER,
				"movefile: cannot copy |%s| to |%s|", from, to);
			return(FAILED);
		}
	}
	(void) close(infd);
	(void) close(outfd);
  }
  if (unlink(from) == -1)
	printmsg(M_TRANSFER, "movefile: cannot remove file %s", from);
  if (chmod(to, 0666) == -1)
	printmsg(M_TRANSFER, "movefile: cannot chmod file %s", to);

  return(OK);
}


/*
 * o n t i m e
 *
 * Catch SIGALRM
 *
 * Return:	nothing
 */
void ontime()
{
  (void) signal(SIGALRM, ontime);
}


/*
 * o p e n d a t a
 *
 * Try to open the data file
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int opendata()
{
  if (fpdata != (FILE *)NULL)		/* trouble, some file already open */
	return(FAILED);

  if (dfileflg) {			/* try the data file */
	if ((fpdata = fopen(datafile, "r")) == (FILE *)NULL) {
		dfileflg = FALSE;
		fpdata = fopen(fromfile, "r");
	}
  }
  else					/* go straight to original */
	fpdata = fopen(fromfile, "r");

  if (fpdata == (FILE *)NULL)		/* if bad file pointer, give up */
	return(FAILED);
  else
	return(OK);
}


/*
 * o p e n s t a t u s
 *
 * Try to open the status file 
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int openstatus(site)
char *site;
{
  if (fpstat != (FILE *)NULL) return(OK);

  sprintf(scratch, "%s%s", STSTPATH, site);
  if ((fpstat = fopen(scratch, "w")) == (FILE *)NULL) {
	printmsg(M_STD, "ASSERT ERROR");
	sprintf(errmsg, "CAN'T OPEN %s (0)", scratch);
	sterflg = TRUE;
	return(FAILED);
  }
  else
	return(OK); 
}


/*
 * p a r s e c m d
 *
 * Parses a command string, expanding the local filenames in accordance with
 * the usual "~" convention.
 *
 * Return:	OK	SUCCESS
 *              FAILED	local file parse failure
 *		(fromfile, tofile, datafile, cmdargs are updated as side-effects)
 *
 * The file "~user" expands to the home directory of "user". "~/" expands
 * to the home directory of the effective userid, which in this case
 * is usually the /usr/spool/uucppublic directory. It expects a command
 * string in 'R' or 'S' format, in the buffer cmd, and calls on expdtilde
 * for part of the work.
 */
int parsecmd(cmd)
char *cmd;
{
  char *cp, *bp, *fields[MAXARGS], buff[132];
  int j, nargs, local, remote, sender, args, data, mode, mail, mbox;
  DIR *dirp = (DIR *)NULL;

  /* clear all the fixed data buffers */
  for (j = 0 ; j < 132 ; j++)
	fromfile[j] = tofile[j] = datafile[j] = tempfile[j] = buff[j] = '\0';

  /* check for a valid command string */
  nargs = getargs(cmd, fields);
  if ((cmd[0] != 'R' && cmd[0] != 'S') || (cmd[0] == 'R' && nargs < 5)
					|| (cmd[0] == 'S' && nargs < 7)) {
	printmsg(M_SPOOL, "parsecmd: invalid command |%s|", curcmd);
	return(FAILED);
  }

  /* sort out which field means what */
  if ((role == MASTER && cmd[0] == 'S')
		|| (role == SLAVE && cmd[0] == 'R')) {
	local = 1;
	remote = 2;
  }
  else {
	local = 2;		/* sometimes tofile is local */
	remote = 1;
  }
  /* always true */
  sender = 3;
  args = 4;

  /* Both 'R' and 'S' messages start 'taskcode fromfile tofile ...' */
  /* but the 'R' command omits the mail, datafile and mode fields */
  if (cmd[0] == 'S') {
	data = 5;
	mode = 6;
	mail = 7;
	mbox = 8;
  }
  else {
	data = 0;
	mode = 0;
	mail = 0;
	mbox = 5;
  }

  /* sort out the options */
  opt_C = opt_c = opt_d = opt_f = opt_m = opt_n = opt_o = FALSE;
  cp = fields[args];
  while (*cp++) {			/* only some arguments are valid here */
	if (*cp == 'C') opt_c = TRUE;
	if (*cp == 'c') opt_C = TRUE;
	if (*cp == 'd') opt_d = TRUE;
	if (*cp == 'f') opt_f = TRUE;
	if (*cp == 'm') {
		opt_m = TRUE;
		strncpy(locmail, fields[sender], 19);
	}
	if (*cp == 'n' && cmd[0] == 'S') {
		opt_n = TRUE;
		strncpy(rmtmail, fields[mail], 19);
	}
	if (*cp == 'o') {
		if(opt_m && nargs >= mbox) {
			opt_o = TRUE;
			strncpy(mailbox, fields[mbox], 131);
		}
	}
  }

  printmsg(M_SPOOL, "parsecmd: %d args in string %s", nargs, curcmd);
  printmsg(M_SPOOL, "parsecmd: flag settings C %d c %d d %d f %d m %d n %d o %d",
			opt_C, opt_c, opt_d, opt_f, opt_m, opt_n, opt_o); 

  if (*fields[local] == '/')		/* leave absolute paths alone */
	strncpy(buff, fields[local], 131);
  else {
	if (expdtilde(fields[local]) == FAILED) return(FAILED);
	strncpy(buff, scratch, 131);
  }

  /* We have a name in buff: is it a directory?  Check for '/', try to open */
  bp = buff + strlen(buff) - 1;
  if (*bp == '\n') *bp-- = '\0';	/* clear a trailing newline */

  if (*bp == '/' || (dirp = opendir(buff)) != (DIR *)NULL) {
	if (dirp != (DIR *)NULL) (void) closedir(dirp);
	if (*bp != '/') strcat(buff, "/");
	bp = strrchr(fields[remote], '/');
	if (bp == (char *)NULL)		/* if there's no remote path */
		bp = fields[remote];	/* use the whole remote name */
	else
		bp++;
	strncat(buff, bp, 131 - strlen(buff));	/* add the filename */
  }

  /* update tofile and fromfile */
  if ((role == MASTER && cmd[0] == 'S') ||
		(role == SLAVE && cmd[0] == 'R')) {
	strncpy(tofile, fields[remote], 131);
	strncpy(fromfile, buff, 131);
  }
  else {
	strncpy(tofile, buff, 131);
	strncpy(fromfile, fields[remote], 131);
  }
  /* update senders name */
  strncpy(rmtuser, fields[sender], 19);
  /* update cmdargs */
  if (strlen(fields[args]) > 1)		/* there are arguments */
	strncpy(cmdargs, fields[args], 19);
  else
	cmdargs[0] = '\0';
  /* update datafile */
  if (cmd[0] == 'S') {
	if (strncmp(fields[data], NODATAFN, 3) == SAME)
		dfileflg = FALSE;
	else {
		dfileflg = TRUE;
		strncpy(datafile, spooldir, 130);
		strcat(datafile, "/");
		strncat(datafile, fields[data], 130 - strlen(spooldir));
	}
  }
  else
	dfileflg = FALSE;

  printmsg(M_SPOOL, "parsecmd: fromfile %s, tofile %s, args %s, datafile %s",
		fromfile, tofile,
		cmdargs[0] ? cmdargs : "none",
		dfileflg ? datafile : "none");
  return(OK);
}


/*
 * p k t e n d
 *
 * Shut down the packet protocol
 *
 * Return:	W	Always
 */
char pktend()
{
  /* turn off the current protocol */
  (*turnoff)(fdin, fdout);

  /* re-initialise the protocol table */
  (void) setproto('\0'); /* UUCP uses ^P ... ^@ bracketed strings for conversation */

  return('W');
}


/*
 * p k t i n i t
 *
 * Initialize the packet protocol
 *
 * Return:	W	Failure
 *		J	Success
 */
char pktinit()
{
  if (setproto(pname) == FAILED || (*turnon)(fdin, fdout) == FAILED) {
	printmsg(M_STD, "FAILED (conversation complete)");
	callstatus |= EC_SFAIL;
	return('W');
  }
  else {
	printmsg(M_STD, "OK (startup)");
	return('J');
  }
}


/*
 * p r i n t m s g
 *
 * Print an error or debugging message into the system error log file.
 *
 * Return:	Nothing
 *
 * All messages at levels less than or equal to the current debug level
 * are printed, unless the M_NOLOG level is used. If the location is LOCAL,
 * and debugging is on, messages are also printed to standard error.
 */
/* VARARGS1 */
void printmsg(level, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int level; char *fmt, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8;
{
  char msg[BUFSIZ];

  (void) time(&now);
  if (level <= dbglvl) {
	sprintf(msg, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
	strcat(msg, "\n");
	if (level != M_NOLOG) {
		if (level == M_STD)
			(void) writelog(msg, TRUE);
		else
			(void) writelog(msg, FALSE);
	}
	if (dbglvl > 0 && location == LOCAL) {
		fprintf(stderr, "%s", msg);
		(void) fflush(stderr);
	}
  }
}


/*
 * r e c v f
 *
 * The state table switcher for receiving files.
 *
 * Return:	J	Success
 *		V	Failure
 *		0	rdmsg()/wrmsg() failure
 */
char recvf()
{
  char buff[256], lstate = 'H';		/* Receive-Init is the start state */
  char *goodmsg = "copy succeeded";
  char *badmsg = "can't copy to file directory - file left in SPOOLDIR";
  char *vbadmsg = "copy failed";

  while (TRUE) {
	printmsg(M_TRANSFER, "recvf: receive lstate %c",
			(lstate > ' ') ? lstate : (lstate + '0'));
	if (lstate == 'H') lstate = rhdr();		/* Receive-File-Header */
	else if (lstate == 'D') {			/* Receive-File-Data */
		if ((*rddata)(fptemp, fdin) == OK) lstate = 'E';
		else lstate = 'A';
	}
	else if (lstate == 'E') lstate = reof();	/* Receive-End-of-File */
	else if (lstate == 'C') break;			/* Completed task */
	else if (lstate == 'Q') break;			/* Remote header failure */
	else if (lstate == 'A') break;			/* Abort */
	else break;					/* Try to fail gracefully */
  }

  /* inform anyone else who needs to know */
  if (lstate == 'C') {
	if (opt_m && role == MASTER) {		/* notify initiator */
		sprintf(buff, "file %s from system %s\n%s\n", tofile, rmtsite,
				copyflg == OK ? goodmsg : badmsg);
		(void) sendmail(locsite, locmail, buff, (char *)NULL, opt_o);
	}
	if (opt_n) {				/* notify recipient */
	 	sprintf(buff, "%s from %s!%s arrived\n",
				tofile, rmtsite, rmtuser);
		if (copyflg == FAILED)
			strncat(buff, badmsg, 255 - strlen(buff));
		(void) sendmail(locsite, rmtmail, buff, (char *)NULL, FALSE);
	}
  }

  /* send final status to remote */
  if (lstate == 'C' && copyflg == OK) {		/* success */
	tasknum++;
	printmsg(M_STD, "COPY (SUCCEEDED)");
	if ((*wrmsg)("CY", fdout) == FAILED) return(0);
  }
  else if (lstate == 'C') {		/* local copy failure */
	printmsg(M_STD, "COPY FAILED (cannot copy TM file)");
	if ((*wrmsg)("CN5", fdout) == FAILED) return(0);
  }
  else if (lstate == 'A') {
	printmsg(M_TRANSFER, "recvf: aborting");
	(void) fclose(fptemp);
	fptemp = (FILE *)NULL;
	(void) unlink(tempfile);
  }
 
  printmsg(M_TRANSFER, "recvf: quitting, lstate %c",
			(lstate > ' ') ? lstate : (lstate + '0'));
  if (lstate == 'C' || lstate == 'Q')
	return('J');
  else if (lstate == 'A')
	return('V');
  else
	return(0);
}


/*
 * r e o f 
 *
 * Close the temporary file and move it to its final position
 *
 * Return:	C	Always
 *		copyflg is updated as a side effect
 */
char reof()
{
  char *dirp[16], *cp, path[132];
  int j, k, status, level, depth;

  (void) fclose(fptemp);
  fptemp = (FILE *)NULL;
  printmsg(M_TRANSFER, "reof: transfer of %s completed", tofile);

  copyflg = FAILED;

  if (tofile[0] != '/') {	/* the path is wrong */
	printmsg(M_TRANSFER, "reof: path %s invalid", tofile);
	if (role == MASTER) taskstatus |= ET_BADCMD;
	return('C');
  }
  if (access(tofile, 0) == 0) { /* the file already exists */
	printmsg(M_TRANSFER, "reof: file %s exists", tofile);
	if (role == MASTER) taskstatus |= ET_LACCESS;
	return('C');
  }

  /* check the parent directory */
  strncpy(scratch, tofile, 131);
  cp = strrchr(scratch, '/');
  *cp = '\0';

  printmsg(M_TRANSFER, "reof: checking directory %s", scratch);
  if (cp == scratch) {			/* root directory */
	if (access("/", 0) == 0)
		copyflg = movefile(tempfile, tofile);
  }
  else if (access(scratch, 0) == 0) {	/* we have write access */
	copyflg = movefile(tempfile, tofile);
  }
  else if (!opt_d) {		/* we cannot make directories */
	printmsg(M_TRANSFER, "reof: need directory %s", scratch);
  }
  else if (opt_d) {		/* we can try and make directories */
	/* parse the path string */
	depth = 0;
	for (cp = scratch ; *cp ; cp++) {
		if (*cp == '/')
			dirp[depth++] = cp;
 	}
	dirp[depth] = cp;

	for (level = depth - 1 ; level > 0 ; level-- ) {
		*dirp[level] = '\0';
		if (access(scratch, 0) == 0) break; /* directory exists */
	}

	/* try to make the required directories */
	while (level < depth) {
		*dirp[level] = '/';

		switch (j = fork()) {	/* run mkdir */
		case -1:		/* can't create new process */
			break;
		case 0:			/* forked, try to execute mkdir */
			(void) fflush(stderr);
			(void) freopen("/dev/null", "w", stderr);
			(void) execlp(MKDIR, "mkdir", scratch, (char *)NULL);
			exit(1);	/* never happen */
			break;
		default:		/* mkdir running, wait for status */
			while ((k = wait(&status)) != j && k != -1)
				;
			break;
		} 
		if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 ||
						WTERMSIG(status) != 0) {
			printmsg(M_TRANSFER, "reof: cannot make directory %s",
							scratch);
			if (role == MASTER)
				taskstatus |= ET_LDIR;
			break;
		}
		else
			(void) chmod(scratch, 0777);
		level++;
	}
	if (j > 1 && k > 1 && WEXITSTATUS(status) == 0 && WTERMSIG(status) == 0)
		copyflg = movefile(tempfile, tofile);
  }
  if (copyflg == OK) {
	sprintf(path, "%s/", spooldir);
	if (strncmp(path, tofile, strlen(path)) == SAME)
		(void) chmod(tofile, 0600);
	printmsg(M_STD, "TEMP FILE STORED (%s)", tofile);
  }
  if (role == MASTER && copyflg == FAILED && taskstatus == 0)
	taskstatus |= ET_LACCESS;

  return('C');
}


/*
 * r h d r
 *
 * Receive file header from remote
 *
 * Return:	0	rdmsg()/wrmsg() failure
 *		A	Local failures (fatal)
 *		Q	Remote failure
 *		D	Success
 */
char rhdr()
{
  printmsg(M_TRANSFER, "rhdr: negotiating transfer");

  if (role == MASTER) {
	/* Response to 'R fromfile tofile user -args [mbox]' */
	if ((*rdmsg)(rbuffer, fdin) == FAILED) 
		return(0);		/* Something wrong in channel */
	if (strncmp(rbuffer, "RY", 2) == SAME) {
		/* saves time */
	}
	else if (rbuffer[2] == '2') {
		printmsg(M_STD, "REQUEST (remote access to path/file denied)");
		taskstatus |= ET_RACCESS;
		return('Q');
	}
	else if (rbuffer[2] == '4') {	/* never happen */
		printmsg(M_STD, "REQUEST (cannot create TM file)");
		taskstatus |= ET_RTMP;
		return('Q');
	}
	else {
		printmsg(M_STD, "COPY FAILED (reason not given by remote)");
		taskstatus |= ET_NOTKNOWN;
		return('Q');
	}
  }

  /* Try to open a temporary file: if this fails, there is something
   * seriously wrong.
   */
  seqnum = getseq();		/* returns 0 on failure */
  sprintf(tempfile, "%s%04d.%03d", TMPATH, seqnum, tasknum);
  if (seqnum == 0 || (fptemp = fopen(tempfile, "w")) == (FILE *)NULL) {
	if (role == SLAVE && (*wrmsg)("SN4", fdout) == FAILED) return(0);
	printmsg(M_STD, "ASSERT ERROR");
	sprintf(errmsg, "CAN'T OPEN %s (0)", tempfile);
	return('\0');
  }
  else {
	if (role == SLAVE && (*wrmsg)("SY", fdout) == FAILED) return(0);
	printmsg(M_STD, "TEMP FILE CREATED (%s)", tempfile);
	return('D');		/* Switch to data state */
  }
}


/*
 * r m e s g
 *
 * read a ^P message from UUCP
 *
 * Return:	FAILED		On sread() failure or EOF
 *		message length	On success
 */
int rmesg(msg, in)
char *msg; int in;
{
  char c, cc[5];
  int j;

  /* get the leading ^P */
  c = 'a';
  while (c != 0x10) {
	/* Don't ask. ACK & MSC need more than a byte to breathe. */
	if (sread(cc, 1, msgtime) < 1) return(FAILED);
	c = cc[0] & 0x7f;
  }

  /* read the message and clear any terminating CR or LF */
  for (j = 0 ; j < MSGLEN && c ; j++) {
	if (sread(cc, 1, msgtime) < 1) return(FAILED);
	c = cc[0] & 0x7f;
	if (c == '\r' || c == '\n') c = '\0';
	msg[j] = c;
  }

  return(OK);
}


/*
 * r u n u u c p
 *
 * Run the uucp program with an 'X' command string
 *
 * Return:	'J'	Ready for next task
 *		'Q'	Hangup (no work)
 *		'W'	Abort
 */
char runuucp()
{
  int j, k, status;

  if (role == MASTER) {
	if ((*rdmsg)(rbuffer, fdin) == FAILED) return('W');
	return('J');			/* get next task anyway */
  }
  else {
	strncpy(tempbuff, curcmd, 255);
	j = getargs(tempbuff, cmdflds);	/* getargs roaches string */

	/* sort out the arguments; -r is always used */
	if (strlen(cmdflds[4]) == 1)	/* '-' only, no arguments */
		*cmdflds[4] = ' ';
	printmsg(M_SPOOL, "runuucp: cmd1 %s, cmd2 %s, cmd4 %s",
		cmdflds[1], cmdflds[2], cmdflds[4]);

	switch (j = fork()) {
		case -1:		/* can't create new process */
			break;
		case 0:			/* forked, try to execute uucp */
			(void) fflush(stderr);
			(void) freopen("/dev/null", "w", stderr);
			(void) execlp(UUCP, "uucp", "-r",
				cmdflds[4], cmdflds[1], cmdflds[2], (char *)NULL);
			exit(1);	/* never happen */
			break;
		default:		/* uucp running, wait for status */
			while ((k = wait(&status)) != j && k != -1)
				;
			break;
	}

	if ((role == MASTER) && (j < 1 || k < 1))
		taskstatus |= ET_BADUUCP;
	printmsg(M_SPOOL, "runuucp: role %s, uucp fork %d, wait %d, status %xx",
		(role == MASTER) ? "Master" : "Slave", j, k, status);

	if (j < 1 || k < 1 || WEXITSTATUS(status) != 0 || WTERMSIG(status) != 0)
		strncpy(sbuffer, "XN", 131);
	else
		strncpy(sbuffer, "XY", 131);
	if ((*wrmsg)(sbuffer, fdout) == FAILED)
		return('W');	/* bad vibes, bale out */
	else
		return('J');	/* get next task */
  }
}

