/*
 * UUCICO - A UUCP File Transport Program.
 *
 *		This is the revised version of DCP, which tries to act as the
 *		"real" UUCICO as much as possible on a native UNIX-like system,
 *		as opposed to the original DCP.
 *
 *		The first port to MINIX was done by Peter S. Housel, more
 *		work was done by F. van Kempen, and then there was a complete
 *		re-write by Will Rose.  This version is part of a full suite
 *		of UUCP programs.
 *
 * Version:	4.2     31 October 1991.
 *
 * Author:	***** Copyright Richard H. Lamb 1985,1986,1987 *****
 *
 * History:	Original by R.H. Lamb
 *		Revised by Stuart Lynne May 1987
 *		Modified heavily for MINIX by Peter S. Housel
 *		Other (also heavy) modifications by Fred van Kempen:
 * 			11/19/89 - 05/12/90
 *		Rewritten completely to match SVR2 UUCP by Will Rose:
 *			05/12/90 - 10/31/91
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <time.h>
#include <unistd.h>
#ifdef _SVR2
#include <ustat.h>
#endif
#include "dial.h"
#include "uucp.h"
#include "uucico.h"

/* Version number in binary */

#ifndef lint
static char *Version = "@(#) uucico 4.2 (31 October 1991)";
#endif

/* Global variables */

char rbuffer[MSGLEN];		/* protocol message input buffer */
char sbuffer[MSGLEN];		/* protocol message output buffer */
char userpath[256];		/* USERFILE path list */
char tempbuff[256];		/* logon / parse buffer */
char curcmd[256];		/* current command */
char scratch[132];		/* scratch buffer */
char spooldir[132];		/* spool directory */
char cmdfile[132];		/* command file name */
char fromfile[132];		/* source of copy */
char tofile[132];		/* destination of copy */
char datafile[132];		/* datafile for copy */
char tempfile[132];		/* temporary file */
char mailbox[132];		/* redefined mailbox */
char errmsg[132];		/* assert error details */
char rmtsite[20];		/* remote site name */
char rmtuser[20];		/* remote user name */
char locsite[20];		/* local site name */
char locuser[20];		/* local user name */
char locmail[20];		/* local mail recipient */
char rmtmail[20];		/* remote mail recipient */
char cmdargs[20];		/* command arguments */
char device[20];		/* device type from L.sys */
char protocols[20];		/* device protocols from L.sys */
char baud[20];			/* device speed from L.sys */
char phone[20];			/* device phone number from L.sys */
char cctime[64];		/* clear calling times from L.sys */
char progname[20];		/* current program name */
char curtime[64];		/* current time */
char state;			/* current state of the UUCP state machine */
char framefirst;		/* first byte of conversation-level message */
char framelast;			/* last byte of conversation-level message */
char *cmdflds[MAXARGS];		/* pointers to parsed command arguments */
char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char spoolist[MAXLOCKS][LOCKLEN + 1];	/* spool file list */

int role;			/* MASTER or SLAVE */
int location;			/* LOCAL or REMOTE */
int fdin;			/* serial input file descriptor */
int fdout;			/* serial output file descriptor */
int commfd;			/* comms line file descriptor */
int retrynum;			/* call retry number */
int retryint;			/* call retry interval */
int maxfld;			/* maximum send/expect field in logon */
int dbglvl;			/* debugging level */
int lsysindex;			/* current position in L.sys */
int seqnum;			/* sequence number */
int jobnum;			/* current job number */
int tasknum;			/* task number */
int taskstatus;			/* task status code number */
int callstatus;			/* call status code number */
int pid;			/* process pid */
int copyflg;			/* file copy flag */
int dfileflg;			/* datafile present flag */
int slckflg;			/* site lock file present flag */
int sterflg;			/* bad status flag */
int usrfflg;			/* USERFILE read flag */
int cbflg;			/* call back flag */
int opt_S;			/* ignore status file */
int opt_T;			/* reset message terminator */
int opt_U;			/* do not run uuxqt */
int opt_C;			/* uucico slave command flags */
int opt_c;
int opt_d;
int opt_f;
int opt_m;
int opt_n;
int opt_o;
int opt_x;

int errno;			/* used by errno.h */

int fdlog;			/* system log file descriptor */
int fderr;			/* error log file descriptor */

unsigned int msgtime;		/* serial input timeout setting */
dev_t devnum;			/* spool directory device number */
time_t now;			/* current time */
time_t begin;			/* session start time */

FILE *fpcmd;			/* command file pointer */
FILE *fpdata;			/* data file pointer */
FILE *fpstat;			/* status file pointer */
FILE *fptemp;			/* temp file pointer */
CALL call;			/* dial(3) structure */
struct TTY stdtty;		/* saved terminal parameters for stdin */

/* Variables used by the device access routines */

char *devname = (char *)NULL;	/* access to connected devices */
int (*devopen)() = empty;
int (*deverror)() = empty;
int (*devclose)() = empty;

struct Dev Devlst[] = {		/* set of connected devices */
  "direct", ttyopen, ttyerror, ttyclose,
  "ACU", dial, merror, hangup,
  "PAD", padopen, paderror, padclose,
  (char *)NULL, empty, empty,
};

/* Variables used by the protocol routines */

char pname = '\0';
int (*turnon)() = empty;
int (*rdmsg)() = empty;
int (*wrmsg)() = empty;
int (*rddata)() = empty;
int (*wrdata)() = empty;
int (*turnoff)() = empty;

/* the first protocol on the list is the default low-level protocol */
/* the last protocol is the default conversation-level protocol */

struct Proto Protolst[] = {
  'g', gopenpk, grdmsg, gwrmsg, grddata, gwrdata, gclosepk,
  'f', fon, frdmsg, fwrmsg, frddata, fwrdata, foff,
  't', ton, trdmsg, twrmsg, trddata, twrdata, toff,
  '\0', empty, rmesg, wmesg, empty, empty, empty,
};


/*
 * a d d l o c k
 *
 * Add a lock to the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int addlock(name)
char *name;
{
  char lock[132];
  int j;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* add it to the list */
  for (j = 0 ; j < MAXLOCKS ; j++) {
	if (locklist[j][0] == '\0') {
		strncpy(locklist[j], lock, LOCKLEN);
		printmsg(M_CALL, "addlock: lockfile %s set", lock);
		break;
	}
  }

  /* see if we succeeded */
  if (j == MAXLOCKS)
	return(FAILED);
  else
	return(OK);
}


/*
 * a t o l l
 *
 * Convert decimal string to positive long int
 *
 * Return:	a positive long number
 *
 * (1.5.10 Ansi/atol.c didn't work, and lint complained about portability)
 */
long int atoll(bp)
char *bp;
{
  int j;
  long int result = 0L;

  while (*bp && isspace(*bp))	/* skip leading white space */
	bp++;

  while (*bp) {
	j = 0;
	while (j < 10 && "0123456789"[j] != *bp) j++; 
	if (j == 10) break;
	result = (result << 1) + (result << 3) + j;
	bp++;
  }

  return(result);
}


/*
 * c a l l b r e a k
 *
 * Continue with call, or quit
 * 
 * Return:	A	Continue
 *		Z	Quit
 */
char callbreak()
{
  char buff[132];

  printmsg(M_CALL, "callbreak: site %s, status error %s, output fd %d",
		rmtsite, sterflg ? "TRUE" : "FALSE", fdout);

  /* if an STST file is open, close it */
  if (fpstat != (FILE *)NULL) {
	(void) fclose(fpstat);
	fpstat = (FILE *)NULL;
  }

  /* If we've been on line, tidy up */
  if (fdout != -1) {
	(void) clearbuff();
	(void) (*devclose)(fdout);
	fdin = fdout = -1;
	devopen = deverror = devclose = empty;
	if (!sterflg) {
		printmsg(M_CALL, "callbreak: removing STST file for site %s",
								rmtsite);
		strncpy(buff, STSTPATH, 131);
		strncat(buff, rmtsite, 131 - strlen(STSTPATH));
		(void) unlink(buff);
	}
  }

  /* remove the site lock file unless communication seems impossible */
  if (!(callstatus & EC_BADSYS) && !(callstatus & EC_HFAIL)) {
	strncpy(buff, LCKPATH, 131);
	strncat(buff, rmtsite, 131 - strlen(LCKPATH));
	(void) unlock(buff); 
  }

  return('Z');
}


/*
 * c h e c k c o d e
 *
 * Check a phone number string, include dialcode
 *
 * Return:	OK	Conversion
 *		FAILED	No conversion
 *		code is updated as a side effect
 */
int checkcode(code)
char *code;
{
  char *cp, *bp, ascii[20], digit[20], telnum[20], buff[132];
  int flg;
  FILE *fp;

  /* park the actual digits */
  bp = telnum;
  cp = code;
  while (isalpha(*cp) && (bp - telnum) < 20)
	*bp++ = *cp++;
  *bp = '\0';

  printmsg(M_CALLMSG, "checkcode: code |%s|, number |%s|", telnum, cp);

  /* convert any leading ascii code */
  flg = FAILED;
  if (telnum[0] != '\0') {
	if ((fp = fopen(LDIAL, "r")) != (FILE *)NULL) {
		while (fgets(buff, 132, fp) != (char *)NULL) {
			if (buff[0] == '#' || buff[0] == '\n') continue;
			if (sscanf(buff, "%s %s", ascii, digit) != 2) continue;
			if (strcmp(telnum, ascii) != SAME) continue;
			printmsg(M_CALLMSG, "checkcode: match %s with %s", ascii, digit);
			strncpy(telnum, digit, 19);
			flg = OK;
			break;
		}
		(void) fclose(fp);
	}
  }
  strncat(telnum, cp, 19 - strlen(telnum));
  strncpy(code, telnum, 19);

  return(flg);
}


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Of the four main signals, all but SIGINT may be ignored, so that only one
 * signal has to be reset for critical parts of the code.  The remaining ten
 * or so signals are scarce enough, and rare enough, to be let run.
 *
 * SIGHUP may be ignored, or caught, depending.  The shell sends SIGHUP to
 * background processes, which in some circumstances may produce an unwanted
 * shutdown of uucico.  Furthermore, the standard PC Minix doesn't generate
 * SIGHUP, so there isn't much point in using it.  However, SIGHUP is found
 * in most versions of Unix, so to use it here simply replace SIG_IGN by
 * cleanup in the setsignal(SIGHUP) call.  Ioend() resets SIGHUP to SIG_IGN
 * in any case, to ensure that the shut-down process runs to completion.
 *
 * SIGLARM is used for non-blocking read; rather than set it on each read()
 * call, it seems better to set it once here.  However, if dial(3) also uses
 * SIGLARM, then it will have to be set and reset each time it is used.
 */
int catchsignal()
{
  if (setsignal(SIGALRM, ontime) == FAILED ||			/* signal 14 */
		setsignal(SIGHUP, SIG_IGN) == FAILED ||		/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGTERM, SIG_IGN) == FAILED ||	/* signal 15 */
		setsignal(SIGQUIT, SIG_IGN) == FAILED)		/* signal 3 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  sig_t sigint, sighup;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* disable SIGINT and SIGHUP, the only un-ignored signal */
  if ((sigint = signal(SIGINT, SIG_IGN)) == BADSIG ||
		(sighup = signal(SIGHUP, SIG_IGN)) == BADSIG)
	return(FAILED);

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  /* re-enable SIGINT and SIGHUP */
  if (signal(SIGINT, sigint) == BADSIG || signal(SIGHUP, sighup) == BADSIG)
	return(FAILED);

  return(flg);
}


/*
 * c h e c k n a m e
 *
 * Check if "name" is a known site
 *
 * Return:	FAILED	Cannot open L.sys file
 *		FAILED	Name not recognised
 *		OK	Name recognised
 */
int checkname(name)
char *name;
{
  char cont[132], line[BUFSIZ];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = strlen(name);
  if (j > SITELEN) j = SITELEN;
  while (fgets(line, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(line)) < (BUFSIZ - 132) &&
			line[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		line[k - 2] = '\0';
		strcat(line, cont);
	}
	for (k = 0 ; k < j ; k++) {
		if (line[k] != name[k])
			break;
	}
	if (k == j) {
		(void) fclose(fplsys);
       		return(OK);
	}
  }
  (void) fclose(fplsys);
  return(FAILED);
}


/*
 * c h e c k s y s t e m
 *
 * Check if a site may be called
 *
 * Return:	D	Call permitted
 *		Y	Call not permitted (wrong time to call)
 *		Z	Call not permitted (in communication already)
 */
char checksystem()
{
  char status = 'D';

  if (!opt_S && checktime(cctime) == FAILED) {
	printmsg(M_STD, "WRONG TIME (call to %s)", rmtsite);
	callstatus |= EC_WTIME;
	status = 'Y';	
  }
  if (status == 'D' && !slckflg) {
	if (checklock(rmtsite, 1) == FAILED) {
		printmsg(M_STD, "LOCKED (call to %s)", rmtsite);
		callstatus |= EC_LCKSYS;
		status = 'Z';
	}
	else
		slckflg = TRUE;
  }
  if (status == 'D') {
	if (openstatus(rmtsite) == FAILED) {
		callstatus |= EC_CPROG;
		status = 'Z';
	}
  }
  printmsg(M_CALL, "checksys: exit status %c", status);

  return(status);
}


/*
 * c h e c k t i m e
 *
 * Check if we may make a call at this time
 *
 * Return:	OK	Call permitted
 *		FAILED	Call not permitted
 */
int checktime(atime)
char *atime;
{
  char *pd, loctime[20], locday[20], buff[BUFSIZ];
  int j, k, present, first, last, delay, trydelay;
  time_t trystart;
  FILE *fp;

  /* A typical field is day[time][,retry][|...] with no intervening spaces */
  /* between day and time.  In this routine it is null-terminated. */

  /* The day subfield is specified using Any, Never, Wk and */
  /* Su Mo Tu We Th Fr Sa.  The time subfield is two 24hr times */
  /* separated by a dash.  The day subfield must be specified if */
  /* the time subfield is specified.  The retry field is decimal */
  /* minutes.  The '|' separator permits several day-time pairs. */

  /* typical ctime() output is "Wed Jul 08 18:43:07 1987\n\0" */

  sprintf(curtime, "%s", ctime(&now));
  curtime[strlen(curtime) - 1] = '\0';

  printmsg(M_CALLMSG, "checktime: now %ld atime |%s| curtime |%s|",
		now, atime, curtime);

  /* get loctime, locday */
  for (j = 0 ; j < 3 ; j++)
	locday[j] = curtime[j];
  locday[j] = '\0';
  for (j = 11 ; j < 13 ; j++)
	loctime[j - 11] = curtime[j];
  for (j = 14 ; j < 16 ; j++)
	loctime[j - 12] = curtime[j];
  loctime[4] = '\0';
  present = atoi(loctime);

  printmsg(M_CALLMSG, "checktime: locday |%s|, loctime |%s|", locday, loctime);

  /* See if there's an STST file current: the first field is the pid of the
   * process setting the file, the second field is the job status, the third
   * field is the try count, the fourth field is the UNIX time of the last
   * attempt, and the fifth field is the retry interval.
   */
  trystart = 0L;
  trydelay = 0;
  strncpy(scratch, STSTPATH, 131);
  strncat(scratch, rmtsite, 131 - strlen(STSTPATH));
  if ((fp = fopen(scratch, "r")) == (FILE *)NULL) {
	if (errno != ENOENT) {
		printmsg(M_STD, "ASSERT ERROR");
		sprintf(errmsg, "SYSTAT OPEN FAIL");
		return(FAILED);
	}
  }
  else if (fgets(buff, BUFSIZ, fp) != (char *)NULL) {
	if (sscanf(buff, "%d %d %d %ld %d", &k, &k, &retrynum, &trystart,
							&trydelay) < 5) {
		trystart = 0L;
		trydelay = 0;
	}
	printmsg(M_CALLMSG,
		"checktime: retry start %ld, interval %d, time now %ld",
		trystart, trydelay, now); 
	(void) fclose(fp);
	retrynum++;
  }

  pd = atime;		/* points to a null-terminated L.sys time field */
  while (*pd) {		/* cycle over multiple entries */
	/* check the day of the week */
	if (strncmp("Never", pd, 5) == SAME) {
		printmsg(M_CALLMSG, "checktime: Never failure");
		return(FAILED);
	}
  	else if (strncmp("Any", pd, 3) == SAME) {
		pd += 3;
	}
	else if (strncmp("Wk", pd, 2) == SAME) {
		if (strncmp(locday, "Sa", 2) == SAME ||
				strncmp(locday, "Su", 2) == SAME) {
			printmsg(M_CALLMSG, "checktime: Wk failure");
			return(FAILED);
		}
		else
			pd += 2;
	}
  	else if (strncmp(locday, pd, 2) == SAME) {
		while (isalpha(*pd))	/* use up the rest of the week */
			pd++;
	}
	else {
		pd += 2;		/* step over this entry */
		if (isalpha(*pd))
			continue;
		else {			/* look for the next field */
			while (*pd && *pd !='|')
				pd++;
			if (*pd == '|') {
				pd++;
				continue;
			}
			else {
				printmsg(M_CALLMSG, "checktime: Day failure");
				return(FAILED);
			}
		}
  	}

  	/* check the time */
	if (*pd == '\0') {		/* we have nothing more */
		first = 0;		/* set the full defaults */
		last = 2359;
		delay = 3300;
	}
	else if (*pd == ',') {		/* we have a retry interval */
		pd++;
		first = 0;
		last = 2359;
		delay = atoi(pd);
	}
  	else {				/* we have a time sequence */
		/* sort out actual and default arguments */
		j = sscanf(pd, "%04d-%04d,%d", &first, &last, &delay);
		if (j < 2) {
			printmsg(M_CALLMSG, "checktime: Time failure");
			return(FAILED);	/* garbage in the string */
		}
		else if (j == 2)
			delay = 3300;	/* standard default */
	}

	/* Note that there may be different retry intervals in the L.sys
	 * and in the status files.  We use the L.sys interval for future
	 * updates, and the current STST file for this check.
	 */
	retryint = delay;

	printmsg(M_CALLMSG, "checktime: first %d last %d wait %d retry %d",
			first, last, trydelay, delay);

	if (first < last && present > first && present < last) {
		if (now < (trystart + (long int)trydelay)) {
			printmsg(M_STD, "NO CALL (RETRY TIME NOT REACHED");
			return(FAILED);
		}
		else
			return(OK);
	}
	else {
		/* lose any retry interval present */
		while (*pd && *pd != '|')
			pd++;
		/* repeat for any subfield */
		if (*pd == '|')
			pd++;
  	}
  }
  return(FAILED);	/* probably garbage in the time field */
}


/*
 * c h e c k u s e r
 *
 * Check if a given user/site has access to a given path
 *
 * Return:	OK		Yes
 *		FAILED		No
 *		usrfflg is updated as a side effect 
 */
int checkuser(site, user, path)
char *site, *user, *path;
{
  char *cp, *bp;
  int flg;

  if (usrfflg == MAYBE)
	usrfflg = usercheck(site, user);

  if (usrfflg == FAILED)
	return(FAILED);

  if (path == (char *)NULL)
	return(OK);
 
  printmsg(M_CALLMSG, "checkuser: path |%s| userpath |%s|", path, userpath);
 
  cp = userpath;
  bp = path;
  flg = FAILED;
  while (TRUE) {
	/* successful match */
	if (*cp == '\0' || *cp == ' ') {
		flg =  OK;
		break;
	}
	/* ok so far */
	else if (*bp == *cp) {
		bp++;
		cp++;
	}
	else {
		/* reset bp */
		bp = path;
		/* use up the rest of this userpath entry */
		while (*cp && !isspace(*cp))
			cp++;
		while (isspace(*cp))
			cp++;
		/* no more possibilities */
		if (*cp == '\0')
			break;
	}
  }
 
  return(flg);
}


/*
 * c l e a r b u f f
 *
 * Clear the input buffer
 *
 * Return:	OK	Always
 */
int clearbuff()
{
  while (sread(scratch, 131, 1) > 0)
		;
  return(OK);
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files before exiting
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  /* shut down the line */
  if (fdout != -1) {
	(void) clearbuff();
	(void) (*devclose)(fdout);
  }

  /* remove all lockfiles */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0') {
		(void) unlink(locklist[j]);
		printmsg(M_CALL, "cleanup: lockfile %s removed", locklist[j]);
	}
	j++;
  }
  /* remove any STST file */
  strncpy(scratch, STSTPATH, 131);
  strncat(scratch, rmtsite, 131 - strlen(STSTPATH));
  (void) unlink(scratch);
  printmsg(M_CALL, "cleanup: status file %s removed", scratch);

  exit(1);
}


/*
 * c o n n e c t
 *
 * Connect to a system, one way or another
 *
 * Return:	B	Dialler failure
 *		E	Success
 *
 * The step following this looks for logon strings, and is made more
 * difficult if there is garbage in the tty buffer.  There are four
 * possible solutions:  1) emulate the ostrich, 2) clear the tty buffer
 * and possibly a genuine logon string, 3) clear the buffer and prompt
 * the remote with \r to get another prompt for expectstr (which may be
 * the next prompt, for a password not a logon id), 4) make the kernel
 * clear the buffer when DCD or DSR is lost.  I prefer 1 or 4, but 2
 * is implemented here for those that want it.
 */
int connect()
{
  char *flds[MAXARGS], speed[20], dialler[20], number[20], buff[132];
  int j, fd, dflg, flg;
  FILE *fp;
  struct Dev *dp;

  printmsg(M_CALL, "connect: device |%s|, baud |%s|, phone |%s|",
					device, baud, phone);

  /* clear status of relevant previous faults */ 
  callstatus &= ~EC_WTIME;
  callstatus &= ~EC_DFAIL;

  if ((fp = fopen(LDEVS, "r")) == (FILE *)NULL) return(FAILED);

  /* L-devices file has (at least) five fields per line:
   *	Class, Data_port, Dial_port, Speed, Dialler, [Token]
   *      0        1          2        3       4        5
   */

  flg = FAILED;
  while (fgets(buff, 132, fp) != (char *)NULL) {
	/* look for the right class of device */
	if (buff[0] == '#' || buff[0] == '\n')
		continue;		/* comment line */
	j = getargs(buff, flds);
	printmsg(M_CALL, "connect: fld 1 |%s| 2 |%s| 3 |%s| 4 |%s| 5 |%s| 6 |%s|",
		flds[0], flds[1], flds[2], flds[3], flds[4], flds[5]);

	if (j < 5)
		continue;		/* invalid line */
	else if (j == 5)
		flds[F_DTOKEN] = "\\D";

	if (strcmp(flds[F_DEVICE], device) != SAME)
		continue;		/* wrong device class */

	/* check the baud rate */
	if (baud[0] != '\0') {
		if (strcmp(flds[F_SPEED], "Any") != SAME &&
				strcmp(flds[F_SPEED], baud) != SAME)
			continue;	/* wrong speed */
		else
			strncpy(speed, baud, 19);
	}
	else {
		if (strcmp(flds[F_SPEED], "Any") == SAME)
			continue;	/* no speed defined */
		else
			strncpy(speed, flds[F_SPEED], 19);
	}

	/* convert any dialcode abbreviation */
	strncpy(number, phone, 19);
	if (strcmp(number, "-") != SAME && 
				strcmp(flds[F_DTOKEN], "\\T") == SAME)
		(void) checkcode(number);

	/* update the dialler type */
	strncpy(dialler, flds[F_DTYPE], 19);

	/* get the appropriate routines from the Devlst table */
	dp = Devlst;
	while (dp->Devname != (char *)NULL) {
		if (strcmp(dp->Devname, dialler) == SAME) {
			devname = dp->Devname;
			devopen = dp->Devopen;
			deverror = dp->Deverror;
			devclose = dp->Devclose;
			break;
		}
		dp++;
	}
	if (dp->Devname == (char *)NULL) continue;	/* unknown dialler */
	printmsg(M_CALL, "connect: dialler %s", dialler);
	if (strcmp(dialler, "direct") == SAME)
		dflg = TRUE;
	else
		dflg = FALSE;

	/* initialise the CALL data structure */
	call.attr = (struct TTY *)NULL;
	call.baud = atoi(speed);
	call.speed = atoi(speed);	/* used by dial(3) */
	if (dflg)
		call.line = flds[F_DATAPORT];
	else
		call.line = (char *)NULL;
	if (strcmp(number, "-") != SAME)
		call.telno = number;
	else
		call.telno = (char *)NULL;
	call.modem = 0;

	/* try to connect */
#ifdef _BCC
	if ((fd = (*devopen)(&call)) < 0) {
#else
	if ((fd = (*devopen)(call)) < 0) {
#endif
 		if (dbglvl > 0) {
			(void) (*deverror)(fd, buff);
			printmsg(M_CALL, "connect: error %s", buff);
		}
		/* no point in closing what we haven't opened */
		devopen = deverror = devclose = empty;
	}
	else {
		fdin = fdout = fd;
		flg = OK;
		break;
	}
  }

  if (flg == OK) {
	printmsg(M_STD, "OK (DIAL %s %s %s)", devname,
			call.line == (char *)NULL ? "-" : call.line,
			call.telno == (char *)NULL ? "-" : call.telno);
	(void) clearbuff();
	return('E');
  }
  else {
	if (fd == DV_NT_A)
		printmsg(M_STD, "LOCKED (call to %s)", rmtsite);
	else
		printmsg(M_STD, "FAILED (call to %s)", rmtsite);
	(void) writestatus("DIAL FAILED");
	callstatus |= EC_DFAIL;
	sterflg = TRUE;
	return('B');
  }
}

 
/*
 * e m p t y
 *
 * Do nothing (gracefully)
 *
 * Return:	0	Always
 */
int empty()
{
  printmsg(M_CALL, "empty: called");
  return(0);
}


/*
 * e x p d t i l d e
 *
 * Expands a filename using the usual "~" convention.
 *
 * Return:	OK	SUCCESS
 *		FAILED	Local filename not parsed
 *		scratch buffer is updated as a side effect
 *
 * The name "~user" expands to the home directory of "user". "~/" expands
 * to the home directory of the effective userid, which in this case
 * is usually the /usr/spool/uucppublic directory. If the file has no
 * leading '/' it is put in SPOOLDIR, largely because that is where
 * programs such as 'mail' look for messages to forward.  Care is taken
 * not to overflow the (static) name buffer.
 */
int expdtilde(filename)
char *filename;
{
  char *cp;
  struct passwd *pw;

  /* if no leading ~, put file in SPOOLDIR */
  if (*filename != '~') {
	strncpy(scratch, spooldir, 131);
	/* strip off any pathname */
	if ((cp = strrchr(filename, '/')) == (char *)NULL) {
		cp = filename;
		strcat(scratch, "/");
	}
	strncat(scratch, cp, 131 - strlen(scratch));
	return(OK);
  }
  /* get the user name from the path */
  ++filename;
  cp = scratch;
  while (cp < (scratch + 131)) {
  	if (*filename && *filename != '/')
		*cp = *filename++;
	else
		*cp = '\0';
	cp++;
  }

  if (strlen(scratch) == 0)
	pw = getpwnam("uucp");
   else
	pw = getpwnam(scratch);

  /* check for valid username */
  if (pw == (struct passwd *)NULL)
	return(FAILED);

  /* check for valid path */
  if ((strlen(pw->pw_dir) + strlen(filename)) > 131)
	return(FAILED);

  strcpy(scratch, pw->pw_dir);
  strcat(scratch, filename);
  return(OK);
}


/*
 * e x p e c t s t r
 *
 * Wait for a specific string from the remote
 *
 * Return:	TRUE	On match with expected string
 *		FALSE	On read failure (timeout)
 */
int expectstr(str, timeout)
char *str; unsigned int timeout;
{
  char c, *sm[30], buff[10];
  int j, k, max, flg;

  printmsg(M_CALL, "expectstr: wanted %s", str);

  if (strcmp(str, "\"\"") == SAME) return(TRUE);

  /* initialise state machines */
  for (j = 0 ; j < 30 ; j++)
	sm[j] = (char *)NULL;

  max = 0;
  flg = FALSE;
  while (!flg) {
	/* for some reason &c doesn't work with sread */ 
	if ((j = sread(buff, 1, timeout)) < 1) {
		printmsg(M_CALLMSG,
			"expectstr: sread failed timeout %d status %d buffer %s",
			timeout, j, visib(buff, 10));
		return(FALSE);
	}
	/* strip high bit, and remove unprintable characters */
	c = buff[0] & 0177;
	if (c < '\040' && c != '\n' && c != '\r')
		c = ' ';
	if (dbglvl >= M_CALLMSG)	/* printmsg() is too slow */
		(void) fputc(c, stderr);
	/* if necessary and possible, start up a state machine */
	if (c == *str && max < 29)	/* last array element must be NULL */
		sm[max++] = str;

	/* look at all the current SMs */
	for (j = 0 ; j < max ; j++) {
		if (*sm[j] == c) {
			/* if there's a match, update the SM */
			if (*++(sm[j]) == '\0') {
				flg = TRUE;
				break;
			}
		}
		else {
			/* if there's no match, remove this SM */
			for (k = j ; k < max ; k++)
				sm[k] = sm[k+1];
			max--;
			j--;
		}
	}
  }
  return(flg);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';

  return(nflds);
}


/*
 * g e t c o u n t
 *
 * Get the current conversation count number
 *
 * Returns	Number	Success
 *		0	Name not found
 *		FAILED	Otherwise
 *
 */
int getcount(site)
char *site;
{
  char tsite[16], buff[BUFSIZ];
  int j, tcount;
  FILE *fp;

  if (access(SQFILE, 0) == -1) return(0);	/* no file, no name */
  if (checklock(LCKSQ, 2) == FAILED) return(FAILED);

  /*  The SQFILE file contains the current call numbers for all the sites
   *  called.  The file format is: site number date_time
   */

  if ((fp = fopen(SQFILE, "r")) != (FILE *)NULL) {
	tcount = 0;
	while (fgets(buff, BUFSIZ, fp) != (char *)NULL) {
		if (buff[0] == '#') continue;
		if ((j = sscanf(buff, "%15s %d", tsite, &tcount)) < 1) break;
		printmsg(M_CALLMSG, "getcount: site %s, count %d", tsite, tcount);
		if (strncmp(site, tsite, SITELEN) == SAME) {
			if (j == 1)
				tcount = 1;
			else
				tcount++;
			break;
		}
		tcount = 0;
	}
	(void) fclose(fp);
  }
  else
	tcount = FAILED;

  (void) unlock(LCKSQ);

  return(tcount);
}


/*
 * g e t d a t e
 *
 * Get the date in a format suitable for the logfile
 *
 * Return	OK	Always
 *		buffer is updated as a side effect
 */
int getdate(buff)
char *buff;
{
  int j;

  sprintf(buff, "%s", ctime(&now));
  for (j = 0 ; j < 20 ; j++)
	buff[j] = buff[j + 4];		/* lose the day field */
  buff[j] = '\0';			/* clear the newline */

  return(OK);
}


/*
 * g e t m t a s k
 *
 * Get the next task, when master.
 *
 * Return:	W	wrmsg() failure
 *		Q	No work found
 *		J	End of task/End of job (command file)
 *		S	Task is to send file
 *		R	Task is to receive file
 *		U	Task is to run uucp
 *
 * Getmtask reads the next line from the current open workfile (cmdfile) and
 * determines from this whether the next task is a send or receive. It then
 * expands the filenames appropriately, and fires up the slave with an 'R'
 * command if necessary.
 */
char getmtask()
{
  char *fcp;
  int j, flg;
#ifdef _SVR2
  struct ustat ubuff;
#endif

  printmsg(M_SPOOL, "getmtask: looking for next command line");

  if (fpcmd == (FILE *)NULL) {
	if (scandir(rmtsite) == OK) {
		tasknum = taskstatus = 0;
	  	printmsg(M_CONVERSE, "getmtask: work found");
	}
	else {
	  	printmsg(M_CONVERSE, "getmtask: no work found");
		return('Q');	/* polling call */
	}
  }

  if (fgets(curcmd, 255, fpcmd) == (char *)NULL) {
	(void) fclose(fpcmd);
	(void) unlink(cmdfile);	/* close and delete completed workfile */
	fpcmd = (FILE *)NULL;
	printmsg(M_SPOOL, "getmtask: finished %s workfile", cmdfile);
	/* write out the current job information */
	(void) writeconn(RSTAT);
	(void) writeconn(LSTAT);
	return('J');			/* get the next command file */
  }

  j = strlen(curcmd) - 1;
  if (curcmd[j] == '\n') curcmd[j] = '\0';
  printmsg(M_STD, "REQUEST (%s)", curcmd);

  /* get the source and destination filenames if they exist */
  if (curcmd[0] == 'S' || curcmd[0] == 'R') {
	if (curcmd[0] == 'S')
		fcp = fromfile;
	else
		fcp = tofile;
	strncpy(tempbuff, curcmd, 255);	/* getargs roaches the string */
	if (parsecmd(tempbuff) == FAILED ||
				checkuser(locuser, locsite, fcp) == FAILED) {
		taskstatus |= ET_BADCMD; 
		flg = FAILED;
	}
	else
		flg = OK;

 	printmsg(M_SPOOL, "getmtask: task %d, mode %c, fromfile %s, tofile %s",
			tasknum, curcmd[0], fromfile, tofile);
	
	/* see if the data file actually exists */
	if (flg == OK && curcmd[0] == 'S') {
		if ((flg = opendata()) == FAILED) { /* not a fatal error */
			printmsg(M_SPOOL, "getmtask: cannot open data file %s",
				datafile);
			taskstatus |= ET_LACCESS;
		}
		else
			(void) fclose(fpdata);
		fpdata = (FILE *)NULL;
	}
	/* see if there's space for the destination file */
	if (flg == OK && curcmd[0] == 'R') {
#ifdef _SVR2
		if (ustat(devnum, &ubuff) == -1 || ubuff.f_tfree < LOWLEVEL) {
			printmsg(M_SPOOL, "getmtask: insufficient space on %d",
								devnum);
			taskstatus |= ET_LACCESS;
			flg = FAILED;
		}
#endif
	}
	if (flg == FAILED) {
		printmsg(M_STD, "PERMISSION (DENIED)");
		return('J');	/* get the next task */
	}
	if ((*wrmsg)(curcmd, fdout) == FAILED) return('W');
	return(curcmd[0]);
  }
  else if (curcmd[0] == 'X') {		/* run a local uucp command */
	if ((*wrmsg)(curcmd, fdout) == FAILED) return('W');
	return('U');			/* regardless */
  }
  else {
	(void) fclose(fpcmd);
	(void) unlink(cmdfile);		/* unknown command, abandon file */
	fpcmd = (FILE *)NULL;
	printmsg(M_SPOOL, "getmtask: unknown command %s in file %s", curcmd, cmdfile);
	printmsg(M_STD, "PERMISSION (DENIED)");
	taskstatus |= ET_BADCMD;
	/* write out the current job information */
	(void) writeconn(RSTAT);
	(void) writeconn(LSTAT);
	return('J');			/* get the next command file */
  }
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t s e q
 *
 * Get the next sequence number, update the sequence file
 *
 * Return:	1-9999	Success
 *		0	Failure
 */
int getseq()
{
  int fd, seq, flg;

  if (checklock(LCKSEQ, 2) == FAILED) {
	printmsg(M_STD, "ASSERT ERROR");
	sprintf(errmsg, "CANNOT GET SEQLOCK (0)");
	return(0);
  }

  /* if the SEQF file doesn't exist, create it */
  flg = TRUE;
  if (access(SEQF, 0) == -1) {
  	if ((fd = open(SEQF, O_CREAT | O_WRONLY | O_TRUNC, 0600)) == -1)
		flg = FALSE;
	else {
		if (write(fd, "0001", 4) != 4)
			flg = FALSE;
		if (close(fd) == -1)
			flg = FALSE;
	}	
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* read the seq file */
  if ((fd = open(SEQF, O_RDONLY, 0600)) == -1)
	flg = FALSE;
  else {
	if (read(fd, scratch, 4) != 4)
		flg = FALSE;
	else {
		scratch[4] = '\0';
		seq = atoi(scratch);
        }
	(void) close(fd);		/* about to overwrite */
	if (seq < 1 || seq > 9999)
		flg = FALSE;
  }
  if (!flg) {
	(void) unlock(LCKSEQ);
	return(0);
  }

  /* write new seq number */
  if ((fd = open(SEQF, O_WRONLY | O_TRUNC, 0600)) == -1)
	flg = FALSE;
  else {
	sprintf(scratch, "%04d", seq < 9999 ? seq + 1 : 1);
	if (write(fd, scratch, 4) != 4)
		flg = FALSE;
	if (close(fd) == -1)
		flg = FALSE;
  }
  (void) unlock(LCKSEQ);
  printmsg(M_CALLMSG, "getseq: got sequence number %d", seq);
  if (flg)
	return(seq);
  else
	return(0);
}


/*
 * g e t s t a s k
 *
 * Get the next task, when slave
 * 
 * Return:	W	rdmsg()/wrmsg() failure
 *		J	Get next task (command failure) 
 *		Q	Hangup command
 *		S	Send file command
 *		R	Receive file command
 *		U	UUCP command
 *		V	Unknown command
 *
 * Getstask reads the next message from the master, and tries to parse
 * it as a command.  If successful, it sets the appropriate state.
 */
char getstask()
{
  char *fcp, *mcp;
  int flg;
#ifdef _SVR2
  struct ustat ubuff;
#endif

  if ((*rdmsg)(rbuffer, fdin) == FAILED) return('W');
  strncpy(curcmd, rbuffer, 255);

  printmsg(M_CONVERSE, "getstask: task %d, command |%s|", tasknum, curcmd);
  printmsg(M_STD, "REQUESTED (%s)", curcmd);

  /* Get the source and destination filenames if they exist */
  if (curcmd[0] == 'S' || curcmd[0] == 'R') {
	if (curcmd[0] == 'S') {
		fcp = tofile;
		mcp = "SN2";
	}
	else {
		fcp = fromfile;
		mcp = "RN2";
	}
	strncpy(tempbuff, curcmd, 255);	/* getargs roaches the string */
	if (parsecmd(tempbuff) == OK && checkuser(locuser, rmtsite, fcp) == OK)
		flg = OK;
	else
		flg = FAILED;
	/* see if the source file actually exists */
	if (flg == OK && curcmd[0] == 'R') {
		if ((flg = opendata()) == OK)
			(void) fclose(fpdata);
		else
			printmsg(M_SPOOL, "getstask: cannot open source file %s",
				fromfile);
		fpdata = (FILE *)NULL;
	}
	/* see if there's space for the destination file */
	if (flg == OK && curcmd[0] == 'S') {
#ifdef _SVR2
		if (ustat(devnum, &ubuff) == -1 || ubuff.f_tfree < LOWLEVEL) {
			printmsg(M_SPOOL, "getstask: insufficient space on %d",
								devnum);
			flg = FAILED;
		}
#endif
	}
	if (flg == FAILED) {
		printmsg(M_STD, "PERMISSION (DENIED)");
		if ((*wrmsg)(mcp, fdout) == FAILED) return('W');
		return('J');	/* get the next task */
	}
  }

  switch (curcmd[0]) {
	case 'H':	return('Q');		/* Hangup command */
	case 'R':	return('S');		/* R command, send a file */
	case 'S':	return('R');		/* S command, receive a file */
	case 'X':	return('U');		/* X command, run uucp */
	default:	return('V');		/* Unknown command */
  }
}


/*
 * g e t s y s t e m
 *
 * Process the "L.sys" file
 *
 * Return:	C	Site found
 *		Y	Site not found on this pass
 *		Z	Site not known at all
 */
char getsystem()
{
  char *cp, status, buff[BUFSIZ], site[20], tmp[30];
  int j, index;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) {
	printmsg(M_STD, "ASSERT ERROR");
	sprintf(errmsg, "CAN'T OPEN %s (0)", LSYS);
	return('Z');
  }

  status = 'Y';
  index = 0;
  while (fgets(buff, BUFSIZ, fplsys) != (char *)NULL) {
	while ((j = strlen(buff)) < (BUFSIZ - 132) &&
			buff[j - 2] == '\\' &&
			fgets(tempbuff, 131, fplsys) != (char *)NULL) {
		buff[j - 2] = '\0';
		strcat(buff, tempbuff);
	}
	if (buff[0] == '#' || buff[0] == '\n') continue; /* ignore comments */
	if (index++ < lsysindex) continue;	/* been here before */
	j = strlen(buff);
	strncpy(tempbuff, buff, 255);		/* getargs roaches the string */
	maxfld = getargs(tempbuff, cmdflds);
	if (maxfld < 6) continue;		/* too few fields to be valid */
	printmsg(M_CALL, "getsystem: %s", buff);
	strncpy(site, cmdflds[F_SYSTEM], 19);
	strncpy(cctime, cmdflds[F_CCTIME], 63);
	strncpy(tmp, cmdflds[F_CLASS], 29);
	/* parse device field for any default protocols */
	if ((cp = strchr(tmp, ',')) != (char *)NULL) {
		strncpy(protocols, cp + 1, 19);
		*cp = '\0';
	}
	strncpy(device, tmp, 19);
	strncpy(baud, cmdflds[F_SPEED], 19);
	strncpy(phone, cmdflds[F_PHONE], 19);

	/* say what we've found */
	if (dbglvl >= M_CALLMSG)
		for (j = F_EXPECT; j < maxfld; j += 2) {
			sprintf(scratch,
			"getsystem: expect [%02d]: %-30s    send [%02d]: %s",
	             	j, cmdflds[j], j + 1, cmdflds[j + 1]);
			printmsg(M_CALLMSG, scratch);
		}
	printmsg(M_CALLMSG, "getsystem: remote site |%s| call time |%s| device |%s|",
				site, cmdflds[F_CCTIME], device);
	printmsg(M_CALLMSG, "getsystem: protocols |%s| speed |%s| phone |%s|",
				protocols, baud, phone);

	/* check for match */
	if (strncmp(rmtsite, site, SITELEN) == SAME) {
		printmsg(M_CALLMSG, "getsystem: site name %s matched", rmtsite);
		status = 'C';
		break;
	}
  }

  (void) fclose(fplsys);
  fplsys = (FILE *)NULL;
  if (status == 'Y' && lsysindex == 0) {	/* system not in L.sys */
	callstatus |= EC_BADSYS;
	status = 'Z';
  }
  else
	lsysindex = index;

  return(status);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}

