/* uucheck - check the UUCP files and permissions.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <ctype.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#include <time.h>
#include <unistd.h>
#include "uucp.h"

#define FALSE		0
#define TRUE		~FALSE
#define OK		1
#define FAILED		-1
#define SAME		0

#ifndef DEFDBG
#define DEFDBG "/usr/lib/uucp/DEFAULT.DBG"
#endif

#define PROTOLST	"fgt"	/* available protocols */

#ifndef lint
static char *Version = "@(#) uucheck 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_v = FALSE;		/* check individual directories and files */
int opt_x = FALSE;

int errno;			/* used by errno.h */
int dbglvl;			/* debugging level */
char locsite[20];		/* name of local site */
char progname[20];		/* program name */

struct File {
  char *name;			/* pathname */
  int need;			/* is essential */
  int dir;			/* is a directory */
} files[] = {			/* UUCP directory and file names */
  PICKDIR, TRUE, TRUE,
  PUBDIR, TRUE, TRUE,
  SPOOLDIR, TRUE, TRUE,
  XQTDIR, TRUE, TRUE,
  "/usr/lib/uucp/cu", FALSE, FALSE,
  "/usr/lib/uucp/uucheck", FALSE, FALSE,
  UUCICO, TRUE, FALSE,
  "/usr/lib/uucp/uuclean", FALSE, FALSE,
  UUCP, TRUE, FALSE,
  "/usr/lib/uucp/uulog", FALSE, FALSE,
  "/usr/lib/uucp/uuname", FALSE, FALSE,
  "/usr/lib/uucp/uupick", FALSE, FALSE,
  "/usr/lib/uucp/uusched", TRUE, FALSE,
  "/usr/lib/uucp/uustat", FALSE, FALSE,
  "/usr/lib/uucp/uusub", FALSE, FALSE,
  "/usr/lib/uucp/uuto", FALSE, FALSE,
  UUX, TRUE, FALSE,
  UUXQT, TRUE, FALSE,
  DEFDBG, FALSE, FALSE,
  LCMDS, TRUE, FALSE,
  LDEVS, TRUE, FALSE,
  LDIAL, FALSE, FALSE,
  LSTAT, FALSE, FALSE,
  LSUB, FALSE, FALSE,
  LSYS, TRUE, FALSE,
  RSUB, FALSE, FALSE,
  RSTAT, FALSE, FALSE,
  SEQF, FALSE, FALSE,
  SQFILE, FALSE, FALSE,
  NODENAME, TRUE, FALSE,
  USERFILE, FALSE, FALSE,
  AUDLOG, FALSE, FALSE,
  ERRLOG, FALSE, FALSE,
  FOREIGN, FALSE, FALSE,
  LOGFILE, FALSE, FALSE,
  SYSLOG, FALSE, FALSE,
  RMAIL, FALSE, FALSE,
  LMAIL, FALSE, FALSE,
  MODEMCAP, FALSE, FALSE,
  (char *)NULL, FALSE, FALSE
  };

/* Externals */

/* Used by getopt(3) package */
extern int getopt(), optind, opterr, optopt;
extern char *optarg;

/* Forward declarations */

int checkcmds();
int checkfile();
int checkperms();
int checksys();
int checktime();
int getargs();
int getgroup();
int getname();
uid_t get_uid();
int getuser();
void usage();


/*
 * c h e c k c m d s
 *
 * Check which commands are permitted
 *
 * Return:	OK	Permitted
 *		FAILED	Not permitted
 *		cmdpath is updated as a side effect
 */
int checkcmds(uuxsite)
char *uuxsite;
{
  char *fields[MAXARGS], cmdpath[132], line[BUFSIZ];
  int j, flg, maxargs;
  FILE *fpcmd;

  fprintf(stdout, "\n   They can execute the commands: ");

#if 0
  /* we can do anything */
  if (strcmp(locsite, uuxsite) == SAME) return(OK);
#endif
  /* but other people are more limited */
  if ((fpcmd = fopen(LCMDS, "r")) == (FILE *)NULL) {
	fprintf(stdout, "   *** cannot open %s, no valid commands\n", LCMDS);
	return(FAILED);
  }

  flg = FAILED;
  cmdpath[0] = '\0';
  while (fgets(line, BUFSIZ, fpcmd) != (char *)NULL) {
	if (line[0] == '#' || line[0] == '\n') continue;
	/* lose any newline */
	j = strlen(line) - 1;
	if (line[j] == '\n') line[j] = '\0';
	/* we've got the PATH string */
	if (strncmp(line, "PATH", 4) == SAME) {
		strncpy(cmdpath, line, 131);
		continue;
	}
	/* make sure of whitespace in L-cmds entry */
	for (j = 0 ; j < strlen(line) ; j++)
		if (line[j] == ',') line[j] = ' ';
	flg = FAILED;
	if ((maxargs = getargs(line, fields)) > 0) {
		if (maxargs == 1)	/* default to all sites */
			flg = OK;
		else {			/* check for this site */
			for (j = 1 ; j < maxargs ; j++) {
				if (strcmp(fields[j], uuxsite) == SAME)
					flg = OK; 
			}
		}
		if (flg == OK) fprintf(stdout, " %s", fields[0]);
	}
  }
  (void) fclose(fpcmd);
  if (cmdpath[0] == '\0') strcpy(cmdpath, UUXPATH);
  fprintf(stdout, "\n   They will use the search path: %s\n", cmdpath);

  return(flg);
}


/*
 * c h e c k f i l e
 *
 * Check if a filename exists, and say what it is
 *
 * Return:	OK	Name exists
 *		FAILED	Otherwise
 */
int checkfile(path, nd, dr)
char *path; int nd, dr;
{
  char *type, *err, *ess, *pad, owner[15], group[15], perms[15], date[40]; 
  int j, k, fd, status;
  struct stat statbuf;

  /* try to access the file */
  status = OK;
  if (access(path, 0) == -1)
	status = FAILED;
  else if ((fd = open(path, O_RDONLY, 0600)) == -1)
	status = FAILED;
  else if (fstat(fd, &statbuf) != 0 || statbuf.st_size == (off_t)0) {
	(void) close(fd);
	status = FAILED;
  }

  if (status == FAILED) {
	type = dr ? "Dir " : "File";
	err = "*** not found";
	ess = nd ? "(essential)" : " ";
	fprintf(stdout, "%s %s:\t%s %s\n", type, path, err, ess);
	return(FAILED);
  }

  /* analyse the results */
  err = " ";
  ess = " ";
  if ((statbuf.st_mode & S_IFMT) == S_IFDIR) {
	type = "Dir ";
	if (!dr) err = "\t*** not directory";
  }
  else if ((statbuf.st_mode & S_IFMT) == S_IFREG) {
	type = "File";
	if (dr) err = "\t*** not file";
  }
  else {
	type = "Unknown";
	err = "\t***";
  }
  if (strcmp(err, " ") != SAME && nd)
	ess = "(essential)";
  if (getuser(statbuf.st_uid, owner) == FAILED)
	strcpy(owner, "***");
  if (getgroup((gid_t)statbuf.st_gid, group) == FAILED)
	strcpy(group, "***");
  k = statbuf.st_mode & 0777;
  for (j = 8 ; j >= 0 ; j--) {
	perms[j] = (k & 1) ? "rwxrwxrwx"[j] : '-';
	k = k >> 1;
  }
  if ((statbuf.st_mode & S_ISUID) == S_ISUID)
	perms[2] = 's';
  if ((statbuf.st_mode & S_ISGID) == S_ISGID)
	perms[5] = 's';
  perms[9] = '\0';
  sprintf(date, "%s", ctime(&statbuf.st_mtime));
  for (j = 0 ; j < 20 ; j++)
	date[j] = date[j + 4];		/* lose the day field */
  date[j] = '\0';			/* clear the newline */
  (void) close(fd);

  /* crude formatting */
  if (strlen(path) < 18)
	pad = "\t\t";
  else
	pad = "\t";

  /* write out the results */
  fprintf(stdout, "%s %s:%s%s.%s %s %s%s %s\n",
	type, path, pad, owner, group, perms, date, err, ess);

  return(OK);
}


/*
 * c h e c k p e r m s
 *
 * Check the user file
 *
 * Return:	OK		Valid entry found
 *		FAILED  	No valid entry found
 */
int checkperms()
{
  char *cp, *bp, *fields[MAXARGS], tsite[20], tuser[20], tflag[20], buff[256];
  int j, level, flg, maxargs;
  FILE *fp;

  if ((fp = fopen(USERFILE, "r")) == (FILE *)NULL) {
	fprintf(stdout, "No %s found: unlimited access permitted\n", USERFILE);
	return(OK);
  }

  /* The USERFILE format is: user_name,site_name callback_flag pathnames
   * The callback flag, 'c', is optional.
   */
  level = 0;
  while (fgets(buff, 255, fp) != (char *)NULL) {
	tsite[0] = tuser[0] = tflag[0] = '\0';
	/* the trailing newline only confuses things */
	j = strlen(buff) - 1;
	if (buff[j] == '\n') buff[j] = '\0';
	if (buff[0] == '#' || buff[0] == '\0') continue;
	fprintf(stdout, "USERFILE entry: %s\n", buff);

	/* lose leading whitespace */
	cp = buff;
	while (isspace(*cp))
		cp++;

	/* we have a ',' or a username or junk */  
	if (*cp == ',')
		cp++;
	else if (isalpha(*cp)) {
		bp = tuser;
		while (isalpha(*cp) && (bp - tuser) < 20)
			*bp++ = *cp++;			
		*bp = '\0';
		while (isspace(*cp))
			cp++;
		if (*cp++ != ',') {
			fprintf(stdout, "   *** Invalid entry\n");
			continue;
		}
	}
	else {
		fprintf(stdout, "   *** Invalid entry\n");
		continue;
	}

	/* we have a sitename, a callback flag, a path, or junk */
	while (isspace(*cp))
		cp++;
	bp = tsite;
	while (isalpha(*cp) && (bp - tsite) < 20)
		*bp++ = *cp++;			
	*bp = '\0';
	while (isspace(*cp))
		cp++;
	bp = tflag;
	while (*cp != '/' && isalpha(*cp) && (bp - tflag) < 20)
		*bp++ = *cp++;			
	*bp = '\0';
	while (isspace(*cp))
		cp++;

	/* we now have a path, or junk */
	if (*cp != '/') {
		fprintf(stdout, "   *** Invalid entry\n");
		continue;
	}

	/* sort out site and callback entries */
	flg = FALSE;
	if (strcmp(tsite, "c") == SAME) {
		flg = TRUE;
		strcpy(tuser, "");
	}
	if (strcmp(tflag, "c") == SAME)
		flg = TRUE;

	/* The rest of the string, if it exists, is pathnames */
	if (*cp) {
		for (j = 0 ; j < 20 ; j++)
			fields[j] = (char *)NULL;
		maxargs = getargs(cp, fields);
	}
	level = 1;

	/* say what we've found */
	if (tuser[0] == '\0') strcpy(tuser, "Anyone");
	if (tsite[0] == '\0') strcpy(tsite, "Anysite");
	if (locsite[0] != '\0' && strcmp(tsite, locsite) == SAME)
		strcpy(tsite, "This Site");
	fprintf(stdout, "   When %s logs in from %s:\n", tuser, tsite);
	if (flg) fprintf(stdout, "   We will call them back.\n");
	fprintf(stdout, "   They can access the directories:");
	for (j = 0 ; j < maxargs ; j++)
		fprintf(stdout, " %s", fields[j]);

	/* check the L-cmds file */
	(void) checkcmds(tsite);
  }
  (void) fclose(fp);

  if (level == 0) {
	fprintf(stdout, "*** No entries in %s\n", USERFILE);
	return(FAILED);
  }
  else
	return(OK);
}


/*
 * c h e c k s y s
 *
 * Check the L.sys file for consistency.
 *
 * Return:	OK	Valid L.sys
 *		FAILED	Otherwise
 */
int checksys()
{
  char *cp, *cmdflds[MAXARGS], site[20], protocols[20], device[20], baud[20];
  char tmp[30], cctime[64], tempbuff[256], buff[BUFSIZ];
  int j, k, speed[10], pflg, sflg, tflg;
  FILE *fplsys;

  speed[0] = 300; speed[1] = 1200; speed[2] = 2400; speed[3] = 4800;
  speed[4] = 9600; speed[5] = 19200; speed[6] = 0;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) {
	fprintf(stdout, "   *** cannot open %s (essential)\n", LSYS);
	return(FAILED);
  }
	
  while (fgets(buff, BUFSIZ, fplsys) != (char *)NULL) {
	while ((j = strlen(buff)) < (BUFSIZ - 132) &&
			buff[j - 2] == '\\' &&
			fgets(tempbuff, 131, fplsys) != (char *)NULL) {
		buff[j - 2] = '\0';
		strcat(buff, tempbuff);
	}
	if (buff[0] == '#' || buff[0] == '\n') continue; /* ignore comments */
	j = strlen(buff);
	strncpy(tempbuff, buff, 255);		/* getargs roaches the string */
	if (getargs(tempbuff, cmdflds) < 6) continue;
	strncpy(site, cmdflds[F_SYSTEM], 19);
	strncpy(cctime, cmdflds[F_CCTIME], 63);
	strncpy(tmp, cmdflds[F_CLASS], 29);
	/* parse device field for any default protocols */
	pflg = TRUE;
	if ((cp = strchr(tmp, ',')) != (char *)NULL) {
		strncpy(protocols, cp + 1, 19);
		*cp = '\0';
		for (j = 0 ; j < strlen(protocols) ; j++) {
			if (strchr(PROTOLST, protocols[j]) == (char *)NULL)
				pflg = FALSE; 
		}
	}
	else
		strncpy(protocols, "none", 19);
	strncpy(device, tmp, 19);
	strncpy(baud, cmdflds[F_SPEED], 19);

	/* say what we've found */
	tflg = (checktime(cctime) == OK);
	if (strcmp(baud, "Any") == SAME)
		sflg = TRUE;
	else {
		sflg = FALSE;
		k = atoi(baud);
		for (j = 0 ; speed[j] != 0 ; j++) {
			if (k == speed[j]) {
				sflg = TRUE;
				break;
			}
		} 
	}
	fprintf(stdout, "L.sys entry: %s", buff);
	fprintf(stdout, "   Call time:     %s is %s\n", cctime,
				tflg ? "valid" : "*** invalid");
	fprintf(stdout, "   Protocol list: %s is %s\n", protocols,
				pflg ? "valid" : "*** invalid");
	fprintf(stdout, "   Speed:         %s is %s\n", baud,
				sflg ? "valid" : "*** invalid");
  }
  (void) fclose(fplsys);

  return(OK);
}


/*
 * c h e c k t i m e
 *
 * Check an L.sys time field for consistency
 *
 * Return:	OK	Call permitted
 *		FAILED	Call not permitted
 */
int checktime(atime)
char *atime;
{
  char *pd, *days = "WkSuMoTuWeThFrSa";
  int j, k, flg;

  /* A typical field is day[time][,retry][|...] with no intervening spaces */
  /* between day and time.  In this routine it is null-terminated. */

  /* The day subfield is specified using Any, Never, Wk and */
  /* Su Mo Tu We Th Fr Sa.  The time subfield is two 24hr times */
  /* separated by a dash.  The day subfield must be specified if */
  /* the time subfield is specified.  The retry field is decimal */
  /* minutes.  The '|' separator permits several day-time pairs. */

  /* typical ctime() output is "Wed Jul 08 18:43:07 1987\n\0" */

  pd = atime;		/* points to a null-terminated L.sys time field */
  while (*pd) {		/* cycle over multiple entries */
	/* check the day of the week */
	if (strncmp("Never", pd, 5) == SAME) {
		return(OK);
	}
  	else if (strncmp("Any", pd, 3) == SAME) {
		pd += 3;
	}
	else {
		/* check for valid weekdays */
		while (isalpha(*pd)) {
			flg = FAILED;
			for (j = 0 ; j < 15 ; j += 2) {
				if (strncmp(days + j, pd, 2) == SAME) {
					flg = OK;
					break;
				}
			}
			if (flg == FAILED) {
				if (dbglvl > 0) {
					fprintf(stderr,
					"Entry %s: Bad day at Black Rock\n",
					atime);
					(void) fflush(stderr);
				}
				return(FAILED);
			}
			pd += 2;		/* step over this entry */
		}
	}

	/* we may now have a time sequence or a retry interval or nothing */
  	if (*pd && *pd != ',' && *pd != '|') {
		/* sort out actual and default arguments */
		j = sscanf(pd, "%04d-%04d,%d", &k, &k, &k);
		if (j < 2) {
			if (dbglvl > 0) {
				fprintf(stderr,
					"Entry %s: Bad time sequence\n", atime);
				(void) fflush(stderr);
			}
			return(FAILED);		/* garbage in the string */
		}
	}
	/* lose any retry interval present */
	while (*pd && *pd != '|') pd++;

	/* repeat for any subfield */
	if (*pd == '|') pd++;
  }

  return(OK);
}


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t g r o u p
 *
 * Get the group name corresponding to the given gid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getgroup(gid, buffer)
gid_t gid; char *buffer;
{
  struct group *grp;

  if ((grp = getgrgid(gid)) != (struct group *)NULL) {
	strncpy(buffer, grp->gr_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename.
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the given uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char *cp;
  int j;
  uid_t tuid;

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* current program name */
  if (getname(locsite) == FAILED)	/* get local site name */
	fprintf(stderr, "uucheck: cannot find local uucpname\n");

  dbglvl = 0;

  /* parse arguments */
  opterr = 0;
  while ((j = getopt(argc, argv, "vx:")) != EOF) {
	switch (j & 0177) {
	case 'v':			/* check individual files */
		opt_v = TRUE;
		break;
	case 'x':
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case '?':
 		/* fall through */
	default:
		usage();
		exit(1);
		break;
	}
  }

  /* validity checks */
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid) {
	fprintf(stderr, "uucheck: option available only to administrators.\n");
	exit(1);
  }
#endif

  if (opt_v) {
	fprintf(stdout, "   ***   Checking UUCP files and directories   ***\n\n");
	for (j = 0 ; files[j].name != (char *)NULL ; j++)
		(void) checkfile(files[j].name, files[j].need, files[j].dir);
	fprintf(stdout, "\n   ***   UUCP file and directory check complete   ***\n");
  }
  fprintf(stdout, "\n   ***   Checking %s   ***\n\n", USERFILE); 
  (void) checkperms();
  fprintf(stdout, "\n   ***   %s check complete   ***\n", USERFILE);
  fprintf(stdout, "\n   ***   Checking %s   ***\n\n", LSYS); 
  (void) checksys();
  fprintf(stdout, "\n   ***   %s check complete   ***\n", LSYS);

  exit(0);
  /* NOTREACHED */
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr, "Usage: %s [-v] [-x #]", progname);
}

