/* "Direct" dialler for Minix UUCP.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose.
 */

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <unistd.h>
#include "dial.h"
#include "uucp.h"

#define FALSE	0
#define TRUE	~FALSE
#define OK	0
#define FAILED	-1
#define SAME	0

#ifndef lint
static char *Version = "@(#) tty 1.2 (31 October 1991)";
#endif

static char linelock[132];
static int gotlock;
static int ttyfd;
static struct TTY initty;
static struct Port {
  int baud;			/* port baudrate */
  int speed;			/* speed setting byte */
} port[] = {
  300, B300,
  600, B600,
  1200, B1200,
  2400, B2400,
  4800, B4800,
  9600, B9600,
  19200, B19200,
  0, 0,
};

/* Implemented in calling program */

extern int checklock();
extern int unlock();

/* Forward declarations */

extern int ttyopen();
extern int ttyerror();
extern int ttyclose();
static int _wrapup();


/*
 * t t y o p e n
 *
 * Open a direct line
 *
 * Return:	fd	Success
 *		-1	Otherwise
 */
#ifdef _BCC			/* BCC can't pass structures */
int ttyopen(callp)
CALL *callp;
{
  CALL call = *callp;
#else
int ttyopen(call)
CALL call;
{
#endif
  char devport[30];
  int j, speed, s_tabs, s_parity, s_bits;
  struct TTY tty;

  /* get the full device and lockfile names */
  if (strlen(call.line) > 25) return(DV_NT_K); 
  sprintf(devport, "%s/%s", DEVDIR, call.line);
  strcpy(linelock, LCKPATH);
  strncat(linelock, call.line, 131 - strlen(linelock));

  /* lock the device */
  if (checklock(linelock, 2) == FAILED) {
	gotlock = FALSE;
	return(DV_NT_A);
  }
  gotlock = TRUE;

  /* open the line */
  if ((ttyfd = open(devport, O_RDWR)) == -1) {
	(void) unlock(linelock);
	gotlock = FALSE;
	return(L_PROB);
  }

  /* park the default settings */
  if (ioctl(ttyfd, TTYGET, &initty) == -1) {
	(void) unlock(linelock);
	gotlock = FALSE;
	return(L_PROB);
  }

  if (call.attr != (struct TTY *)NULL)	/* set the line to the given state */
	tty = *(call.attr);
  else {
	/* set the given speed */
	speed = j = 0;
	while (port[j].baud != 0) {
		if (call.baud == port[j].baud) {
			speed = port[j].speed;
			break;
		}
		j++;
	}
	if (speed == 0) {
		(void) unlock(linelock);
		gotlock = FALSE;
		return(ILL_BD);
	}
	tty = initty;		/* use defaults */
#ifdef _MINIX
	tty.sg_ispeed = tty.sg_ospeed = speed;
  }

  /* set RAW, NOECHO regardless */
  s_tabs = tty.sg_flags & XTABS;
  s_parity = tty.sg_flags & (EVENP | ODDP);
  s_bits = tty.sg_flags & BITS8;		/* BITS8 is a mask */
  tty.sg_flags = RAW | s_parity | s_bits | s_tabs;
#endif
#ifdef _SVR2
	tty.c_cflag = (tty.c_cflag & ~(CBAUD)) | speed;
  }

  /* set RAW, NOECHO regardless */
  tty.c_iflag &= ~(INLCR | ICRNL | IUCLC | ISTRIP | IXON | BRKINT);
  tty.c_oflag &= ~OPOST;
  tty.c_lflag &= ~(ICANON | ISIG | ECHO);
  tty.c_cc[4] = 5;
  tty.c_cc[5] = 2;
#endif

  if (ioctl(ttyfd, TTYSET, &tty) == -1) {
	(void) _wrapup();
	return(L_PROB);
  }

  return(ttyfd);
}


/*
 * t t y e r r o r
 *
 * Direct line error reporting
 *
 * Return:	OK		Always
 */
int ttyerror(err, str)
int err; char *str;
{
  switch (err) {
	case L_PROB:
	  	strncpy(str, "Line open/ioctl failure", 131);
		break;
	case ILL_BD:
		strncpy(str, "Illegal baud rate", 131);
		break;
	case DV_NT_A:
		strncpy(str, "Device not available", 131);
		break;
	case DV_NT_K:
		strncpy(str, "Device not known", 131);
		break;
	default:
  		sprintf(str, "Error %d", -err);
		break;
  }
  return(OK);
}


/*
 * t t y c l o s e
 *
 * Close a direct line
 *
 * Return:	OK	Always
 */
int ttyclose(fd)
int fd;
{
  /* return the line to its original state */
  (void) ioctl(ttyfd, TTYSET, &initty);
  (void) close(ttyfd);
  ttyfd = -1;
  (void) unlock(linelock);
  gotlock = FALSE;

  return(0);
}


/*
 * w r a p u p
 *
 * Tidy up on error exit
 *
 * Return:	OK	Always
 */
static int _wrapup()
{ 
  (void) close(ttyfd);
  ttyfd = -1;
  (void) unlock(linelock);
  gotlock = FALSE;
  return(OK);
}
