/* 't' protocol driver for Minix UUCP.
 *
 * Version 1.2 of 31 October 1991.
 *
 * The 't' protocol sends messages in a block containing an exact number of
 * (512 byte) packets, with both the message and the last packet null-
 * terminated.  It isn't clear why both these conditions are needed, since
 * the message itself is plain ASCII.  Is TPACKSIZE really identical in
 * all versions of this protocol?
 *
 * The 't' protocol sends files in a series of blocks, with each block
 * preceded by a count.  This count is in the form of a long integer, with
 * a zero count denoting EOF.  Note that under PC Minix, the size of a long
 * is hard-wired into the code as 4 bytes in htonl() and ntohl(); this will
 * need changing for the 68000 versions.
 *
 * The six main routines return OK (1) if successful, and FAILED (-1)
 * otherwise.
 *
 * Based on:
 *		 tio.c 4.6 dated 1/24/86 by Rick Adams.
 * Rewritten for PC Minix 1.5:
 *		C W Rose, San Diego, September 1991.
 */

#include <sys/types.h>
#include <sys/times.h>
#ifdef _MINIX
#include <minix/config.h>
#include <minix/const.h>
#undef TRUE
#endif
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <time.h>
#include <unistd.h>
#include "dial.h"
#include "uucp.h"
#include "uucico.h"


#ifdef VERBOSE				/* remove these calls for speed */
#define print2msg printmsg
#define print3msg printmsg
#define print4msg printmsg
#define print5msg printmsg
#define print6msg printmsg
#define print7msg printmsg
#else
#define print2msg(a, b)
#define print3msg(a, b, c)
#define print4msg(a, b, c, d)
#define print5msg(a, b, c, d, e)
#define print6msg(a, b, c, d, e, f)
#define print7msg(a, b, c, d, e, f, g)
#endif

#define TPACKSIZE	512
#define TBUFFSIZE	1024

#ifndef lint
static char *Version = "@(#) tio 1.2 (31 October 1991)";
#endif

struct tbuff {
	char t_bytes[4];	/* ### this assumes a long is four bytes */
	char t_data[TBUFFSIZE];
};

extern int sread();		/* read from the serial line */
extern int swrite();		/* write to the serial line */
extern int syslog();		/* print file transfer statistics */
extern char *visib();		/* print buffer in ASCII for debugging */

/* Global variables */

static int fd_in;
static int fd_out;

/* Forward declarations */

extern int ton();
extern int trdmsg();
extern int twrmsg();
extern int trddata();
extern int twrdata();
extern int toff();
static void htonl();
static void ntohl();


/*
 * t o n
 *
 * Turn on the protocol
 *
 * Return:	OK	Always
 */
int ton(in, out)
int in, out;
{
  /* initialise some local variables */

  fd_in = in;
  fd_out = out;

  return(OK);
}


/*
 * t r d m s g
 *
 * Read a message
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int trdmsg(msg, in)
char *msg; int in;
{
  char c, tmp[10];
  size_t sz;

  sz = 0;
  while (TRUE) {
	if (sz == MSGLEN) return(FAILED);
	if (sread(tmp, 1, WAIT) < 1) return(FAILED);
	c = tmp[0] & 0x7f;
	if (c == '\r' || c == '\n') c = '\0';
	msg[sz++] = c;
	if (c == '\0') {
		/* messages are an exact number of packets in length */
		while ((sz++ % TPACKSIZE) != 0) {
			if (sread(tmp, 1, WAIT)  < 1) return(FAILED);
			if (sz < MSGLEN) msg[sz] = '\0';
		}
		sz--;
		break;
	}
  }
  print4msg(M_DATA, "trdmsg: length %u buffer start |%s|", sz, visib(msg, 16));

  return(OK);
}


/*
 * t w r m s g
 *
 * Write a message
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int twrmsg(msg, out)
char *msg; int out;
{
  char buff[TBUFFSIZE];
  int j;
  size_t sz;

  /* load the buffer and clear any trailing newline */
  strncpy(buff, msg, TBUFFSIZE);
  sz = strlen(buff);
  if (sz != 0 && buff[sz - 1] == '\n') sz--;
  buff[sz] = '\0';

  /* null the unused bytes at the end of the last packet */
  j = sz + 1 + TPACKSIZE - ((sz + 1) % TPACKSIZE);
  if (j > TBUFFSIZE) return(FAILED);
  while (sz < j)
	buff[sz++] = '\0';

  /* write the message */
  print4msg(M_DATA, "twrmsg: length %u buffer start |%s|", sz, visib(buff, 16));
  if (swrite(buff, sz) == FAILED)
	return(FAILED);
  else
	return(OK);
}


/*
 * t r d a t a
 *
 * Read data until EOF
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 */
int trddata(fpout, in)
FILE *fpout; int in;
{
  char num[10], buff[TBUFFSIZE];
  int j;
  long int count, bytes;
  size_t sz;
  time_t secs;
  clock_t start, end;
  struct tms ttms;

  bytes = 0L;
  start = times(&ttms);
  while (TRUE) {
	/* read the data block length, a four-byte number */
	if ((j = sread(num, 4, WAIT)) != 4) {
		print5msg(M_DATA, "trddata: count read failed, %d bytes, num |%s|",
					j, visib(num, 4));
		return(FAILED);
	}
	ntohl(&count, num);		/* convert if necessary */

	if (count == 0L)		/* EOF */
		break;
	if (count > (long int)TBUFFSIZE) {
		print4msg(M_DATA, "trdata: count too big, 0x%lx, num |%s|",
					count, visib(num, 4));
		return(FAILED);
	}
	bytes += count;
	sz = (size_t)count;

 	/* read the data block */
	if ((j = sread(buff, sz, WAIT)) != sz) {
		print3msg(M_DATA, "trddata: data read failed, %d bytes", j);
		return(FAILED);
	}

	/* write the data block */
	if (fwrite((void *)buff, sizeof(char), sz, fpout) != sz) {
		print2msg(M_DATA, "trddata: fwrite failure");
		return(FAILED);
	}
  }
  end = times(&ttms);

  /* write out the transfer statistics */
  end -= start;
  secs = end / HZ;
  end %= HZ;
  (void) syslog(bytes, secs, end);

  return(OK);
}


/*
 * t w r d a t a
 *
 * Write data until EOF
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 */
int twrdata(fpin, out)
FILE *fpin; int out;
{
  int j;
  long int count, bytes;
  time_t secs;
  clock_t start, end;
  struct tbuff buff;
  struct tms ttms;

  count = 1L;
  bytes = 0L;
  start = times(&ttms);
  while (count != 0L) {
	/* read a data block */
	if ((j = fread((void *)buff.t_data, sizeof(char), TBUFFSIZE, fpin))
					== 0 && feof(fpin) == 0) {
		print2msg(M_DATA, "twrdata: fread failure");
		return(FAILED);
	}
 	count = j;
	bytes += count;
	htonl(buff.t_bytes, &count);
	print3msg(M_DATA, "twrdata: t_bytes |%s|, count 0x%08lx",
					visib(buff.t_bytes, 4), count);
	j += 4;
	/* write a data block */
	if (swrite((char *)&buff, (size_t)j) == FAILED) {
		print2msg(M_DATA, "twrdata: write failure");
		return(FAILED);
	}
  }
  end = times(&ttms);

  /* write out transfer statistics */
  end -= start;
  secs = end / HZ;
  end %= HZ;
  (void) syslog(bytes, secs, end);

  return(OK);
}


/*
 * t o f f
 *
 * Turn off the protocol
 *
 * Return:	OK	Always
 *
 */
int toff(in, out)
int in, out;
{
  return(OK);
}


/*
 *  htonl is a function that converts 4 bytes from host to network order
 *  ntohl is a function that converts 4 bytes from network to host order
 *
 *  network order is 		0 1 2 3 (bytes in a long)
 *  host order on a vax is	3 2 1 0
 *  host order on a pdp11 is	1 0 3 2
 *  host order on a 68000 is	0 1 2 3
 *  most other machines are	0 1 2 3
 *
 * A 68010 has the order 3 2 1 0
 * An 8086 has the order 0 1 2 3
 *
 * The order on a 68010 is most_significant -> least_significant,
 * with increasing buffer index.  Presumably this is the 'standard'
 * network order shown above.  cwr.
 */
static void htonl(c, n)
char *c; long int *n;
{
  char *np = (char *)n;

  /* horrible, but 16-bit Minix won't shift longs */

#if (MACHINE == IBM_PC)
  *(c + 3) = *np; 
  *(c + 2) = *(np + 1); 
  *(c + 1) = *(np + 2); 
  *c = *(np + 3); 
#else
  *c = *np; 
  *(c + 1) = *(np + 1); 
  *(c + 2) = *(np + 2); 
  *(c + 3) = *(np + 3); 
#endif 

}


static void ntohl(n, c)
long int *n; char *c;
{
  char *np = (char *)n;

  /* horrible, but 16-bit Minix won't shift longs */

#if (MACHINE == IBM_PC)
  *np = *(c + 3); 
  *(np + 1) = *(c + 2); 
  *(np + 2) = *(c + 1); 
  *(np + 3) = *c; 
#else
  *np = *c; 
  *(np + 1) = *(c + 1); 
  *(np + 2) = *(c + 2); 
  *(np + 3) = *(c + 3); 
#endif

}
