/* PAD dialler for Minix UUCP.
 *
 * Version 1.2 of 31 October 1991.
 *
 * Based on pad.c 1.1 dated 85/01/08 by Chris Lewis
 */

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <unistd.h>
#include "dial.h"
#include "uucp.h"

#define FALSE	0
#define TRUE	~FALSE
#define OK	0
#define FAILED	-1
#define SAME	0

#define COMMWAIT	40

#ifndef lint
static char *Version = "@(#) pad 1.2 (31 October 1991)";
#endif

/* PAD initialisation parameters.  What do they actually set ? */
static char *padparms[] = {
  "set1:0,2:0,3:2,4:1,5:1,6:5,7:4,9:0,10:0,13:0",
  "set14:0,15:0,16:0,17:0,18:0,19:1,20:255",
  "set102:0,103:17,104:19,105:0,106:0,107:0,108:0",
  "set109:0,110:0,111:0,112:0,113:0,114:0,115:0,116:0",
  (char *)NULL,
};

static char linelock[132];
static int gotlock;
static int padfd;
static struct TTY initty;
static struct Port {
  int baud;			/* port baudrate */
  int speed;			/* speed setting byte */
} port[] = {
  300, B300,
  600, B600,
  1200, B1200,
  2400, B2400,
  4800, B4800,
  9600, B9600,
  19200, B19200,
  0, 0,
};

/* Implemented in calling program */

extern void ontime();		/* SIGALRM catcher */
extern int checklock();
extern int unlock();

/* Forward declarations */

extern int padopen();
extern int paderror();
extern int padclose();
static int _expectstr();
static int _getdev();
static int _sread();
static int _wrapup();


/*
 * p a d o p e n
 *
 * Establish connection through a PAD.
 *
 * Return:	fd	Success
 *		errno	Failure
 */
#ifdef _BCC			/* BCC can't pass structures */
int padopen(callp)
CALL *callp;
{
  CALL call = *callp;
#else
int padopen(call)
CALL call;
{
#endif
  char *cp, devport[30], dataport[30], buff[132];
  char **parmptr;
  int j, ok, speed;
  size_t sz;
  struct TTY tty;

  /* get and lock an appropriate device line */
  if ((j = _getdev(&call, devport, dataport)) < 0)
	return(j);

  /* try and open the line */
  sprintf(buff, "%s/%s", DEVDIR, devport);
  if ((padfd = open(buff, O_RDWR)) == -1) {
	(void) unlock(linelock);
	gotlock = FALSE;
	return(L_PROB);
  }

  /* park the default settings */
  if (ioctl(padfd, TTYGET, &initty) == -1) {
	(void) unlock(linelock);
	gotlock = FALSE;
	return(L_PROB);
  }

  /* set the required attributes */
  if (call.attr != (struct TTY *)NULL)
	tty = *(call.attr);
  else {
	/* convert the given baud rate to a port speed setting byte */
	speed = j = 0;
	while (port[j].baud != 0) {
		if (call.baud == port[j].baud) {
			speed = port[j].speed;
			break;
		}
		j++;
	}
	if (speed == 0) {
		(void) _wrapup();
		return(ILL_BD);
	}
	tty = initty;
#ifdef _MINIX
	tty.sg_ospeed = tty.sg_ispeed = speed;
  }

  /* set RAW, NOECHO regardless */
  tty.sg_flags |= RAW;
  tty.sg_flags &= ~ECHO;
  tty.sg_erase = -1;
  tty.sg_kill = -1;
#endif
#ifdef _SVR2
	tty.c_cflag = (tty.c_cflag & ~(CBAUD)) | speed;
  }

  /* set RAW, NOECHO regardless */
  tty.c_iflag &= ~(INLCR | ICRNL | IUCLC | ISTRIP | IXON | BRKINT);
  tty.c_oflag &= ~OPOST;
  tty.c_lflag &= ~(ICANON | ISIG | ECHO);
  tty.c_cc[4] = 5;
  tty.c_cc[5] = 2;
#endif

  if (ioctl(padfd, TTYSET, &tty) == -1) {
	(void) _wrapup();
	return(L_PROB);
  }

  /* Synchronize with the PAD prompt */
  if (write(padfd, "\r", 1) != 1) {
	(void) padclose(padfd);
	return(D_HUNG);
  }
  if ((ok = _expectstr(">", 0)) == FALSE) {
	(void) padclose(padfd);
	return(NO_ANS);
  }

  /* Initialization of PAD */
  ok = FALSE;
  for (parmptr = padparms ; *parmptr != (char *)NULL && !ok ; parmptr++) {
	if (write(padfd, *parmptr, strlen(*parmptr)) != strlen(*parmptr) ||
			write(padfd, "\r", 1) != 1) {
		(void) padclose(padfd);
		return(D_HUNG);
	}
	ok = _expectstr(">", 0);
  }
  if (!ok) {
	(void) padclose(padfd);
	return(NO_ANS);
  }
#ifdef TEST
  fprintf(stderr, "PAD %s:\n", "configuration");
  if (write(padfd, "par?\r", 6) != 6) {
	(void) padclose(padfd);
	return(D_HUNG);
  }
  if ((ok = _expectstr(">", 0)) == FALSE) {
	(void) padclose(padfd);
	return(NO_ANS);
  }
#endif

  /* if no number, task complete */
  if (call.telno == (char *)NULL)	/* ### "com" response ? */
	return(padfd);

  /* otherwise, call the number */
  cp = call.telno;
  sz = strlen(cp);
  if (write(padfd, "c ", 2) != 2 || write(padfd, cp, sz) != sz ||
		write(padfd, "\r", 1) != 1) {
	(void) padclose(padfd);
	return(D_HUNG);
  }
  if ((ok = _expectstr("com", COMMWAIT)) == FALSE) {
	(void) padclose(padfd);
	return(NO_ANS);
  } 

  return(padfd);
}


/*
 * p a d e r r o r
 *
 * PAD error reporting
 *
 * Return:	OK	Always
 */
int paderror(err, str)
int err; char *str;
{
  switch (err) {
	case L_PROB:
	  	strncpy(str, "Line open/ioctl failure", 131);
		break;
	case ILL_BD:
		strncpy(str, "Illegal baud rate", 131);
		break;
	case DV_NT_A:
		strncpy(str, "Device not available", 131);
		break;
	case DV_NT_K:
		strncpy(str, "Device not known", 131);
		break;
	default:
  		sprintf(str, "Error %d", -err);
		break;
  }
  return(OK);
}


/*
 * p a d c l o s e
 *
 * Close the PAD
 *
 * Return:	0	Success
 *		-1	Otherwise
 */
int padclose(fd)
int fd;
{
  if (padfd > 2) {
	(void) ioctl(padfd, TTYSET, &initty);
	(void) _wrapup();
	return(0);
  }
  else
	return(-1);
}


/*
 * e x p e c t s t r
 *
 * Wait for a specific string from the remote
 *
 * Return:	TRUE	On match with expected string
 *		FALSE	On read failure (timeout)
 */
static int _expectstr(str, timeout)
char *str; unsigned int timeout;
{
  char c, *sm[30], buff[10];
  int j, k, max, flg;

  if (strcmp(str, "\"\"") == SAME) return(TRUE);

  /* initialise state machines */
  for (j = 0 ; j < 30 ; j++)
	sm[j] = (char *)NULL;

  max = 0;
  flg = FALSE;
  while (!flg) {
	/* for some reason &c doesn't work with sread */ 
	if ((j = _sread(buff, 1, timeout)) < 1)
		return(FALSE);
	/* strip high bit, and remove unprintable characters */
	c = buff[0] & 0177;
	if (c < '\040')
		c = ' ';

	/* if necessary and possible, start up a state machine */
	if (c == *str && max < 29)	/* last array element must be NULL */
		sm[max++] = str;

	/* look at all the current SMs */
	for (j = 0 ; j < max ; j++) {
		if (*sm[j] == c) {
			/* if there's a match, update the SM */
			if (*++(sm[j]) == '\0') {
				flg = TRUE;
				break;
			}
		}
		else {
			/* if there's no match, remove this SM */
			for (k = j ; k < max ; k++)
				sm[k] = sm[k+1];
			max--;
			j--;
		}
	}
  }
  return(flg);
}

 
/*
 * g e t d e v
 *
 * Get and lock a device satisfying current requirements
 *
 * Return:	OK	Success
 *		errno	Otherwise
 */
static int _getdev(callp, dataport, dialport)
CALL *callp; char *dataport, *dialport;
{
  char *cp, *flds[10], buff[132];	/* L-devices line buffer */
  int j, flg, exists, tried;
  FILE *fp;				/* L-devices file pointer */

  /* L-devices file has (at least) five fields per line:
   *      0        1          2        3       4
   *	Class, Dataport, Dialport, Speed, Dialler
   */

  if ((fp = fopen(LDEVS, "r")) == (FILE *)NULL) return(NO_Ldv);

  exists = tried = FALSE;
  while (fgets(buff, sizeof(buff), fp) != (char *)NULL) {
	if (buff[0] == '#' || buff[0] == '\n') continue;
	/* cheaper than sscanf */
	j = 0;
	flg = TRUE;
	cp = buff;
	while (*cp && j < 10) {
		if (isspace(*cp)) {
			if (!flg) *cp = '\0';
			flg = TRUE;
		}
		else {
			if (flg) flds[j++] = cp;
			flg = FALSE;
		}
		cp++;
	}
	if (j < 5 || j == 10) continue;		/* duff line */
	if (*(--cp) == '\n') *cp = '\0';	/* lose the newline */
			
	if (strcmp(flds[F_DTYPE], "PAD") != 0)	/* not a PAD */
		continue;
	if (callp->line != (char *)NULL) {	/* device specified? */
		if (strcmp(flds[F_DATAPORT], callp->line) != SAME)
			continue;		/* wrong device */
	}
	exists = TRUE;

	if (strcmp(flds[F_SPEED], "Any") != SAME &&
			atoi(flds[F_SPEED]) != callp->baud)
		continue;			/* wrong speed */
	tried = TRUE;

	if (checklock(flds[F_DATAPORT], 2) == FAILED)
		continue;			/* couldn't lock line */
	sprintf(linelock, "%s/%s", LCKPATH, flds[F_DATAPORT]);
	gotlock = TRUE;

	strncpy(dataport, flds[F_DATAPORT], 15);
	strncpy(dialport, flds[F_DIALPORT], 15);
	(void) fclose(fp);
	return(OK);
  }
  (void) fclose(fp);

  if (tried)
	return(DV_NT_A);
  else if (exists)
	return(ILL_BD);
  else
	return(DV_NT_K);
}


/*
 * s r e a d
 *
 * Read from the input fd, allowing for timeout
 *
 * Return:	FAILED			On read() failure
 *		No. of bytes read	Otherwise
 */
static int _sread(data, num, timeout)
char *data; unsigned int num, timeout;
{
  int status;
  unsigned int setting;
  sig_t oldsig;

  if ((oldsig = signal(SIGALRM, ontime)) == BADSIG)
	return(FAILED);
  if (timeout != 0)
	setting = alarm(timeout);
  else
	setting = alarm(5);

  /* read returns the number of bytes read, 0 on EOF, or -1 on error */
  /* error returns immediately, without timing out - does EOF ? */

  status = read(padfd, data, num);
  (void) alarm(setting);
  (void) signal(SIGALRM, oldsig);

  return(status);
}


/*
 * w r a p u p
 *
 * Tidy up on error exit
 *
 * Return:	OK	Always
 */
static int _wrapup()
{ 
  (void) close(padfd);
  padfd = -1;
  (void) unlock(linelock);
  gotlock = FALSE;
  return(OK);
}

