/*$$ file: newgpkt.c
 * Author: Peter S. Housel
 *
 * The |cksum()| routine is taken from UUPC, Copyright 1985, 1986, 1987 by
 * Richard H. Lamb, with (possible) changes Copyright 1987 by Stuart Lynne
 *
 * All other code is Copyright 1989 by Peter S. Housel.
 * Redistribution for any purpose is permitted provided this message
 * is included intact. No warranty of any sort is provided.
 *
 * newgpkt version 1.6 31/10/91
 *
 * Added rejected packet number patch from Ralf Wenk 1 June 91.
 * Added long command patch suggested by Martin Grundy 25 June 91.
 * Added second patch from Ralph Wenk 26 June 91.
 * Changed typing of buffers and buffer lengths 18 July 91.
 * Stopped using short ints 30 July 91.
 * Standardised calling interface 30 Sep 91.
 * Cleaned up signed/unsigned typing 31 Oct 91.
 */

/* This program was written based on the original UUPC 'g' driver,
 * John Gilmore's version of uuslave, and Greg Chesson's protocol
 * description article. The last was especially helpful.
 *
 * This code was written around a severely hacked version of UUPC.
 * The call interface (was) almost identical to the original, but
 * the internal implementation is quite different. Also, many
 * functions are called that were not available in the original
 * UUPC support functions. It should serve as an adequate framework.
 *
 * The framing strategy requires that a |read()| be interruptable
 * by a |SIGALRM|. No "busy wait" or nonblocking read is required.
 *
 * Note that strncpy() needs to follow the spec in that it should
 * zero-pad the destination string.
 *
 * The six main protocol routines return 1 (OK) on success, and FAILED
 * (-1) otherwise.  They call the routine syslog(long int bytes,
 * time_t secs, clock_t ticks) to update transfer statistics records
 * as required.
 */

#include <sys/types.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <time.h>
#include <unistd.h>
#include "dial.h"
#include "uucp.h"
#include "uucico.h"


#ifdef VERBOSE				/* remove these calls for speed */
#define print2msg printmsg
#define print3msg printmsg
#define print4msg printmsg
#define print5msg printmsg
#define print6msg printmsg
#define print7msg printmsg
#else
#define print2msg(a, b)
#define print3msg(a, b, c)
#define print4msg(a, b, c, d)
#define print5msg(a, b, c, d, e)
#define print6msg(a, b, c, d, e, f)
#define print7msg(a, b, c, d, e, f, g)
#endif

#ifndef lint
static char *Version = "@(#) newgpkt 1.6 (31 October 1991)";
#endif

extern int sread();		/* read from the serial line */
extern int swrite();		/* write to the serial line */
extern int syslog();		/* print file transfer statistics */
extern char *visib();		/* print buffer in ASCII for debugging */

/* Forward declarations. */

int gopenpk();			/* declared as external in uucico.h */
int gclosepk();			/* declared as external in uucico.h */
int grdmsg();			/* declared as external in uucico.h */
int gwrmsg();			/* declared as external in uucico.h */
int grddata();			/* declared as external in uucico.h */
int gwrdata();			/* declared as external in uucico.h */
static int ggetpkt();
static int gsendpkt();
static int gmachine();
static int ggpack();
static int gspack();
static unsigned int cksum();
static int ilog2();

#define DLE		'\020'	/* ASCII 'DLE' character */
#define MAXPKT		256	/* incredibly conservative... actually 4096 */
#define SWINDOW		3	/* initial send window size */
#define RWINDOW		1	/* window size we want to receive */
#define SPKTSIZE	64	/* initial send packet size */
#define RPKTSIZE	64	/* packet size we want to receive */

#define MAXLOST		5	/* max lost packets (closes or such) */
#define TIMEOUT		10	/* max seconds of silence before timeout */

#define LOSTPKT		-2	/* packet lost, got timeout */
#define BADPKT		-3	/* bad checksum, or data read timed out */

#define ENV_DLE		0	/* framing char at start of envelope */
#define ENV_K		1	/* packet length specifier */
#define ENV_C0		2	/* low-order checksum */
#define ENV_C1		3	/* high-order checksum */
#define ENV_C		4	/* control byte */
#define ENV_X		5	/* xor check byte */
#define ENV_LEN		6	/* overall envelope length */

#define TT_CTRL		0	/* control packet */
#define TT_DATA		2	/* data packet */
#define TT_SDATA	3	/* short data packet */
#define TT_ALTCHAN	1	/* 'alternate channel' - invalid */

#define X_CLOSE		1	/* close down protocol */
#define X_RJ		2	/* reject received packet */
#define X_SRJ		3	/* selectively reject packet - invalid */
#define X_RR		4	/* receiver ready */
#define X_INITC		5	/* third init packet */
#define X_INITB		6	/* second init packet */
#define X_INITA		7	/* first init packet */

#define OP_OPEN		1	/* request to open/init protocol */
#define OP_CLOSE	2	/* request to close protocol */
#define OP_WRITE	3	/* request to send packet */
#define OP_READ		4	/* request to read packet */

#define MAGIC (unsigned int) 0xAAAA	/* checksum magic value */

/* From Dale Schumacher - determine if a <= b < c, for mod 8 seq numbers. */
#define between(a,b,c)	(((a)<(c)) ? (((a)<=(b))&&((b)<(c))) \
				   : (((a)<=(b))||((b)<(c))))	/* check */

struct {
	int sdata;		/* 'is this a short data packet' flag */
	size_t length;		/* length of this packet */
	char *bufloc;		/* location of this data pkt's buffer */
} inpbufs[8], outbufs[8];	/* input/output queues */

static int ackneeded;		/* do we need to acknowledge a rcv'd pkt? */
static int nackneeded;		/* do we need to reject a received pkt? */
static int recv;		/* seq. number of last correctly rcv'd pkt */
static int lastread;		/* seq. number of last pkt ret. to caller */
static int send;		/* first packet in output window */
static int next;		/* next pkt to send  send <= next < nstuff */
static int nstuff;		/* next loc. to stuff a pkt in output queue */
				/* (last pkt in output window) + 1 */
static int initpk;		/* current init sequence send packet */
static int skipping;		/* skipping out-of-seq packets after RJ */
static size_t spktsize;		/* send data size (requested by other end) */
static int swindow;		/* send window size (ditto) */
static size_t rpktsize;		/* receive data packet size */
static int rwindow;		/* receive window size */
static int nlost;		/* number of consecutive timeouts */
static int fd_in = -1;		/* input fd */
static int fd_out = -1;		/* output fd */
static int chanopen = 0;	/* 'channel open' flag */

static char rbuff[MAXPKT];			/* data receive buffer */
static char wbuff[MAXPKT];			/* data transmit buffer */
static unsigned char buf[ENV_LEN + RPKTSIZE];	/* packet framing buffer */
static unsigned char *low, *high;		/* framing buffer limits */
static char *ibufset, *obufset;			/* i/o packet buffer sets */


/* Gopenpk() opens the 'g' packet protocol on the serial line. It
 * initializes its state variables, allocates buffers, etc., then calls
 * gmachine() to do the actual protocol negotiation/initialization.
 *
 * Return:		OK	Success
 *			FAILED	gmachine(OP_OPEN) failure
 *				malloc() failure
 */
int gopenpk(in, out)
int in, out;
{
  char *p, *q;			/* pointers into buffer set */
  int j;			/* index */

  high = low = buf;		/* empty input buffer */
  ackneeded = nackneeded = 0;	/* don't need to accept or reject anything */
  initpk = X_INITA;		/* send INITA during initialization */
  recv = lastread = 0;		/* initial empty read queue, seq=0 */
  send = next = nstuff = 1;	/* first in output queue, seq=1 */
  skipping = nlost = 0;		/* nothing lost yet, not skipping */
  rpktsize = RPKTSIZE;		/* for symmetry - fixed at compile time */
  rwindow = RWINDOW;		/* ### can this be negotiated ? */
  fd_in = in;			/* protocol always needs access to */
  fd_out = out;			/* both these descriptors */
  
  if (gmachine(OP_OPEN) != 0) return(FAILED);	/* do the open */

  /* Allocate send and receive buffers. */
  if ((p = ibufset = (char *)malloc(8 * rpktsize)) == (char *)NULL)
	return(FAILED);
  if ((q = obufset = (char *)malloc(8 * spktsize)) == (char *)NULL)
	return(FAILED);

  for (j = 0 ; j < 8 ; j++) {
	inpbufs[j].bufloc = p;
	p += rpktsize;
	outbufs[j].bufloc = q;
	q += spktsize;
  }

  return(OK);
}


/* Gclosepk() closes down the packet protocol using the OP_CLOSE
 * operation of gmachine().
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 */
int gclosepk(in, out)
int in, out;
{
  int j;

  /* ### should there be a timer here, for aborts ? */
  j = gmachine(OP_CLOSE);
  fd_in = fd_out = -1;

  return(j == 0 ? OK : FAILED);
}


/*
 * g r d m s g
 *
 * 'g' protocol message reading routine
 *
 * Return:	OK	Success
 *		FAILED	ggetpkt() failure
 *
 * Grdmsg() gets a null-terminated message into "dest".  It may need one
 * or more packets, but all of them will be "long-data" packets containing
 * a full 64 bytes.  No allowance is made for messages longer than the
 * assigned buffer...
 */
int grdmsg(message, in)
char *message; int in;
{
  size_t len;

  while (TRUE) {
	if (ggetpkt(message, &len) < 0) return(FAILED);
	*(message + len) = '\0';		/* make sure it is terminated */
	if (strlen(message) != len)		/* we reached the terminator */
		return(OK);
	message += len;
  }
}


/*
 * g w r m s g
 *
 * 'g' protocol message sending routine
 *
 * Return:	OK		Success
 *		FAILED		gsendpkt() failure
 *
 * Gwrmsg() sends a null-terminated message.
 */
int gwrmsg(message, out)
char *message; int out;
{
  size_t len;

  len = strlen(message) + 1;		/* total length including '\0' */
  while (TRUE) {
	if (gsendpkt(message, len, TRUE) < 0) return(FAILED);
	if (len < rpktsize) return(OK);	/* end of string */
	message += rpktsize;
	len -= rpktsize;
  }
}


/*
 * g r d d a t a
 *
 * 'g' protocol data reading routine
 *
 * Return:	OK		Success (no more data)
 *		FAILED		fwrite()/ggetpkt() failure
 */
int grddata(fpout, in)
FILE *fpout; int in;
{
  long int tbytes;
  size_t len;
  time_t start, end, tsecs;

  print2msg(M_TRANSFER, "grddata: reading data");

  tbytes = 0L;
  (void) time(&start);
  while (TRUE) {
	if (ggetpkt(rbuff, &len) < 0) return(FAILED);
	if (len == 0) {
		(void) time(&end);
		tsecs = end - start;
		(void) syslog(tbytes, tsecs, (clock_t)0);
		return(OK);
	}
	if (fwrite((void *)rbuff, sizeof(char), len, fpout) != len) {
		return(FAILED);		/* just close down */
	}
	tbytes += (long) len;
  }
}


/*
 * g w r d a t a
 *
 * 'g' protocol data sending routine
 * 
 * Return: 	OK		Success (EOF)
 *		FAILED		fread()/gsendpkt() failure
 */
int gwrdata(fpin, out)
FILE *fpin; int out;
{
  long tbytes;
  size_t len;
  time_t start, end, tsecs;

  print2msg(M_TRANSFER, "gwrdata: sending data");

  tbytes = 0L;
  (void) time(&start);
  while (TRUE) {			/* Get data until EOF or failure */
	if ((len = fread((void *)wbuff, sizeof(char), rpktsize, fpin)) == 0) {
		/* packet of length zero indicates EOF */
		if (gsendpkt(wbuff, 0, FALSE) < 0) return(FAILED);
		(void) time(&end);
		tsecs = end - start;
		(void) syslog(tbytes, tsecs, (clock_t)0);
		return(OK);
	}
	if (gsendpkt(wbuff, len, FALSE) < 0)
		return(FAILED);
	tbytes += (long) len;
  }
}


/* Ggetpkt() gets one packet and returns it. The data is stored in
 * the buffer pointed to by |cdata|, and the length is stored in |*lenp|.
 * It calls gmachine() to get the data, and copies it from the proper input
 * buffer to the user's buffer area. "Short data" packets are handled here,
 * as opposed to within gmachine().
 *
 * Return:	-1	Channel not open
 *			gmachine(OP_READ) failure
 *			Invalid packet length
 *		0	Otherwise
 */
static int ggetpkt(cdata, lenp)
char *cdata; size_t *lenp;
{
  char *bufp;
  int nextread;

  if (!chanopen) return(-1);

  nextread = (lastread + 1) & 7;
  print3msg(M_HIGHPROTO, "ggetpkt: waiting for packet %d", nextread);

  if (gmachine(OP_READ) == -1) return(FAILED);

  *lenp = inpbufs[nextread].length;
  bufp = inpbufs[nextread].bufloc;
  if (inpbufs[nextread].sdata) {
	/* Less than 128 bytes shorter than packet length. */
	if (*bufp < 128) {
		*lenp -= *bufp++;
	}
	else {
		/* more than 128 bytes shorter */
		*lenp -= (*bufp++ & 127) * 256;
		*lenp -= *bufp++;
    	}
  }
  if (*lenp > (ENV_LEN + RPKTSIZE)) return(-1); /* invalid packet length */
  (void) memcpy((void *)cdata, (void *)bufp, *lenp);
  lastread = nextread;
  return(0);
}


/* Gsendpkt() queues the packet pointed to by cdata of length len
 * into the packet driver output buffer, and calls gmachine() to send
 * it. Gmachine() will return when the packet has been transmitted (but
 * not necessarily acknowledged, with window size greater than 1.) If
 * |flag| is true, cdata is considered a null-terminated string
 * which will be null-padded to the packet size and transmitted.
 *
 * Return:	-1				Channel not open
 *		result of gmachine(OP_WRITE)	Otherwise
 */
static int gsendpkt(cdata, len, flag)
char *cdata; size_t len; int flag;
{
  char *destp;
  int diff;

  if (!chanopen) return(-1);

  destp = outbufs[nstuff].bufloc;
  if (flag && len < spktsize) {
	print3msg(M_HIGHPROTO, "gsendpkt: padded packet |%s|", cdata);
	strncpy(destp, cdata, spktsize);
	outbufs[nstuff].sdata = 0;
  } else {
	if ((diff = spktsize - len) > 127) {	/* short packet ? */
		*destp++ = (diff >> 8) | 128;	/* use two bytes for diff */
		*destp++ = diff & 255;
		outbufs[nstuff].sdata = 1;
	} else if (diff > 0) {			/* really short packet */
		*destp++ = diff;		/* diff fits in one byte */
		outbufs[nstuff].sdata = 1;
	} else
		outbufs[nstuff].sdata = 0;
						/* copy into buffer */
	(void) memcpy((void *)destp, (void *)cdata,
			(len = (len > spktsize) ? spktsize : len));
  }
  print4msg(M_HIGHPROTO, "gsendpkt: queued packet %d length %u", nstuff, len);
  outbufs[nstuff].length = spktsize;
  nstuff = (nstuff + 1) & 7;

  return(gmachine(OP_WRITE));
}


/* Gmachine() is the heart of the 'g' packet driver. Its basic strategy
 * is:
 *	- transmit a packet if necessary
 *	- return if possible,
 *	- else read a packet and act on it
 *	- repeat
 *
 * OP_OPEN requests that the channel be opened, and OP_CLOSE requests that
 * it be closed. If why is OP_WRITE, gmachine() will return when the
 * last packet in the output queue has been transmitted (but not necessarily
 * acknowledged). OP_READ requests will return as soon as a new packet arrives.
 *
 * Return:	0	Success
 *		-1	Otherwise 
 */
static int gmachine(why)
int why;
{
  char *bufp;
  char *pktname[3];
  int xxx, yyy;
  int shortdat;
  int size;
  unsigned int len;

  pktname[0] = "INITC";
  pktname[1] = "INITB";
  pktname[2] = "INITA";

  while (TRUE) {
	if (OP_CLOSE == why) {
		print2msg(M_MEDPROTO, "gmachine: sending CLOSE request");
		(void) gspack(TT_CTRL, X_CLOSE, 0, (char *)NULL, 0);
		chanopen = 0;
	} else if (nackneeded) {
		print3msg(M_MEDPROTO, "gmachine: sending RJ, last packet %d", recv);
		/* corrected protocol violation. Ralf Wenk 25.Nov.90 */
		(void) gspack(TT_CTRL, X_RJ, recv, (char *)NULL, 0);
		nackneeded = 0;
	} else if (send != nstuff		/* nonzero output queue? */
		       	&& between(send, next, nstuff)	/* 'next' in queue? */
		       	&& between(send, next, (send + swindow) & 7)) {
		print3msg(M_MEDPROTO, "gmachine: sending data packet %d", next);
		(void) gspack(outbufs[next].sdata ? TT_SDATA : TT_DATA,
			next, recv, outbufs[next].bufloc,
			(unsigned int)outbufs[next].length);
		ackneeded = 0;
		next = (next + 1) & 7;
		/* Go back for more. */
       		if (OP_WRITE == why && next == nstuff)
			return(0);
	} else if (ackneeded) {
		print3msg(M_MEDPROTO, "gmachine: sending RR, last packet %d", recv);
		(void) gspack(TT_CTRL, X_RR, recv, (char *)NULL, 0);
		ackneeded = 0;
	} else if (OP_OPEN == why) {
		if (X_INITB == initpk)	/* INITB contains packet size, */
					/* given by size = 32(2^yyy) */
			size = ilog2(RPKTSIZE) - 5;
		else
			size = rwindow;	/* INITA, INITC contain window size */
		if (X_INITA == initpk || X_INITB == initpk || X_INITC == initpk) {
			print4msg(M_MEDPROTO, "gmachine: sending %s, yyy %d",
					pktname[initpk - 5], size);
			if (X_INITC == initpk)
				print4msg(M_MEDPROTO,
				"gmachine: input packet size %d, window size %d",
						RPKTSIZE, rwindow);
		}
		else {	/* never happen */
			print4msg(M_MEDPROTO,
				"gmachine: sending OPEN packet xxx %d yyy %d",
					initpk, size);
		}
		(void) gspack(TT_CTRL, initpk, size, (char *)NULL, 0);
        }

	/* quit if we're reading and we've got a new packet */
	if (OP_READ == why && recv != lastread) return(0);

	/* we've sent a packet - now get an answer */
	shortdat = 0;
	bufp = (char *)(buf + ENV_LEN);
	switch(ggpack(&xxx, &yyy, &len)) {
		case LOSTPKT:
			print2msg(M_MEDPROTO, "gmachine: timeout - lost packet");
			if (nlost > MAXLOST)
				return(-1);
			next = send;	/* retransmit last un-ack'ed pkt, */
			if (OP_READ == why)	/* request retransmit */
			   	nackneeded = 1;
			skipping = 0;
			break;
		case BADPKT:
			print2msg(M_MEDPROTO, "gmachine: bad packet");
			nackneeded = 1;	/* reject! */
			skipping = 1;	/* ignore the rest of the 'window' */
			break;
		case TT_SDATA:
			shortdat = 1;
			/* fall through */
		case TT_DATA:
			send = (yyy + 1) & 7;	/* receive acknowledgement */
			if (xxx != ((recv + 1) & 7)) {
				print4msg(M_MEDPROTO,
				"gmachine: input out of sequence (got %d != %d)",
							xxx, (recv + 1) & 7);
				/* We must have missed one. */
				if (!skipping) {
					nackneeded = 1;
					skipping = 1;
				}
				else
					print2msg(M_MEDPROTO,
						"gmachine: ignoring packet");
			}
			else {	/* This is most recent correct pkt. */
				recv = xxx;
				ackneeded = 1; /* we will ack it */
				skipping = 0;
				inpbufs[xxx].length = len;
				inpbufs[xxx].sdata = shortdat;
				(void) memcpy((void *)inpbufs[xxx].bufloc,
						(void *)bufp, (size_t)len);
			}
			break;
		case TT_CTRL:
			skipping = 0;
			switch(xxx) {
			    case X_CLOSE:
				print2msg(M_MEDPROTO, "gmachine: got CLOSE");
				(void) gspack(TT_CTRL, X_CLOSE, 0, (char *)NULL, 0);
				chanopen = 0;
				if (OP_CLOSE == why)
					return(0);	/* expected? */
				  else
					return(-1);	/* nope */
			    case X_RJ:
				print3msg(M_MEDPROTO,
					"gmachine: got RJ, last packet %d", yyy);
				/* rejected packet number patch. Ralf Wenk */
				yyy = ( yyy + 1 ) & 7;  /* resend packet # */
				if (between(send,yyy,next))
					next = send = yyy;	/* check! */
				else
					print2msg(M_MEDPROTO,
						"gmachine: RJ out of window");
				break;
			    case X_RR:
				print3msg(M_MEDPROTO,
					"gmachine: got RR, last packet %d", yyy);
				if (between(send,yyy,next))
					send = (yyy + 1) & 7;
				else {
					print2msg(M_MEDPROTO,
						"gmachine: RR out of window");
					/* Back up all the way to beginning */
					/* of window. */
					next = send;
				}
				break;
			    case X_INITC:
				print3msg(M_MEDPROTO,
					"gmachine: got INITC, yyy %d", yyy);
				swindow = yyy;
				if (X_INITC == initpk) {
					print4msg(M_MEDPROTO,
			"gmachine: output packet size %d, window size %d",
							spktsize, swindow);
					chanopen = 1;
					return(0);
				}
				break;
			    case X_INITB:
				print3msg(M_MEDPROTO,
					"gmachine: got INITB, yyy %d", yyy);
				spktsize = 32 << yyy;
				if (X_INITB == initpk)
					initpk = X_INITC;
				break;
			    case X_INITA:
				print3msg(M_MEDPROTO,
					"gmachine: got INITA, yyy %d", yyy);
				swindow = yyy;
				initpk = X_INITB;
				break;
			    default:
				print3msg(M_MEDPROTO,
				"gmachine:  got bad control packet xxx %d", xxx);
			}
			break;
		default:
			print2msg(M_MEDPROTO, "gmachine: bad packet type");
			break;
	}
  }
}


/*
 * Ggpack() is responsible for getting one 'g'-protocol packet from the
 * input communications channel. This includes framing, detecting timeouts,
 * and checksum validation.
 * The basic strategy is to keep a count of how many bytes have currently
 * been read and how many are needed. When enough bytes for a packet header
 * ("envelope") have been read, it is validated. If it is valid, and it
 * has a nonzero data segment, the data portion is read. When it has
 * everything, it does a checksum test and returns, with the control
 * information stored in *xxxp, *yyyp, and the data segment length stored
 * in *lenp.
 *
 * Return:	A result status code	Always
 *		packet variables are updated as a side-effect
 */
static int ggpack(xxxp, yyyp, lenp)
int *xxxp, *yyyp;
unsigned int *lenp;
{
  unsigned char *bufp;		/* scratch pointer */
  int j;			/* scratch integer */
  int env;			/* 'have pkt envelope' flag */
  int gotdle;			/* do we have envelope hdr? */
  int tt;			/* packet type */
  unsigned int sum;		/* checksum */
  size_t need;			/* total bytes needed */
  size_t remain;		/* bytes which we still need */

  env = gotdle = 0;
  need = ENV_LEN;		/* initially, need a header */

  while (TRUE) {
	if (low == high)	/* prevent framebuffer overruns */
	  	low = high = buf;
	print5msg(M_LOWPROTO, "ggpack: l=%d h=%d g=%d", (low - buf),
						(high - buf), gotdle);
	while (need > (high - low)) {
		remain = (need - (high - low));
		if ((j = sread((char *)high, remain, TIMEOUT)) < 0) {
			++nlost;
			/* empty out any partial packet */
			low = high = buf;
			return(env ? BADPKT : LOSTPKT);
		}
		high += j;	/* got some data - move upper limit up */
	}
	if (!gotdle) {
		while (low < high) {	/* look for header 'DLE' prefix */
			if (DLE == *low) {
				gotdle = 1;
				break;
			} else ++low;
		}
		continue;
	} else if (!env) {	/* found DLE, but haven't found header yet */
			if (low > buf) { /* move envelope to buf beginning */
				bufp = buf;
				while (low < high) *bufp++ = *low++;
				low = buf;
				high = bufp;
			}
			if (buf[ENV_X] !=
				(buf[ENV_K]^buf[ENV_C0]^buf[ENV_C1]^buf[ENV_C])
	  		  || buf[ENV_K] < 1 || buf[ENV_K] > 9) {  /* valid? */
				++low;
				gotdle = 0;
				print2msg(M_LOWPROTO,
					"ggpack: rejecting an envelope");
				continue;
			}
			env = 1;		/* we have an envelope */
			tt = (buf[ENV_C] >> 6) & 3; 	/* store control info */
			*xxxp = (buf[ENV_C] >> 3) & 7;
			*yyyp = (buf[ENV_C]     ) & 7;
			if (buf[ENV_K] == 9)	/* does it have data? */
				*lenp = 0;
			else
				/* Size = 32 * 2^(k-1) */
				*lenp = 16 << buf[ENV_K];
			need += *lenp;		/* now need that many more */
			print6msg(M_LOWPROTO,
				"ggpack: tt=%d, xxx=%d, yyy=%d, need=%d",
							tt, *xxxp, *yyyp, need);
			continue;
	} else {			/* have everything we need */
		if (*lenp)
		     sum = MAGIC - (cksum((char *)(buf + ENV_LEN), *lenp) ^ buf[ENV_C]);
		else
		     sum = MAGIC - buf[ENV_C];
		if (((sum >> 8) & 0xFF) != buf[ENV_C1]
					|| (sum & 0xFF) != buf[ENV_C0]) {
			nlost = 0;
			print5msg(M_LOWPROTO,
				"ggpack: bad checksum c1=%u, c0=%u, cksum=%u",
				buf[ENV_C1], buf[ENV_C0], sum);
			low += ENV_LEN;	/* search bad data seg for a header */
			return(BADPKT);
		} else {
			nlost = 0;
			low += need;
			if (*lenp)
				print3msg(M_DATA, "ggpack: |%s|",
					visib((char *)(buf + ENV_LEN), *lenp));
			return(tt);
		}
	}
  }
}


/*
 * gspack simply sends out a packet, by assembling an envelope and writing
 * it, and the data segment, to the communications channel.
 *
 * Return:	0	Always
 */
static int gspack(tt, xxx, yyy, data, len)
int tt, xxx, yyy;
char *data;
unsigned int len;
{
  unsigned char envelope[ENV_LEN];	/* packet envelope */
  unsigned char ctrl;			/* header control byte */
  int k;				/* size = 32 * 2^(k-1) */
  unsigned int sum;			/* checksum */
  unsigned int tmp;

  ctrl = ((tt & 3) << 6) | ((xxx & 7) << 3) | (yyy & 7);

  if (0 == len) {
	k = 9;				/* no data seg */
	sum = MAGIC - ctrl;
  } else {
	tmp = len;
	for (k = -5; tmp != 0; ++k)	/* find k */
		tmp >>= 1;
	sum = MAGIC - (cksum(data, len) ^ ctrl);
  }
  envelope[ENV_DLE] = DLE;
  envelope[ENV_K]   = k;
  envelope[ENV_C1]  = (sum >> 8) & 0xFF;
  envelope[ENV_C0]  = sum & 0xFF;
  envelope[ENV_C]   = ctrl;
  envelope[ENV_X]   = k ^ envelope[ENV_C0] ^ envelope[ENV_C1] ^ ctrl;

  print7msg(M_LOWPROTO, "gspack: tt=%d xxx=%d yyy=%d k=%d cksum=%u",
						tt, xxx, yyy, k, sum);
  if (0 != len) print3msg(M_DATA, "gspack: |%s|", visib(data, len));

  (void) swrite((char *)envelope, ENV_LEN); /* send envelope */
  if (0 != len) (void) swrite(data, len); /* send data segment */

  return(0);
}


/*
 * chksum(data, len) came directly from dcp. It checksums the given data
 * area using the g protocol algorithm.
 *
 * Return:	The checksum value	Always
 */
static unsigned int cksum(data, len)
char *data;
unsigned int len;
{
  unsigned int j, k, tmp, chk1, chk2;

  chk1 = 0xffff;
  chk2 = 0;
  j = len;
  for (k = 0 ; k < len ; k++) {
	if (chk1 & 0x8000) {
		chk1 <<= 1;
		chk1++;
	} else {
		chk1 <<= 1;
	}
	tmp = chk1;
	chk1 += (data[k] & 0xff);
	chk2 += chk1 ^ j;
	if ((chk1 & 0xffff) <= (tmp & 0xffff)) chk1 ^= chk2;
	j--;
  }
  return(chk1 & 0xffff);
}


/* ilog2(value)
 *
 * Returns floor(log2(value)).
 */
static int ilog2(value)
unsigned int value;
{
  int j;

  if (value == 0) return(-1);
  for (j = 0 ; value > 1 ; j++) value >>= 1;

  return(j);
}
