/* 'f' protocol driver for Minix UUCP.
 *
 * Version 1.2 of 31 October 1991.
 *
 * flow control protocol.
 *
 * This protocol relies on flow control of the data stream.
 * It is meant for working over links that can (almost) be
 * guaranteed to be errorfree, specifically X.25/PAD links.
 * A checksum is carried out over a whole file only. If a
 * transport fails the receiver can request retransmission(s).
 * This protocol uses a 7-bit datapath only, so it can be
 * used on links that are not 8-bit transparent.
 *
 * When using this protocol with an X.25 PAD:
 *
 * Although this protocol uses no control chars except CR,
 * control chars NULL and ^P are used before this protocol
 * is started; since ^P is the default char for accessing
 * PAD X.28 command mode, be sure to disable that access
 * (PAD par 1). Also make sure both flow control pars
 * (5 and 12) are set. The CR used in this proto is meant
 * to trigger packet transmission, hence par 3 should be 
 * set to 2; a good value for the Idle Timer (par 4) is 10.
 * All other pars should be set to 0.
 *
 * Normally a calling site will take care of setting the
 * local PAD pars via an X.28 command and those of the remote
 * PAD via an X.29 command, unless the remote site has a
 * special channel assigned for this protocol with the proper
 * par settings.
 *
 * Additional comments for hosts with direct X.25 access:
 * - reads are done in small chunks, which can be smaller than
 *   the packet size; your X.25 driver must support that.
 *
 * The 'f' protocol sends messages as '\r' terminated strings,
 * whose length must be less than MSGLEN.
 *
 * The 'f' protocol sends files as a continuous stream, followed by
 * a four-byte string representing a hexadecimal checksum, which
 * in turn is terminated by '\r'.  Data is sent using 7 bits only,
 * and bytes with the 8th bit set are translated to 7 bits and 
 * escaped by a preceeding byte from the range 0172-0176; the data
 * reading routine reverses the translation.  If an error occurs,
 * then the transfer is aborted and another attempt made, up to the
 * limit MAXRETRIES.
 *
 * The six main routines return OK (1) if successful, and FAILED (-1)
 * otherwise.
 *
 * Author:
 *	Piet Beertema, CWI, Amsterdam, Sep 1984
 * Modified for X.25 hosts:
 *	Robert Elz, Melbourne Univ, Mar 1985
 * Rewritten for PC Minix 1.5:
 *	C W Rose, San Diego, Sep 1991
 */

#include <sys/types.h>
#include <sys/times.h>
#ifdef _MINIX
#include <minix/const.h>
#undef TRUE
#endif
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <time.h>
#include <unistd.h>
#include "dial.h"
#include "uucp.h"
#include "uucico.h"


#ifdef VERBOSE 				/* remove these calls for speed */
#define print2msg printmsg
#define print3msg printmsg
#define print4msg printmsg
#define print5msg printmsg
#define print6msg printmsg
#define print7msg printmsg
#else
#define print2msg(a, b)
#define print3msg(a, b, c)
#define print4msg(a, b, c, d)
#define print5msg(a, b, c, d, e)
#define print6msg(a, b, c, d, e, f)
#define print7msg(a, b, c, d, e, f, g)
#endif

#define FIBUFFSIZE	256	/* for X.25 interfaces: set equal to packet size,
				 * but see comment above
				 */

#define FOBUFFSIZE	256	/* for X.25 interfaces: set equal to packet size;
				 * otherwise make as large as feasible to reduce
				 * number of write system calls 
				 */

#define MAXRETRIES	2	/* max. attempts to retransmit a file */

#ifndef lint
static char  *Version = "@(#) fio 1.2 (31 October 1991)";
#endif

extern int sread();		/* read from the serial port */
extern int swrite();		/* write to the serial port */
extern int syslog();		/* print file transfer statistics */
extern char *visib();		/* print buffer in ASCII for debugging */

/* Global variables */

static int checksum;
static int endflg;
static int fd_in;
static int fd_out;

/*Forward declarations */

extern int fon();
extern int frdmsg();
extern int fwrmsg();
extern int frddata();
extern int fwrdata();
extern int foff();
static int frdblk();
static int fwrblk();


/*
 * f o n
 *
 * Turn on the protocol
 * 
 * Return:	OK	Always
 */
int fon(in, out)
int in, out;
{
  /* initialise some local variables */

  checksum = 0xffff;
  endflg = FALSE;
  fd_in = in;
  fd_out = out;

  return(OK);
}


/*
 * f r d m s g
 *
 * Read a message
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int frdmsg(msg, in)
char *msg; int in;
{
  char c, tmp[10];		/* ### sread won't work with &c */
  size_t sz;

  sz = 0;
  while (TRUE) {
	if (sz == MSGLEN) return(FAILED);
	if (sread(tmp, 1, WAIT) < 1) return(FAILED);
	c = tmp[0] & 0x7f;
	if (c == '\r' || c == '\n') c = '\0';
	msg[sz++] = c;
	if (c == '\0') break;
  }
  print3msg(M_DATA, "frdmsg: buffer |%s|", visib(msg, sz));

  return(OK);
}


/*
 * f w r m s g
 *
 * Write a message
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int fwrmsg(msg, out)
char *msg; int out;
{
  char buff[MSGLEN];
  size_t sz;

  /* load the buffer and clear any trailing newline */
  strncpy(buff, msg, MSGLEN - 2);
  sz = strlen(buff);
  if (sz != 0 && buff[sz - 1] == '\n') sz--;
  buff[sz++] = '\r';
  buff[sz] = '\0';

  /* write the message */
  print3msg(M_DATA, "fwrmsg: buffer |%s|", visib(buff, sz));
  if (swrite(buff, sz) != sz)
	return(FAILED);
  else
	return(OK);
}


/*
 * f r d d a t a
 *
 * Read data
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int frddata(fpout, in)
FILE *fpout; int in;
{
  char buff[FIBUFFSIZE];
  int retry, maxtry, count;
  long int bytes;
  time_t secs;
  clock_t start, end;
  struct tms ttms;

  checksum = 0xffff;
  bytes = 0L;
  maxtry = MAXRETRIES;
  retry = 0;
  while (retry < maxtry) {
	/* receive a file */
	start = times(&ttms);
	while (TRUE) {
	 	/* read the data block */
		if ((count = frdblk(buff, FIBUFFSIZE, fd_in)) == 0 ||
						count == FAILED) 
			break;
		bytes += (long)count;
		/* write the data block */
		if (fwrite((void *)buff, sizeof(char), (size_t)count, fpout)
							!= (size_t)count) {
			count = FAILED;
			break;
		}
	}
	end = times(&ttms);

	if (count == FAILED) {
		/* clear input */
		while (sread(buff, FIBUFFSIZE, 1) != FAILED)
			;
		/* request repeat */
		print2msg(M_HIGHPROTO, "sending ack: 'R'");
		if (fwrmsg("R", fd_out) == FAILED) return(FAILED);
		if (fseek(fpout, 0L, 0) != 0) return(FAILED);
		checksum = 0xffff;
		bytes = 0L;
		retry++;
	}
	else
		break;
  }

  if (retry == maxtry || count == FAILED) {
	/* clear input */
	while (sread(buff, FIBUFFSIZE, 1) != FAILED)
		;

	/* request abort */
	count = FAILED;
	print2msg(M_HIGHPROTO, "sending ack: 'Q'");
	(void) fwrmsg("Q", fd_out);
  }
  else {
	print2msg(M_HIGHPROTO, "sending ack: 'G'");
	(void) fwrmsg("G", fd_out);

	/* write transfer statistics */
	end -= start;
	secs = end / HZ;
	end %= HZ;
	(void) syslog(bytes, secs, end);
  }

  return(count == FAILED ? FAILED : OK);
}


/*
 * f w r d a t a
 *
 * Write data
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int fwrdata(fpin, out)
FILE *fpin; int out;
{
  char ack, buff[FOBUFFSIZE], msg[MSGLEN];
  int count, maxtry, retry, status;
  long int bytes;
  time_t secs;
  clock_t start, end;
  struct tms ttms;

  /* give the other end time to sort itself out */
  (void) sleep(2);

  checksum = 0xffff;
  bytes = 0L;
  maxtry = MAXRETRIES;
  retry = 0;
  while (retry < maxtry) {
	print3msg(M_DATA, "fwrdata: pass %d", retry);
	/* transmit a file */
	count = 1;
	start = times(&ttms);
	while (count != 0) {
		if ((count = (int) fread((void *)buff, sizeof(char),
				FOBUFFSIZE, fpin)) == 0 && feof(fpin) == 0) {
			count = FAILED;
			break;
		}
		bytes += (long)count;
		if (fwrblk(buff, count, fd_out) == FAILED) {
			count = FAILED;
			break;
		}
	}
	end = times(&ttms);
	
	if (frdmsg(msg, fd_in) == FAILED || (ack = msg[0]) == 'Q') {
		print2msg(M_DATA, "fwrdata: failed");
		status = FAILED;
		break;
	}
	else if (ack == 'R') {
		print2msg(M_DATA, "fwrdata: retry");
		retry++;
		if (fseek(fpin, 0L, 0) != 0) return(FAILED);
		checksum = 0xffff;
		bytes = 0L;
		status = FAILED;
	}
	else if (ack == 'G') {
		print2msg(M_DATA, "fwrdata: success");
		status = OK;
		break;
	}
  }

  /* write transfer statistics */
  if (status == OK) {
	end = times(&ttms);
	end -= start;
	secs = end / HZ;
	end %= HZ;
	(void) syslog(bytes, secs, end);
  }

  return(status);
}


/*
 * f o f f
 *
 * Turn off the protocol
 *
 * Return:	OK	Always
 */
int foff(in, out)
int in, out;
{
  return(OK);
}


/*
 * Byte conversion, to allow transmission over 7-bit paths:
 *
 *   from	 pre	   to
 * 000-037	 172	 100-137
 * 040-171		 040-171
 * 172-177	 173	 072-077
 * 200-237	 174	 100-137
 * 240-371	 175	 040-171
 * 372-377	 176	 072-077
 */


/*
 * f r d b l k
 *
 * Read a block of data
 *
 * Return:	Bytes read	Success
 *		0		EOF
 *		FAILED		Otherwise
 *
 * This routine converts incoming data from 7-bit to 8-bit bytes,
 * and evaluates the checksum that terminates a file.
 */
static frdblk(buff, len, fd) 
char *buff; int len, fd;
{
  char c, tmp[10];
  static char special;
  int j, k, sum;

  /* return 0 to denote eof */
  if (endflg) {
	endflg = FALSE;
	return(0);
  }

  special = '\0';
  sum = checksum;
  j = 0;
  while (j < len) {
	/* abort on read failure or premature EOF */
	if (sread(tmp, 1, WAIT) < 1) return(FAILED);
	c = tmp[0] & 0x7f;		/* ### mask or discard ? */
	/* current character is an escape */
	if (c > 0171) {
		if (c == 0176 && special == 0176) {
			/* set the eof flag */
			if (j != 0) endflg = TRUE;
			/* get the terminating checksum */
			if (sread(tmp, 5, WAIT) < 5) return(FAILED);
			for (k = 0 ; k < 5 ; k++) tmp[k] &= 0177;
			if (tmp[4] != '\r') return(FAILED);
			tmp[4] = '\0';
			if (sscanf(tmp, "%4x", &checksum) != 1) return(FAILED);
			print4msg(M_DATA, "frdblk: sum %04x checksum %04x",
							sum, checksum);
			/* see if it matches */
			if (sum == checksum)
				return(j);
			else
				return(FAILED);
		}
		else
			special = c;
	}
	/* current character is ordinary */
	else {
		/* convert the data to 8 bits */
		switch (special) {
		case '\172':
			c -= 0100;
			break;
		case '\173':
			/* fall through */
		case '\174':
			c += 0100;
			break;
		case '\175':
			c += 0200;
			break;
		case '\176':
			c += 0300;
			break;
		default:
			break;
		}
		/* update the checksum */
		if (sum & 0x8000) {
			sum <<= 1;
			sum++;
		} else
			sum <<= 1;
		sum += c & 0377;
		sum &= 0xffff;
		buff[j++] = c;
		special = 0;
	}
  }
  checksum = sum;

  return(j);
}


/*
 * f w r b l k
 *
 * Write a block
 *
 * Return:	Bytes written	Success
 *		FAILED		Otherwise
 *
 * This routine converts outgoing data from 8-bit to 7-bit bytes;
 * if called with len == 0, it generates the terminating checksum.
 */
static fwrblk(buff, len, fd)
char *buff; int len, fd;
{
  char c, data[FOBUFFSIZE * 2];
  int j, k, sum;
  size_t sz;

  /* write a block of data */
  j = k = 0;
  sum = checksum;
  while (j < len) {
	c = buff[j];
	/* update the checksum */
	if (sum & 0x8000) {
		sum <<= 1;
		sum++;
	} else
		sum <<= 1;
	sum += c;
	sum &= 0xffff;
	/* convert the data to 7 bits */
	if (c < 040) {
		data[k++] = '\172';
		data[k++] = c + 0100;
	}
	else if (c <= 0171) {
		data[k++] = c;
	}
	else if (c <= 0177) {
		data[k++] = '\173';
		data[k++] = c - 0100;
	}
	else {
		c &= 0177;
		if (c < 040) {
			data[k++] = '\174';
			data[k++] = c + 0100;
		}
		else if (c <= 0171) {
			data[k++] = '\175';
			data[k++] = c;
		}
		else {
			data[k++] = '\176';
			data[k++] = c - 0100;
		}
	}
	j++;
  }
  checksum = sum;

  /* add the checksum at EOF */
  if (len == 0) {
	sprintf(data, "\176\176%04x\r", checksum);
	k = strlen(data);
  }

  /* write out the data buffer */
  sz = (unsigned int)k;
  if (swrite(data, sz) != sz)
	return(FAILED);
  else
	return(OK);
}
