/* cu2.c
 *
 * Version 1.2 of 31 October 1991.
 */

#ifdef _MINIX
#include <minix/config.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <time.h>
#include <unistd.h>
#include "dial.h"
#include "uucp.h"
#include "cu.h"


/*
 * g e t a r g s
 *
 * Parse a string of arguments
 * 
 * Return:	The number of fields allocated		Success
 *		0					Failure 
 *
 * The string must contain whitespace-separated fields.  Getargs puts a pointer
 * to each field into the array pointed to by argv, and increments a count.
 */
int getargs(str, argv)
char *str; char *argv[];
{
  int nflds = 0;

  while (*str && isspace(*str)) ++str;		/* leading space */

  while (*str && *str != '\n') {
	if (++nflds > MAXARGS) return(0);	/* too many fields */

	*argv++ = str;
	while (*str && !isspace(*str)) ++str;	/* skip to end of field */
	*str++ = '\0';			/* and terminate it with a null */

	while (*str && isspace(*str)) ++str;	/* trailing space */
  }
  *str = '\0';
  return(nflds);
}


/*
 * g e t _ l i n e
 *
 * Get a line from the terminal, allowing for editing characters
 *
 * Return:	Number of characters read	Success
 *		-1				Read error
 */
int get_line(fd, bp, max)
int fd; char *bp; unsigned max;
{
  int j;
  struct TTY tty;

  (void) ioctl(fd, TTYGET, &tty);
  (void) ioctl(fd, TTYSET, &stdtty);

  fprintf(stderr, "[%s] ", locsite);

  switch (j = read(fd, bp, max - 1)) {
	case -1:
		break;
	case 0:
		break;
	default:
		if (bp[j - 1] == '\n')
			j--;
		bp[j] = '\0';
		break;
  }

  (void) ioctl(fd, TTYSET, &tty);

  printmsg(M_CALL, "get_line: command string |%s|", bp);

  return(j);
}


/*
 * g e t c m d
 *
 * Get command string reflected from remote
 *
 * Return:	TRUE	Writing to takefile
 *		FALSE	Otherwise
 */
int getcmd(in, out, buff, buffp, tfile, takeflg, end)
char *buff, *buffp, *tfile, *end; int takeflg; int in, out;
{
  char c, tc, fbuff[256], nbuff[132], sbuff[256];
  int j, state, bpos, fpos, npos, spos, lastpos, firstpos;

  firstpos = buffp - buff;		/* initial guess at start of command */
  lastpos = firstpos - strlen(end);	/* we've seen the whole trigger string */
  if (lastpos < 0) lastpos = 0;		/* but it may not be in this buffer */
  tc = '\r';				/* character terminating command */

  /* debug this in ascii */
  if (dbglvl > 0) {
	tc = 'R';
	for (j = 0 ; j < count ; j++) {
		if (buff[j] == '\r') buff[j] = 'R';
		if (buff[j] == '\n') buff[j] = 'N';
	}
	fprintf(stderr,
		"\r\ngetcmd: buffer lastpos %d, firstpos %d, count %d\r\n",
		lastpos, firstpos, count);
	fprintf(stderr, "contents |%s|\r\n", buff);
  }

  /*
   * The state machine changes as the string is scanned L to R
   *
   * State 1 is the scan of the characters before the trigger
   * State 2 is the scan of the trigger string
   * State 3 is the scan of the command string
   * State 4 is the scan of the characters after the command
   *
   * Note that the trigger may occur in the middle of the buffer,
   * and that the command may lack a file name.
   * We can receive commands only when not taking a file.
   */
  bpos = fpos = npos = spos = 0;
  if (!takeflg) {
 	printmsg(M_CALL, "getcmd: starting command entry");
	if (lastpos > 0)
		state = 1;
	else {
		state = 2;
		sbuff[0] = '\0';
	}
	while (state != 4) {
		/* scan the buffer */
		for (j = bpos ; j < count ; j++) {
			c = buff[j];
	
			/* strip all bytes - state 1 & 2 already done */
			c &= bitmask; 
	
			if (state == 1) {	/* screen characters */
				sbuff[spos++] = c;
				if (j == lastpos) {
					sbuff[spos] = '\0';
					state++;
				}
			}
			else if (state == 2) {	/* trigger characters */
				if (j == firstpos) state++;
			}
			else if (state == 3) {	/* command characters */
				if (c == tc) {
					nbuff[npos] = '\0';
					firstpos = j + 1;
					state++;
				}
				else
					nbuff[npos++] = c;
			}
			else if (state == 4) {	/* takefile characters */
				fbuff[fpos++] = c;
			}
		}
		/* we must have the filename terminator */
		if (state != 4) {
			bpos = count;
			if (read(in, buff + count, 1) == -1)
				wrapup("cannot read from device", "");
			if (dbglvl > 0) {
				if (buff[count] == '\r') buff[count] = 'R';
				if (buff[count] == '\n') buff[count] = 'N';
			}
			count++;
		}
	}
	fbuff[fpos] = '\0';
	count = fpos;

	/* complete any output to the terminal */
	if (write(out, sbuff, (unsigned int)spos) != spos)
		wrapup("cannot write to the terminal", "");

	/* move the remaining data back into the buffer */
	for (j = 0 ; j < count ; j++)
		buff[j] = fbuff[j];

	/* see if we've got a command or a file name */
	printmsg(M_CALL, "getcmd: command/file |%s|", nbuff);
	if (strncmp(nbuff, "cd ", 3) == SAME) {	/* change directory */	
		(void) chdir(nbuff + 3);
		takeflg = !takeflg;
	}
	else {				/* try to open the take file */
		if (dbglvl > 0)
			strcpy(tfile, "taskfile");
		else
			strncpy(tfile, nbuff, 131);

		lines = characters = 0;

		if (creat(tfile, 0644) == -1 || (fdt = open(tfile, 1)) == -1) {
			(void) write(2, progname, strlen(progname));
			(void) write(2, ": cannot access ", 16);
			(void) write(2, tfile, strlen(tfile));
			(void) write(2, "\r\n", 2);
			takeflg = !takeflg;
		}
		(void) chmod(tfile, 0666);
	}
  }
  else {
	printmsg(M_CALL, "getcmd: shutting down take");
	if (lastpos > 0)
		state = 4;
	else {
		state = 3;
		fbuff[0] = '\0';
	}
	/* scan the buffer */
	for (j = 0 ; j < count ; j++) {
		c = buff[j];

		/* strip all bytes - state 4 & 2 already done */
		c &= bitmask;

		if (state == 1) {	/* screen characters */
			sbuff[spos++] = c;
		}
		else if (state == 2) {	/* trigger characters */
			if (j == firstpos)
				state--;
		}
		else if (state == 3) {	/* tidy up */
			characters += fpos;
			state--;
			if (j == firstpos)
				state--;
		}
		else if (state == 4) {	/* takefile characters */
			fbuff[fpos++] = c;
			if (c == '\n') lines++;
			if (j == lastpos) {
				fbuff[fpos] = '\0';
				state--;
			}
		}
	}
 	sbuff[spos] = '\0';
	count = spos;

	/* write what remains to the take file */
	if (write(fdt, fbuff, (unsigned int)fpos) != fpos)
		wrapup("cannot write to ", tfile);
	else
		printmsg(M_CALL, "getcmd: %d bytes to %s", fpos, tfile);

	/* move the terminal data back into the main buffer */
	for (j = 0 ; j < count ; j++)
		buff[j] = sbuff[j];

	/* close the takefile */
	(void) close(fdt);

	/* write out the summary */
	fprintf(stderr, "%d lines / %d characters\r\n", lines, characters);
  }

  takeflg = !takeflg;
  printmsg(M_CALL, "getcmd: return takeflg %c", takeflg ? 'T' : 'F');

  return(takeflg);
}


/*
 * g e t h d i r
 *
 * Get the HOME directory corresponding to the given name
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		Directory is updated as a side effect
 */
int gethdir(user, directory)
char *user, *directory;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) == (struct passwd *)NULL)
	return(FAILED);

  strncpy(directory, pw->pw_dir, 131);

  return(OK);
}


/*
 * g e t n a m e
 *
 * Get the local UUCP nodename
 *
 * Return:	OK	Sucess
 *		FAILED	Otherwise
 *		name is updated as side effect
 *
 * There are several possible means of determining this, depending
 * on the operating system version. For now, this version just reads
 * one line from the NODENAME file, which is usually LIBDIR/SYSTEMNAME.
 */
int getname(name)
char *name;
{
  int j, fd;
  char buff[132];

  for (j = 0 ; j < 132 ; j++) buff[j] = '\0';
  if ((fd = open(NODENAME, 0)) != -1 && read(fd, buff, 131) != -1) {
	if (!isalpha(buff[0])) {
		*name = '\0';
		(void) close(fd);
		return(FAILED);
	}
	for (j = 0 ; j < strlen(buff) ; j++) {
		if (buff[j] == '\0' || buff[j] == ' ' ||
				buff[j] == '\t' || buff[j] == '\n') {
			buff[j] = '\0';
			break;
		}
	}
	(void) close(fd);
	buff[SITELEN] = '\0';
	strncpy(name, buff, SITELEN);
	return(OK);
  }
  else {
	*name = '\0';
	(void) close(fd);
	return(FAILED);
  }
}


/*
 * g e t s y s t e m
 *
 * Process the "L.sys" file
 *
 * Return:	C	Site found
 *		(Y	Site not found on this pass)
 *		Z	Site not known at all
 */
char getsystem(site)
char *site;
{
  char  *cp, status, buff[BUFSIZ], tempbuff[256], scratch[132], tmp[30];
  int j, index, maxfld;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) {
	printmsg(M_CALL, "getsystem: can't open L.sys");
	return('Z');
  }

  status = 'Y';
  index = 0;
  while (fgets(buff, BUFSIZ, fplsys) != (char *)NULL) {
	while ((j = strlen(buff)) < (BUFSIZ - 132) &&
			buff[j - 2] == '\\' &&
			fgets(tempbuff, 131, fplsys) != (char *)NULL) {
		buff[j - 2] = '\0';
		strcat(buff, tempbuff);
	}
	if (buff[0] == '#' || buff[0] == '\n') continue; /* ignore comments */
	if (index++ < lsysindex) continue;	/* been here before */
	j = strlen(buff);
	strncpy(tempbuff, buff, 255);		/* getargs roaches the string */
	maxfld = getargs(tempbuff, cmdflds);
	if (maxfld < 6) continue;		/* too few fields to be valid */
	printmsg(M_CALL, "getsystem: %s", buff);
	strncpy(Lsite, cmdflds[F_SYSTEM], 19);
	strncpy(Lcctime, cmdflds[F_CCTIME], 63);
	strncpy(tmp, cmdflds[F_CLASS], 29);
	/* strip protocols from device field */
	if ((cp = strchr(tmp, ',')) != (char *)NULL) *cp = '\0';
	strncpy(Ldevice, tmp, 19);
	strncpy(Lbaud, cmdflds[F_SPEED], 19);
	strncpy(tmp, cmdflds[F_PHONE], 29);

	/* convert any dialcode abbreviation */
	if (strcmp(tmp, "-") != SAME && isalpha(tmp[0]) &&
					checkcode(tmp) == FAILED)
		continue;
	strncpy(Lphone, tmp, 19);

	/* say what we've found */
	if (dbglvl >= M_CALLMSG) {
		for (j = F_EXPECT; j < maxfld; j += 2) {
			sprintf(scratch,
			"getsystem: expect [%02d]: %-30s    send [%02d]: %s",
	             	j, cmdflds[j], j + 1, cmdflds[j + 1]);
			printmsg(M_CALLMSG, scratch);
		}
		printmsg(M_CALLMSG, "getsystem: rmt=%s ctm=%s",
			Lsite, cmdflds[F_CCTIME]);
		printmsg(M_CALLMSG, "getsystem: dev=%s pro=%s spd=%s tel=%s",
			Ldevice, Lprotocols, Lbaud, Lphone);
	}

	/* check for match */
	if (strcmp(site, Lsite) == SAME) {
		printmsg(M_CALLMSG, "getsystem: site name %s matched", site);
		status = 'C';
		break;
	}
  }

  (void) fclose(fplsys);
  fplsys = (FILE *)NULL;
  if (status == 'Y' && lsysindex == 0)		/* system not in L.sys */
	status = 'Z';
  else
	lsysindex = index;

  return(status);
}


/*
 * g e t _ u i d
 *
 * Get the user id corresponding to the given name
 *
 * Return:	uid	Success
 *		~0	Otherwise
 */
uid_t get_uid(user)
char *user;
{
  struct passwd *pw;

  if ((pw = getpwnam(user)) != (struct passwd *)NULL)
	return(pw->pw_uid);
  else 
	return((uid_t) ~0);
}


/*
 * g e t u s e r 
 *
 * Get the username corresponding to the current uid
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *		buffer is updated as side effect
 */
int getuser(uid, buff)
uid_t uid; char *buff;
{
  struct passwd *pw;

  if ((pw = getpwuid(uid)) != (struct passwd *)NULL) {
	strncpy(buff, pw->pw_name, USERLEN);
	return(OK);
  }
  else 
	return(FAILED);
}


/*
 * h e l p
 *
 * Display a list of ~ escapes
 *
 */
void help()
{
  fprintf(stderr, "~.                   quit cu\r\n");
  fprintf(stderr, "~?                   list escapes\r\n");
  fprintf(stderr, "~!                   escape to a shell\r\n");
  fprintf(stderr, "~!cmd                run cmd here, put output here\r\n");                        
  fprintf(stderr, "~$cmd                run cmd here, put output there\r\n");
  fprintf(stderr, "~~                   send ~ to remote\r\n");
  fprintf(stderr, "~l                   show line termio structure\r\n");
  fprintf(stderr, "~t                   show terminal termio structure\r\n");
  fprintf(stderr, "~%%>:file             divert received data to file\r\n");
  fprintf(stderr, "~%%>:                 end diversion\r\n");
  fprintf(stderr, "~%%put from [to]      put a file to the remote\r\n");
  fprintf(stderr, "~%%take from [to]     take a file from the remote\r\n");
  fprintf(stderr, "~%%cd [dir]           change local directory\r\n");
  fprintf(stderr, "~%%break              send BREAK to remote\r\n");
  fprintf(stderr, "~%%debug              toggle debugging\r\n");
}


/*
 * m a i n
 *
 * Main program
 */
int main(argc, argv)
int argc ; char *argv[] ;
{
  char c, *cp, buff[132];
  char g_dev[32], g_site[32], g_baud[32], g_phone[32];
  int j, pipefd[2];
  uid_t tuid;

  /* get local site name, local user, pid, and home directory */
  if (getname(locsite) == FAILED) {
	fprintf(stderr, "cu: cannot find local uucpname\n");
	exit(1);
  }
  if (getuser(getuid(), locuser) == FAILED) {
	fprintf(stderr, "cu: cannot find own username\n");
	exit(1);
  }
  if ((pid = getpid()) < 2) {
	fprintf(stderr, "cu: cannot find own pid\n");
	exit(1);
  }
  if (gethdir(locuser, homedir) == FAILED) {
	fprintf(stderr, "cu: cannot find own HOME\n");
	exit(1);
  }

  (void) umask(022);			/* set protection mask to rw-r--r-- */
  (void) time(&now);			/* get the current unix time */

  if ((cp = strrchr(argv[0], '/')) == (char *)NULL) cp = argv[0];
  strncpy(progname, cp, 19);		/* program name */
  strncpy(g_baud, DEFSPD, 31);
  g_dev[0] = g_phone[0] = '\0';
  bitmask = 0xff;
  baudrate = parity = bits = -1;	/* use current values */
  devopen = deverror = devclose = empty;
  dbglvl = 0;

  /* parse arguments */
  opterr = 0;
  while ((j = getopt(argc, argv, "b:cehl:os:tx:")) != EOF) {
	switch (j & 0177) {
	case 'b':			/* use given number of bits */
		if (opt_b) wrapup("too many character sizes", "");
		(void) checkparam(optarg);
		bitmask = (2 ^ bits) - 1;
		opt_b = TRUE;
		break;
	case 'c':			/* strip high bit */
		bitmask = 0x7f;
		break;
	case 'e':			/* use even parity */
		if (opt_e || opt_o) wrapup("too many parities", "");
		(void) checkparam("even");
		opt_e = TRUE;
		break;
	case 'h':			/* half duplex (local echo) ? */
		opt_h = TRUE;
		break;
	case 'l':			/* use given device */
		if (opt_l) wrapup("too many devices", "");
		if (optarg[0] == '/') {	/* need device name not path */
			cp = strrchr(optarg, '/');
			strncpy(g_dev, ++cp, 31);
		}
		else
			strncpy(g_dev, optarg, 31);
		opt_l = TRUE;
		break;
	case 'o':			/* use odd parity */
		if (opt_e || opt_o) wrapup("too many parities", "");
		(void) checkparam("odd");
		opt_o = TRUE;
		break;
	case 's':			/* use given speed */
		if (opt_s) wrapup("too many speeds", "");
		strncpy(g_baud, optarg, 31);
		(void) checkparam(g_baud);
		opt_s = TRUE;
		break;
	case 't':			/* convert CR to CRLF ? */
		opt_t = TRUE;
		break;
	case 'x':			/* set debugging level */
		dbglvl = atoi(optarg);
		opt_x = TRUE;
		break;
	case '?':
	default:
		usage();
		exit(1);
		break;
	}
  }

  /* validity checks */
#ifdef IDCHK
  /* some jobs can be run only by the superuser or by the uucp administrator */
  if ((tuid = get_uid(UUCPADM)) == (uid_t) ~0) tuid = 0;
  if (getuid() > MAXUID && getuid() != tuid) dbglvl = 0;
#endif

  /* get a string which may be a system or a phone number */
  if (optind < argc - 1) {
	fprintf(stderr, "cu: too many arguments\n");
	exit(1);
  }
  else if (optind < argc) {
	strncpy(buff, argv[optind], 131);
	if (checkname(buff) == OK) {
		strncpy(g_site, buff, SITELEN);
		printmsg(M_CALL, "cu: checkname returned OK\n");
	}
	else if (isalpha(buff[0])) {
		if (checkcode(buff) == OK) {
			strncpy(g_phone, buff, 31);
			printmsg(M_CALL, "cu: checkcode returned OK\n");
		}
		else {
			fprintf(stderr, "cu: invalid argument %s\n", buff);
			exit(1);
		}
	}
	else
		/* ### testing for digits is probably too limiting */
		strncpy(g_phone, buff, 31);
  }

  /* record the initial terminal state */
  (void) ioctl(0, TTYGET, &stdtty);

  /* set up to catch signals */
  if (catchsignal() == FAILED)
	fprintf(stderr, "cu: cannot catch signals\n");

  /* try to connect to a system */
  if (g_site[0] != '\0') {
	printmsg(M_CALL, "cu: connecting to %s\n", g_site);
	lsysindex = 0;
	while ((c = getsystem(g_site)) == 'C') {
		if (checktime(Lcctime) == OK && connect(g_dev) == OK)
			break;
	}
	if (c != 'C')
		wrapup("cannot connect to site ", g_site);
  }
  /* try to dial a number */
  else if (g_phone[0] != '\0') {
	printmsg(M_CALL, "cu: calling %s\n", g_phone);
	Lsite[0] = '\0';
	Lcctime[0] = '\0';
	strcpy(Ldevice, "ACU");
	strcpy(Lbaud, g_baud); 
	strcpy(Lphone, g_phone);
	if (connect(g_dev) == FAILED)
		wrapup("cannot call number ", g_phone);
  }
  /* just talk to the trees... */
  else if (g_dev[0] != '\0') {
	printmsg(M_CALL, "cu: opening device %s\n", g_dev);
	Lsite[0] = '\0';
	Lcctime[0] = '\0';
	strncpy(Ldevice, "Direct", 19);
	strncpy(Lbaud, g_baud, 19); 
	strncpy(Lphone, g_phone, 19);
	if (connect(g_dev) == FAILED)
		wrapup("cannot open ", g_dev);
  }
  else
	wrapup("no device given", "");

  /* RAW mode on stdin, others parameters as current */
  (void) set_mode(0, -1, -1, -1, &stdtty);

  /* Main body of the terminal simulator. */
  if (pipe(pipefd) < 0)
	wrapup("cannot create pipe", "");

  /* Piped stdin to tty */
  switch ((writepid = fork())) {
	case -1:
		wrapup("cannot create process to write to comm device", "");
	case 0:
		(void) close(pipefd[1]);
		/* only the parent closes the device */
		devclose = empty;
		/* copy runs forever in the child process */
		copy(pipefd[0], "piped stdin", commfd, g_dev);
  }
  (void) close(pipefd[0]);

  if (dbglvl > 0) {
	fprintf(stderr, "Sleeping\r\n");
	(void) sleep(3);
	fprintf(stderr, "Commfd parameters:\r\n");
	(void) showio(commfd);
	fprintf(stderr, "Stdin parameters:\r\n");
	(void) showio(0);
  }

  /* Pipe tty to stdout */
  switch ((readpid = fork())) {
	case -1:
		wrapup("cannot create process to read from comm device", "");
	case 0:
		/* only the parent closes the device */
		devclose = empty;
		/* t_copy runs forever in the child process */
		t_copy(commfd, g_dev, 1, "stdout", takeseq);
  }

  /* Pipe stdin to pipe */
  /* s_copy runs forever in this process */
  s_copy(0, "stdin", pipefd[1], "redirect stdin", endseq);

  exit(0);
  /* NOTREACHED */
}


/*
 * o n t i m e
 *
 * Catch SIGALRM
 *
 * Return:	nothing
 */
void ontime()
{
  (void) signal(SIGALRM, ontime);
}


/*
 * p r i n t m s g
 *
 * Print an error or debugging message onto the console.
 *
 * Return:	Nothing
 *
 * All messages at levels less than or equal to the current debug level
 * are printed, unless the M_NOLOG level is used.
 */
/* VARARGS1 */
void printmsg(level, fmt, a1, a2, a3, a4, a5, a6, a7, a8)
int level; char *fmt, *a1, *a2, *a3, *a4, *a5, *a6, *a7, *a8;
{
  char msg[BUFSIZ];

  if (level <= dbglvl) {
	sprintf(msg, fmt, a1, a2, a3, a4, a5, a6, a7, a8);
	if (msg[strlen(msg) - 1] != '\n') strcat(msg, "\r\n");
	if (level != M_NOLOG) {
		fprintf(stderr, "%s", msg);
		(void) fflush(stderr);
	}
  }
}


/*
 * p u t f i l e
 *
 * Put a file to a remote machine
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		Aborts on write failure to stdout
 */
int putfile(fdout, nameout, command)
int fdout; char *nameout, *command;
{
  char fromfile[132], tofile[132], buff[CHUNK];
  int j, fdin;
  size_t sz;

  if ((j = sscanf(command, "%*s %s %s", fromfile, tofile)) == 0)
	return(FAILED);
  else if (j == 1)
	strncpy(tofile, fromfile, 131);

  printmsg(M_STD, "putfile: Args: %d, From: %s, To: %s",
			j, fromfile, tofile);

  /* try to open the input file */
  if ((fdin = open(fromfile, 0)) == -1) {
	printmsg(M_STD, "putfile: input file open failure");
	return(FAILED);
  }

  /* send the start-up string */
  sprintf(buff, "stty -echo ; cat - > %s ; stty echo\r\n", tofile);
  sz = strlen(buff);
  if (write(fdout, buff, sz) != sz)
	wrapup("cannot write to ", nameout);

  (void) sync();
  (void) sleep(1);

  characters = lines = 0;
  while (TRUE) {
	/* read the data */
	if ((count = read(fdin, buff, CHUNK - 1)) == -1) {
		(void) write(2, progname, strlen(progname));
		(void) write(2, ": cannot read from ", 19);
		(void) write(2, fromfile, strlen(fromfile));
		(void) write(2, "\r\n", 2);
		count = 0;
		cleanup(2);
	}
	else
		printmsg(M_CALL, "putfile: %d bytes read from file", count);

	/* count the characters */
	for (j = 0 ; j < count ; j++) {
		characters++;
		if (buff[j] == '\n')
			lines++;
	}
	/* finish up if necessary */
	if (count == 0) {
		(void) close(fdin);
		if (write(fdout, "\004\004", 1) != 1)
			wrapup("cannot write to ", nameout);
  		else {
			fprintf(stderr, "%d lines / %d characters\r\n",
						lines, characters);
		}
		return(OK);
	}
	/* write the data */
	if (write(fdout, buff, (unsigned int)count) != count)
		wrapup("cannot write to ", nameout);
  }
}


/*
 * s _ c o p y
 *
 * Copy between file descriptors
 *
 * Return:	Does not return
 *		Aborts on failure of read/write to input/output
 *
 * Copy from the screen to the pipe process, allowing for the special
 * character sequences of the ~ commands.  Strip high bit if required.
 */
void s_copy(in, inname, out, outname, end)
int in, out; char *inname, *outname, *end;
{
  static char buff[CHUNK + 5];
  char *buffp;
  char *buffend, *ep = end;

  while (TRUE) {
	/* read the data */
	if ((count = read(in, buff, CHUNK)) <= 0)
		wrapup("scopy cannot read from ", inname);

	/* check the data */
	buffend = buff + count;
	for (buffp = buff; buffp < buffend; ++buffp) {
		*buffp &= bitmask;
		if (*ep != *buffp)
			ep = end;
		else
			ep++;
		if (*ep == '\0') {
			/* get a command from the user */
			if (get_line(in, buff, CHUNK) == -1)
				wrapup("cannot read from ", inname);
			switch (buff[0]) {
			case '.':
				cleanup(2);
				break;
			case '?':
				help();
				break;
			case '!':
				if (buff[1] == '\0') {
					(void) ex("sh", 0, 1);
				}
				else {
					(void) ex(buff + 1, 0, 1);
				}
				break;
			case '$':
				(void) ex(buff + 1, 0, commfd);
				break;
			case '%':
				/* change directory */
				if (strncmp(buff, "%cd", 3) == SAME) {
					if (strlen(buff) == 3) {
						(void) chdir(homedir);
						(void) changedir(out,
							outname, homedir);
					}
					else if (buff[3] == ' ') {
						(void) chdir(buff + 4);
						(void) changedir(out,
							outname, buff + 4);
					}
					else {
						(void) chdir(buff + 3);
						(void) changedir(out,
							outname, buff + 3);
					}
				}
				/* open or close a diversion file */
				else if (strncmp(buff, "%>:", 3) == SAME) {
					if (buff[3] == ' ')
						(void) divert(out,
							outname, buff + 4);
					else
						(void) divert(out,
							outname, buff + 3);
				}
				/* send a BREAK to the remote */
				else if (strncmp(buff, "%break", 6) == SAME) {
					(void) sendbreak(0);
				}
				/* change diagnostics output level */
				else if (strncmp(buff, "%debug", 6) == SAME) {
					if (dbglvl == 0)
						dbglvl = 1;
					else
						dbglvl = 0;
				}
				/* put a file */
				else if (strncmp(buff, "%put", 4) == SAME) {
					(void) putfile(out, outname, buff);
				}
				/* set up to take a file */
				else if (strncmp(buff, "%take", 5) == SAME) {
					(void) takefile(out, outname, buff);
				}
				break;
			case '~':
				/* send a ~ to the remote */
				if (write(out, "~", 1) != 1)
					wrapup("cannot write to ", outname);
				break;
			case 'l':
				/* show line termio structure */
				(void) showio(commfd);
				break;
			case 't':
				/* show terminal termio structure */
				(void) showio(in);
				break;
			default:
				/* ignore it */
				break;
			}
			count = 0;
		}
	}

	/* write the data */
	if (write(out, buff, (unsigned int)count) != count)
		wrapup("scopy cannot write to ", outname);
  }
}


/*
 * s e n d b r e a k
 *
 * Send a BREAK, or pretend to.
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int sendbreak(num)
int num;
{
  int j;
  struct TTY tty;

  /* sort out invalid arguments */
  if (num <= 0 || num > 20) num = 8;

#ifdef _MINIX
  /* park the settings */
  if (ioctl(commfd, TTYGET, &tty) == -1) return(FAILED);

  /* drop the speed */
  j = tty.sg_ospeed;
  tty.sg_ospeed = B110;
  if (ioctl(commfd, TTYSET, &tty) == -1) return(FAILED);

  /* write a bunch of nulls */
  (void) write(commfd, "\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0\0",
		(unsigned int)num);

  /* revert the speed */
  tty.sg_ospeed = j;
  if (ioctl(commfd, TTYSET, &tty) == -1) return(FAILED);
#endif

#ifdef _SVR2
  if (ioctl(commfd, TCSBRK, 0) == -1) return(FAILED);
#endif

  /* say what we've done */
  (void) write(commfd, "@", 1);

  return(OK);
}


/*
 * s e t _ m o d e
 *
 * Set an fd to RAW mode, leaving other modes untouched
 *
 * Return:	OK	Always
 */
int set_mode(fd, s_speed, s_parity, s_bits, sgsavep)
int fd; int s_speed, s_parity, s_bits; struct TTY *sgsavep;
{
  /* Set open file fd to RAW mode with the given other modes. If fd is
   * not a tty, this may do nothing but connecting ordinary files as
   * ttys may have some use. */

  int tabs;
  struct TTY tty;

  tty = *sgsavep;

#ifdef _MINIX
  tabs = tty.sg_flags & XTABS;
  if (s_speed == -1) s_speed = tty.sg_ispeed;
  if (s_parity == -1) s_parity = tty.sg_flags & (EVENP | ODDP);
  if (s_bits == -1)
	s_bits = tty.sg_flags & BITS8;	/* BITS8 is actually a mask */

  printmsg(M_CALL,
	"set_mode: tabs = %d, parity = %d, bits = %d, sg_flags & BITS8 = %d",
	tabs, s_parity, s_bits, tty.sg_flags & BITS8);

  /* set the things we need to set */
  tty.sg_ispeed = s_speed;
  tty.sg_ospeed = s_speed;

  /* and set RAW mode */
  tty.sg_flags = RAW | s_parity | s_bits | tabs;
#endif

#ifdef _SVR2
  if (s_speed == -1) s_speed = tty.c_cflag & CBAUD;
  if (s_parity == -1) s_parity = tty.c_cflag & (PARENB | PARODD);
  if (s_bits == -1) s_bits = tty.c_cflag & CSIZE;

  printmsg(M_CALL,
	"set_mode: parity = %d, bits = %d, c_flag & CSIZE = %d",
	s_parity, s_bits, tty.c_cflag & CSIZE);

  /* set the things we need to set */
  tty.c_cflag = (tty.c_cflag & ~CBAUD) | s_speed;
  tty.c_cflag = (tty.c_cflag & ~(PARENB | PARODD)) | s_parity;
  tty.c_cflag = (tty.c_cflag & ~CSIZE) | s_bits;

  /* and set RAW mode */
  tty.c_iflag &= ~(IGNBRK | IGNCR | INLCR | ICRNL | IUCLC |
			IXANY | IXON | IXOFF | INPCK | ISTRIP);
  tty.c_iflag |= (BRKINT | IGNPAR);
  tty.c_oflag &= ~OPOST;
  tty.c_lflag &= ~(ICANON | ISIG | ECHO);
#if 0
  tty.c_cflag &= ~(CSIZE | PARENB);	/* ### is this flexibility needed ? */
  tty.c_cflag |= (CS8 | CREAD);
#endif
  tty.c_cflag |= CREAD;
  tty.c_cc[4] = 1;
  tty.c_cc[5] = 5;
#endif

  (void) ioctl(fd, TTYSET, &tty);

  return(OK);
}


/*
 * s e t s i g n a l
 *
 * Set up a catcher for a signal if it is not already being ignored
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int setsignal(sig, func)
int sig; sig_t func;
{
  sig_t ret;

  ret = signal(sig, SIG_IGN);

  if (ret == BADSIG)		/* no dealing with this failure */
	return(FAILED);
  else if (ret == SIG_IGN)	/* ignore ignored signals */
	return(OK);
  else if (ret == SIG_DFL) {	/* set catcher for defaulted signals */
	if (signal(sig, func) == BADSIG)
		return(FAILED);
  }
  else				/* never happen */
	return(FAILED);

  return(OK);
}


/*
 * s h o w i o
 *
 * Show the termio paramters for an fd
 *
 * Return:	OK	Always
 */
int showio(fd)
int fd;
{
  char inspeed[20], outspeed[20], flags[132];
  int j;
  struct TTY tty;

#ifdef _MINIX
  struct tchars tchar;

  /* get the parameters */

  (void) ioctl(fd, TIOCGETP, &tty); 
  (void) ioctl(fd, TIOCGETC, &tchar);

  /* show the tty parameters */
  j = 0;
  while (params[j].type != BAD) {
	if (params[j].type == SPEED && params[j].value == tty.sg_ispeed) {
		strncpy(inspeed, params[j].pattern, 15);
		break;
	}
	j++;
  }
  if (params[j].type == BAD)
	strcpy(inspeed, "Unknown speed");
  j = 0;
  while (params[j].type != BAD) {
	if (params[j].type == SPEED && params[j].value == tty.sg_ospeed) {
		strncpy(outspeed, params[j].pattern, 15);
		break;
	}
	j++;
  }
  if (params[j].type == BAD)
	strcpy(outspeed, "Unknown speed");

  flags[0] = '\0';
  /* XTABS is a mask */
  if ((tty.sg_flags & XTABS) == XTABS) strcat(flags, " XTABS");
  /* BITS8 is a mask */
  if ((tty.sg_flags & BITS8) == BITS8) strcat(flags, " BITS8");
  else if (tty.sg_flags & BITS7) strcat(flags, " BITS7");
  else if (tty.sg_flags & BITS6) strcat(flags, " BITS6");
  else if (tty.sg_flags & BITS5) strcat(flags, " BITS5");
  if (tty.sg_flags & EVENP) strcat(flags, " EVENP");
  if (tty.sg_flags & ODDP) strcat(flags, " ODDP");
  if (tty.sg_flags & RAW) strcat(flags, " RAW");
  if (tty.sg_flags & CRMOD) strcat(flags, " CRMOD");
  if (tty.sg_flags & ECHO) strcat(flags, " ECHO");
  if (tty.sg_flags & CBREAK) strcat(flags, " CBREAK");
  if (tty.sg_flags & COOKED) strcat(flags, " COOKED");
  if (tty.sg_flags & DCD) strcat(flags, " DCD");
  if (flags[0] == '\0') strcpy(flags, " none");

  fprintf(stderr, "Input speed         %s baud\r\n", inspeed);
  fprintf(stderr, "Output speed        %s baud\r\n", outspeed);
  fprintf(stderr, "Erase character     %3d decimal\r\n", tty.sg_erase & 0377);
  fprintf(stderr, "Kill character      %3d decimal\r\n", tty.sg_kill & 0377);
  fprintf(stderr, "Mode flags          %s\r\n", flags);

  /* show the tchars parameters */

  fprintf(stderr, "SIGINT character    %3d decimal\r\n", tchar.t_intrc & 0377);
  fprintf(stderr, "SIGQUIT character   %3d decimal\r\n", tchar.t_quitc & 0377);
  fprintf(stderr, "Start character     %3d decimal\r\n", tchar.t_startc & 0377);
  fprintf(stderr, "Stop character      %3d decimal\r\n", tchar.t_stopc & 0377);
  fprintf(stderr, "EOF character       %3d decimal\r\n", tchar.t_eofc & 0377);
  fprintf(stderr, "Delimiter character %3d decimal\r\n", tchar.t_brkc & 0377);
#endif

#ifdef _SVR2
  (void) ioctl(fd, TCGETA, &tty); 

  flags[0] = '\0';
  if (tty.c_iflag & IGNBRK) strcat(flags, " IGNBRK");
  if (tty.c_iflag & BRKINT) strcat(flags, " BRKINT");
  if (tty.c_iflag & IGNPAR) strcat(flags, " IGNPAR");
  if (tty.c_iflag & PARMRK) strcat(flags, " PARMRK");
  if (tty.c_iflag & INPCK) strcat(flags, " INPCK");
  if (tty.c_iflag & ISTRIP) strcat(flags, " ISTRIP");
  if (tty.c_iflag & INLCR) strcat(flags, " INLCR");
  if (tty.c_iflag & IGNCR) strcat(flags, " IGNCR");
  if (tty.c_iflag & ICRNL) strcat(flags, " ICRNL");
  if (tty.c_iflag & IUCLC) strcat(flags, " IUCLC");
  if (tty.c_iflag & IXON) strcat(flags, " IXON");
  if (tty.c_iflag & IXANY) strcat(flags, " IXANY");
  if (tty.c_iflag & IXOFF) strcat(flags, " IXOFF");
  if (flags[0] == '\0') strcpy(flags, " none");
  fprintf(stderr, "Input mode flags    %s\r\n", flags);

  flags[0] = '\0';
  if (tty.c_oflag & OPOST) {
	strcat(flags, " OPOST");
	if (tty.c_oflag & OLCUC) strcat(flags, " OLCUC");
	if (tty.c_oflag & ONLCR) strcat(flags, " ONLCR");
	if (tty.c_oflag & OCRNL) strcat(flags, " OCRNL");
	if (tty.c_oflag & ONOCR) strcat(flags, " ONOCR");
	if (tty.c_oflag & ONLRET) strcat(flags, " ONLRET");
	if (tty.c_oflag & OFILL) strcat(flags, " OFILL");
	if (tty.c_oflag & OFDEL) strcat(flags, " OFDEL");
	if ((tty.c_oflag & NLDLY) == NL0) strcat(flags, " NL0");
	if ((tty.c_oflag & NLDLY) == NL1) strcat(flags, " NL1");
	if ((tty.c_oflag & CRDLY) == CR0) strcat(flags, " CR0");
	if ((tty.c_oflag & CRDLY) == CR1) strcat(flags, " CR1");
	if ((tty.c_oflag & CRDLY) == CR2) strcat(flags, " CR2");
	if ((tty.c_oflag & CRDLY) == CR3) strcat(flags, " CR3");
	if ((tty.c_oflag & TABDLY) == TAB0) strcat(flags, " TAB0");
	if ((tty.c_oflag & TABDLY) == TAB1) strcat(flags, " TAB1");
	if ((tty.c_oflag & TABDLY) == TAB2) strcat(flags, " TAB2");
	if ((tty.c_oflag & TABDLY) == TAB3) strcat(flags, " TAB3");
	if ((tty.c_oflag & BSDLY) == BS0) strcat(flags, " BS0");
	if ((tty.c_oflag & BSDLY) == BS1) strcat(flags, " BS1");
	if ((tty.c_oflag & VTDLY) == VT0) strcat(flags, " VT0");
	if ((tty.c_oflag & VTDLY) == VT1) strcat(flags, " VT1");
	if ((tty.c_oflag & FFDLY) == FF0) strcat(flags, " FF0");
	if ((tty.c_oflag & FFDLY) == FF1) strcat(flags, " FF1");
  }
  else
	strcat(flags, " ~OPOST");
  fprintf(stderr, "Output mode flags   %s\r\n", flags);

  flags[0] = '\0';
  if ((tty.c_cflag & CBAUD) == B0) strcat(flags, " BO");
  if ((tty.c_cflag & CBAUD) == B50) strcat(flags, " B50");
  if ((tty.c_cflag & CBAUD) == B75) strcat(flags, " B75");
  if ((tty.c_cflag & CBAUD) == B110) strcat(flags, " B110");
  if ((tty.c_cflag & CBAUD) == B134) strcat(flags, " B134");
  if ((tty.c_cflag & CBAUD) == B150) strcat(flags, " B150");
  if ((tty.c_cflag & CBAUD) == B200) strcat(flags, " B200");
  if ((tty.c_cflag & CBAUD) == B300) strcat(flags, " B300");
  if ((tty.c_cflag & CBAUD) == B600) strcat(flags, " B600");
  if ((tty.c_cflag & CBAUD) == B1200) strcat(flags, " B1200");
  if ((tty.c_cflag & CBAUD) == B1800) strcat(flags, " B1800");
  if ((tty.c_cflag & CBAUD) == B2400) strcat(flags, " B2400");
  if ((tty.c_cflag & CBAUD) == B4800) strcat(flags, " B4800");
  if ((tty.c_cflag & CBAUD) == B9600) strcat(flags, " B9600");
  if ((tty.c_cflag & CBAUD) == B19200) strcat(flags, " B19200");
  if ((tty.c_cflag & CBAUD) == EXTB) strcat(flags, " EXTB");
  if ((tty.c_cflag & CSIZE) == CS5) strcat(flags, " CS5");
  if ((tty.c_cflag & CSIZE) == CS6) strcat(flags, " CS6");
  if ((tty.c_cflag & CSIZE) == CS7) strcat(flags, " CS7");
  if ((tty.c_cflag & CSIZE) == CS8) strcat(flags, " CS8");
  if (tty.c_cflag & CSTOPB) strcat(flags, " CSTOPB");
  if (tty.c_cflag & CREAD) strcat(flags, " CREAD");
  if (tty.c_cflag & PARENB) strcat(flags, " PARENB");
  if (tty.c_cflag & PARODD) strcat(flags, " PARODD");
  if (tty.c_cflag & HUPCL) strcat(flags, " HUPCL");
  if (tty.c_cflag & CLOCAL) strcat(flags, " CLOCAL");
  if (tty.c_cflag & CTSCD) strcat(flags, " CTSCD");
  if (tty.c_cflag & HDX) strcat(flags, " HDX");
  if (flags[0] == '\0') strcpy(flags, " none");
  fprintf(stderr, "Control mode flags  %s\r\n", flags);

  flags[0] = '\0';
  if (tty.c_lflag & ISIG) strcat(flags, " ISIG");
  if (tty.c_lflag & ICANON) strcat(flags, " ICANON");
  if (tty.c_lflag & XCASE) strcat(flags, " XCASE");
  if (tty.c_lflag & ECHO) strcat(flags, " ECHO");
  if (tty.c_lflag & ECHOE) strcat(flags, " ECHOE");
  if (tty.c_lflag & ECHOK) strcat(flags, " ECHOK");
  if (tty.c_lflag & ECHONL) strcat(flags, " ECHONL");
  if (tty.c_lflag & NOFLSH) strcat(flags, " NOFLSH");
  if (flags[0] == '\0') strcpy(flags, " none");
  fprintf(stderr, "Local mode flags    %s\r\n", flags);

  fprintf(stderr, "SIGINT character    %3d decimal\r\n", tty.c_cc[0] & 0377);
  fprintf(stderr, "SIGQUIT character   %3d decimal\r\n", tty.c_cc[1] & 0377);
  fprintf(stderr, "Start character     %3d decimal\r\n", tty.c_cc[2] & 0377);
  fprintf(stderr, "Stop character      %3d decimal\r\n", tty.c_cc[3] & 0377);
  fprintf(stderr, "EOF character       %3d decimal\r\n", tty.c_cc[4] & 0377);
  fprintf(stderr, "Delimiter character %3d decimal\r\n", tty.c_cc[5] & 0377);
#endif

  return(OK);
}


/*
 * t a k e f i l e
 *
 * Set up a file to take from a remote machine
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 *		Aborts on write failure to stdout
 */
int takefile(fdout, nameout, command)
int fdout; char *nameout, *command;
{
  char fromfile[132], tofile[132], buff[CHUNK];
  int j;
  size_t sz;

  if ((j = sscanf(command, "%*s %s %s", fromfile, tofile)) == 0)
	return(FAILED);
  else if (j == 1)
	strncpy(tofile, fromfile, 131);

  printmsg(M_CALL, "takefile: assigned: %d, From: %s, To: %s",
			j, fromfile, tofile);

  /* send the start-up string:  ### use mesg n and mesg y ? */
  sprintf(buff,
  "stty -echo ; echo \'~>:\'%s ; cat %s ; echo \'~>:\' ; stty echo\r\n",
		tofile, fromfile);
 
  sz = strlen(buff);
  if (write(fdout, buff, sz) != sz)
	wrapup("cannot write to ", nameout);

  (void) sync();
  (void) sleep(1);

  return(OK);
}


/*
 * t _ c o p y
 *
 * Copy between file descriptors
 *
 * Return:	Does not return
 *		Aborts on failure of read/write to input/output
 *
 * Copy from the tty to the screen, allowing for the special character
 * sequence used by %commands.  Strip high bit if required.
 */
void t_copy(in, inname, out, outname, end)
int in, out; char *inname, *outname, *end;
{
  static char buff[CHUNK + 5];
  char *buffp;
  char tfile[132], *buffend, *ep;
  int j, takeflg;

  takeflg = FALSE;
  ep = end;
  while (TRUE) {
	/* read the data */
	if ((count = read(in, buff, CHUNK)) <= 0)
		wrapup("tcopy cannot read from ", inname);

	/* check the data */
	buffend = buff + count;
	for (buffp = buff; buffp < buffend; ++buffp) {

		/* if required, strip parity */
		*buffp &= bitmask;
		/* lose garbage in the input */
		if (*buffp < 32 && *buffp != '\r' && *buffp != '\n'
				&& *buffp != '\t' && *buffp != '\b')
			*buffp = 32;

		/* match the start/end sequence */
		if (*ep != *buffp)
			ep = end;
		else
			ep++;
		/* special sequence received - change state */
		if (*ep == '\0') {
			ep = end;
			/* ignore simple echoes of command string */
			/* if the next char is \', it's an echo */
			if (buffp == buffend - 1) {	/* need a character */
				if (read(in, buffend, 1) <= 0)
					wrapup("cannot read from ", inname);

				buffend++;
			}
			if (*(buffp + 1) != '\'')	/* it's a real command */
				takeflg = getcmd(in, out, buff,
					buffp, tfile, takeflg, end);
		}
	}
	/* write the data */
	if (takeflg) {
		for (j = 0 ; j < count ; j++)
			if (buff[j] == '\n')
				lines++;
		characters += count;
		if (write(fdt, buff, (unsigned int)count) != count) {
			(void) write(2, progname, strlen(progname));
			(void) write(2, ": cannot write to ", 16);
			(void) write(2, tfile, strlen(tfile));
			(void) write(2, "\r\n", 2);
			takeflg = !takeflg;
		}
	}
	else if (write(out, buff, (unsigned int)count) != count)
			wrapup("tcopy cannot write to ", outname);
  }
}


/*
 * u n l o c k
 *
 * Remove a lockfile, update the lock file list
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 */
int unlock(file)
char *file;
{
  int j, flg;

  j = 0;
  flg = FAILED;
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0' && strcmp(locklist[j], file) == SAME) {
		(void) unlink(locklist[j]);
		locklist[j][0] = '\0';
		flg = OK;
		break;
	}
	j++;
  }

  return(flg);
}


/*
 * u s a g e
 *
 * Usage message
 */
void usage()
{
  fprintf(stderr,
  "Usage: %s [-b bits] [-ceh] [-l line] [-o] [-s speed] [-t] [-x #] [sys|phn]\n",
	progname);
}


/*
 * w r a p u p
 *
 * Print message and die, earthling
 */
void wrapup(s1, s2)
char *s1, *s2;
{
  (void) write(2, progname, strlen(progname));
  (void) write(2, ": ", 2);
  (void) write(2, s1, strlen(s1));
  (void) write(2, s2, strlen(s2));
  (void) write(2, "\r\n", 2);

  (void) cleanup(2);
}

