/* cu - call another unix system
 *
 * Version 1.2 of 31 October 1991.
 *
 * Written by C W Rose, based on Minix 1.5.10 term.c by ast and bde.
 */

#ifdef _MINIX
#include <minix/config.h>
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#ifdef _MINIX
#include <sgtty.h>
#endif
#include <signal.h>
#include <stdio.h>
#ifdef _MINIX
#undef NULL
#include <stdlib.h>
#endif
#include <string.h>
#ifdef _SVR2
#include <termio.h>
#endif
#include <time.h>
#include <unistd.h>
#include "dial.h"
#include "uucp.h"
#include "cu.h"

#ifndef lint
static char *Version = "@(#) cu 1.2 (31 October 1991)";
#endif

/* Global variables */

int opt_b = FALSE;		/* use given number of bits */
int opt_e = FALSE;		/* set even parity */
int opt_h = FALSE;		/* set half-duplex mode (local echo) */
int opt_l = FALSE;		/* use given line */
int opt_o = FALSE;		/* set odd parity */
int opt_s = FALSE;		/* set speed */
int opt_t = FALSE;		/* map CR to CRLF ? */
int opt_x = FALSE;		/* debugging flag */

char locklist[MAXLOCKS][LOCKLEN + 1];	/* lock file list */
char *cmdflds[MAXARGS];		/* pointers to parsed command arguments */
char homedir[132];		/* current user's HOME directory */
char locsite[20];		/* local site name */
char locuser[20];		/* local user name */
char progname[20];		/* program name */
char Lsite[20];			/* site from L.sys */
char Ldevice[20];		/* device type from L.sys */
char Lprotocols[20];		/* device protocols from L.sys */
char Lbaud[20];			/* device baud rate from L.sys */
char Lphone[20];		/* device phone number from L.sys */
char Lcctime[64];		/* clear calling times from L.sys */
char Ldialler[20];		/* dialler from L-devices */
char curtime[64];		/* current time */
char endseq[] = "\r~";		/* local command prefix */
char takeseq[] = "~>:";		/* capture command prefix */

int lsysindex;			/* current position in L.sys file */
int commfd;			/* comm device file descriptor */
int pid;			/* current pid */
int readpid;			/* pid of child reading commfd */
int writepid;			/* pid of child writing commfd */
int bits;			/* bits per char for comms line */
int parity;			/* parity of comms line */
int baudrate;			/* speed of comms line */
int characters;			/* bytes in file */
int lines;			/* lines in file */
int count;			/* characters in i/o buffer */
int bitmask;			/* mask for valid bits */
int fdt;			/* take transfer file descriptor */
int dbglvl;			/* debugging level */
int errno;			/* used by errno.h */

struct TTY stdtty;		/* saved terminal parameters for stdin */

time_t now;			/* current unix time */
CALL call;			/* dial(3) structure */

#ifdef _MINIX
struct Params params[] = {	/* permissible i/o port parameters */
  "5", BITS5, BITS,
  "6", BITS6, BITS,
  "7", BITS7, BITS,
  "8", BITS8, BITS,

  "even", EVENP, PARITY,
  "odd", ODDP, PARITY,
  "strip", 0, STRIP,

  "50", B50, SPEED,
  "75", B75, SPEED,
  "110", B110, SPEED,
  "134", B134, SPEED,
  "200", B200, SPEED,
  "300", B300, SPEED,
  "600", B600, SPEED,
  "1200", B1200, SPEED,
  "1800", B1800, SPEED,
  "2400", B2400, SPEED,
  "3600", B3600, SPEED,
  "4800", B4800, SPEED,
  "7200", B7200, SPEED,
  "9600", B9600, SPEED,
  "19200", B19200, SPEED,
  "EXTA", EXTA, SPEED,
  "EXTB", EXTB, SPEED,
  "38400", B38400, SPEED,
  "57600", B57600, SPEED,
  "115200", B115200, SPEED,
  "", 0, BAD,			/* BAD type to end list */
};
#endif

#ifdef _SVR2
struct Params params[] = {	/* permissible i/o port parameters */
  "5", CS5, BITS,
  "6", CS6, BITS,
  "7", CS7, BITS,
  "8", CS8, BITS,

  "even", PARENB, PARITY,	/* actually parity enable */
  "odd", (PARENB | PARODD), PARITY,

  "50", B50, SPEED,
  "75", B75, SPEED,
  "110", B110, SPEED,
  "134", B134, SPEED,
  "150", B150, SPEED,
  "200", B200, SPEED,
  "300", B300, SPEED,
  "600", B600, SPEED,
  "1200", B1200, SPEED,
  "1800", B1800, SPEED,
  "2400", B2400, SPEED,
  "4800", B4800, SPEED,
  "9600", B9600, SPEED,
  "19200", B19200, SPEED,
  "EXTB", EXTB, SPEED,
  "", 0, BAD,			/* BAD type to end list */
};
#endif

char *devname;			/* access to connected devices */
int (*devopen)();
int (*deverror)();
int (*devclose)();

struct Dev Devlst[] = {		/* set of connected devices */
  "direct", ttyopen, ttyerror, ttyclose,
  "ACU", dial, merror, hangup,
  "PAD", padopen, paderror, padclose,
  (char *)NULL, empty, empty, empty,
};


/*
 * c a t c h s i g n a l
 *
 * Deal with the four main signals
 *
 * Return:	OK	Success
 *		FAILED	Otherwise
 *
 * Since cu is interactive, the four main signals are all caught, not
 * ignored as in the rest of the UUCP suite.
 */
int catchsignal()
{
  if (setsignal(SIGHUP, cleanup) == FAILED || 			/* signal 1 */
		setsignal(SIGINT, cleanup) == FAILED ||		/* signal 2 */
		setsignal(SIGQUIT, cleanup) == FAILED ||	/* signal 3 */
		setsignal(SIGPIPE, cleanup) == FAILED ||	/* signal 13 */
		setsignal(SIGTERM, cleanup) == FAILED)		/* signal 15 */
	return(FAILED);
  else
	return(OK);
}


/*
 * c h e c k l o c k
 *
 * Check/create a lock file
 *
 * Return:	OK	Lock file created
 *		FAILED	Lock file already present
 */
int checklock(name, tries)
char *name; int tries;
{
  char buff[132], lock[132];
  int j, fd, flg;
  size_t sz;

  /* build the lock file name */
  if (*name == '/') {
	strncpy(lock, name, 131);
  }
  else {
	strcpy(lock, LCKPATH);
	strncat(lock, name, 131 - strlen(lock));
  }

  /* check/create LCK file semaphore (open is atomic) */
  for (j = 0 ; j < tries ; j++) {
	if ((fd = open(lock, O_CREAT | O_WRONLY | O_EXCL, 0644)) == -1
				 && errno == EEXIST) {
		flg = FAILED;
		(void) sleep(3);
	}
	else {
		flg = OK;
		sprintf(buff, "%04d %s %s\n", pid, progname, locuser);
		sz = strlen(buff);
		if (write(fd, buff, sz) != sz)
			flg = FAILED;
		if (close(fd) == -1)
			flg = FAILED;
		if (flg == FAILED)
			(void) unlink(lock);
		break;
  	}
  }

  /* Update the lockfile list for cleanup() */
  if (flg == OK) {
	j = 0;
	while (j < MAXLOCKS) {
		if (locklist[j][0] == '\0') {
			strncpy(locklist[j], lock, LOCKLEN);
			break;
		}
		j++;
	}
  }

  return(flg);
}


/*
 * c l e a n u p
 *
 * Remove any outstanding lock files before exiting
 *
 * When a signal arrives, cleanup() is called with a single integer
 * argument that is the number of the caught signal.
 */
void cleanup(sig)
int sig;
{
  int j = 0;

  (void) signal(sig, SIG_IGN);	/* nothing to be done on failure anyway */

  /* close down the dialler */
  (void) (*devclose)(commfd);

  /* restore stdin */
  (void) ioctl(0, TTYSET, &stdtty);

  /* clear any locks */
  while (j < MAXLOCKS) {
	if (locklist[j][0] != '\0') {
		(void) unlink(locklist[j]);
		printmsg(M_CALL, "cleanup: lockfile %s removed", locklist[j]);
	}
	j++;
  }
#if 0
  /* remove any STST file */
  strcpy(scratch, STSTPATH, 131);
  strncat(scratch, rmtsite, 131 - strlen(STSTPATH));
  (void) unlink(scratch);
  printmsg(M_CALL, "cleanup: status file %s removed, scratch);
#endif

  /* kill the other processes */
  (void) signal(SIGINT, SIG_IGN);
  (void) kill(0, SIGINT);

  exit(1);
}


/*
 * c h a n g e d i r
 *
 * Change a directory by reflection through the connected machine
 *
 * Return:	OK		Success
 */
int changedir(fdout, nameout, command)
int fdout; char *nameout, *command;
{
  char newdir[132], buff[CHUNK];
  size_t sz;

  if (strlen(command) > 131)
	return(FAILED);
  strcpy(newdir, command);

  printmsg(M_CALL, "changedir: new directory %s", newdir);

  /* send the start-up string */
  sprintf(buff, "stty -echo ; echo \'~>:\'cd %s ; stty echo\r\n", newdir);

  sz = strlen(buff); 
  if (write(fdout, buff, sz) != sz)
	wrapup("cannot write to ", nameout);

  (void) sync();
  (void) sleep(1);

  return(OK);
}


/*
 * c h e c k c o d e
 *
 * Check a phone number string, include dialcode
 *
 * Return:	OK	Conversion
 *		FAILED	No conversion
 *		code is updated as a side effect
 */
int checkcode(code)
char *code;
{
  char *cp, *bp, ascii[20], digit[20], telnum[20], buff[132];
  int flg;
  FILE *fp;

  /* park the actual digits */
  bp = telnum;
  cp = code;
  while (isalpha(*cp) && (bp - telnum) < 20)
	*bp++ = *cp++;
  *bp = '\0';

  printmsg(M_CALLMSG, "checkcode: code |%s|, number |%s|", telnum, cp);

  /* convert any leading ascii code */
  flg = FAILED;
  if (telnum[0] != '\0') {
	if ((fp = fopen(LDIAL, "r")) != (FILE *)NULL) {
		while (fgets(buff, 132, fp) != (char *)NULL) {
			if (buff[0] == '#' || buff[0] == '\n') continue;
			if (sscanf(buff, "%s %s", ascii, digit) != 2) continue;
			if (strcmp(telnum, ascii) != SAME) continue;
			printmsg(M_CALLMSG, "checkcode: match %s with %s", ascii, digit);
			strncpy(telnum, digit, 19);
			flg = OK;
			break;
		}
		(void) fclose(fp);
	}
  }
  strncat(telnum, cp, 19 - strlen(telnum));
  strncpy(code, telnum, 19);

  return(flg);
}


/*
 * c h e c k n a m e
 *
 * Check if "name" is a known system
 *
 * Return:	FAILED	Cannot open L.sys file
 *		FAILED	Name not recognised
 *		OK	Name recognised
 */
int checkname(name)
char *name;
{
  char cont[132], line[BUFSIZ];
  int j, k;
  FILE *fplsys;

  if ((fplsys = fopen(LSYS, "r")) == (FILE *)NULL) return(FAILED);

  j = strlen(name);
  if (j > SITELEN) j = SITELEN;
  while (fgets(line, BUFSIZ, fplsys) != (char *)NULL) {
	while ((k = strlen(line)) < (BUFSIZ - 132) &&
			line[k - 2] == '\\' &&
			fgets(cont, 131, fplsys) != (char *)NULL) {
		line[k - 2] = '\0';
		strcat(line, cont);
	}
	for (k = 0 ; k < j ; k++) {
		if (line[k] != name[k])
			break;
	}
	if (k == j) {
		(void) fclose(fplsys);
       		return(OK);
	}
  }
  (void) fclose(fplsys);
  return(FAILED);
}


/*
 * c h e c k p a r a m
 *
 * Check that a parameter is valid 
 *
 * Return:	OK	Valid parameter
 *		Abort on failure
 *		Parameter global variables are updated as a side effect
 */
int checkparam(string)
char *string;
{
  struct Params *param;

  /* Check parameter for legality. */
  for (param = &params[0] ; param->type != BAD ; ++param) {
	if (strcmp(string, param->pattern) == SAME) {
		break;
	}
  }
  switch (param->type) {
	case BITS:
		bits = param->value;
		break;
	case PARITY:
		parity = param->value;
		break;
	case SPEED:
		baudrate = param->value;
		if (baudrate == 0) wrapup("invalid speed: ", string);
		break;
	case STRIP:
		bitmask = 0x7f;
		break;
	case BAD:
	default:
		wrapup("invalid parameter: ", string);
		break;
  }

  return(OK);
}


/*
 * c h e c k t i m e
 *
 * Check if we may make a call at this time
 *
 * Return:	OK	Call permitted
 *		FAILED	Call not permitted
 */
int checktime(atime)
char *atime;
{
  char *pd, loctime[20], locday[20], buff[BUFSIZ];
  int j, k, present, first, last, delay, trydelay;
  time_t trystart;
  FILE *fp;

  /* A typical field is day[time][,retry][|...] with no intervening spaces */
  /* between day and time.  In this routine it is null-terminated. */

  /* The day subfield is specified using Any, Never, Wk and */
  /* Su Mo Tu We Th Fr Sa.  The time subfield is two 24hr times */
  /* separated by a dash.  The day subfield must be specified if */
  /* the time subfield is specified.  The retry field is decimal */
  /* minutes.  The '|' separator permits several day-time pairs. */

  /* typical ctime() output is "Wed Jul 08 18:43:07 1987\n\0" */

  sprintf(curtime, "%s", ctime(&now));
  curtime[strlen(curtime) - 1] = '\0';

  printmsg(M_CALLMSG, "checktime: now %ld atime |%s| curtime |%s|",
		now, atime, curtime);

  /* get loctime, locday */
  for (j = 0 ; j < 3 ; j++)
	locday[j] = curtime[j];
  locday[j] = '\0';
  for (j = 11 ; j < 13 ; j++)
	loctime[j - 11] = curtime[j];
  for (j = 14 ; j < 16 ; j++)
	loctime[j - 12] = curtime[j];
  loctime[4] = '\0';
  present = atoi(loctime);

  printmsg(M_CALLMSG, "checktime: locday |%s|, loctime |%s|", locday, loctime);

  /* See if there's an STST file current: the first field is the pid of the
   * process setting the file, the second field is the job status, the third
   * field is the try count, the fourth field is the UNIX time of the last
   * attempt, and the fifth field is the retry interval.
   */
  trystart = 0L;
  trydelay = 0;
#if 0
  strncpy(scratch, STSTPATH, 131);
  strncat(scratch, rmtsite, 131 - strlen(STSTPATH));
  if ((fp = fopen(scratch, "r")) == (FILE *)NULL) {
	if (errno != ENOENT) {
		printmsg(M_STD, "ASSERT ERROR");
		sprintf(errmsg, "SYSTAT OPEN FAIL");
		return(FAILED);
	}
  }
  else if (fgets(buff, BUFSIZ, fp) != (char *)NULL) {
	if (sscanf(buff, "%d %d %d %ld %d", &k, &k, &retrynum, &trystart,
							&trydelay) < 5) {
		trystart = 0L;
		trydelay = 0;
	}
	printmsg(M_CALLMSG,
		"checktime: retry start %ld, interval %d, time now %ld",
		trystart, trydelay, now); 
	(void) fclose(fp);
	retrynum++;
  }
#endif

  pd = atime;		/* points to a null-terminated L.sys time field */
  while (*pd) {		/* cycle over multiple entries */
	/* check the day of the week */
	if (strncmp("Never", pd, 5) == SAME) {
		printmsg(M_CALLMSG, "checktime: Never failure");
		return(FAILED);
	}
  	else if (strncmp("Any", pd, 3) == SAME) {
		pd += 3;
	}
	else if (strncmp("Wk", pd, 2) == SAME) {
		if (strncmp(locday, "Sa", 2) == SAME ||
				strncmp(locday, "Su", 2) == SAME) {
			printmsg(M_CALLMSG, "checktime: Wk failure");
			return(FAILED);
		}
		else
			pd += 2;
	}
  	else if (strncmp(locday, pd, 2) == SAME) {
		while (isalpha(*pd))	/* use up the rest of the week */
			pd++;
	}
	else {
		pd += 2;		/* step over this entry */
		if (isalpha(*pd))
			continue;
		else {			/* look for the next field */
			while (*pd && *pd !='|')
				pd++;
			if (*pd++ == '|')
				continue;
			else {
				printmsg(M_CALLMSG, "checktime: Day failure");
				return(FAILED);
			}
		}
  	}

  	/* check the time */
	if (*pd == '\0') {		/* we have nothing more */
		first = 0;		/* set the full defaults */
		last = 2359;
		delay = 3300;
	}
	else if (*pd == ',') {		/* we have a retry interval */
		pd++;
		first = 0;
		last = 2359;
		delay = atoi(pd);
	}
  	else {				/* we have a time sequence */
		/* sort out actual and default arguments */
		j = sscanf(pd, "%04d-%04d,%d", &first, &last, &delay);
		if (j < 2) {
			printmsg(M_CALLMSG, "checktime: Time failure");
			return(FAILED);	/* garbage in the string */
		}
		else if (j == 2)
			delay= 3300;	/* standard default */
	}

	/* Note that there may be different retry intervals in the L.sys
	 * and in the status files.  We use the L.sys interval for future
	 * updates, and the current STST file for this check.
	 */
#if 0
	retryint = delay;
#endif
	printmsg(M_CALLMSG, "checktime: first %d last %d wait %d retry %d",
			first, last, trydelay, delay);

	if (first < last && present > first && present < last) {
		if (now < (trystart + (long int)trydelay)) {
			printmsg(M_STD, "NO CALL (RETRY TIME NOT REACHED");
			return(FAILED);
		}
		else
			return(OK);
	}
	else {			/* look for the next field */
		while (*pd && *pd != '|')
			pd++;
		if (*pd++ == '|')
			continue;
		else
			return(FAILED);
  	}
  }
  return(FAILED);	/* probably garbage in the time field */
}


/*
 * c o n n e c t
 *
 * Connect to a system, one way or another
 *
 * Return:	OK		Success
 *		FAILED		Otherwise
 */
int connect(dev)
char *dev;
{
  char *flds[MAXARGS], speed[20], number[20], buff[132];
  int j, fd, dflg, flg;
  FILE *fp;
  struct Dev *dp;
  struct TTY tty;

  printmsg(M_CALL, "connect: device |%s|, baud |%s|, phone |%s|",
					Ldevice, Lbaud, Lphone);

  if ((fp = fopen(LDEVS, "r")) == (FILE *)NULL) return(FAILED);

  /* L-devices file has (at least) five fields per line:
   *	Class, Data_port, Dial_port, Speed, Dialler, [Token]
   *      0        1          2        3       4        5
   */

  flg = FAILED;
  while (fgets(buff, 132, fp) != (char *)NULL) {
	/* look for the right class of device */
	if (buff[0] == '#' || buff[0] == '\n')
		continue;		/* comment line */
	j = getargs(buff, flds);
	printmsg(M_CALL, "fields: 1 |%s| 2 |%s| 3 |%s| 4 |%s| 5 |%s| 6 |%s|",
		flds[0], flds[1], flds[2], flds[3], flds[4], flds[5]);

	if (j < 5)
		continue;		/* invalid line */
	else if (j == 5)
		flds[F_DTOKEN] = "\\D";

	if (strcmp(flds[F_DEVICE], Ldevice) != SAME)
		continue;		/* wrong device class */

	/* check the baud rate */
	if (Lbaud[0] != '\0') {
		if (strcmp(flds[F_SPEED], "Any") != SAME &&
				strcmp(flds[F_SPEED], Lbaud) != SAME)
			continue;	/* wrong speed */
		else
			strncpy(speed, Lbaud, 19);
	}
	else {
		if (strcmp(flds[F_SPEED], "Any") == SAME)
			continue;	/* no speed defined */
		else
			strncpy(speed, flds[F_SPEED], 19);
	}

	/* check if a specific device is needed/available */
	if (dev[0] != '\0') {
		if (strcmp(flds[F_DATAPORT], dev) != SAME)
			continue;	/* wrong device */
	}

	/* convert any dialcode abbreviation */
	strncpy(number, Lphone, 19);
	if (strcmp(number, "-") != SAME &&
			strcmp(flds[F_DTOKEN], "\\T") == SAME)
		(void) checkcode(number);

	/* update the dialler type */
	strncpy(Ldialler, flds[F_DTYPE], 19);

	/* get the appropriate open/close routines from the Devlst table */
	dp = Devlst;
	while (dp->Devname != (char *)NULL) {
		if (strcmp(dp->Devname, Ldialler) == SAME) {
			devname = dp->Devname;
			devopen = dp->Devopen;
			deverror = dp->Deverror;
			devclose = dp->Devclose;
			break;
		}
		dp++;
	}
	if (dp->Devname == (char *)NULL) continue;	/* unknown dialler */
	printmsg(M_CALL, "connect: dialler %s\n", Ldialler);
	if (strcmp(Ldialler, "direct") == SAME)
		dflg = TRUE;
	else
		dflg = FALSE;

	/* initialise the tty data structure */
	if (dflg || opt_b || opt_e || opt_o || opt_s) {
		tty = stdtty;			/* reasonable defaults */
		(void) checkparam(speed);	/* aborts on failure */
#ifdef _MINIX
		tty.sg_ispeed = tty.sg_ospeed = baudrate;
		if (opt_b)
			tty.sg_flags &= bits;
		if (opt_o || opt_e)
			tty.sg_flags |= parity;
#endif

#ifdef _SVR2
		tty.c_cflag = (tty.c_cflag & ~CBAUD) | baudrate;
		if (opt_b)
			tty.c_cflag = (tty.c_cflag & ~CSIZE) | bits;
		if (opt_o || opt_e)
			tty.c_cflag = (tty.c_cflag & ~(PARENB | PARODD)) | parity;
#endif
	}

	/* initialise the CALL data structure */
	if (dflg || opt_b || opt_e || opt_o || opt_s)
		call.attr = &tty;
	else
		call.attr = (struct TTY *)NULL;
	call.baud = atoi(Lbaud);
	call.speed = atoi(Lbaud);	/* used by dial(3) */
	if (dflg || dev[0] != '\0')
		call.line = flds[F_DATAPORT];
	else
		call.line = (char *)NULL;
	if (strcmp(number, "-") != SAME)
		call.telno = number;
	else
		call.telno = (char *)NULL;
	call.modem = 0;

	/* try to connect */
#ifdef _BCC
	if ((fd = (*devopen)(&call)) < 0) {
#else
	if ((fd = (*devopen)(call)) < 0) {
#endif
 		if (dbglvl > 0) {
			(void) (*deverror)(fd, buff);
			printmsg(M_CALL, "connect: error %s", buff);
		}
		/* no point in closing what we haven't opened */
		devopen = deverror = devclose = empty;
	}
	else {
		commfd = fd;
		flg = OK;
		break;
	}
  }

#if 0
  if (flg == OK) {
	printmsg(M_STD, "OK (DIAL %s %s %s)", devname,
			call.line == (char *)NULL ? "-" : call.line,
			call.telno == (char *)NULL ? "-" : call.telno);
	(void) clearbuff();
	return('E');
  }
  else {
	if (fd == DV_NT_A)
		printmsg(M_STD, "LOCKED (call to %s)", rmtsite);
	else
		printmsg(M_STD, "FAILED (call to %s)", rmtsite);
	(void) writestatus("DIAL FAILED");
	callstatus |= EC_DFAIL;
	sterflg = TRUE;
	return('B');
  }
#endif
  return(flg);
}


/*
 * c o p y
 *
 * Copy from one open file to another, without changes
 *
 * Return:	Nothing
 *		Aborts on failure
 */
void copy(in, inname, out, outname)
int in, out; char *inname, *outname;
{
  static char buf[CHUNK + 5];
  int j;

  while (TRUE) {
	if ((j = read(in, buf, CHUNK)) <= 0)
		wrapup("copy cannot read from ", inname);
	if (write(out, buf, (unsigned int)j) != j)
		wrapup("copy cannot write to ", outname);
  }
}


/*
 * d i v e r t
 *
 * Divert input to a file
 *
 * Return:	OK		Always
 *		Aborts on write failure to stdout
 */
int divert(fdout, nameout, command)
int fdout; char *nameout, *command;
{
  char tofile[132], buff[CHUNK];
  size_t sz;

  strncpy(tofile, command, 131);

  printmsg(M_CALL, "divert: diverting to: |%s|", tofile);

  /* send the start-up/shutdown string:  ### use mesg n and mesg y ? */
  if (strlen(tofile) != 0)
	sprintf(buff, "stty -echo ; echo \'~>:\'%s\r\n", tofile);
  else
	sprintf(buff, "echo \'~>:\' ; stty echo\r\n");

  sz = strlen(buff);
  if (write(fdout, buff, sz) != sz)
	wrapup("cannot write to ", nameout);

  (void) sync();
  (void) sleep(1);

  return(OK);
}


/*
 * e m p t y
 *
 * Do nothing (gracefully)
 *
 * Return:	0	Always
 */
int empty()
{
  printmsg(M_CALL, "empty: called");
  return(0);
}


/*
 * e x
 *
 * Run a shell from within a program, either one-shot or interactive
 *
 * Return:	Status of shell process		Always
 *		Abort on failure of read/write
 */
int ex(arg, fdin, fdout)
char *arg; int fdin, fdout;
{
  int j, k, status;
  struct TTY tty;

  printmsg(M_CALL, "ex: program %s, input fd, %d output fd %d", arg, fdin, fdout);

  if (fdin != commfd) {
	(void) ioctl(fdin, TTYGET, &tty);
	(void) ioctl(fdin, TTYSET, &stdtty);
  }
 
  /* run an interactive shell */
  if (strcmp(arg, "sh") == SAME) {
	switch (j = fork()) {			/* fork, dup and exec */
		case -1:			/* error */
			wrapup("fork failed", "");
			break;
		case 0:				/* pregnant */
			/* become who we really are */
			(void) setgid(getgid());
			(void) setuid(getuid());
			(void) execlp(SHELL, "sh", "-i", (char *)0);
			wrapup("exec failed", "");	/* never happen */
			break;
	}
	
	/* if we're here, we're a parent */
  }
  else {
	/* run a command */
	switch (j = fork()) {			/* fork, dup and exec */
		case -1:			/* error */
			wrapup("fork failed", "");
			break;
		case 0:				/* pregnant */
			if (fdin != 0 && (close(0) == -1 || dup(fdin) != 0))
				wrapup("input close or dup failed", "");
			if (fdout != 1 && (close(1) == -1 || dup(fdout) != 1))
				wrapup("output close or dup failed", "");

			/* become who we really are */
			(void) setgid(getgid());
			(void) setuid(getuid());
			(void) execlp(SHELL, "sh", "-c", arg, (char *)0);
			wrapup("exec failed", "");	/* never happen */
			break;
	}
	/* if we're here, we're a parent */
  }

  while ((k = wait(&status)) != j && k != -1)
	;

  if (fdin != commfd)
	(void) ioctl(fdin, TTYSET, &tty);

  printmsg(M_CALL, "ex: exec fork %d, wait %d, status %xx", j, k, status);

  return(status);
}

