#include <sys/types.h>
#include <sgtty.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>
#include "modemcap.h"
#include "dial.h"

#define FAILED -1
typedef void (*sig_t)();

static	int	gotlock = 0;
static	char	lockfile[132];

extern	int	errno;
static	int	modemfd = -1;			/* modem file descriptor */


static int findline(callp, tty, brand)
CALL *callp;
char *tty;
char *brand;
{
  int exists = 0;			/* device exists at some baud rate */
  int tried = 0;			/* found a device but it was locked */
  char l_class[DVC_LEN+1];		/* device class */
  char l_line[DVC_LEN+1];		/* tty device name */
  char l_modem[DVC_LEN+1];		/* modem name */
  char l_baud[DVC_LEN+1];		/* tty baud rate */
  char l_type[DVC_LEN+1];		/* ACU etc. */
  char buf[128];			/* L-devices line buffer */
  FILE *fp;				/* L-devices file pointer */
  int fd;				/* device lock file descriptor */

  if ((fp = fopen(LDEVS, "r")) == (FILE *) NULL) return(NO_Ldv);

  while (fgets(buf, sizeof(buf), fp) != (char *) NULL) {
	if (buf[0] == '#' || buf[0] == '\n') continue;	/* ignore comment lines */

	if (sscanf(buf, "%s%s%s%s%s", l_class, l_line, l_modem, l_baud,
						l_type) != 5)
		continue;				/* mangled line */

	if (strcmp(l_type, "ACU") != 0) continue;	/* not a dialer line */

	if (callp->line != (char *) NULL)		/* device specified? */
		if (strcmp(l_line, callp->line) != 0)
			continue;			/* wrong device */

	exists++;		/* say device exists at some baud rate */

	if (strcmp(l_baud, "Any") != 0 && atoi(l_baud) != callp->baud)
		continue;				/* wrong speed */

	tried++;		/* found device at desired baud rate */

	if (checklock(l_line, 2) == FAILED) continue;	/* couldn't lock line */
	sprintf(lockfile, "%s%s", LOCK, l_line);
	gotlock = 1;

	strncpy(tty, l_line, 15);
	strncpy(brand, l_modem, 15);
	fclose(fp);
	return(1);
  }

  if (tried)
	return(DV_NT_A);
  else if (exists)
	return(ILL_BD);
  else
	return(DV_NT_K);
}


static struct speedlist {
  int value;
  int name;
} speeds[] = {
  {    0,      0 },
  {  110,   B110 },
  {  300,   B300 },
  { 1200,  B1200 },
  { 2400,  B2400 },
  { 4800,  B4800 },
  { 9600,  B9600 },
#if defined(B19200)
  {19200, B19200 },
#endif B19200
  {   -1,     -1 }
};


static int findspeed(speed)
int speed;
{
  register struct speedlist *ps;

  for (ps = speeds; ps->value >= 0; ps++)
	if (ps->value == speed) return(ps->name);

  return(0);
}


static void catcher()
{
  (void) signal(SIGALRM, catcher);
}


static void clear(fd)
int fd;
{
  char c;
  unsigned int setting;
  sig_t sig;

  sig = signal(SIGALRM, catcher);
  setting = alarm(2);
  while (read(fd, &c, 1) == 1)
	;
  (void) alarm(setting);
  (void) signal(SIGALRM, sig);
}


#ifdef CATCH
int alarmcatch(void)
{
  long timebuf[2];

  time(&timebuf[0]);
  timebuf[1] = timebuf[0];

  utime(lockfile, timebuf);
  signal(SIGALRM, alarmcatch);
  alarm(3600);
}


int hupcatch(void)
{
  close(modemfd);
  (void) unlock(lockfile);
  gotlock = 0;
  signal(SIGHUP, SIG_DFL);
}
#endif CATCH

#ifdef _BCC			/* BCC can't pass structures */
int dial(callp)
CALL *callp;
{
  CALL call = *callp;
#else
int dial(call)
CALL call;
{
#endif
  char line[64];
  char modemname[16];		/* modemcap name of modem */
  char device[16];		/* name of device */
  char *strcpy(), *strcat();
#ifdef	UNIX_S5
  struct termio termio;
#endif
#ifdef	UNIX_V7
  struct sgttyb termio;
#endif
  int fd;
  int err;

  modemname[0] = device[0] = '\0';

  fd = -1;			/* channel illegal until line is opened	*/

  if ((err = findline(&call, device, modemname)) <= 0) goto error;

  if (device[0] == '\0') {	/* can't determine name of line */
	err = L_PROB;
	goto error;
  }
  if (modemname[0] == '\0') {	/* can't determine the type of modem */
	err = DV_NT_K;
	goto error;
  }
  sprintf(line, "%s/%s", DEVDIR, device);
  if ((fd = open(line, O_RDWR)) < 0) { /* can't open modem line */
	err = L_PROB;
	goto error;
  }
#ifdef	UNIX_S5
  if (call.attr != (struct termio *) 0) { /* set attributes */
	if (ioctl(fd, TCSETA, call.attr) == -1) { /* some ioctl() problem */
		err = L_PROB;
		goto error;
	}
  } else {
	  ioctl(fd, TCGETA, &termio);
	  if ((termio.c_cflag = findspeed(call.baud)) == 0) {
		err = ILL_BD;
		goto error;
	  }
	  termio.c_cflag |= (CS8 | CREAD | HUPCL);
	  termio.c_iflag = 0;
	  termio.c_oflag = 0;
	  termio.c_lflag = 0;
	  termio.c_cc[VMIN] = 1;
	  termio.c_cc[VTIME] = 1;
	  if (ioctl(fd, TCSETA, &termio) == -1) {
		err = L_PROB;
		goto error;
	  }
  }
#endif
#ifdef	UNIX_V7
  if (call.attr != (struct sgttyb *) NULL) { /* set attributes */
	if (ioctl(fd, TIOCGETP, call.attr) == -1) { /* some gtty() problem */
		err = L_PROB;
		goto error;
	}
  } else {
	  ioctl(fd, TIOCGETP, &termio);
	  if ((termio.sg_ispeed = findspeed(call.baud)) == 0) {
		err = ILL_BD;
		goto error;
	  }
	  termio.sg_ospeed = termio.sg_ispeed;
	  termio.sg_flags |= RAW;
	  termio.sg_flags &= ~ECHO;
	  termio.sg_erase = -1;
	  termio.sg_kill = -1;
	  if (ioctl(fd, TIOCSETP, &termio) == -1) {
			err = L_PROB;
			goto error;
	  }
  }
#endif
  clear(fd);			/* clear the input buffer */
  initmodem(modemname, fd);	/* setup modemcap variables */
  if (call.telno == (char *) NULL)	/* no phone number, connection complete */
	goto okay;

  if (! DI) {			/* modem has no ACU!!! */
	err = A_PROB;		/* no ACU to attach to */
	goto error;
  }
  if (BD != call.speed) {	/* is connection desired at high speed? */
	if (BL != call.speed){	/* is connection desired at low speed? */
		err = ILL_BD;	/* modem can't handle this speed */
		goto error;
	}
	BD = BL;		/* set baud to low baud rate */
	CO = CL;		/* set connect reply to low baud reply */
  }
  if (err = mdial(call.telno, fd)) 	/* some error trying to dial */
				goto error;
#ifdef CATCH
  signal(SIGALRM, alarmcatch);	/* set catcher for ALARM clock */
  signal(SIGHUP, hupcatch);	/* set catcher for HANG UP */
  alarm(3600);			/* set clock to touch lock in 1 hour */
#endif CATCH

okay:
  return(modemfd = fd);

error:
  if (gotlock == 1) {		/* don't unlink '.'! */
	(void) unlock(lockfile);
	gotlock = 0;
  }
  if (fd > 2) close(fd);
  return(err);
}


void undial(fd)
int fd;
{
  clear(fd);			/* clear the input buffer */
  if (fd > 2) close (fd);

  if (gotlock == 1) {		/* don't unlink '.'! */
	(void) unlock(lockfile);
	gotlock = 0;
  }

#ifdef CATCH
  alarm(0);
#endif CATCH
}

