/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		Miscellaneous Internal Servers.  This file holds the
 *		code and data for all internal servers, like the ECHO,
 *		DISCARD et al protocols.  In the INETD.CONF file, these
 *		services are marked "internal".
 *
 * Version:	@(#)servers.c		1.00	07/12/92
 *
 * Authors:	Original by Phil Karn KA9Q.
 *		Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <stdio.h>

#include "mbuf.h"
#include "netuser.h"
#include "timer.h"
#include "tcp.h"


_PROTOTYPE( void r_discard, (struct tcb *, int)				);
_PROTOTYPE( void r_echo, (struct tcb *, int)				);
_PROTOTYPE( void t_echo, (struct tcb *, int)				);
_PROTOTYPE( void t_state, (struct tcb *, char, char)			);


static struct tcb *disc_tcb;		/* TCB for the DISCARD service	*/
static struct tcb *echo_tcb;		/* TCB for the ECHO service	*/


/* DISCARD server receiver upcall. */
static void r_discard(tcb, cnt)
struct tcb *tcb;
int cnt;
{
  struct mbuf *bp;

  if (recv_tcp(tcb, &bp, cnt) > 0) {
	/* do nothing */
	(void) free_p(bp);
  }
}


/*
 * ECHO server receiver upcall.
 * Copies only as much will fit on the transmit queue.
 */
static void r_echo(tcb, cnt)
struct tcb *tcb;
int cnt;
{
  struct mbuf *bp;
  int acnt;

  if (cnt == 0){
	close_tcp(tcb);
	return;
  }
  acnt = min(cnt, tcb->snd.wnd);
  if (acnt > 0){
	/* Get only as much will fit in the send window. */
	recv_tcp(tcb, &bp, tcb->snd.wnd);
	send_tcp(tcb, bp);
  }
}


/*
 * ECHO server transmitter upcall.
 * Copies anything that might have been left in the receiver queue.
 */
static void t_echo(tcb, cnt)
struct tcb *tcb;
int cnt;
{
  struct mbuf *bp;

  if (tcb->rcvcnt > 0){
	/* Get only as much will fit in the send window. */
	recv_tcp(tcb, &bp, cnt);
	send_tcp(tcb, bp);
  }
}


/* Log connection state changes; also respond to remote closes */
static void t_state(tcb, old, new)
register struct tcb *tcb;
char old, new;
{
  switch(new){
	case ESTABLISHED:
		break;
	case CLOSE_WAIT:
		close_tcp(tcb);
		break;
	case CLOSED:
		del_tcp(tcb);
		/* Clean up if server is being shut down */
		if (tcb == disc_tcb) disc_tcb = NULLTCB;
		  else if (tcb == echo_tcb) echo_tcb = NULLTCB;
		break;
  }
}


/* Start the DISCARD server. */
void di_start()
{
  struct socket lsocket;

  lsocket.address = ip_addr;
  lsocket.port = DISCARD_PORT;
  disc_tcb = open_tcp(&lsocket, NULLSOCK, TCP_PASSIVE,
		      0, r_discard, NULLVFP, t_state, 0);
}


/* Start the ECHO server. */
void ec_start()
{
  struct socket lsocket;

  lsocket.address = ip_addr;
  lsocket.port = ECHO_PORT;
  echo_tcb = open_tcp(&lsocket, NULLSOCK, TCP_PASSIVE,
		      0, r_echo, t_echo, t_state, 0);
}


/* Shutdown the DISCARD server. */
void di_stop()
{
  if (disc_tcb != NULLTCB) close_tcp(disc_tcb);
}


/* Shutdown the ECHO server. */
void ec_stop()
{
  if (echo_tcb != NULLTCB) close_tcp(echo_tcb);
}
