/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file manages the IP routing table.
 *
 * Version:	@(#)route.c		1.00	07/12/92
 *
 * Authors:	Original by Phil Karn KA9Q.
 *		Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <stdio.h>

#include "mbuf.h"
#include "timer.h"
#include "netuser.h"
#include "ip.h"
#include "icmp.h"
#include "iface.h"


static struct route *routes[NROUTE];	/* routing table		*/
static struct route r_default;		/* default route entry		*/


/* Compute hash function on IP address. */
static int16 hash_ip(addr)
register int32 addr;
{
  register int16 ret;

  ret = hiword(addr);
  ret ^= loword(addr);
  ret %= NROUTE;
  return(ret);
}


/* Look up target in hash table. Return NULL if not found. */
struct route *rt_lookup(target)
int32 target;
{
  register struct route *rp;
  int16 hash_ip();

  if (target == (int32) 0) return(&r_default);

  for (rp = routes[hash_ip(target)]; rp != NULLROUTE; rp = rp->next) {
	if (rp->target == target) break;
  }
  return(rp);
}


/*
 * Add an entry to the IP routing table.
 * Returns 0 on success, -1 on failure
 */
int rt_add(target,gateway,metric,interface)
int32 target;
int32 gateway;
int metric;
struct interface *interface;
{
  struct route *rp,**hp,*rt_lookup();
  int16 hash_ip();

  /* A target address of 0.0.0.0 refers to the default entry */
  if (target == 0) {
	rp = &r_default;
  } else if ((rp = rt_lookup(target)) == NULLROUTE) {
	/*
	 * If the target is not already in the table, create a new
	 * entry and put it in. Otherwise just overwrite the old one.
	 */
	if ((rp = (struct route *)malloc(sizeof(struct route))) == NULLROUTE)
			return(-1);	/* No space */
	/* Insert at head of table. */
	rp->prev = NULLROUTE;
	hp = &routes[hash_ip(target)];
	rp->next = *hp;
	if (rp->next != NULLROUTE) rp->next->prev = rp;
	*hp = rp;
  }
  rp->target = target;
  rp->gateway = gateway;
  rp->metric = metric;
  rp->interface = interface;
  return(0);
}


/*
 * Remove an entry from the IP routing table.
 * Returns 0 on success, -1 if entry was not in table.
 */
int rt_drop(target)
int32 target;
{
  register struct route *rp;
  struct route *rt_lookup();

  if (target == (int32) 0){
	/* Clean out the "default" table entry. */
	r_default.interface = NULLIF;
	return(0);
  }
  if ((rp = rt_lookup(target)) == NULLROUTE) return(-1);   /* not in table */

  if (rp->next != NULLROUTE) rp->next->prev = rp->prev;
  if (rp->prev != NULLROUTE) rp->prev->next = rp->next;
    else routes[hash_ip(target)] = rp->next;

  free((char *)rp);
  return(0);
}


/* Dump IP routing table
 * Dest              Interface    Gateway          Metric
 * 192.001.002.003   sl0          192.002.003.004       4
 */
int rt_dump(fdout)
int fdout;
{
  register int i;
  register struct route *rp;

  rprintf(fdout, "Dest              Interface    Gateway          Metric\r\n");
  if (r_default.interface != NULLIF) {

  	rprintf(fdout, "default           ");
	rprintf(fdout, "%-13s",r_default.interface->name);
	if (r_default.gateway != 0)
		rprintf(fdout, "%-17s", inet_ntoa(htonl(r_default.gateway)));
	  else rprintf(fdout, "%-17s", "");
	rprintf(fdout, "%6u\r\n",r_default.metric);
  }
  for(i = 0; i < NROUTE; i++) {
	for(rp = routes[i];rp != NULLROUTE;rp = rp->next){
		rprintf(fdout, "%-18s", inet_ntoa(htonl(rp->target)));
		rprintf(fdout, "%-13s", rp->interface->name);
		if (rp->gateway != 0) rprintf(fdout, "%-17s",
					inet_ntoa(htonl(rp->gateway)));
		  else rprintf(fdout, "%-17s", "");
		rprintf(fdout, "%6u\r\n",rp->metric);
	}
  }
  return(0);
}
