/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles all incoming requests from remote
 *		(external) clients and servers.  It is the base where
 *		all "system calls" find their origin...
 *
 * Version:	@(#)rmt.c		1.00	07/12/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <stdio.h>
#include <arpa/internet.h>

#include "rmt.h"
#include "iface.h"
#include "netuser.h"
#include "nproc.h"
#include "driver.h"

#if HAVE_SELECT
extern fd_set fds_in_use;
#endif

static struct rmtentry rmt[NRMT];
static int rmtfdout = 1;
static int rmtfdin  = -1;
static int rmtflags = 0;
static int rmtpause = 0;
static char my_hostname[128];
static char my_domainname[128];


_PROTOTYPE( int rmt_blk, (int fd)					);
_PROTOTYPE( int rmt_nblk, (int fd)					);
_PROTOTYPE( int do_shid, (RMTHDR *)					);
_PROTOTYPE( int do_ghid, (RMTHDR *)					);
_PROTOTYPE( int do_shname, (RMTHDR *, int)				);
_PROTOTYPE( int do_ghname, (RMTHDR *, int)				);
_PROTOTYPE( int do_sdname, (RMTHDR *, int)				);
_PROTOTYPE( int do_gdname, (RMTHDR *, int)				);
_PROTOTYPE( int do_connect, (RMTHDR *, int, int *)			);
_PROTOTYPE( int do_attach, (RMTHDR *, int)				);
_PROTOTYPE( int do_detach, (RMTHDR *, int)				);
_PROTOTYPE( int do_ifconfig, (RMTHDR *, int, int)			);
_PROTOTYPE( int do_route, (RMTHDR *, int, int)				);
_PROTOTYPE( int do_arp, (RMTHDR *, int, int)				);
_PROTOTYPE( int do_ip, (RMTHDR *)					);
_PROTOTYPE( int do_tcp, (RMTHDR *)					);
_PROTOTYPE( int do_stat, (RMTHDR *, int, int)					);


/*
 * Put the "COMMAND CHANNEL" pipe into NON-BLOCK mode.
 */
static int rmt_nblk(fd)
int fd;
{
  int st;

  /* Already running? */
  if (rmtpause == 0) return(0);

  /* No. Fetch current flags. */
  st = fcntl(fd, F_GETFL, &rmtflags);
  if (st < 0) return(errno);

  /* Set new flags. */
  st = fcntl(fd, F_SETFL, O_NONBLOCK);
  if (st < 0) return(errno);

  rmtpause = 0;
  return(0);
}


/*
 * Put the "COMMAND CHANNEL" pipe into BLOCKING mode.
 */
static int rmt_blk(fd)
int fd;
{
  int st;

  /* Already blocked? */
  if (rmtpause == 1) return(0);

  /* No, set new flags. */
  st = fcntl(fd, F_SETFL, rmtflags);
  if (st < 0) return(errno);
  rmtflags = -1;
  rmtpause = 1;
  return(0);
}


/*
 * Perform the "SETHOSTID" system call.
 *
 * Entry:	c_l1		my new host ID
 * Exit:	c_opcode:	status
 */
static int do_shid(hdr)
register RMTHDR *hdr;
{
  register struct interface *ifp;
  int32 old_addr;

  /* Check the caller's permission. */
  if (hdr->c_uid != 0) return(EPERM);

  /* Fetch our new address, convert to host order. */
  old_addr = ip_addr;
  ip_addr = (int32) ntohl(hdr->c_l1);
  if (ip_addr == (int32) 0) return(EINVAL);

  /* Remove our current address from the IP routing table. */
  rt_drop(old_addr);

  /* Add a route to our new address. */
  for(ifp = ifaces; ifp != NULLIF; ifp = ifp->next)
	if(strcmp(ifp->name, "loopback") == 0) break;
  if (ifp != NULLIF) rt_add(ip_addr, 0L, 0, ifp);

  hdr->c_length = 0;
  return(0);
}


/*
 * Perform the "GETHOSTID" system call.
 * 
 * Exit:	c_l1		host ID
 *		c_opcode	status
 */
static int do_ghid(hdr)
register RMTHDR *hdr;
{
  /* Have we been previously assigned an IP address? */
  if (ip_addr == (int32) 0) return(ENOENT);

  /* Stash our current IP address. */
  hdr->c_l1 = (u_long) htonl(ip_addr);

  hdr->c_length = 0;
  return(0);
}


/*
 * Perform the "SETHOSTNAME" system call.
 *
 * Entry:	c_i1		length of name
 * Exit:	c_opcode	status
 */
static int do_shname(hdr, fd)
register RMTHDR *hdr;
int fd;
{
  int len, st;

  /* Check the caller's permission. */
  if (hdr->c_uid != 0) return(EPERM);

  len = hdr->c_i1;
  if (len > sizeof(my_hostname)) len = (sizeof(my_hostname) - 1);

  st = read(fd, my_hostname, len);
  my_hostname[len] = '\0';

  hdr->c_length = 0;
  return(0);
}


/*
 * Perform the "GETHOSTNAME" system call.
 *
 * Exit:	c_i1		length of name
 *		c_opcode	status
 */
static int do_ghname(hdr, fd)
register RMTHDR *hdr;
int fd;
{
  if (my_hostname[0] == '\0') return(ENOENT);

  hdr->c_i1 = strlen(my_hostname);
  hdr->c_length = hdr->c_i1;

  /* Set the return status. */
  hdr->c_opcode = (u_short) 0;

  /* Write the result message. */
  if (fd > 0) (void) write(fd, (char *) hdr, RMTHDR_LEN);

  /* Write the host name ASCII string. */
  (void) write(fd, my_hostname, hdr->c_length);

  return(-999);
}


/*
 * Perform the "SETDOMAINNAME" system call.
 *
 * Entry:	c_i1		length of name
 * Exit:	c_opcode	status
 */
static int do_sdname(hdr, fd)
register RMTHDR *hdr;
int fd;
{
  int len, st;

  /* Check the caller's permission. */
  if (hdr->c_uid != 0) return(EPERM);

  len = hdr->c_i1;
  if (len > sizeof(my_domainname)) len = (sizeof(my_domainname) - 1);

  st = read(fd, my_domainname, len);
  my_domainname[len] = '\0';

  hdr->c_length = 0;
  return(0);
}


/*
 * Perform the "GETDOMAINNAME" system call.
 *
 * Exit:	c_i1		length of name
 *		c_opcode	status
 */
static int do_gdname(hdr, fd)
register RMTHDR *hdr;
int fd;
{
  if (my_domainname[0] == '\0') return(ENOENT);

  hdr->c_i1 = strlen(my_domainname);
  hdr->c_length = hdr->c_i1;

  /* Set the return status. */
  hdr->c_opcode = (u_short) 0;

  /* Write the result message. */
  if (fd > 0) (void) write(fd, (char *) hdr, RMTHDR_LEN);

  /* Write the domain name ASCII string. */
  (void) write(fd, my_domainname, hdr->c_length);

  return(-999);
}


/*
 * Perform the "CONNECT" system call.
 *
 * Entry:	c_i1		protocol to use
 *		c_i2		remote port number
 *		c_i3		local port number
 *		c_l1		remote IP address
 *		c_l2		connection flags
 *
 * Return:	c_opcode	operation status
 */
static int do_connect(hdr, fd, keepit)
register RMTHDR *hdr;
int fd;
int *keepit;
{
  char buff[128];
  int i, s, fdin;
  u_long dst;

  /* Client checks */
  /* We cannot connect to 0.0.0.0 ??? */
  if ((hdr->c_l2 & 1L) && hdr->c_i1 == TCP_PTCL)
	if (hdr->c_l1 == 0L) return(EINVAL);

  /* Convert from NETWORK to HOST byte order. Barf!!! */
  dst = ntohl(hdr->c_l1);

  /* Allocate an I/O process for this channel. */
  i = cnproc(hdr->c_i1, &hdr->c_i3, &hdr->c_i2, &dst, hdr->c_l2);
  s = ((i >= 0) ? 0 : ENOMEM);
  if (s != 0) return(s);

  /* Create a return channel for this process. */
  sprintf(hdr->c_return, TNET_COUT, hdr->c_pid);
  sprintf(buff, "%s/%s/%s", TNET_DIR, TNET_PDIR, hdr->c_return);

  /* FIXME: should we check the success of the next instructions??? */
  (void) mkfifo(buff, 0666);
  (void) chmod(buff, 0666);
  fdin = open(buff, O_RDONLY | O_NONBLOCK);

  nproc[i].fdin = fdin;
#if HAVE_SELECT
  FD_SET(fdin, &fds_in_use);
#endif
  nproc[i].fdout = fd;
  *keepit = 1;

  hdr->c_length = 0;
  return(0);
}


/*
 * Perform the "ATTACH" system call.
 */
static int do_attach(hdr, fd)
register RMTHDR *hdr;
int fd;
{
  struct attach at;
  int st;

  /* Check the caller's permission. */
  if (hdr->c_uid != 0) return(EPERM);

  st = read(fd, (char *) &at, sizeof(struct attach));
  if (st < 0) return(errno);

  st = drv_attach(&at);
  
  hdr->c_length = 0;
  return(st);
}


/*
 * Perform the "DETACH" system call.
 & FIXME: this has to be completed yet!
 */
static int do_detach(hdr, fd)
register RMTHDR *hdr;
int fd;
{
  char name[128];
  int st;

  /* Check the caller's permission. */
  if (hdr->c_uid != 0) return(EPERM);

  st = read(fd, name, hdr->c_i1);
  if (st < 0) return(errno);

  st = drv_detach(name);

  hdr->c_length = 0;
  return(st);
}


/*
 * Perform the "IFCONFIG" system call.
 * FIXME: this has to be completed yet!
 */
static int do_ifconfig(hdr, fdin, fdout)
register RMTHDR *hdr;
int fdin, fdout;
{
  char name[32];
  int st;

  hdr->c_length = 0;

  /* Check the caller's permission. */
  if (hdr->c_uid != 0) return(EPERM);

  switch(hdr->c_i1) {
	case 0:
		hdr->c_opcode = (u_short) 0;
		(void) write(fdout, (char *) hdr, RMTHDR_LEN);
		(void) drv_config((char *)NULL, (char *)NULL, fdout);
		return(-999);
		break;
	case 1:
		st = read(fdin, name, hdr->c_i1);
		if (st < 0) return(errno);
		st = drv_config(name, (char *)NULL, fdout);
		hdr->c_length = 0;
		return(st);
	default:
		return(EINVAL);
  }
  /* NOTREACHED */
}


/*
 * Perform the "ROUTE" system call.
 */
static int do_route(hdr, fdin, fdout)
register RMTHDR *hdr;
int fdin, fdout;
{
  int32 dest, gateway;
  int metric, st;
  struct interface *ifp;
  struct uroute rt;

  /* Check the caller's permission. */
  if(hdr->c_i1 != ROUTE_DUMP)
	if (hdr->c_uid != 0) return(EPERM);

  switch(hdr->c_i1) {
	case ROUTE_ADD:
		st = read(fdin, (char *) &rt, sizeof(struct uroute));
		if (st != sizeof(struct uroute)) return(ENXIO);
		dest = ntohl(rt.target);
		for (ifp = ifaces; ifp != NULLIF; ifp = ifp->next) {
			if (strcmp(rt.iface, ifp->name) == 0) break;
		}
		if (ifp == NULLIF) {
			rprintf(2, "ROUTE ADD: iface \"%s\" unknown\n",
								rt.iface);
			return(EINVAL);
		}
		gateway = ntohl(rt.gateway);
		metric = rt.metric;
		return(rt_add(dest, gateway, metric, ifp));
		/* NOTREACHED */
	case ROUTE_DROP:
		dest = ntohl(hdr->c_l1);
		return(rt_drop(dest));
		/* NOTREACHED */
	case ROUTE_DUMP:
		hdr->c_opcode = (u_short) 0;
		(void) write(fdout, (char *) hdr, RMTHDR_LEN);
		rt_dump(fdout);
		return(-999);
		/* NOTREACHED */
	default:
		return(ENOSYS);
  }
  /* NOTREACHED */
  return(0);
}


/*
 * Perform the "ARP" system call.
 * FIXME: this has to be completed yet!
 */
static int do_arp(hdr, fdin, fdout)
register RMTHDR *hdr;
int fdin, fdout;
{
  int st;

  hdr->c_length = 0;

  /* Check the caller's permission. */
  if(hdr->c_i1 != ARP_DUMP)
	if (hdr->c_uid != 0) return(EPERM);

  switch(hdr->c_i1) {
	case ARP_DUMP:
		hdr->c_opcode = (u_short) 0;
		(void) write(fdout, (char *) hdr, RMTHDR_LEN);
		(void) doarp(fdout);
		return(-999);
		break;
	case ARP_ADD:
		return(EINVAL);
		break;
	case ARP_DROP:
		return(EINVAL);
		break;
	case ARP_FLUSH:
		return(EINVAL);
		break;
	case ARP_PUBLISH:
		return(EINVAL);
		break;
	default:
		return(EINVAL);
  }
  /* NOTREACHED */
}


/*
 * Perform the "IP" system call.
 */
static int do_ip(hdr)
register RMTHDR *hdr;
{
  int st;
  extern int ip_ttl;			/* in ip.c			*/

  /* Check the caller's permission. */
  if(hdr->c_i2 != 0)
	if (hdr->c_uid != 0) return(EPERM);

  switch(hdr->c_i1) {
	case IPVAL_TTL:
		if (hdr->c_i2 == 0) hdr->c_i2 = ip_ttl;
		  else ip_ttl = hdr->c_i2;
		st = 0;
		break;
	default:
		st = EINVAL;
  }
  return(st);
}


/*
 * Perform the "TCP" system call.
 */
static int do_tcp(hdr)
register RMTHDR *hdr;
{
  int st;
  extern int tcp_mss, tcp_window;	/* in tcp_in.c			*/

  /* Check the caller's permission. */
  if(hdr->c_i2 != 0)
	if (hdr->c_uid != 0) return(EPERM);

  switch(hdr->c_i1) {
	case TCPVAL_MSS:
		if (hdr->c_i2 == 0) hdr->c_i2 = tcp_mss;
		  else tcp_mss = hdr->c_i2;
		st = 0;
		break;
	case TCPVAL_WINDOW:
		if (hdr->c_i2 == 0) hdr->c_i2 = tcp_window;
		  else tcp_window = hdr->c_i2;
		st = 0;
		break;
	default:
		st = EINVAL;
  }
  return(st);
}


/*
 * Perform the "STAT" system call.
 */
static int do_stat(hdr, fdin, fdout)
register RMTHDR *hdr;
int fdin, fdout;
{
extern int tcpstt();
extern int doipstat();
extern int doudpstat();
extern int doicmpstat();

  switch(hdr->c_i1) {
	case STAT_DUMP:
		hdr->c_opcode = (u_short) 0;
		(void) write(fdout, (char *) hdr, RMTHDR_LEN);
		tcpstt(fdout, (struct tcb *)hdr->c_i2);
		if(hdr->c_i2 == 0) {
			doipstat(fdout);
			doudpstat(fdout);
			doicmpstat(fdout);
		}
		return(-999);
		/* NOTREACHED */
	default:
		return(ENOSYS);
  }
  /* NOTREACHED */
  return(0);
}


/* Open the TNET CONTROL CHANNEL pipe for READ and WRITE. */
void rmtinit()
{
  char buff[128];
  int i;

  sprintf(buff, "%s/%s", TNET_DIR, TNET_PIPE);
  if ((mkfifo(buff, 0666) < 0)) {
	if (errno != EEXIST) {
		rprintf(2, "Cannot mkpipe(%s): %d!\n", buff, errno);
		exit(-1);
	}
  }

  (void) chmod(buff, 0666);
  if ((rmtfdin = open(buff, O_RDONLY | O_NONBLOCK)) < 0) {
	rprintf(2, "Cannot open(%s, R|N): %d!\n", buff, errno);
	exit(-1);
  }
#if HAVE_SELECT
  FD_SET(rmtfdin, &fds_in_use);
#endif

  if ((rmtfdout = open(buff, O_WRONLY)) < 0) {
	rprintf(2, "Cannot open(%s, W): %d!\n", buff, errno);
	exit(-1);
  }

  for(i = 0; i < NRMT; i++)
		rmt[i].used = RMTFREE;
}


void rmtstop() 
{
  int i;

  (void) close(rmtfdout);
  (void) close(rmtfdin);
  for(i = 0; i < NRMT; i++)
	if (rmt[i].used != RMTFREE) {
		(void) close(rmt[i].fdin);
		(void) close(rmt[i].fdout);
	}
}


void rmtdel(entry)
int entry;
{
  (void) close(rmt[entry].fdin);
  (void) close(rmt[entry].fdout);
  rmt[entry].used = RMTFREE;
}


/* check if we have any commands to process */
void DOrmt()
{
  RMTHDR hdr;
  char pipename[128];
  int i, fdout;
  int keepopen;

  /* Try reading a command header. */
  i = read(rmtfdin, (char *) &hdr, RMTHDR_LEN);
  if (i != RMTHDR_LEN) return;

  /* Create the caller's return channel name. */
  sprintf(pipename, "%s/%s/%s", TNET_DIR, TNET_PDIR, hdr.c_return);
  fdout = open(pipename, O_WRONLY);

  keepopen = 0;
  i = ENOSYS;
  if ((rmtpause == 1 && hdr.c_opcode == RMT_START) ||
      (rmtpause == 0)) switch(hdr.c_opcode) {
	case RMT_EXIT:
		if (hdr.c_uid == 0) {
			shutdown = TRUE;
			i = 0;
		} else i = EPERM;
		break;
	case RMT_STOP:
		if (hdr.c_uid == 0) i = rmt_blk(rmtfdin);
		  else i = EPERM;
		break;
	case RMT_START:
		if (hdr.c_uid == 0) i = rmt_nblk(rmtfdin);
		  else i = EPERM;
		break;
	case RMT_SHID:
		i = do_shid(&hdr);
		break;
	case RMT_GHID:
		i = do_ghid(&hdr);
		break;
	case RMT_SHNAME:
		i = do_shname(&hdr, rmtfdin);
		break;
	case RMT_GHNAME:
		i = do_ghname(&hdr, fdout);
		break;
	case RMT_SDNAME:
		i = do_sdname(&hdr, rmtfdin);
		break;
	case RMT_GDNAME:
		i = do_gdname(&hdr, fdout);
		break;
	case RMT_CONNECT:
		i = do_connect(&hdr, fdout, &keepopen);
		break;
	case RMT_ATTACH:
		i = do_attach(&hdr, rmtfdin);
		break;
	case RMT_DETACH:
		i = do_detach(&hdr, rmtfdin);
		break;
	case RMT_IFCONFIG:
		i = do_ifconfig(&hdr, rmtfdin, fdout);
		break;
	case RMT_ROUTE:
		i = do_route(&hdr, rmtfdin, fdout);
		break;
	case RMT_ARP:
		i = do_arp(&hdr, rmtfdin, fdout);
		break;
	case RMT_IP:
		i = do_ip(&hdr);
		break;
	case RMT_TCP:
		i = do_tcp(&hdr);
		break;
	case RMT_STAT:
		i = do_stat(&hdr, rmtfdin, fdout);
		break;
	default:
		rprintf(2, "TNET Huh? Weird command %d (0x%04.4x)\n",
					hdr.c_opcode, hdr.c_opcode);
		i = EINVAL;
  }

  /* Send a reply message if needed. */
  if (i != -999) {
	/*
	 * Set the return status.
	 * We are acting as an extension of the OS kernel, so we should
	 * return _negative_ values to indicate an error.  The rmtlink()
	 * function at the client side will then see the error, set the
	 * global value "errno" to its positive value, and return the
	 * commonly used -1 value.
	 */
	hdr.c_opcode = (u_short) -i;

	/* Write the result message. */
	if (fdout >= 0) (void) write(fdout, (char *) &hdr, RMTHDR_LEN);
  }

  /* If have an error, or if we dont need it anymore, close the channel. */
  if ((i != 0 && i != -999) || keepopen == 0) (void) close(fdout);
}
