/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles the IP protocol. It contains send and
 *		receive primitives, fragment reassembly functions, gateway
 *		routines and does routing and options processing.
 *
 * Version:	@(#)ip.c		1.00	07/12/92
 *
 * Authors:	Original by Phil Karn KA9Q.
 *		Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <stdio.h>

#define	TLB		30	/* Reassembly limit time */

#include "mbuf.h"
#include "timer.h"
#include "ip.h"
#include "icmp.h"
#include "iface.h"


#define IP_LOCAL_ADDR	0x7F000001L		/* IP loopback addr	*/
#define IP_LOCAL_MASK	0x000000FFL		/* IP loopback netmask	*/


int32 ip_addr;
int ip_ttl = MAXTTL;
struct ip_stats ip_stats;


int ip_recv();	/* Should be void, but C complains */
void ip_timeout();


/* Local recipient interface structure */
struct interface if_lo = {
  NULLIF,
  "loopback",
  65535,		/* We take them as big as they come */
  ip_recv,
  NULLFP,
  NULLFP,
  NULLFP,
  0,
  0,
  0,
  0,
  IF_TAKENF,		/* Accept packets marked "no forwarding" */
  NULLCHAR,
  "(internal)",
  IP_LOCAL_ADDR,
  IP_LOCAL_MASK,
  0x0000,
  0,
  0
};
struct reasm *reasmq;

#define	INSERT	0
#define	APPEND	1
#define	PREPEND	2

/* Send an IP datagram. Modeled after the example interface on p 32 of
 * RFC 791
 */
void
ip_send(source,dest,protocol,tos,ttl,bp,length,id,df)
int32 source;		/* source address */
int32 dest;		/* Destination address */
char protocol;		/* Protocol */
char tos;		/* Type of service */
char ttl;		/* Time-to-live */
struct mbuf *bp;	/* Data portion of datagram */
int16 length;		/* Optional length of data portion */
int16 id;		/* Optional identification */
char df;		/* Don't-fragment flag */
{
	struct mbuf *hbp;	/* mbuf containing IP header */
	struct ip_header *iph;	/* Pointer to IP header */
	static int16 id_cntr;	/* Datagram serial number */
	int16 hdr_len;		/* IP header length, bytes */
	void ip_route();	/* Datagram router */

	if(length == 0 && bp != NULLBUF)
		length = len_mbuf(bp);
	if(id == 0)
		id = id_cntr++;		
	if(ttl == 0)
		ttl = ip_ttl;

	/* Allocate an mbuf for the IP header */
	hdr_len = sizeof(struct ip_header);
	if((hbp = alloc_mbuf(hdr_len)) == NULLBUF){
		/* We really ought to source-quench the sender, but that would
		 * probably fail too.
		 */
		free_p(bp);
		return;
	}
	hbp->cnt = hdr_len;

	/* and fill it in */
	iph = (struct ip_header *)hbp->data;

	iph->v_ihl = (IPVERSION << 4) | (hdr_len/sizeof(int32));
	iph->tos = tos;
	iph->length = htons(hdr_len + length);
	iph->id = htons(id);
	if(df)
		iph->fl_offs = htons(DF);
	else
		iph->fl_offs = 0;
	iph->ttl = ttl;
	iph->protocol = protocol;
	iph->checksum = 0;
	iph->source = htonl(source);
	iph->dest = htonl(dest);
	iph->checksum = cksum((struct pseudo_header *)NULL,hbp,hdr_len);
	hbp->next = bp;
	if_lo.pkt_out++;
	ip_route(hbp,0);		/* Toss it to the router */
}

/* Reassemble incoming IP fragments and dispatch completed datagrams
 * to the proper transport module
 */
int	/* Should really be void */
ip_recv(bp)
struct mbuf *bp;
/*int16 dev;	/* Ignored */
/*int32 gateway;	/* Ignored */
{
	register struct ip_header *ipp;	/* Pointer to original IP header */
	struct ip_header ip;	/* Extracted copy of header */
	int16 ip_len;		/* Length of IP header */
	struct mbuf *fraghandle(), *bp2;
	void (*recv)();	/* Function to call with completed datagram */
	void tcp_input();
	void udp_input();
	void icmp_input();

	ipp = (struct ip_header *)bp->data;

	bp2 = NULLBUF;

	/* Initial check for protocols we can't handle */
	switch(ipp->protocol & 0xff){
	case TCP_PTCL:
		recv = tcp_input;
		break;
	case UDP_PTCL:
		recv = udp_input;
		bp2 = copy_p(bp, len_mbuf(bp));
		break;
	case ICMP_PTCL:
		recv = icmp_input;
		break;
	default:
		/* Send an ICMP Protocol Unknown response */
		ip_stats.badproto++;
		icmp_output(bp,DEST_UNREACH,PROT_UNREACH,(union icmp_args *)NULL);
		free_p(bp);
		return;
	}
	ip_len = lonibble(ipp->v_ihl) * sizeof(int32);
	/* Extract IP header */
	pullup(&bp,(char *)&ip,sizeof(ip));
	if((ip_len - sizeof(ip)) > 0)
		pullup(&bp, NULLCHAR, ip_len - sizeof(ip));

	/* Convert to host byte order */
	ip.length = ntohs(ip.length) - ip_len;	/* Length of data portion */
	ip.id = ntohs(ip.id);
	ip.fl_offs = ntohs(ip.fl_offs);
	ip.source = ntohl(ip.source);
	ip.dest = ntohl(ip.dest);

	/* If we have a complete packet, call the next layer
	 * to handle the result
	 */
	if((bp = fraghandle(&ip,bp)) != NULLBUF) {
		if_lo.pkt_in++;
		(*recv)(bp,ip.protocol,ip.source,ip.dest,ip.tos,ip.length, bp2);
	} else
		free_p(bp2);
}
/* Process IP datagram fragments
 * If datagram is complete, return it with ip->length containing its
 * entire length; otherwise return NULLBUF
 */
static
struct mbuf *
fraghandle(ip,bp)
struct ip_header *ip;	/* IP header, host byte order */
struct mbuf *bp;	/* The fragment itself */
{
	void freefrag(),free_reasm();
	struct reasm *lookup_reasm(),*creat_reasm();
	register struct reasm *rp; /* Pointer to reassembly descriptor */
	struct frag *lastfrag,*nextfrag,*tfp,*newfrag();
	struct mbuf *tbp;
	int16 i;
	int16 offset;		/* Index of first byte in fragment */
	int16 last;		/* Index of first byte beyond fragment */
	char mf;		/* 1 if not last fragment, 0 otherwise */

	offset = (ip->fl_offs & F_OFFSET) << 3;	/* Convert to bytes */
	last = offset + ip->length;
	mf = (ip->fl_offs & MF) ? 1 : 0;

	rp = lookup_reasm(ip);
	if(offset == 0 && !mf){
		/* Complete datagram received. Discard any earlier fragments */
		if(rp != NULLREASM)
			free_reasm(rp);

		return bp;
	}
	if(rp == NULLREASM){
		/* First fragment; create new reassembly descriptor */
		if((rp = creat_reasm(ip)) == NULLREASM){
			/* No space for descriptor, drop fragment */
			free_p(bp);
			return NULLBUF;
		}
	}
	/* Keep restarting timer as long as we keep getting fragments */
	stop_timer(&rp->timer);
	start_timer(&rp->timer);

	/* If this is the last fragment, we now know how long the
	 * entire datagram is; record it
	 */
	if(!mf)
		rp->length = last;

	/* Set nextfrag to the first fragment which begins after us,
	 * and lastfrag to the last fragment which begins before us
	 */
	lastfrag = NULLFRAG;
	for(nextfrag = rp->fraglist;nextfrag != NULLFRAG;nextfrag = nextfrag->next){
		if(nextfrag->offset > offset)
			break;
		lastfrag = nextfrag;
	}
	/* Check for overlap with preceeding fragment */
	if(lastfrag != NULLFRAG  && offset < lastfrag->last){
		/* Strip overlap from new fragment */
		i = lastfrag->last - offset;
		pullup(&bp,NULLCHAR,i);
		if(bp == NULLBUF)
			return NULLBUF;	/* Nothing left */
		offset += i;
	}
	/* Look for overlap with succeeding segments */
	for(; nextfrag != NULLFRAG; nextfrag = tfp){
		tfp = nextfrag->next;	/* save in case we delete fp */

		if(nextfrag->offset >= last)
			break;	/* Past our end */
		/* Trim the front of this entry; if nothing is
		 * left, remove it.
		 */
		i = last - nextfrag->offset;
		pullup(&nextfrag->buf,NULLCHAR,i);
		if(nextfrag->buf == NULLBUF){
			/* superseded; delete from list */
			if(nextfrag->prev != NULLFRAG)
				nextfrag->prev->next = nextfrag->next;
			else
				rp->fraglist = nextfrag->next;
			if(tfp->next != NULLFRAG)
				nextfrag->next->prev = nextfrag->prev;
			freefrag(nextfrag);
		} else
			nextfrag->offset = last;
	}
	/* Lastfrag now points, as before, to the fragment before us;
	 * nextfrag points at the next fragment. Check to see if we can
	 * join to either or both fragments.
	 */
	i = INSERT;
	if(lastfrag != NULLFRAG && lastfrag->last == offset)
		i |= APPEND;
	if(nextfrag != NULLFRAG && nextfrag->offset == last)
		i |= PREPEND;
	switch(i){
	case INSERT:	/* Insert new desc between lastfrag and nextfrag */
		tfp = newfrag(offset,last,bp);
		tfp->prev = lastfrag;
		tfp->next = nextfrag;
		if(lastfrag != NULLFRAG)
			lastfrag->next = tfp;	/* Middle of list */
		else
			rp->fraglist = tfp;	/* First on list */
		if(nextfrag != NULLFRAG)
			nextfrag->prev = tfp;
		break;
	case APPEND:	/* Append to lastfrag */
		append(&lastfrag->buf,bp);
		lastfrag->last = last;	/* Extend forward */
		break;
	case PREPEND:	/* Prepend to nextfrag */
		tbp = nextfrag->buf;
		nextfrag->buf = bp;
		append(&nextfrag->buf,tbp);
		nextfrag->offset = offset;	/* Extend backward */
		break;
	case (APPEND|PREPEND):
		/* Consolidate by appending this fragment and nextfrag
		 * to lastfrag and removing the nextfrag descriptor
		 */
		append(&lastfrag->buf,bp);
		append(&lastfrag->buf,nextfrag->buf);
		nextfrag->buf = NULLBUF;
		lastfrag->last = nextfrag->last;

		/* Finally unlink and delete the now unneeded nextfrag */
		lastfrag->next = nextfrag->next;
		if(nextfrag->next != NULLFRAG)
			nextfrag->next->prev = lastfrag;
		freefrag(nextfrag);
		break;
	}
	if(rp->fraglist->offset == 0 && rp->fraglist->next == NULLFRAG 
		&& rp->length != 0){
		/* We've gotten a complete datagram, so extract it from the
		 * reassembly buffer and pass it on.
		 */
		bp = rp->fraglist->buf;
		rp->fraglist->buf = NULLBUF;
		ip->length = rp->length;	/* Tell IP the entire length */
		free_reasm(rp);
		return bp;
	} else
		return NULLBUF;
}
static struct reasm *
lookup_reasm(ip)
struct ip_header *ip;
{
	register struct reasm *rp;

	for(rp = reasmq;rp != NULLREASM;rp = rp->next){
		if(ip->source == rp->source && ip->dest == rp->dest
		 && ip->protocol == rp->protocol && ip->id == rp->id)
			return rp;
	}
	return NULLREASM;
}

/* Create a reassembly descriptor,
 * put at head of reassembly list
 */
static struct reasm *
creat_reasm(ip)
register struct ip_header *ip;
{
	register struct reasm *rp;

	if((rp = (struct reasm *)calloc(1,sizeof(struct reasm))) == NULLREASM)
		return rp;	/* No space for descriptor */
	rp->source = ip->source;
	rp->dest = ip->dest;
	rp->id = ip->id;
	rp->protocol = ip->protocol;
	rp->timer.start = TLB;
	rp->timer.func = ip_timeout;
	rp->timer.arg = (int *)rp;

	rp->next = reasmq;
	if(rp->next != NULLREASM)
		rp->next->prev = rp;
	reasmq = rp;
	return rp;
}

/* Free all resources associated with a reassembly descriptor */
static void
free_reasm(rp)
register struct reasm *rp;
{
	register struct frag *fp;

	stop_timer(&rp->timer);
	/* Remove from list of reassembly descriptors */
	if(rp->prev != NULLREASM)
		rp->prev->next = rp->next;
	else
		reasmq = rp->next;
	if(rp->next != NULLREASM)
		rp->next->prev = rp->prev;
	/* Free any fragments on list, starting at beginning */
	while((fp = rp->fraglist) != NULLFRAG){
		rp->fraglist = fp->next;
		free_p(fp->buf);
		free((char *)fp);
	}
	free((char *)rp);
}

/* Handle reassembly timeouts by deleting all reassembly resources */
static void
ip_timeout(arg)
int *arg;
{
	register struct reasm *rp;

	rp = (struct reasm *)arg;
	free_reasm(rp);
}
/* Create a fragment */
static
struct frag *
newfrag(offset,last,bp)
int16 offset,last;
struct mbuf *bp;
{
	struct frag *fp;

	if((fp = (struct frag *)calloc(1,sizeof(struct frag))) == NULLFRAG){
		/* Drop fragment */
		free_p(bp);
		return NULLFRAG;
	}
	fp->buf = bp;
	fp->offset = offset;
	fp->last = last;
	return fp;
}
/* Delete a fragment, return next one on queue */
static
void
freefrag(fp)
struct frag *fp;
{
	free_p(fp->buf);
	free((char *)fp);
}

/* Route an IP datagram. This is the "hopper" through which all IP datagrams,
 * coming or going, must pass.
 *
 * This router is a temporary hack, since it only does host-specific or
 * default routing (no hierarchical routing yet).
 *
 * If "noforward" is set, the router will discard the packet unless the
 * target interface will accept it (only the local interface should do
 * this). If the destination is unreachable, no ICMP unreachables will
 * be generated. Any interfaces accepting broadcast packets should set
 * this flag.
 */
void
ip_route(bp,noforward)
struct mbuf *bp;
int noforward;
{
	register struct ip_header *ip;	/* IP header being processed */
	int16 ip_len;			/* IP header length */
	int16 buflen;			/* Length of mbuf */
	int16 length;			/* Total datagram length */
	int32 target;			/* Target IP address */
	int32 gateway;			/* Gateway IP address */
	register struct route *rp;	/* Route table entry */
	struct route *rt_lookup();
	int opi;			/* Index into options field */
	int opt_len;			/* Length of current option */
	int strict;			/* Strict source routing flag */
	struct mbuf *sbp;		/* IP header for fragmenting */
	int16 fl_offs;			/* fl_offs field of datagram */
	int16 offset;			/* Offset of fragment */
	char precedence;		/* Extracted from tos field */
	char delay;
	char throughput;
	char reliability;

	ip_stats.total++;
	buflen = len_mbuf(bp);
	if(buflen < sizeof(struct ip_header)){
		/* The packet is shorter than a legal IP header */
		ip_stats.runt++;
		free_p(bp);
		return;
	}
	ip = (struct ip_header *)bp->data;
	length = ntohs(ip->length);
	if(buflen > length){
		/* Packet has excess garbage (e.g., Ethernet padding); trim */
		if(bp->next == NULLBUF){
			/* One mbuf, just adjust count */
			bp->cnt = length;
		} else {
			struct mbuf *nbp;
			/* Copy to a new one */
			nbp = copy_p(bp,length);
			free((char *)bp);
			bp = nbp;
			ip = (struct ip_header *)bp->data;
		}
	}
	ip_len = lonibble(ip->v_ihl) * sizeof(int32);
	if(ip_len < sizeof(struct ip_header)){
		/* The IP header length field is too small */
		ip_stats.length++;
		free_p(bp);
		return;
	}
	if(cksum(NULLHEADER,bp,ip_len) != 0){
		/* Bad IP header checksum; discard */
		ip_stats.checksum++;
		free_p(bp);
		return;
	}
	if(hinibble(ip->v_ihl) != IPVERSION){
		/* We can't handle this version of IP */
		ip_stats.version++;
		free_p(bp);
		return;
	}
	/* Decrement TTL and discard if zero */
	if(--ip->ttl == 0){
		/* Send ICMP "Time Exceeded" message */
		icmp_output(bp,TIME_EXCEED,0,NULLICMP);
		free_p(bp);
		return;
	}
	/* Process options, if any. Also compute length of secondary IP
	 * header in case fragmentation is needed later
	 */
	strict = 0;
	for(opi = sizeof(struct ip_header);opi < ip_len; opi += opt_len){
		char *opt;	/* Points to current option */
		int opt_type;	/* Type of current option */
		int pointer;	/* Pointer field of current option */
		int32 *addr;	/* Pointer to an IP address field in option */

		opt = (char *)ip + opi;
		opt_type = opt[0] & OPT_NUMBER;

		/* Handle special 1-byte do-nothing options */
		if(opt_type == IP_EOL)
			break;		/* End of options list, we're done */
		if(opt_type == IP_NOOP){
			opt_len = 1;	/* No operation, skip to next option */
			continue;
		}
		/* Other options have a length field */
		opt_len = opt[1] & 0xff;

		/* Process options */
		switch(opt_type){
		case IP_SSROUTE:/* Strict source route & record route */
			strict = 1;
		case IP_LSROUTE:/* Loose source route & record route */
			/* Source routes are ignored unless the datagram appears to
			 * be for us
			 */
			if(ntohl(ip->dest) != ip_addr)
				continue;
		case IP_RROUTE:	/* Record route */
			pointer = (opt[2] & 0xff) - 1;
			if(pointer + sizeof(int32) <= opt_len){
				/* Insert our address in the list */
				addr = (int32 *)&opt[pointer];
				if(opt_type != IP_RROUTE)
					/* Old value is next dest only for source routing */
					ip->dest = *addr;
				*addr = htonl(ip_addr);
				opt[2] += 4;
			} else {
				/* Out of space; return a parameter problem and drop */
				union icmp_args icmp_args;

				icmp_args.unused = 0;
				icmp_args.pointer = sizeof(struct ip_header) + opi;
				icmp_output(bp,PARAM_PROB,0,&icmp_args);
				free_p(bp);
				return;
			}
			break;
		}
	}
	/* Note this address may have been modified by source routing */
	target = ntohl(ip->dest);

	/* Look up target address in routing table */
	if((rp = rt_lookup(target)) == NULLROUTE || rp->interface == NULLIF){
		/* Not found, use default route instead */
		rp = rt_lookup((int32) 0);
	}
	if(rp->interface == NULLIF){
		/* No route exists, return unreachable message */
		if(!noforward)
			icmp_output(bp,DEST_UNREACH,HOST_UNREACH,NULLICMP);
		else
			ip_stats.noforward++;
		free_p(bp);
		return;
	}
	/* Find gateway; zero gateway in routing table means "send direct" */
	if(rp->gateway == (int32)0)
		gateway = target;
	else
		gateway = rp->gateway;

	if(strict && gateway != target){
		/* Strict source routing requires a direct entry */
		if(!noforward)
			icmp_output(bp,DEST_UNREACH,ROUTE_FAIL,NULLICMP);
		else
			ip_stats.noforward++;
		free_p(bp);
		return;
	}
	/* If the input interface has marked this packet "don't forward",
	 * and the output interface won't accept it, then discard
	 */
	if(noforward && !(rp->interface->flags & IF_TAKENF)){
		ip_stats.noforward++;
		free_p(bp);
		return;
	}
	precedence = PREC(ip->tos);
	delay = ip->tos & DELAY;
	throughput = ip->tos & THRUPUT;
	reliability = ip->tos & RELIABILITY;

	if(length <= rp->interface->mtu){
		/* Datagram smaller than interface MTU; send normally */
		/* Recompute header checksum */
		ip->checksum = 0;
		ip->checksum = cksum(NULLHEADER,bp,ip_len);
		(*rp->interface->send)(bp,rp->interface,gateway,
			precedence,delay,throughput,reliability);
		return;
	}
	/* Fragmentation needed */
	fl_offs = ntohs(ip->fl_offs);
	if(fl_offs & DF){
		/* Don't Fragment set; return ICMP message and drop */
		icmp_output(bp,DEST_UNREACH,FRAG_NEEDED,NULLICMP);
		free_p(bp);
		return;
	}
	/* Create copy of IP header for each fragment */
	sbp = copy_p(bp,ip_len);
	pullup(&bp,NULLCHAR,ip_len);
	length -= ip_len;

	/* Create fragments */
	offset = (fl_offs & F_OFFSET) << 3;
	while(length != 0){
		int16 fragsize;		/* Size of this fragment's data */
		struct mbuf *f_header;	/* Header portion of fragment */
		struct ip_header *fip;	/* IP header */
		struct mbuf *f_data;	/* Data portion of fragment */

		f_header = copy_p(sbp,ip_len);
		fip = (struct ip_header *)f_header->data;
		fip->fl_offs = htons(offset >> 3);
		if(length + ip_len <= rp->interface->mtu){
			/* Last fragment; send all that remains */
			fragsize = length;
		} else {
			/* More to come, so send multiple of 8 bytes */
			fragsize = (rp->interface->mtu - ip_len) & 0xfff8;
			fip->fl_offs |= htons(MF);
		}
		fip->length = htons(fragsize + ip_len);
		/* Recompute header checksum */
		fip->checksum = 0;
		fip->checksum = cksum(NULLHEADER,f_header,ip_len);

		/* Extract portion of data and link in */
		f_data = copy_p(bp,fragsize);
		pullup(&bp,NULLCHAR,fragsize);
		f_header->next = f_data;

		(*rp->interface->send)(f_header,rp->interface,gateway,
			precedence,delay,throughput,reliability);
		offset += fragsize;
		length -= fragsize;
	}
	free_p(sbp);
}

/* Internet checksum routine. The 1's complement algorithm has some nice
 * properties. If you sum a block of data in network order, you end up with
 * the proper network-order checksum regardless of the byte order of the
 * machine doing the summing! Also, summing a byteswapped block is equivalent
 * to byteswapping the sum of an unbyteswapped block. Nice optimizations here.
 */
int16
cksum(ph,bp,length)
struct pseudo_header *ph;	/* Optional IP pseudo-header, MACHINE ORDER */
struct mbuf *bp;
int16 length;
{
	register int32 sum;
	int16 *wp;
	int16 wcnt;
	int16 cnt, tot;
	int16 carries;
	char *buf;

	sum = 0L;
	if(ph != NULLHEADER){
		sum = (int32) hiword(ph->source);
		sum = sum + (int32) loword(ph->source);
		sum = sum + (int32) hiword(ph->dest);
		sum = sum + (int32) loword(ph->dest);
		sum = sum + (int32) (ph->protocol & 0x00FF);
		sum = sum + (int32) ph->length;
		/* Adjust for 1's complement by doing end-around-carries */
		while((carries = ((sum >> 16) & 0xFFFFL)) != 0)
			sum = (sum & 0xFFFFL) + (int32) carries;
		/* Swap to network order */
		sum = (int32) htons((int16)(sum & 0xFFFFL));
	}

	/* Now compute sum of data blocks, already in network order */
	tot = 0;
	while(bp != NULLBUF){
		cnt = min(bp->cnt,length-tot);
		buf = bp->data;
		/* check for leading odd byte */
		if(tot & 1){
			sum += (int32) ntohs((int16)(*buf++ & 0xFF));
			cnt--;
			tot++;
		}
		/* Do the main bulk of the data */
		wp = (int16 *)buf;
		wcnt = cnt >> 1;
		while(wcnt--) {
			sum += (int32) (*wp++ & 0xFFFF);
#ifdef __BCC__
			if(wcnt == 0) break;	/* BCC bug!! */
#endif
		}

		/* Check for trailing odd byte */
		if(cnt & 1){
			buf = buf + (cnt & (~1));
			sum += (int32) ntohs((int16)((*buf << 8) & 0xFF00));
		}
		tot = tot + cnt;
		if(tot == length)
			break;
		bp = bp->next;
	}
	/* Adjust for 1's complement by doing end-around-carries */
	while((carries = ((sum >> 16) & 0xFFFFL)) != 0)
		sum = (sum & 0xFFFFL) + (int32) carries;
	/* All done */
	return(((int16)(~sum & 0xFFFFL)));
}

int doipstat(fdout)
int fdout;
{
	extern struct ip_stats ip_stats;
	extern struct reasm *reasmq;
	register struct reasm *rp;
	register struct frag *fp;
	char *inet_ntoa();

	rprintf(fdout, "IP status:\r\n");
	rprintf(fdout, "total %ld runt %u len err %u vers err %u",
		ip_stats.total,ip_stats.runt,ip_stats.length,ip_stats.version);
	rprintf(fdout, " chksum err %u nofwd %u badproto %u\r\n",
		ip_stats.checksum,ip_stats.noforward,ip_stats.badproto);

	if(reasmq != NULLREASM)
		rprintf(fdout, "Reassembly fragments:\r\n");
	for(rp = reasmq;rp != NULLREASM;rp = rp->next){
		rprintf(fdout, "src %s",inet_ntoa(rp->source));
		rprintf(fdout, " dest %s",inet_ntoa(rp->dest));
		rprintf(fdout, " id %u pctl %u time %u len %u\r\n",
			rp->id,rp->protocol,rp->timer.count,rp->length);
		for(fp = rp->fraglist;fp != NULLFRAG;fp = fp->next)
			rprintf(fdout, " offset %u last %u\r\n",fp->offset,fp->last);
	}
	return 0;
}

#if HAVE_TRACE

/* Dump an IP datagram header. If it's the first fragment, also dump
 * the next layer header (if known). Dumping is controlled by the low-order
 * 4 bits of the external variable "trace":
 * Level 0: no dump
 * Level 1-2: link level dumps only
 * Level 3: IP and ARP dumps only
 * Level 4: IP header + TCP header
 * Level 5: All headers + hex dump
 */
void
ip_dump(bp, tflag)
struct mbuf *bp;
int tflag;
{
	void dump_tcp(), dump_udp();
	register struct ip_header *ip;
	int32 source,dest;
	int16 ip_len;
	int16 length;
	struct mbuf *tbp;
	int16 offset;
	int i;

	if(bp == NULLBUF)
		return;	
	ip = (struct ip_header *)bp->data;
	ip_len = lonibble(ip->v_ihl) * sizeof(int32);
	length = ntohs(ip->length);
	offset = (ntohs(ip->fl_offs) & F_OFFSET) << 3 ;
	source = ntohl(ip->source);
	dest = ntohl(ip->dest);
	rprintf(2, "IP: %s",inet_ntoa(ip->source));
	rprintf(2, "->%s len %u ihl %u ttl %u prot %u",
		inet_ntoa(ip->dest),length,ip_len,ip->ttl & 0xff,
		ip->protocol & 0xff);

	if(ip->tos != 0)
		rprintf(2, " tos %u",ip->tos);
	if(offset != 0 || (ntohs(ip->fl_offs) & MF))
		rprintf(2, " id %u offs %u",ntohs(ip->id),offset);

	if(ntohs(ip->fl_offs) & DF)
		rprintf(2, " DF");
	if(ntohs(ip->fl_offs) & MF)
		rprintf(2, " MF");
	if((i = cksum((struct pseudo_header *)NULL,bp,ip_len)) != 0)
		rprintf(2, " CHECKSUM ERROR (%u)",i);
	rprintf(2, "\r\n");

	if(tflag > 3){
		if(offset == 0){
			dup_p(&tbp,bp,ip_len,length - ip_len);
			switch(ip->protocol & 0xff){
			case TCP_PTCL:
				dump_tcp(tbp,source,dest);
				break;
			case UDP_PTCL:
				dump_udp(tbp, source, dest);
				break;
			case ICMP_PTCL:
				dump_icmp(tbp,source,dest);
				break;
			}
			free_p(tbp);
		}
	}
}

#endif
