/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file contains the ETH interface.
 *
 * Version:	@(#)eth.c		1.00	07/12/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#if HAVE_ETHER

#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <stdio.h>

#include "timer.h"
#include "arp.h"
#include "eth.h"
#include "iface.h"
#include "mbuf.h"
#include "ip.h"


_PROTOTYPE( int ethstop, (struct interface *)				);
_PROTOTYPE( int ethsend, (struct mbuf *, struct interface *, int32,	\
			  char, char, char, char)			);
_PROTOTYPE( int ethout, (struct interface *, char *, char *,		\
			  int16, struct mbuf *)				);
_PROTOTYPE( int ethrecv, (interface *)					);


static int ethstop(interface)
struct interface *interface;
{
  (void) close(interface->fdin);
  (void) close(interface->fdout);
  if (interface->pid > 0) (void) kill(interface->pid, SIGKILL);
  if (interface->lock != NULLCHAR) (void) unlink(interface->lock);
  return(0);
}


/*
 * Send an IP datagram to an Ethernet interface.
 * This means, that we have to call the ARP client code to
 * find out what the Ethernet (physical) address of the
 * machine specified in 'gw' is.  If we get NULL as the
 * answer, the address has yet to be resolved, and the
 * ARP code will take care of sending this datagram.
 */
static int ethsend(bp,ifa,gw,precedence,delay,throughput,reliability)
struct mbuf *bp;
struct interface *ifa;
int32 gw;
char precedence;
char delay;
char throughput;
char reliability;
{
  struct mbuf *abp;
  char *eth_addr;
  time_t now;

#if HAVE_TRACE
  if (ifa->trace & 0x0f) {
	(void) time(&now);
	rprintf(2, "%24.24s %s send:\n", ctime(&now), ifa->name);
	if ((ifa->trace & 0x0f) > 2) ip_dump(bp, (ifa->trace & 0x0f));
	if ((ifa->trace & 0x0f) > 4) hexdump(bp);
  }
#endif
  ifa->pkt_out++;

  abp = bp;	/* make sure bp is not changed */
  eth_addr = res_arp(ifa, ARP_ETHER, gw, abp);

  /* If ARP could not resolve it, it will have queued this frame. */
  if (eth_addr != NULLCHAR)
	(*ifa->output)(ifa, eth_addr, myethad, IP_TYPE, bp);
}


/*
 * Send an IP datagram to some ETHER-class device driver.
 * This means, that we have to send an ETHER-class packet
 * header first, which tells the driver how much data it
 * must expect.  After that, we send out the Ethernet
 * header, followed by the IP datagram itself.
 */
static int ethout(interface, dest, src, prot_id, bp)
struct interface *interface;
char *dest, *src;
int16 prot_id;
struct mbuf *bp;
{
  char buff[2000];
  struct mbuf *abp;
  char *cp, *ap;
  int16 hdr_len, size;

  /* Create an Ethernet header mbuf and fill it in. */
  hdr_len = sizeof(struct ether);
  if ((abp = alloc_mbuf(hdr_len)) == NULLBUF){
	free_p(bp);
	rprintf(2, "eth_out: cannot allocate eth hdr!\n");
	return;
  }
  abp->cnt = hdr_len;
  ap = abp->data;
  bcopy(dest, ap, SIZE_ADDR);
  ap = ap + SIZE_ADDR;
  bcopy(src, ap, SIZE_ADDR);
  ap = ap + SIZE_ADDR;
  *(int16 *)ap = htons(prot_id);	/* Ethernet TYPE field		*/
  abp->next = bp;			/* append IP datagram		*/

  /* Pull up this frame into a contiguous buffer. */
  size = dqdata(abp, &buff[sizeof(int)], sizeof(buff) - sizeof(int));
  *(int *)buff = size;	/* YUCK! */
  size += sizeof(int);
  (void) write(interface->fdout, buff, size);
  return(0);
}


static int ethrecv(interface)
struct interface *interface;
{
  char buff[2000];
  struct mbuf *bp;
  int cnt, size;
  int16 prot_id;
  time_t now;
  extern void ip_route();

  cnt = read(interface->fdin, (char *)&size, sizeof(size));
  if (cnt > 0) {
	cnt = read(interface->fdin, buff, size);
	if ((bp = alloc_mbuf(cnt)) == NULLBUF) return(-1);
	bp->cnt = cnt;
	bcopy(buff, bp->data, cnt);
	bp->next = NULLBUF;
	bp->anext = NULLBUF;

	/* Examine and pull up Ethernet destination address. */
	if (pullup(&bp, buff, SIZE_ADDR) != SIZE_ADDR) {
		rprintf(2, "eth_recv: cannot pullup Ethernet dest addr!\n");
		free_p(bp);
		return(0);
	}

	/* Examine and pull up Ethernet source address. */
	if (pullup(&bp, &buff[16], SIZE_ADDR) != SIZE_ADDR) {
		rprintf(2, "eth_recv: cannot pullup Ethernet src addr!\n");
		free_p(bp);
		return(0); 
	}

	/* Examine Ethernet TYPE field. */
	if (pullup(&bp, (char *) &prot_id, sizeof(int16)) != sizeof(int16)) {
		rprintf(2, "eth_recv: cannot pullup Ethernet TYPE field!\n");
		free_p(bp);
		return(0);
	}

	/*
	 * Determine what protocol layer this frame is for.  We can
	 * determine by looking at the Ethernet TYPE field in the
	 * Ethernet Frame header.  Currently, we only understand the
	 * ARP and IP frame types.
	 */
	prot_id = htons(prot_id);
	switch(prot_id) {
		case ARP_TYPE:
			arp_input(interface, bp);
			interface->pkt_in++;
			break;
		case IP_TYPE:
#if HAVE_TRACE
			if (interface->trace & 0xf0) {
				(void) time(&now);
				rprintf(2, "%24.24s %s recv:\n",
					ctime(&now), interface->name);
				if ((interface->trace & 0xf0) > 0x20)
				    ip_dump(bp,(interface->trace & 0xf0)>>4);
				if ((interface->trace & 0xf0) > 0x40)
								hexdump(bp);
			}
#endif
			interface->pkt_in++;
			ip_route(bp, 0);
			break;
		default:
			pether(&buff[64], buff);
			pether(&buff[128], &buff[16]);
			rprintf(2, "eth_recv: %s->%s type 0x%04.4x (%u)\n",
				&buff[64], &buff[128], prot_id, prot_id);
			free_p(bp);
			break;
	}
  }
  return(0);
}


/*
 * Fill in the device-class specific stuff for the interface.
 * We do it this way to keep this information hidden for the
 * upper level layers.
 */
int eth_attch(ifa)
struct interface *ifa;
{
  ifa->send = ethsend;
  ifa->output = ethout;
  ifa->recv = ethrecv;
  ifa->stop = ethstop;

  ifa->flags |= (IF_ACTIVE | IF_BROADCAST);
  return(0);
}


void pether(out, addr)
char *out, *addr;
{
  sprintf(out, "%02x:%02x:%02x:%02x:%02x:%02x",
	addr[0] & 0xFF, addr[1] & 0xFF, addr[2] & 0xFF,
	addr[3] & 0xFF, addr[4] & 0xFF, addr[5] & 0xFF);
}
#endif
