/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles the attachment, configuration and
 *		detachment of device drivers ("interfaces") to the
 *		TNET kernel.  It mostly replaces the former "asy"
 *		and "eth" modules.
 *
 * Version:	@(#)driver.c		1.00	07/12/92
 *
 * Author:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#include <sys/types.h>
#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <stdio.h>

#include "iface.h"
#include "mbuf.h"
#include "timer.h"
#include "ip.h"

#if HAVE_SELECT
extern fd_set fds_in_use;
#endif

#define DEV_DIR		"/dev"
#define LCK_DIR		"/usr/spool/uucp"


static char *drv_progs[] = {		/* names of the driver programs	*/
  "(internal)",
  "slip",				/* BSD SLIP protocol		*/
  "cslip",				/* BSD Compressed SLIP protocol	*/
  "ppp",				/* Point To Point protocol	*/
  "ether",				/* IEEE Ethernet protocol	*/
  "ax25",				/* KA9Q AX.25 protocol driver	*/
  (char *)NULL
};
static char *drv_names[] = {		/* names of the device drivers	*/
  "LOOPBACK",
  "SLIP",				/* BSD SLIP protocol		*/
  "CSLIP",				/* BSD Compressed SLIP protocol	*/
  "PPP",				/* Point To Point protocol	*/
  "ETHER",				/* IEEE Ethernet protocol	*/
  "AX.25",				/* KA9Q AX.25 protocol driver	*/
  (char *)NULL
};


_PROTOTYPE( int drv_start, (struct interface *, char *, char *, char **) );


static int drv_start(ifa, in_name, out_name, argv)
struct interface *ifa;
char *in_name, *out_name;
char *argv[];
{
  char buff[128];

  /* Create a new set of FIFO pipes. */
  (void) unlink(in_name); (void) unlink(out_name);
  (void) mkfifo(in_name, 0600);
  (void) mkfifo(out_name, 0600);

  sprintf(buff, "%s/%s", TNET_DIR, argv[0]);
  if ((ifa->pid = fork()) < 0) {
	rprintf(2, "drv_start: cannot fork (%d)\n", errno);
	return(-1);
  }

  /* We are now a child of the TNET kernel. */
  if (ifa->pid == 0) {
	(void) close(0);
	(void) close(1);
	closefiles();
	(void) execv(buff, argv);
	rprintf(2, "DRIVER: cannot exec %s !!!\n", buff);
	return(-1);
  }

  /* Try to open the WRITE pipe. */
  ifa->fdout = open(in_name, O_WRONLY);
  if (ifa->fdout < 0) {
	rprintf(2, "drv_start: open(%s, W) == %d\n", in_name, errno);
	return(-errno);
  }

  /* Try to open the READ pipe. */
  ifa->fdin = open(out_name, O_RDONLY);
  if (ifa->fdin < 0) {
	rprintf(2, "drv_start: open(%s, R) == %d\n", out_name, errno);
	return(-errno);
  }

  /* Put the READ pipe into "non-blocking" mode so we can scan it. */
  if (fcntl(ifa->fdin, F_SETFL, O_NONBLOCK) < 0) {
	rprintf(2, "drv_start: fcntl(%d, NB) == %d\n", ifa->fdin, errno);
	return(ENXIO);
  }

  /* Remove the pipes since they are not needed after they are used. */
  (void) unlink(in_name);
  (void) unlink(out_name);
#if HAVE_SELECT
  FD_SET(ifa->fdin, &fds_in_use);
#endif
  return(0);
}

	
int drv_attach(ptr)
struct attach *ptr;
{
  char device[128], lockfn[128];
  char buff1[128], buff2[128];
  char temp1[32], temp2[32];
  struct interface *ifa;
  char *args[32];
  time_t now;
  int i;

  /* Create device file pathname. */
  sprintf(device, "%s/%s", DEV_DIR, ptr->device);

#if HAVE_LOCK
  /* Try locking the device file. */
  sprintf(lockfn, "%s/LCK..%s", LCK_DIR, ptr->device);
  if ((i = open(lockfn, O_CREAT | O_WRONLY | O_EXCL, 0644)) < 0) {
	if (errno == EEXIST) {
		rprintf(2, "ATTACH: %s is locked.\n", device);
		return(EBUSY);
	}

	/* Ouch!  We cannot lock the device... :-(  */
	rprintf(2, "ATTACH: cannot lock %s... aborting!\n", device);
	return(errno);
  }
  /* Close the lockfile. */
  (void) close(i);
#endif

  /*
   * The device is now locked, so we have exclusive use over
   * it.  Allocate a new IFACE structure, and fill in the device
   * class I/O routines.  To keep things clean, we ask the respective
   * device class handlers to do this.
   */
  ifa = (struct interface *)malloc(sizeof(struct interface));
  if (ifa == (struct interface *)NULL) {
	rprintf(2, "ATTACH: cannot allocate interface... aborting!\n");
#if HAVE_LOCK
	(void) unlink(lockfn);
#endif
	return(ENOMEM);
  }
  ifa->mtu = ptr->mtu;
  ifa->class = ptr->class;
  ifa->flags = 0;
  ifa->ipaddr = ip_addr;
  ifa->netmask = 0;
  if (debug == 1) ifa->trace = 0xFF;
    else ifa->trace = 0;
  ifa->pkt_in = 0;
  ifa->pkt_out = 0;

  switch(ptr->class) {
#if HAVE_SLIP
	case ATTCH_SLIP:
#endif
#if HAVE_CSLIP
	case ATTCH_CSLIP:
#endif
#if HAVE_PPP
	case ATTCH_PPP:
#endif
#if (HAVE_SLIP || HAVE_CSLIP || HAVE_PPP)
		asy_attch(ifa);
		break;
#endif
#if HAVE_ETHER
	case ATTCH_ETHER:
		eth_attch(ifa);
		break;
#endif
	default:
		rprintf(2, "ATTACH: unsupported device class %d\n",
							ptr->class);
		free(ifa);
#if HAVE_LOCK
		(void) unlink(lockfn);
#endif
		return(EINVAL);
  }

  /*
   * The device I/O functions have been filled in.  Continue
   * filling in the new IFACE structure, and add it to the list of
   * attached interfaces.
   */
  ifa->name = (char *)malloc(strlen(ptr->label) + 1);
  if (ifa->name == (char *)NULL) {
	rprintf(2, "ATTACH: cannot allocate iface name... aborting!\n");
	free(ifa);
#if HAVE_LOCK
	(void) unlink(lockfn);
#endif
	return(ENOMEM);
  } else strcpy(ifa->name, ptr->label);
  ifa->devname = (char *)malloc(strlen(device) + 1);
  if (ifa->devname == (char *)NULL) {
	rprintf(2, "ATTACH: cannot allocate device name... aborting!\n");
	free(ifa->name);
	free(ifa);
#if HAVE_LOCK
	(void) unlink(lockfn);
#endif
	return(ENOMEM);
  } else strcpy(ifa->devname, device);
  ifa->lock = (char *)malloc(strlen(lockfn) + 1);
  if (ifa->lock == (char *)NULL) {
	rprintf(2, "ATTACH: cannot allocate iface lock... aborting!\n");
	free(ifa->name);
	free(ifa->devname);
	free(ifa);
#if HAVE_LOCK
	(void) unlink(lockfn);
#endif
	return(ENOMEM);
  } else strcpy(ifa->lock, lockfn);
  ifa->next = ifaces;
  ifaces = ifa;

  /* Create the device driver pipes. */
  (void) time(&now);
  sprintf(buff1, "%s/%s/.ioI%08.8lx", TNET_DIR, TNET_PDIR, now);
  sprintf(buff2, "%s/%s/.ioO%08.8lx", TNET_DIR, TNET_PDIR, now);

  /*
   * Fill in the args[] array.
   * We call the driver with the following syntax:
   *
   * slip.drv [-dv] [-b speed] [-m mtu] [-n nbufs] \
   *          [-i input] [-o output] /dev/ttyNN [telno|script]
   */
  sprintf(lockfn, "%s.drv", drv_progs[ifa->class]);
  i = 0;
  args[i++] = lockfn;
  if (debug == 1) args[i++] = "-v";
  if (ptr->speed[0] != '\0') {
	args[i++] = "-b";
	args[i++] = ptr->speed;
  }
  args[i++] = "-i";
  args[i++] = buff1;
  args[i++] = "-o";
  args[i++] = buff2;
  args[i++] = "-m";
  sprintf(temp1, "%d", ifa->mtu);
  args[i++] = temp1;
  args[i++] = "-n";
  sprintf(temp2, "%d", ptr->buffers);
  args[i++] = temp2;
  args[i++] = ifa->devname;
  if (ptr->telno[0] != '\0') args[i++] = ptr->telno;

  /* Terminate the argument list. */
  while (i < 32) args[i++] = (char *)NULL;

  /* All went well... start up the driver process! */
  return(drv_start(ifa, buff1, buff2, args));
}


/*
 * Detach a device driver.
 */
int drv_detach(name)
char *name;
{
  rprintf(2, "drv_detach: trying to detach interface \"%s\" !\n", name);
  return(ENXIO);
}


/*
 * Configure an attached interface.
 */
int drv_config(name, ptr, fd)
char *name;
char *ptr;	/* FIXME: struct ifconfig */
int fd;
{
  struct interface *ifa;

  if (name != (char *)NULL) {
	rprintf(2, "drv_config: trying to configure interface \"%s\" !\n",
								name);
  } else {
	ifa = ifaces;
	while (ifa != (struct interface *)NULL) {
		rprintf(fd, "%-16.16s ", ifa->name);
		rprintf(fd, "IP addr %s  MTU=%u  Link encap %s\n",
			inet_ntoa(htonl(ifa->ipaddr)), ifa->mtu,
			drv_names[ifa->class]);
		rprintf(fd, "%17.17sUNIX dev %s  PID=%d FDIN=%d FDOUT=%d\n",
			"", ifa->devname, ifa->pid, ifa->fdin, ifa->fdout);
		rprintf(fd, "%17.17sFlags 0x%04.4x  Trace %04.4x  ", "",
					ifa->flags, ifa->trace);
		rprintf(fd, "Netmask 0x%08.8lx\n", ntohl(ifa->netmask));
		rprintf(fd, "%17.17sPackets SENT=%u  RECEIVED=%u\n", "",
					ifa->pkt_out, ifa->pkt_in);
		rprintf(fd, "\n");
		ifa = ifa->next;
	}
  }
  return(0);
}
