/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file handles the "ASY" class of device drivers,
 *		also known as "serial lines".  This includes the SLIP,
 *		CSLIP and PPP interface drivers.
 *
 * Version:	@(#)asy.c		1.00	07/12/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include "tnet.h"

#if (HAVE_SLIP | HAVE_CSLIP | HAVE_PPP)

#include <fcntl.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <time.h>
#include <stdio.h>

#include "asy.h"
#include "iface.h"
#include "mbuf.h"
#include "timer.h"
#include "ip.h"


_PROTOTYPE( int asystop, (struct interface *)				);
_PROTOTYPE( int asysend, (struct mbuf *, struct interface *, int32,	\
			  char, char, char, char)			);
_PROTOTYPE( int asyout, (struct interface *, struct mbuf *)		);
_PROTOTYPE( int asyrecv, (struct interface *)				);


static int asystop(interface)
struct interface *interface;
{
  (void) close(interface->fdin);
  (void) close(interface->fdout);
  if (interface->pid > 0) (void) kill(interface->pid, SIGKILL);
  if (interface->lock != NULLCHAR) (void) unlink(interface->lock);
  return(0);
}


static int asysend(bp,ifa,gw,precedence,delay,throughput,reliability)
struct mbuf *bp;
struct interface *ifa;
int32 gw;			/* IGNORED */
char precedence;		/* IGNORED */
char delay;			/* IGNORED */
char throughput;		/* IGNORED */
char reliability;		/* IGNORED */
{
  time_t now;

#if HAVE_TRACE
  if (ifa->trace & 0x0f) {
	(void)time(&now);
	rprintf(2, "%24.24s %s send:\n", ctime(&now), ifa->name);
	if ((ifa->trace & 0x0f) > 2) ip_dump(bp, (ifa->trace & 0x0f));
	if ((ifa->trace & 0x0f) > 4) hexdump(bp);
  }
#endif
  ifa->pkt_out++;
  (*ifa->output)(ifa, bp);
}


/*
 * Send an IP datagram to some ASY-class device driver.
 * This means, that we have to send an ASY-class packet
 * header first, which tells the driver how much data it
 * must expect.  To make the write(2) call efficient, we
 * marshall all data (header + datagram) into one buffer,
 * and then write out that buffer in one blow.
 */
static int asyout(interface,bp)
struct interface *interface;
struct mbuf *bp;
{
  char buff[2048];
  int size;
  int s;

  s = sizeof(int);
  size = dqdata(bp, &buff[s], sizeof(buff) - s);
  *(int *)buff = size;	/* YUCK! */
  size += s;
  (void) write(interface->fdout, buff, size);
  return(0);
}


static int asyrecv(interface)
struct interface *interface;
{
  char buff[2048];
  struct mbuf *bp;
  int cnt, size;
  time_t now;
  extern void ip_route();

  cnt = read(interface->fdin, (char *)&size, sizeof(size));
  if (cnt == sizeof(size)) {
	cnt = read(interface->fdin, buff, size);
	if ((bp = alloc_mbuf(cnt)) == NULLBUF) return(-1);
	bp->cnt = cnt;
	bcopy(buff, bp->data, cnt);
	bp->next = NULLBUF;
	bp->anext = NULLBUF;
#if HAVE_TRACE
	if (interface->trace & 0xf0) {
		(void)time(&now);
		rprintf(2, "%24.24s %s recv:\n",
			ctime(&now), interface->name);
		if ((interface->trace & 0xf0) > 0x20)
				ip_dump(bp,(interface->trace & 0xf0)>>4);
		if ((interface->trace & 0xf0) > 0x40) hexdump(bp);
	}
#endif
	interface->pkt_in++;
	ip_route(bp, 0);
   }
  return(0);
}


/*
 * Fill in the device-class specific stuff for the interface.
 * We do it this way to keep this information hidden for the
 * upper level layers.
 */
int asy_attch(ifa)
struct interface *ifa;
{
  ifa->send = asysend;
  ifa->output = asyout;
  ifa->recv = asyrecv;
  ifa->stop = asystop;

  ifa->flags |= IF_ACTIVE;

  return(0);
}
#endif
