/*
 * slip.drv	A program for TNET, which reads SLIP encapsulated IP
 *		datagrams from a serial port, and sends them to TNET
 *		for further processing.
 *
 *		SLIP protocol module.
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include "slip.h"

/* SLIP protocol constants. */
#define	END		0xC0		/* end-of-datagram marker	*/
#define	ESC		0xDB		/* Binary Escape marker		*/
#define	ESC_END		0xDC		/* Escaped END character	*/
#define	ESC_ESC		0xDD		/* Escaped ESCAPE character	*/


_PROTOTYPE( int sl_recvp, (char *p, int maxlen)				);
_PROTOTYPE( void sl_sendp, (char *ptr, int len)				);

#ifdef _CSLIP
char *gbuffer;
#endif

/*
 * Receive a full IP datagram from a SLIP link.
 * This means, that we have to store incoming characters in
 * a buffer, and, when a full datagram has been received, we
 * have to send it to the TNET program for further processing.
 *
 * We perform the upcall to TNET by writing our IP datagram,
 * with its total length in bytes prepended, to the write end
 * of our pipe link with TNET.
 */
static int sl_recvp(p, maxlen)
char *p;
int maxlen;
{
  int received;
  int all_of_it;
  register int c;
  register char *x;

  received = 0;
  all_of_it = 0;
  x = p;

  while(!all_of_it) {
	c = tty_getc();
	if (c == -1) return(-1);

	switch(c) {
		case END:
			if (received > 0) all_of_it = 1;
			  else p = x;
			break;
		case ESC:
			c = tty_getc();
			if (c == -1) {
				all_of_it = 1;
				c = 0;
			}
			switch(c) {
				case ESC_END:
					c = END;
					break;
				case ESC_ESC:
					c = ESC;
					break;
			}
			/* FALL_THROUGH */
		default:
			if (received++ < maxlen) *p++ = (char) c;
	}
  }

  /* We now have a full IP datagram in the buffer. */
  if (opt_v) fprintf(stderr, "SLIP: IP LEN=%d received.\n", received);

  /* Dump contents of datagram if needed. */
  if (opt_d) ip_dump(x, received);

  return(received);
}


/*
 * Encapsulate incoming IP datagram bytes into a SLIP frame, and
 * send the frame to ths serial line when a complete frame is done.
 */
static void sl_sendp(ptr, len)
register char *ptr;
int len;
{
  register int c;
#ifdef _CSLIP
  int tlen;
  char *tptr;

  tlen = len; tptr = ptr;
  tcp_comp(&tptr, &tlen);
  len = tlen; ptr = tptr;
#endif

  tty_putc(END);
  while(len--) {
	c = ((int) *ptr) & 0xFF;
	switch(c) {
		case END:
			tty_putc(ESC);
			tty_putc(ESC_END);
			break;
		case ESC:
			tty_putc(ESC);
			tty_putc(ESC_ESC);
			break;
		default:
			tty_putc(c);
	}
	ptr++;
  }
  tty_putc(END);
  tty_putc(-1);			/* there it goes... */
}


/*
 * This is the input half of the SLIP driver.
 * We read bytes from the serial line (via a buffered READ),
 * and de-slip the encapsulated IP datagram on the fly. The
 * resulting datagram is then written to the output channel,
 * prepended with the total datagram size in bytes.
 */
void in_slip(fd)
int fd;
{
  int s, n;
  int maxlen;
  char *buffer;

#ifndef _CSLIP
  /* Allocate a buffer for the decoded IP datagrams. */
  maxlen = 2 * sl_mtu;
  buffer = (char *) malloc(maxlen + sizeof(int));
  if (buffer == (char *)NULL) {
	fprintf(stderr, "SLIP: IN: cannot allocate my buffer!\n");
	return;
  }
#else
  buffer = gbuffer;
#endif

  while(1) {
	n = sl_recvp(buffer + sizeof(int), maxlen);
	if (n < 0)
#ifndef _CSLIP
		continue;
#else
		break;
#endif

#ifdef _CSLIP
	n = tcp_uncomp(buffer+sizeof(int), n);
#endif
	if (n <= 0) continue;
	*(int *)buffer = n;
	s = write(fd, buffer, sizeof(int) + n);
	if (s != (sizeof(int) + n)) {
		fprintf(stderr, "SLIP: IN: dgram write error %d(%d)\n",
								errno, fd);
	}
#ifdef _CSLIP
	break;
#endif
  }
}


/*
 * This is the SLIP datagram sender.
 * We start an endless loop, reading the "SLIP header" from the
 * I/O input channel.  This header is simply a word which tells
 * us the total length of the IP datagram which we have to en-
 * capsulate into a SLIP frame.  When the encapsulation is done,
 * the frame is sent to the serial line.
 */
void out_slip(fd)
int fd;
{
  int s, n;
  int maxlen;
  char *buffer;

#ifndef _CSLIP
  /* Allocate a buffer for the outgoing SLIP frame. */
  maxlen = 2 * sl_mtu;
  buffer = (char *) malloc(maxlen);
  if (buffer == (char *)NULL) {
	fprintf(stderr, "SLIP: OUT: cannot allocate my buffer!\n");
	return;
  }
#else
  buffer = gbuffer;
#endif

  while(1) {
	s = read(fd, (char *) &n, sizeof(int));
	if (s == 0) return;
	if (s != sizeof(int)) {
		fprintf(stderr, "SLIP OUT: dghdr read error %d(%d)\n",
								errno, fd);
#ifndef _CSLIP
		continue;
#else
		break;
#endif
	}
	if (n > maxlen)
#ifndef _CSLIP
		continue;
#else
		break;
#endif

	/* Read in the entire datagram. */
	s = read(fd, buffer, n);
	if (s != n) {
		fprintf(stderr, "SLIP OUT: dg read error %d(%d)\n",
								errno, fd);
#ifndef _CSLIP
		continue;
#else
		break;
#endif
	}

	/* We now have a full IP datagram in the buffer. */
	if (opt_v) fprintf(stderr, "SLIP: IP LEN=%d sent.\n", n);

	/* Dump contents of datagram if needed. */
	if (opt_d) ip_dump(buffer, n);

	(void) sl_sendp(buffer, n);
#ifdef _CSLIP
	break;
#endif
  }
}

#ifdef _CSLIP
void do_slip(fd_in, fd_out)
int fd_in, fd_out;
{
extern int select();
extern int tty_fd;
fd_set fds;
int maxfd;

  /* Allocate a buffer for the decoded IP datagrams. */
  gbuffer = (char *) malloc(2 * sl_mtu + sizeof(int));
  if (gbuffer == (char *)NULL) {
	fprintf(stderr, "SLIP: IN: cannot allocate my buffer!\n");
	return;
  }
  maxfd = fd_in;
  if(tty_fd > maxfd) maxfd = tty_fd;
  while(1) {
	FD_ZERO(&fds);
	FD_SET(fd_in, &fds);
	FD_SET(tty_fd, &fds);
	select(FD_SETSIZE, &fds, NULL, NULL, NULL);
	if(FD_ISSET(fd_in, &fds))
		out_slip(fd_in);
	else
		in_slip(fd_out);
  }
}
#endif
