/*
 * slip.drv	A program for TNET, which reads SLIP encapsulated IP
 *		datagrams from a serial port, and sends them to TNET
 *		for further processing.
 *
 * Usage:	slip [-b baud] [-dv] [-m mtu] [-n buffs] [-i inpipe] \
 *		     [-o outpipe] ttyname [telno | script]
 *
 * Version:	2.00	07/12/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>
#include <sgtty.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <arpa/inet.h>
#include <arpa/ip.h>
#define GLOBAL
#include "slip.h"


#define VERSION	"3.00"


static char *Version = "@(#) slip.drv 2.00 (07/12/92)";


_PROTOTYPE( void DoRaw, (void)						);
_PROTOTYPE( void DoSlip, (void)						);
_PROTOTYPE( void DoStop, (void)						);
_PROTOTYPE( int tnio_init, (char *in_name, char *out_name)		);
_PROTOTYPE( void usage, (void)						);


extern int getopt(), opterr, optind;
extern char *optarg;

/* Restore the TTY device and exit. */
static void DoStop()
{
  tty_stop();
  exit(0);
}


/* Initialize the TNET I/O channels. */
static int tnio_init(in_name, out_name, ipipe, opipe)
char *in_name, *out_name;
int *ipipe, *opipe;
{
  /* Set up the I/O channels to the TNET kernel. */
  if (strcmp(in_name, "(stdin)")) {
	*ipipe = open(in_name, O_RDONLY);
	if (*ipipe < 0) {
		fprintf(stderr, "SLIP: cannot open(%s, R): %d\n",
							in_name, errno);
		return(-errno);
	}
  } else *ipipe = 0;
  if (opt_v) fprintf(stderr, "IN  = %s (%d)\n", in_name, *ipipe);
  if (strcmp(out_name, "(stdout)")) {
	*opipe = open(out_name, O_WRONLY);
	if (*opipe < 0) {
		fprintf(stderr, "SLIP: cannot open(%s, W): %d\n",
							out_name, errno);
		return(-errno);
	}
  } else *opipe = 1;
  if (opt_v) fprintf(stderr, "OUT = %s (%d)\n", out_name, *opipe);

  return(0);
}


void usage()
{
  fprintf(stderr,
    "Usage: slip [-b baud] [-dv] [-m mtu] [-n buffs] [-i in_pipe]\n");
  fprintf(stderr,
    "            [-o out_pipe] tty_name [telno | script]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char telscript[128], tty_name[128];
  char in_pipe[128], out_pipe[128];
  char baud[16];
  int i, s, fd_in, fd_out;

  /* Set program defaults. */
  sl_mtu = DEF_MTU;
  sl_bufs = DEF_BUFS;
  strcpy(baud, "(default)");
  strcpy(in_pipe, "(stdin)");
  strcpy(out_pipe, "(stdout)");
  strcpy(tty_name,  "");
  strcpy(telscript, "");
  opt_d = 0;
  opt_v = 0;

  /* Scan command line for any arguments. */
  opterr = 0;
  while ((s = getopt(argc, argv, "b:dm:n:i:o:v")) != EOF) switch(s) {
	case 'b':
		strncpy(baud, optarg, 16);
		break;
	case 'd':
		opt_d = 1 - opt_d;
		break;
	case 'm':
		sl_mtu = atoi(optarg);
		if (sl_mtu <= 0 || sl_mtu > 32767) usage();
		break;
	case 'n':
		sl_bufs = atoi(optarg);
		if (sl_bufs < 1 || sl_bufs > 16) usage();
		break;
	case 'i':
		strcpy(in_pipe, optarg);
		break;
	case 'o':
		strcpy(out_pipe, optarg);
		break;
	case 'v':
		opt_v = 1 - opt_v;
		opt_d = 1 - opt_d;
		break;
	default:
		usage();
  }
  opt_d = 0;
  opt_v = 0;

  /* At least one more argument needed. */
  if (optind >= argc) usage();
  strcpy(tty_name, argv[optind++]);

  /* At most one more argument allowed. */
  if (optind != argc) strcpy(telscript, argv[optind]);

  /* Set up signal catchers to allow TNET to have us switch modes. */
  signal(SIGHUP, DoStop);
  signal(SIGTERM, DoStop);
  signal(SIGINT, SIG_IGN);
  signal(SIGQUIT, SIG_IGN);

  fprintf(stderr, "TNET SLIP Protocol Driver version %s\n", VERSION);
  fprintf(stderr, "Written by Michael Temari and Fred N. van Kempen.\n\n");

#if 0
  /* lets do some stuff to remove controlling terminal and keep stderr though */
  (void) close(0); (void) close(1); (void) close(2);
  (void)setpgrp(getpid(), getpid());
#endif
  /* Initialize the serial line. */
  if ((s = tty_init(tty_name, baud)) != 0) return(s);
#if 0
  (void)setpgrp(getpid(), getppid());
#endif

  /* Initialize the TNET I/O channels. */
  if ((s = tnio_init(in_pipe, out_pipe, &fd_in, &fd_out)) != 0) return(s);

  /* Try dialing a host. */
  if ((s = do_dial(telscript)) != 0) {
	fprintf(stderr, "NO CARRIER from %s - ABORTING\n", telscript);
	return(s);
  } else {
	if (opt_v) fprintf(stderr, "CONNECTED\n");
  }

  /*
   * Fork off a child process that handles the WRITE DGRAM
   * operations.  The parent then takes care of receiving
   * any data from the serial line.
   */
#ifndef _CSLIP
  if ((s = fork()) < 0) {
	fprintf(stderr, "SLIP: cannot fork (%s)\n", errno);
	return(-errno);
  }
  if (s == 0) out_slip(fd_in);
    else in_slip(fd_out);
#else
  do_slip(fd_in, fd_out);
#endif

  /* NOTREACHED */
  exit(0);
}
