/*
 * TNET		A server program for MINIX which implements the TCP/IP
 *		suite of networking protocols.  It is based on the
 *		TCP/IP code written by Phil Karn et al, as found in
 *		his NET package for Packet Radio communications.
 *
 *		This file contains an implementation of the TIMEP
 *		protocol (see RFC 868) for fetching the current date
 *		and time off a network server.
 *
 * Usage:	netdate [-adtv] [-s] [server_name]
 *
 * Version:	@(#)netdate.c	1.00	07/02/92
 *
 * Authors:	Michael Temari, <temari@temari.ae.ge.com>
 *		Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <arpa/internet.h>
#include <arpa/inet.h>
#include <inet/in.h>
#include <inet/socket.h>
#include <tnet/tnet.h>
#include <tnet/client.h>
#include <errno.h>
#include <netdb.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <stdio.h>


static char *Version = "@(#) netdate 1.00 (07/02/92)";


int opt_a = 0;				/* display in ARPA style	*/
int opt_d = 0;				/* debugging output flag	*/
int opt_s = 0;				/* set the system's RTC clock	*/
int opt_t = 0;				/* use time(2) and do as date(1)*/
static char *days[] = {
  "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};
static char *months[] = {
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};



extern int getopt(), optind, opterr;
extern char *optarg;
extern char *tzname[2];



_PROTOTYPE( int timep, (char *, time_t *)				);
_PROTOTYPE( void do_disp, (time_t *, int)				);
_PROTOTYPE( int do_set, (time_t *)					);


/*
 * Send a TIMEP request to a server.  If no server is
 * given, we use ourself as the TIMEP server.
 */
int timep(host, ptr)
char *host;
time_t *ptr;
{
  char buff[128];
  u_long rmt_addr;
  u_short rmt_port;
  struct hostent *hp;
  struct servent *sp;
  u_long val;
  time_t nettime, now;
  int fd_in, fd_out;
  int s;

  if (opt_d == 1) fprintf(stderr, "Resolving %s...", host);
  if ((hp = gethostbyname(host)) == (struct hostent *)NULL) {
	fprintf(stderr, "Unknown host %s!\n", host);  
	return(-1);
  } else {
	memcpy((char *) &rmt_addr, (char *) hp->h_addr, hp->h_length);
	strcpy(host, hp->h_name);
  }

  /* Now, to which port must we connect? */
  if ((sp = getservbyname("time", "tcp")) == (struct servent *)NULL) {
	fprintf(stderr, "TIME port is unknown????\n");
	return(-1);
  } else rmt_port = sp->s_port;

  strcpy(buff, inet_ntoa(rmt_addr));
  if (opt_d == 1) fprintf(stderr, " trying %s:%d...\n", buff, rmt_port);

  s = client(TCP_PTCL, 0, rmt_port, rmt_addr, &fd_in, &fd_out);
  if (s != 0) {
	fprintf(stderr, "Connection failed: status = %d\n", s);
	return(s);
  }

  /* Wait for the TIMEP server to send us its time. */
  s = read(fd_in, (char *) &val, sizeof(u_long));
  if (s != sizeof(u_long)) {
	fprintf(stderr, "SERVER failed: bad reply.\n");
	return(-1);
  }
  nettime = (time_t) ntohl(val);
  now = nettime - (time_t) 2208988800L;	/* adjust from Network Time */
  *ptr = now;

  (void) close(fd_in);
  (void) close(fd_out);

  return(0);
}


/* Display the given time in ARPA standard format. */
void do_disp(now, arpa)
time_t *now;
int arpa;
{
  struct tm *tm;

  if (arpa == 1) {
	tm = localtime(now);
	printf("%s, %d %s %d %02d:%02d:%02d %s\n",
		days[tm->tm_wday],
		tm->tm_mday, months[tm->tm_mon], tm->tm_year,
		tm->tm_hour, tm->tm_min, tm->tm_sec,
		tzname[tm->tm_isdst]);
  } else printf("%s", ctime(now));
}


/* Set this machine's clock to this time. */
int do_set(now)
time_t *now;
{
  time_t ourtime;
  int s;

  (void) time(&ourtime);
  if ((s = stime(now)) >= 0) {
	printf("Network Time: %s", ctime(now));
	printf("Our Time    : %s", ctime(&ourtime));
	printf("Difference  : %ld\n", (long) (ourtime - *now));
  } else fprintf(stderr, "Unable to set system clock.\n");
  return(s);
}


void usage()
{
  fprintf(stderr, "Usage: netdate [-adtv] [-s] [server]\n");
  exit(-1);
}


int main(argc, argv)
int argc;
char *argv[];
{
  char host[128];
  time_t net_time;
  time_t now;
  register int c;

  opterr = 0;
  while ((c = getopt(argc, argv, "adtvs")) != EOF) switch(c) {
	case 'a':
		opt_a = 1;
		break;
	case 'd':
	case 'v':
		opt_d = 1;
		break;
	case 's':
		opt_s = 1;
		break;
	case 't':
		opt_t = 1;
		break;
	default:
		usage();
  }

  /* If the -t option is given, act as "date(1)". */
  if (opt_t == 1) {
	if (argc != optind) usage();

	(void) time(&now);
	do_disp(&now, opt_a);
	return(0);
  }

  /* At most one more argument allowed. */
  if (optind != argc && optind != (argc - 1)) usage();

  /* If an argument is given, use that as the TIMEP server. */
  if (optind != argc) strcpy(host, argv[optind]);
    else gethostname(host, 128);

  /* Ask this server to send us its time-of-day. */
  if ((c = timep(host, &net_time)) < 0) return(c);

  /* Display that time, if needed. */
  if ((opt_s == 0) || (opt_d == 1)) {
	do_disp(&net_time, opt_a);
	c = 0;
  }

  /* Do we need to set this system's clock? */
  if (opt_s == 1) c = do_set(&net_time);

  return(c);
}
