/**  ether.c by MOst, 1993
 **
 ** This file contains the driver for the following special file:
 **     /dev/ether	- Ethernet network card (major 7, minor 0)
 **
 ** The driver supports the following operations (using message format m2):
 **
 **    m_type       DEVICE    PROC_NR    COUNT   TTY_FLAGS   ADRRESS
 ** -----------------------------------------------------------------
 ** |  HARD_INT   |         |         |         |         |         |
 ** |-------------+---------+---------+---------+---------+---------|
 ** | ETHER_READ  | device  | proc nr |  bytes  |nonblock | buf ptr |
 ** |-------------+---------+---------+---------+---------+---------|
 ** | ETHER_WRITE | device  | proc nr |  bytes  |         | buf ptr |
 ** |-------------+---------+---------+---------+---------+---------|
 ** | ETHER_IOCTL | device  | proc nr |func code|  data   |         |
 ** |-------------+---------+---------+---------+---------+---------|
 ** |   CANCEL    | device  | proc nr |  mode   |         |         |
 ** -----------------------------------------------------------------
 **
 ** The file contains five entry points:
 **
 **   ether_task:	main entry when system is brought up
 **   ether_int:	Ethernet interrupt
 **   ether_stp:	stop the Ethernet
 **   recv_find:	driver wants to know what to do with received packet
 **   recv_copy:	driver finished copying data to the user space
 **/

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <sgtty.h>
#include "ether.h"

#if ETHERNET	/* whole module */

PUBLIC Eth_addr curr_hw_addr;	/* board's hardware address */

PRIVATE message mess;		/* message buffer */

/* calling process parameters (incomplete read request is stored here) */
PRIVATE int in_status;		/* read request status */
PRIVATE int in_caller;		/* process that made the call (usually FS) */
PRIVATE int in_proc;		/* process that wants to read from ether */
PRIVATE vir_bytes in_vir;	/* virtual address where data is to go */
PRIVATE vir_bytes in_count;	/* how many bytes */

/* This ethernet driver can perform low-level frame filtering of incoming
 * frames. In this mode it passes only specific types of frames to upper
 * layers of software, while other frames are rejected.
 * Filters can be set from user level using IOCTL system call.
 * The driver can also work in "reverse mode": it rejects only specific kinds
 * of frames. Frame filtering in task level can radically improve performance
 * of the whole system, especially on multi-protocol networks.
 * Frame filtering can be used only on Ethernet II (not an IEEE 802.3) nets.
 */
PRIVATE int filt_count;			/* number of filters */
PRIVATE int filt_flags;			/* filter flags */
PRIVATE int filters[NR_FILTERS];	/* array of filters */

/* A small statistics code is also included in ether driver.
 * By pressing F5 key you can check board type and its settings,
 * number of packets/bytes sent/received and frame filters.
 */
PRIVATE struct
{
  unsigned	sent_frm;	/* number of frames sent */
  unsigned long	sent_bytes;	/* number of bytes sent */
  unsigned	recv_frm;	/* number of frames received */
  unsigned long recv_bytes;	/* number of bytes received */
} eth_stat;

FORWARD void do_int();
FORWARD int do_read();
FORWARD int do_write();
FORWARD void do_ioctl();
FORWARD int do_cancel();
FORWARD void ether_reply();

/*===========================================================================*
 *				ether_task				     * 
 *===========================================================================*/
PUBLIC void ether_task()
{
/* Main program of the ether task. */

  int r;

  in_status = IN_NONE;
  filt_count = 0;
  filt_flags = FF_ACCEPT;

  etopen();			/* initialize interface */
  enable_irq(ETHER_IRQ);	/* enable interrupts */

  /* Here is the main loop of the ether task.
   *  It waits for a message, carries it out, and sends a reply.
   */
  while (TRUE) {
	/* First wait for a request to read or write. */
	receive(ANY, &mess);

	/* Now carry out the work.  It depends on the opcode. */
	switch(mess.m_type) {
	    case HARD_INT:	do_int(); continue;	/* no reply, please */
	    case ETHER_READ:	r = do_read(&mess);	break;
	    case ETHER_WRITE:	r = do_write(&mess);	break;
	    case ETHER_IOCTL:	do_ioctl(&mess); continue; /* replies itself */
	    case CANCEL:	r = do_cancel(&mess);	break;
	    default:		r = EINVAL;		break;
	}

	/* Finally, send the reply message. */
	ether_reply(TASK_REPLY, mess.m_source, mess.PROC_NR, r);
  }
}

/*===========================================================================*
 *				do_int					     *
 *===========================================================================*/
PRIVATE void do_int()
{
  recv();			/* do process interrupt */
  enable_irq(ETHER_IRQ);	/* now we can safely reenable interrupts */
}

/*===========================================================================*
 *				do_read					     * 
 *===========================================================================*/
PRIVATE int do_read(m_ptr)
register message *m_ptr;	/* pointer to read message */
{
  int device, count, res;
  phys_bytes user_phys;

  /* Get minor device number */
  device = m_ptr->DEVICE;
  if (device != 0) return(ENXIO);	/* bad minor device */

  count = m_ptr->COUNT;
  if (count < GIANT) return(ENOMEM);	/* not enough space */

  /* Determine address where data is to go. */
  user_phys = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS,
		    (vir_bytes) count);
  if (user_phys == 0) return(E_BAD_ADDR);

  if (in_status != IN_NONE)
    return(EIO);	/* if someone else requested a frame, give up */

/* store message information */
  in_caller = m_ptr->m_source;
  in_proc = m_ptr->PROC_NR;
  in_vir = (vir_bytes) m_ptr->ADDRESS;
  in_count = (vir_bytes) m_ptr->COUNT;

  in_status = IN_WAITING;	/* set "busy" flag */

  res=recv_pkt();		/* ask the driver for a frame */
  if (res == OK)		/* was there any one? */
  {
    in_status = IN_NONE;	/* processing finished */
    return(in_count);		/* return frame size (see recv_copy) */
  }

  if (m_ptr->TTY_FLAGS)		/* non-blocking operation? */
  {
    in_status = IN_NONE;	/* no further processing */
    return(EAGAIN);		/* say: try again */
  }

/* no frames were available, calling process should be suspended */
  in_status = IN_HANGING;
  return(SUSPEND);			/* tell FS to suspend process */
}

/*===========================================================================*
 *				do_write				     *
 *===========================================================================*/
PRIVATE int do_write(m_ptr)
register message *m_ptr;	/* pointer to write message */
{
  int device, count, res;
  phys_bytes user_phys;

  /* Get minor device number */
  device = m_ptr->DEVICE;
  if (device != 0) return(ENXIO);	/* bad minor device */

  count = m_ptr->COUNT;

  /* Determine address where data is to come from. */
  user_phys = numap(m_ptr->PROC_NR, (vir_bytes) m_ptr->ADDRESS,
		    (vir_bytes) count);
  if (user_phys == 0) return(E_BAD_ADDR);

  /* Send the data. */
  res=send_pkt(user_phys, (vir_bytes) count);
  if (res != OK)	/* error sending packet? */
    return res;

  eth_stat.sent_frm++;
  eth_stat.sent_bytes += (unsigned long)count;

  return(count);	/* sent O.K. */
}

/*===========================================================================*
 *				do_ioctl				     *
 *===========================================================================*/
PRIVATE void do_ioctl(m_ptr)
register message *m_ptr;	/* pointer to the newly arrived message */
{
  int r, caller, func_code, data_lo, data_hi;
  int filters_count, filters_flags, count_bytes;
  long data;
  vir_bytes user_vir;
  phys_bytes user_phys, mem_phys;

  r = OK;
  caller = m_ptr->PROC_NR;
  func_code = m_ptr->COUNT;
  data = m_ptr->TTY_FLAGS;
  data_lo = (int) (data & 0xFFFF);
  data_hi = (int) ((data >> 16) & 0xFFFF);

  switch (func_code)
  {
    case EIOCGETADDR:
     /* data_lo => pointer to receive buffer in user memory
      * data_hi => unused
      */
      user_vir = (vir_bytes) data_lo;
      user_phys = numap(caller, user_vir, sizeof(Eth_addr));
      if (user_phys != 0)
      {
        mem_phys = numap(ETHER, (vir_bytes) &curr_hw_addr, sizeof(Eth_addr));
        phys_copy(mem_phys, user_phys, (phys_bytes) sizeof(Eth_addr));
      }
      else
        r = E_BAD_ADDR;
      break;

    case EIOCGETFILTER:
    case EIOCSETFILTER:
     /* data_lo => ptr to array of filters: int filters[]
      * data_hi =>  low byte: number of filters (i.e. array size)
      *            high byte: flags
      */
      user_vir = (vir_bytes) data_lo;
      filters_count = data_hi & BYTE;
      filters_flags = (data_hi >> 8) & BYTE;
      if (filters_count < 0 ||
	  (func_code == EIOCGETFILTER && filters_count < filt_count) ||
	  (func_code == EIOCSETFILTER && filters_count > NR_FILTERS))
      {
	r = ENOMEM;
	break;
      }
      if (filters_count == 0)	/* in set mode only */
      {
	filt_count = filters_count;
	filt_flags = filters_flags;
	break;
      }
      count_bytes = filters_count * sizeof(int);
      user_phys = numap(caller, user_vir, count_bytes);
      if (user_phys == 0)
      {
	r = E_BAD_ADDR;
	break;
      }
      mem_phys = numap(ETHER, (vir_bytes) filters, NR_FILTERS*sizeof(int));
      if (func_code == EIOCGETFILTER)
      {
	count_bytes = filt_count * sizeof(int);
	phys_copy(mem_phys, user_phys, (phys_bytes) count_bytes);
	data = ((long) filt_flags << 24) & 0xFF000000 | 
	       ((long) filt_count << 16) & 0x00FF0000;
      }
      else
      {
	phys_copy(user_phys, mem_phys, (phys_bytes) count_bytes);
	filt_count = filters_count;
	filt_flags = filters_flags;
      }
      break;

    default:		/* unknown func code */
      r = EINVAL;
      break;
  }

  /* Send the reply. Like tty_reply() with extra argument(s). */
  m_ptr->m_type = TASK_REPLY;
  m_ptr->REP_PROC_NR = m_ptr->PROC_NR;
  m_ptr->REP_STATUS = r;
  m_ptr->TTY_FLAGS = data;
  r = send(m_ptr->m_source, m_ptr);
  if (r != OK)
    printf("\r\nsend in do_ioctl (ether task) failed with status %d\r\n", r);
}

/*===========================================================================*
 *				do_cancel				     *
 *===========================================================================*/
PRIVATE int do_cancel(m_ptr)
register message *m_ptr;	/* pointer to the newly arrived message */
{
/* Cancel a request that has already started.  Usually this means that
 * the process doing the printing has been killed by a signal.  It is not
 * clear if there are race conditions.  Try not to cancel the wrong process,
 * but rely on FS to handle the EINTR reply and de-suspension properly.
 */
  int caller, mode;

  caller = m_ptr->PROC_NR;
  mode = m_ptr->COUNT;

  if ((mode & R_BIT) &&		/* read request canceled? */
      in_status != IN_NONE &&	/* was anybody hanging? */
      caller == in_proc)	/* this one? */
  {
    in_status = IN_NONE;	/* if so, cancel read request */
  }
  return EINTR;
}

/*===========================================================================*
 *				ether_reply				     *
 *===========================================================================*/
PRIVATE void ether_reply(code, replyee, proc_nr, status)
int code;			/* TASK_REPLY or REVIVE */
int replyee;			/* destination address for the reply */
int proc_nr;			/* to whom should the reply go? */
int status;			/* reply code */
{
/* Send a reply to a process that wanted to read or write data. */

  message ether_mess;

  ether_mess.m_type = code;
  ether_mess.REP_PROC_NR = proc_nr;
  ether_mess.REP_STATUS = status;
  if ((status = send(replyee, &ether_mess)) != OK)
	printf("\r\nether_reply failed with status %d\r\n", status);
}

/*===========================================================================*
 *				ether_int				     * 
 *===========================================================================*/
PUBLIC void ether_int()			/* Ethernet interrupt */
{
  interrupt(ETHER);		/* signal interrupt to ETHER task */
				/* DO NOT reenable ethernet interrupts now */
}

/*===========================================================================*
 *				ether_stp				     * 
 *===========================================================================*/
PUBLIC void ether_stp()			/* stop the ethernet upon reboot */
{
  reset_board();
}

/*===========================================================================*
 *				ether_dmp				     * 
 *===========================================================================*/
PUBLIC void ether_dmp()			/* dump ethernet statistics */
{
  char *board_name;
  int i;

#if (ETHERNET==ETH_WD8003)
  board_name = "WD8003";
#endif
#if (ETHERNET==ETH_3C503)
  board_name = "3c503";
#endif
#if (ETHERNET==ETH_NE1000)
  board_name = "NE1000";
#endif
#if (ETHERNET==ETH_NE2000)
  board_name = "NE2000";
#endif
#if (ETHERNET==ETH_3C509)
  board_name = "3c509";
#endif

  printf("\r\n%s board: IRQ %d, port %xh",
	 board_name, ETHER_IRQ, ETHER_PORT);
#ifdef ETHER_BASE
  printf(", base %lxh", ETHER_BASE);
#endif
  printf("\r\nHardware address: ");
  for(i=0 ; i<=4 ; i++)
    printf("%02x:", curr_hw_addr.ea[i] & BYTE);
  printf("%02x\r\n", curr_hw_addr.ea[5] & BYTE);
  printf(" sent: %u frames (%lu bytes)\r\n",
	 eth_stat.sent_frm, eth_stat.sent_bytes);
  printf(" recv: %u frames (%lu bytes)\r\n",
	 eth_stat.recv_frm, eth_stat.recv_bytes);
  printf(" filters: ");
  if (filt_count == 0)
    printf("(none)\r\n");
  else
  {
    printf("%s", (filt_flags & FF_MODEMASK)==FF_ACCEPT ? "accept" : "reject");
    for(i=0 ; i<filt_count ; i++)
      printf(" %04x", filters[i]);
    printf("\r\n");
  }
  
}

/*===========================================================================*
 *				recv_find				     * 
 *===========================================================================*/
PUBLIC phys_bytes recv_find(pkt_length, pkt_type, pkt_class)
unsigned pkt_length;	/* packet length */
unsigned pkt_type;	/* packet type */
unsigned pkt_class;	/* packet class */
{
/* Called by ethernet driver when it wants to know what to do with a packet.
 * Return physical address of buffer to be filled by the driver 
 * if somebody does want this packet.
 * Return null pointer (LEAVE_PKT) if there is no process requesting
 * for a frame. Driver should leave it in board's memory.
 * Return -1 (DISCARD_PKT) if we don't want this particular packet,
 * so the driver can throw it away.
 */
  phys_bytes user_phys;
  register int i;

/* frame filtering stuff */
  if (filt_count > 0)			/* any filters? perform filtering? */
  {
    for (i=0 ; i<filt_count ; i++)
      if (filters[i] == pkt_type)	/* compare packet type to filter */
	if (filt_flags & FF_REJECT)
	  return DISCARD_PKT;		/* in "reject" mode: reject packet */
	else
	  break;			/* in "accept" mode: accept it */

    if ((filt_flags & FF_MODEMASK) == FF_ACCEPT &&	/* "accept" mode? */
	 i == filt_count)		/* tested all filters and no match? */
      return DISCARD_PKT;		/* if so, reject unwanted packet */
  }

  switch(in_status)
  {
    case IN_WAITING:		/* someone wants a frame */
    case IN_HANGING:
    /* Determine address where data is go to. */
      user_phys=numap(in_proc, in_vir, in_count);
      return user_phys;

    case IN_NONE:		/* nobody wants a frame */
      return LEAVE_PKT;

    default:			/* what? impossible, but it's wise to check */
      panic("bad call to recv_find() in ether task", NO_NUM);
  }
}

/*===========================================================================*
 *				recv_copy				     * 
 *===========================================================================*/
PUBLIC void recv_copy(pkt_buf, pkt_length)
phys_bytes pkt_buf;	/* packet buffer */
unsigned pkt_length;	/* packet length */
{
/* Called by ethernet driver after it has copied a packet into the buffer.
 */

  switch(in_status)
  {
    case IN_WAITING:
      in_count = (vir_bytes) pkt_length;	/* store packet length */
      break;

    case IN_HANGING:
    /* If process was suspended then it's time to wake it up. */
      ether_reply(REVIVE, in_caller, in_proc, pkt_length);
      in_status = IN_NONE;	/* processing finished */
      break;

    case IN_NONE:		/* what? impossible, but it's wise to check */
    default:
      panic("bad call to recv_copy() in ether task", NO_NUM);
  }
  eth_stat.recv_frm++;
  eth_stat.recv_bytes += (unsigned long) pkt_length;
}

#endif /* ETHERNET - no code after this line ! */
