/* log.c */

#include <stdio.h>
#include <varargs.h>
#include <sys/time.h>

#include "config.h"
#include "externs.h"
#include "interface.h"

#ifdef LOG_WIPES_OKAY
#include <pwd.h>
#endif

char *quick_unparse(object)
     dbref object;
{
  static char buff[BUFFER_LEN];

  switch (object) {
  case NOTHING:
    sprintf(buff, "*NOTHING*");
    break;
  case HOME:
    sprintf(buff, "*HOME*");
    break;
  default:
    sprintf(buff, "%s(#%d%s)", 
	    Name(object), object, unparse_flags(object, GOD));
  }

  return buff;
}

void start_log(fp, filename)
     FILE **fp;
     const char *filename;
{
  *fp = fopen(filename, "w");
  if (*fp == NULL) {
    fprintf(stderr, "WARNING: cannot open log %s\n", filename);
    *fp = stderr;
  }
}

void end_log(fp)
     FILE *fp;
{
  if (fp != stderr) {
    fprintf(fp, "END OF LOG.\n");
    fflush(fp);
    fclose(fp);
  }
}

void do_log(logtype, player, object, va_alist)
     int logtype;
     dbref player;
     dbref object;
     va_dcl
{
  /* take a log type and format list and args, write to appropriate logfile.
   * log types are defined in db.h
   */

  time_t tt;
  struct tm *ttm;
  char timebuf[16];
  char tbuf1[BUFFER_LEN];
  va_list args;
  char *fmt;
  char unp1[BUFFER_LEN], unp2[BUFFER_LEN];

  va_start(args);
  fmt = va_arg(args, char *);
  (void) vsprintf(tbuf1, fmt, args);
  va_end(args);

  time(&tt);
  ttm = localtime(&tt);
  sprintf(timebuf, "%d%d/%d%d %d%d:%d%d:%d%d", 
	  (((ttm->tm_mon) + 1) / 10), (((ttm->tm_mon) + 1) % 10),
	  (ttm->tm_mday / 10), (ttm->tm_mday % 10),
	  (ttm->tm_hour / 10), (ttm->tm_hour % 10),
	  (ttm->tm_min / 10), (ttm->tm_min % 10),
	  (ttm->tm_sec / 10), (ttm->tm_sec %10));

  switch (logtype) {
  case LT_ERR:
    fprintf(stderr, "%s  ERR: %s\n", timebuf, tbuf1);
    fflush(stderr);
    break;
  case LT_CMD:
    if (options.log_commands) {
      strcpy(unp1, quick_unparse(player));
      strcpy(unp2, quick_unparse(getloc(player)));
      if (Suspect(player))
	fprintf(cmdlog_fp, "%s  CMD: SUSPECT %s in %s: %s\n", 
		timebuf, unp1, unp2, tbuf1);
      else
	fprintf(cmdlog_fp, "%s  CMD: %s in %s: %s\n",
		timebuf, unp1, unp2, tbuf1);
    } else {
      if (Suspect(player)) {
	strcpy(unp1, quick_unparse(player));
	strcpy(unp2, quick_unparse(getloc(player)));
	fprintf(cmdlog_fp, "%s  CMD: SUSPECT %s in %s: %s\n", 
		timebuf, unp1, unp2, tbuf1);
      }
    }
    fflush(cmdlog_fp);
    break;
  case LT_WIZ:
    strcpy(unp1, quick_unparse(player));
    if (GoodObject(object)) {
      strcpy(unp2, quick_unparse(object));
      fprintf(wizlog_fp, "%s  WIZ: %s --> %s: %s\n",
	      timebuf, unp1, unp2, tbuf1);
    } else {
      fprintf(wizlog_fp, "%s  WIZ: %s: %s\n", timebuf, unp1, tbuf1);
    }
    fflush(wizlog_fp);
    break;
  case LT_CONN:
    fprintf(connlog_fp, "%s  NET: %s\n", timebuf, tbuf1);
    fflush(connlog_fp);
    break;
  case LT_TRACE:
    break;
  case LT_RPAGE:
    break;
  case LT_CHECK:
    break;
  case LT_HUH:
    if (!controls(player, getloc(player))) {
      strcpy(unp1, quick_unparse(player));
      strcpy(unp2, quick_unparse(getloc(player)));
      fprintf(cmdlog_fp, "%s  HUH: %s in %s [%s]: %s\n",
	      timebuf, unp1, unp2, Name(Owner(getloc(player))), tbuf1);
      fflush(cmdlog_fp);
    }
    break;
  default:
    fprintf(stderr, "%s  ERR: %s", timebuf, tbuf1);
    fflush(stderr);
  }
}

#ifdef LOG_WIPES_OKAY
void do_logwipe(player, log_name, str)
     dbref player;
     char *log_name;
     char *str;
{
    /* Wipe out a game log. This is intended for those emergencies where
     * the log has grown out of bounds, overflowing the disk quota, etc.
     * Because someone with the god password can use this command to wipe
     * out 'intrusion' traces, we also require the password of the account
     * the game is running on to be given.
     */

    struct passwd *entry;
    char salt[2];

    if (!God(player)) {
	notify(player, "Permission denied.");
	return;
    }

    entry = getpwuid(getuid());
    strncpy(salt, entry->pw_passwd, 2);

    if (strcmp(crypt(salt, str), entry->pw_passwd) != 0) {
	do_log(LT_WIZ, player, NOTHING, 
	       "Invalid attempt to wipe log %s, password %s");
	return;
    }
}
#endif				/* LOG_WIPES_OKAY */
