/*
 * Random artifact generator (randart) by Greg Wooledge.
 * If you're trying to compile this without the Angband 2.8.1 source, it won't
 * work.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include "angband.h"

#ifdef GJW_RANDART

#define NAMES_FILE "names.txt"
#define BUFLEN 1024
#define MAX_NAMES 512
#define MAX_TRIES 200

/* Temporary space for names, while reading them in from the file. */
char *names[MAX_NAMES];
int nnames = 0;
char *a_base;

/* Cache the results of lookup_kind(), which is expensive and would otherwise
   be called much too often. */
s16b kinds[MAX_A_IDX];

/* Global just for convenience. */
int randart_verbose = 0;

int init_names (void);
int scramble (void);
int scramble_names (void);
void choose_item (int a_idx, u32b activates);
void add_ability (artifact_type *);
void remove_contradictory (artifact_type *);
void do_pval (artifact_type *);
void do_curse (artifact_type *);
int artifact_power (int a_idx, bool cannot_use_kind_cache);
int artifacts_acceptable (void);
int bow_multiplier (int sval);
char *my_strdup (const char *);

#define abs(x)	((x) > 0 ? (x) : (-(x)))

/* Hack -- if we're just in the standalone randart a_info generator, then we
   can't use msg_format(). */
#ifdef STANDALONE_RANDART
#include <stdarg.h>
#define msg_format ra_msg_format
void ra_msg_format (cptr fmt, ...)
{
	va_list ap;
	va_start (ap, fmt);
	vprintf (fmt, ap);
	printf ("\n");
	va_end (ap);
}
#endif

int do_randart (u32b randart_seed)
{
	int rc;

	/* Prepare to use the Angband "simple" RNG.  We want this to follow
	   the precedents set by the town layout and object flavor code. */
	Rand_value = randart_seed;
	Rand_quick = TRUE;

	msg_format ("Reading new artifact names...");
	if ((rc = init_names()) != 0) return rc;

	msg_format ("Scrambling artifacts...");
	if ((rc = scramble()) != 0) return rc;

	/* When done, resume use of the Angband "complex" RNG.  Right
	   now this does nothing, but later it will be important! */
	Rand_quick = FALSE;

	return 0;
}

/* Read our list of usable names from an external file. */
int init_names (void)
{
	FILE *f;
	char buf [BUFLEN];

	path_build (buf, BUFLEN, ANGBAND_DIR_FILE, NAMES_FILE);
	if ((f = fopen (buf, "r")) == NULL)
	{
		msg_format ("could not open '" NAMES_FILE "'\n");
		return 1;
	}

	while (nnames < MAX_NAMES && fgets (buf, BUFLEN, f) != NULL)
	{
		int i = strlen (buf) - 1;
		if (buf[i] == '\n') buf[i] = '\0';
		names[nnames] = my_strdup (buf);
		if (names[nnames] == NULL)
		{
			msg_format ("memory allocation error\n");
			return 2;
		}
		nnames++;
	}

	if (nnames == MAX_NAMES)
		msg_format ("(using the first %d names) ", MAX_NAMES);
	fclose (f);

	if (nnames < MAX_A_IDX)
	{
		msg_format ("Not enough names!");
		return 3;
	}

	return 0;
}

/* Note the two special cases (Grond, Morgoth).  Note also that if an artifact
   has an activation, it must be preserved by artifact number, since
   activations are hard-coded into the game. */
int scramble (void)
{
	int a_idx;
	int rc;

	if ((rc = scramble_names()) != 0)
		return rc;

	/* This outer loop is for post-processing.  If our artifact set
	   fails to meet certain criteria, we start over. :-( */
	do
	{
		/* Generate all the artifacts. */
		for (a_idx = 1; a_idx < MAX_A_IDX; a_idx++)
		{
			artifact_type *a_ptr = &a_info[a_idx];
			u32b activates = a_ptr->flags3 & TR3_ACTIVATE;
			int power;
			int tries;
			int ap;
			bool curse_me = FALSE;

			/* Special cases -- don't randomize these! */
			if (a_idx == ART_POWER || a_idx == ART_GROND ||
			    a_idx == ART_MORGOTH)
				continue;

			/* Skip unused artifacts, too! */
			if (a_ptr->tval == 0) continue;

			/* Evaluate the original artifact to determine the power
			   level. */
			power = artifact_power (a_idx, TRUE);
			if (power < 3)
				power = 3;
			if (power > 120)	/* Deathwreaker, Aule, ... */
				power = 120;

			if (randart_verbose)
				msg_format ("Artifact %d: power = %d",
					a_idx, power);

			if (a_idx >= ART_MIN_NORMAL)
			{
				/* Normal artifact - choose a random base item
				   type.  Not too powerful, so we'll have to
				   add something to it.  Not too weak, for the
				   opposite reason. */
				int ap2, count = 0;
				do
				{
					choose_item (a_idx, activates);
					ap2 = artifact_power (a_idx, FALSE);
					count++;
				} while ((count < MAX_TRIES) &&
					   ((ap2 > (power * 8) / 10 + 1) ||
					    (ap2 < (power / 10))));

				/* These will always be hideous. */
				if (a_idx == ART_BERUTHIEL ||
				    a_idx == ART_GORLIM ||
				    a_idx == ART_CAMLOST ||
				    a_idx == ART_MORMEGIL)
				    	curse_me = TRUE;
			}
			else
			{
				/* Special artifact (light source, ring, or
				   amulet).  Clear the following fields; leave
				   the rest alone. */
				a_ptr->pval = 0;
				a_ptr->to_h = a_ptr->to_d = a_ptr->to_a = 0;
				a_ptr->flags1 = a_ptr->flags2 = 0;
				a_ptr->flags3 = (TR3_IGNORE_ACID |
						 TR3_IGNORE_ELEC |
						 TR3_IGNORE_FIRE |
						 TR3_IGNORE_COLD);
			}

			/* Select a random set of abilities which roughly
			   matches the original's in terms of overall
			   power/usefulness. */
			for (tries = 0; tries < MAX_TRIES; tries++)
			{
				artifact_type atmp;

				/* Copy artifact info temporarily. */
				atmp = *a_ptr;
				add_ability (a_ptr);
				ap = artifact_power (a_idx, FALSE);
				if (ap > (power * 11) / 10 + 1)
				{	/* too powerful -- put it back */
					*a_ptr = atmp;
					continue;
				}

				else if (ap >= (power * 9) / 10)	/* just right */
				{
					/* If we've got a "bad" one, curse it. */
					if (curse_me)
					{
						do_curse (a_ptr);
						do_curse (a_ptr);
						remove_contradictory (a_ptr);
						ap = artifact_power (a_idx, FALSE);
					}
					break;
				}

				/* Stop if we're going negative, so we don't
				   overload the artifact with great powers to
				   compensate. */
				else if ((ap < 0) && (ap < (-(power * 1)) / 10))
				{
					break;
				}
			}			/* end of power selection */

			a_ptr->cost = ap * (s32b)1000;

			if (a_ptr->cost < 0) a_ptr->cost = 0;

			/* One last hack: if the artifact is very powerful,
			   raise the rarity.  This compensates for artifacts
			   like (original) Bladeturner, which have low artifact
			   rarities but came from extremely-rare base kinds. */
			if ((ap > 0) && ((ap / 8) > a_ptr->rarity))
				a_ptr->rarity = ap / 8;

			if (activates) a_ptr->flags3 |= TR3_ACTIVATE;
			if (a_idx < ART_MIN_NORMAL) a_ptr->flags3 |= TR3_INSTA_ART;

			/* Add TR3_HIDE_TYPE to all artifacts because we're
			   too lazy to find out which ones need it and which
			   ones don't. */
			a_ptr->flags3 |= TR3_HIDE_TYPE;
		} 				/* end of this artifact */
	} while (!artifacts_acceptable());	/* end of all artifacts */

	return 0;
}

int scramble_names (void)
{
	int i;
	size_t name_size;
	char *a_next;

	/* Mix up the names. */
	for (i = 0; i < nnames; i++)
	{
		int j = rand_int (nnames);
		if (i != j)
		{
			char *tmp = names[i];
			names[i] = names[j];
			names[j] = tmp;
		}
	}

	/* Special cases -- keep these three names separate. */
	free (names [ART_POWER - 1]);
	free (names [ART_GROND - 1]);
	free (names [ART_MORGOTH - 1]);
	if ((names [ART_POWER - 1] = my_strdup ("of Power (The One Ring)")) == NULL)
	{
		msg_format ("Memory allocation error");
		return 1;
	}
	if ((names [ART_GROND - 1] = my_strdup ("'Grond'")) == NULL)
	{
		msg_format ("Memory allocation error");
		return 1;
	}
	if ((names [ART_MORGOTH - 1] = my_strdup ("of Morgoth")) == NULL)
	{
		msg_format ("Memory allocation error");
		return 1;
	}

	/* Convert our names array into an a_name structure for later use. */
	name_size = 0;
	for (i = 1; i < MAX_A_IDX; i++)
	{
		name_size += strlen (names[i-1]) + 2;	/* skip first char */
	}
	if ((a_base = malloc (name_size)) == NULL)
	{
		msg_format ("Memory allocation error");
		return 1;
	}
	a_next = a_base + 1;	/* skip first char */
	for (i = 1; i < MAX_A_IDX; i++)
	{
		strcpy (a_next, names[i-1]);
		if (a_info[i].tval > 0)		/* skip unused! */
			a_info[i].name = a_next - a_base;
		a_next += strlen (names[i-1]) + 1;
	}

	/* Free some of our now unneeded memory. */
	KILL (a_name, char);
	for (i = 0; i < nnames; i++)
	{
		free (names [i]);
	}
	a_name = a_base;

	return 0;
}

/* Randomly select a base item type (tval,sval).  Assign the various fields
   corresponding to that choice. */
void choose_item (int a_idx, u32b activates)
{
	artifact_type *a_ptr = &a_info[a_idx];
	int tval, sval;
	object_kind *k_ptr;
	int r;
	s16b k_idx, r2;

	r = rand_int (100);
	if (r < 5)
	{
		/* Create a missile weapon. */
		tval = TV_BOW;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 10) sval = SV_SLING;
		else if (r2 < 20) sval = SV_SHORT_BOW;
		else if (r2 < 50) sval = SV_LONG_BOW;
		else if (r2 < 70) sval = SV_LIGHT_XBOW;
		else sval = SV_HEAVY_XBOW;
	}
	else if (r < 9)
	{
		/* Create a digging tool. */
		tval = TV_DIGGING;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 10) sval = SV_SHOVEL;
		else if (r2 < 20) sval = SV_PICK;
		else if (r2 < 40) sval = SV_GNOMISH_SHOVEL;
		else if (r2 < 50) sval = SV_ORCISH_PICK;
		else if (r2 < 60) sval = SV_DWARVEN_SHOVEL;
		else sval = SV_DWARVEN_PICK;
	}
	else if (r < 19)
	{
		/* Create a "blunt" weapon. */
		tval = TV_HAFTED;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 8) sval = SV_WHIP;
		else if (r2 < 15) sval = SV_QUARTERSTAFF;
		else if (r2 < 25) sval = SV_MACE;
		else if (r2 < 30) sval = SV_BALL_AND_CHAIN;
		else if (r2 < 40) sval = SV_WAR_HAMMER;
		else if (r2 < 45) sval = SV_LUCERN_HAMMER;
		else if (r2 < 50) sval = SV_MORNING_STAR;
		else if (r2 < 60) sval = SV_FLAIL;
		else if (r2 < 70) sval = SV_LEAD_FILLED_MACE;
		else if (r2 < 80) sval = SV_TWO_HANDED_FLAIL;
		else sval = SV_MACE_OF_DISRUPTION;
	}
	else if (r < 33)
	{
		/* Create a long, sharp-edged weapon. */
		tval = TV_SWORD;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < -10) sval = SV_BROKEN_DAGGER;
		else if (r2 < 0) sval = SV_BROKEN_SWORD;
		else if (r2 < 4) sval = SV_DAGGER;
		else if (r2 < 6) sval = SV_MAIN_GAUCHE;
		else if (r2 < 8) sval = SV_RAPIER;	/* or at least pointy ;-) */
		else if (r2 < 12) sval = SV_SMALL_SWORD;
		else if (r2 < 16) sval = SV_SHORT_SWORD;
		else if (r2 < 20) sval = SV_SABRE;
		else if (r2 < 24) sval = SV_CUTLASS;
		else if (r2 < 26) sval = SV_TULWAR;
		else if (r2 < 32) sval = SV_BROAD_SWORD;
		else if (r2 < 36) sval = SV_LONG_SWORD;
		else if (r2 < 44) sval = SV_SCIMITAR;
		else if (r2 < 50) sval = SV_KATANA;
		else if (r2 < 55) sval = SV_BASTARD_SWORD;
		else if (r2 < 70) sval = SV_TWO_HANDED_SWORD;
		else if (r2 < 80) sval = SV_EXECUTIONERS_SWORD;
		else sval = SV_BLADE_OF_CHAOS;
	}
	else if (r < 42)
	{
		/* Create a weapon that's not blunt or sword-shaped. */
		tval = TV_POLEARM;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 10) sval = SV_SPEAR;
		else if (r2 < 15) sval = SV_AWL_PIKE;
		else if (r2 < 20) sval = SV_TRIDENT;
		else if (r2 < 25) sval = SV_PIKE;
		else if (r2 < 30) sval = SV_BEAKED_AXE;
		else if (r2 < 35) sval = SV_BROAD_AXE;
		else if (r2 < 40) sval = SV_GLAIVE;
		else if (r2 < 45) sval = SV_HALBERD;
		else if (r2 < 50) sval = SV_SCYTHE;
		else if (r2 < 55) sval = SV_LANCE;
		else if (r2 < 60) sval = SV_BATTLE_AXE;
		else if (r2 < 70) sval = SV_GREAT_AXE;
		else if (r2 < 80) sval = SV_LOCHABER_AXE;
		else sval = SV_SCYTHE_OF_SLICING;
	}
	else if (r < 50)
	{
		/*
		 * Create light body armor.  This category and the
		 * TV_HARD_ARMOR category are oddballs -- they're really the
		 * same category, but inconveniently split apart.  The bad
		 * part is that I can't make the high-end "soft" armors
		 * appear at the right level, or we'd be inundated with
		 * leather scale mail.  Ideally, the choice between "soft" and
		 * "hard" would be level-based as well....
		 */
		tval = TV_SOFT_ARMOR;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 0) sval = SV_FILTHY_RAG;
		else if (r2 < 10) sval = SV_ROBE;
		else if (r2 < 20) sval = SV_SOFT_LEATHER_ARMOR;
		else if (r2 < 30) sval = SV_SOFT_STUDDED_LEATHER;
		else if (r2 < 50) sval = SV_HARD_LEATHER_ARMOR;
		else if (r2 < 70) sval = SV_HARD_STUDDED_LEATHER;
		else sval = SV_LEATHER_SCALE_MAIL;
	}
	else if (r < 64)
	{
		/* Create heavy body armor.  See above for gripes.  In a
		   similar vein, these aren't the "right" levels, but making
		   them appear at the right level would inundate us with
		   rusty chain mail. :-( */
		tval = TV_HARD_ARMOR;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 5) sval = SV_RUSTY_CHAIN_MAIL;
		else if (r2 < 10) sval = SV_METAL_SCALE_MAIL;
		else if (r2 < 20) sval = SV_CHAIN_MAIL;
		else if (r2 < 25) sval = SV_AUGMENTED_CHAIN_MAIL;
		else if (r2 < 30) sval = SV_DOUBLE_CHAIN_MAIL;
		else if (r2 < 35) sval = SV_BAR_CHAIN_MAIL;
		else if (r2 < 40) sval = SV_METAL_BRIGANDINE_ARMOUR;
		else if (r2 < 45) sval = SV_PARTIAL_PLATE_ARMOUR;
		else if (r2 < 50) sval = SV_METAL_LAMELLAR_ARMOUR;
		else if (r2 < 60) sval = SV_FULL_PLATE_ARMOUR;
		else if (r2 < 70) sval = SV_RIBBED_PLATE_ARMOUR;
		else if (r2 < 75) sval = SV_MITHRIL_CHAIN_MAIL;
		else if (r2 < 80) sval = SV_MITHRIL_PLATE_MAIL;
		else sval = SV_ADAMANTITE_PLATE_MAIL;
	}
	else if (r < 71)
	{
		/* Make shoes. */
		tval = TV_BOOTS;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 20) sval = SV_PAIR_OF_SOFT_LEATHER_BOOTS;
		else if (r2 < 40) sval = SV_PAIR_OF_HARD_LEATHER_BOOTS;
		else sval = SV_PAIR_OF_METAL_SHOD_BOOTS;
	}
	else if (r < 78)
	{
		/* Make gloves. */
		tval = TV_GLOVES;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 20) sval = SV_SET_OF_LEATHER_GLOVES;
		else if (r2 < 40) sval = SV_SET_OF_GAUNTLETS;
		else sval = SV_SET_OF_CESTI;
	}
	else if (r < 83)
	{
		/* Make headgear. */
		tval = TV_HELM;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 10) sval = SV_HARD_LEATHER_CAP;
		else if (r2 < 30) sval = SV_METAL_CAP;
		else if (r2 < 50) sval = SV_IRON_HELM;
		else sval = SV_STEEL_HELM;
	}
	else if (r < 87)
	{
		/* Make thrilling headgear. */
		tval = TV_CROWN;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 20) sval = SV_IRON_CROWN;
		else if (r2 < 50) sval = SV_GOLDEN_CROWN;
		else sval = SV_JEWELED_CROWN;
	}
	else if (r < 94)
	{
		/* Make a shield. */
		tval = TV_SHIELD;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 10) sval = SV_SMALL_LEATHER_SHIELD;
		else if (r2 < 25) sval = SV_SMALL_METAL_SHIELD;
		else if (r2 < 40) sval = SV_LARGE_LEATHER_SHIELD;
		else sval = SV_LARGE_METAL_SHIELD;
	}
	else
	{
		/* Make a cloak. */
		tval = TV_CLOAK;
		r2 = randnor (a_ptr->level + 10, 20);
		if (r2 < 40) sval = SV_CLOAK;
		else sval = SV_SHADOW_CLOAK;
	}

	k_idx = lookup_kind (tval, sval);
	k_ptr = &k_info[k_idx];
	kinds[a_idx] = k_idx;

	a_ptr->tval = k_ptr->tval;
	a_ptr->sval = k_ptr->sval;
	a_ptr->pval = k_ptr->pval;
	a_ptr->to_h = k_ptr->to_h;
	a_ptr->to_d = k_ptr->to_d;
	a_ptr->to_a = k_ptr->to_a;
	a_ptr->ac = k_ptr->ac;
	a_ptr->dd = k_ptr->dd;
	a_ptr->ds = k_ptr->ds;
	a_ptr->weight = k_ptr->weight;
	a_ptr->flags1 = k_ptr->flags1;
	a_ptr->flags2 = k_ptr->flags2;
	a_ptr->flags3 = k_ptr->flags3;
	a_ptr->flags3 |= (TR3_IGNORE_ACID | TR3_IGNORE_ELEC |
			   TR3_IGNORE_FIRE | TR3_IGNORE_COLD);

	switch (a_ptr->tval)
	{
		case TV_BOW: case TV_DIGGING: case TV_HAFTED:
		case TV_SWORD: case TV_POLEARM:
			a_ptr->to_h += a_ptr->level / 10 + rand_int (4) + rand_int (4);
			a_ptr->to_d += a_ptr->level / 10 + rand_int (4) + rand_int ((a_ptr->dd * a_ptr->ds) / 2 + 1);
			break;
		case TV_BOOTS: case TV_GLOVES: case TV_HELM: case TV_CROWN:
		case TV_SHIELD: case TV_CLOAK: case TV_SOFT_ARMOR:
		case TV_HARD_ARMOR:
			a_ptr->to_a += a_ptr->level / 10 + a_ptr->ac / 3 + rand_int (8);
			if (a_ptr->to_a < 10) a_ptr->to_a += 2 + rand_int (4) + rand_int (4);
			/*
			 * ** MEGA-HACK **  Make sure armor gets some !^&@)*
			 * resists!  Hard body armor is generally high-level
			 * stuff, with good ac and to_a.  That sucks up all the
			 * points....
			 */
			switch (a_ptr->tval)
			{
			case TV_SOFT_ARMOR: case TV_HARD_ARMOR:
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_RES_ACID;
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_RES_ELEC;
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_RES_COLD;
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_RES_FIRE;
				break;
			}
			break;
	}
}

/* Randomly select an extra ability to be added to the artifact in question.
   This function is way too large. */
void add_ability (artifact_type *a_ptr)
{
	int r;

	r = rand_int (10);
	if (r < 5)		/* Pick something dependent on item type. */
	{
		r = rand_int (100);
		switch (a_ptr->tval)
		{
			case TV_BOW:
			{
				if (r < 15)
				{
					a_ptr->flags1 |= TR1_SHOTS;
					do_pval (a_ptr);
				}
				else if (r < 35)
				{
					a_ptr->flags1 |= TR1_MIGHT;
					do_pval (a_ptr);
				}
				else if (r < 65) a_ptr->to_h += 2 + rand_int (2);
				else a_ptr->to_d += 2 + rand_int (3);

				break;
			}
			case TV_DIGGING:
			case TV_HAFTED:
			case TV_POLEARM:
			case TV_SWORD:
			{
				if (r < 4)
				{
					a_ptr->flags1 |= TR1_WIS;
					do_pval (a_ptr);
					if (rand_int (2) == 0) a_ptr->flags2 |= TR2_SUST_WIS;
					if (a_ptr->tval == TV_SWORD || a_ptr->tval == TV_POLEARM)
						a_ptr->flags3 |= TR3_BLESSED;
				}
				else if (r < 7)
				{
					a_ptr->flags1 |= TR1_BRAND_ACID;
					if (rand_int (4) > 0) a_ptr->flags2 |= TR2_RES_ACID;
				}
				else if (r < 10)
				{
					a_ptr->flags1 |= TR1_BRAND_ELEC;
					if (rand_int (4) > 0) a_ptr->flags2 |= TR2_RES_ELEC;
				}
				else if (r < 15)
				{
					a_ptr->flags1 |= TR1_BRAND_FIRE;
					if (rand_int (4) > 0) a_ptr->flags2 |= TR2_RES_FIRE;
				}
				else if (r < 20)
				{
					a_ptr->flags1 |= TR1_BRAND_COLD;
					if (rand_int (4) > 0) a_ptr->flags2 |= TR2_RES_COLD;
				}
				else if (r < 28)
				{
					a_ptr->dd += 1 + rand_int (2) + rand_int (2);
					if (a_ptr->dd > 9) a_ptr->dd = 9;
				}
				else if (r < 31) a_ptr->flags1 |= TR1_KILL_DRAGON;
				else if (r < 35) a_ptr->flags1 |= TR1_SLAY_DRAGON;
				else if (r < 40) a_ptr->flags1 |= TR1_SLAY_EVIL;

				else if (r < 45) a_ptr->flags1 |= TR1_SLAY_ANIMAL;
				else if (r < 50)
				{
					a_ptr->flags1 |= TR1_SLAY_UNDEAD;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_DEMON;
				}
				else if (r < 54)
				{
					a_ptr->flags1 |= TR1_SLAY_DEMON;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_UNDEAD;
				}
				else if (r < 59)
				{
					a_ptr->flags1 |= TR1_SLAY_ORC;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_TROLL;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_GIANT;
				}
				else if (r < 63)
				{
					a_ptr->flags1 |= TR1_SLAY_TROLL;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_ORC;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_GIANT;
				}
				else if (r < 67)
				{
					a_ptr->flags1 |= TR1_SLAY_GIANT;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_ORC;
					if (rand_int (2) == 0) a_ptr->flags1 |= TR1_SLAY_TROLL;
				}
				else if (r < 72) a_ptr->flags3 |= TR3_SEE_INVIS;
				else if (r < 76)
				{
					if (a_ptr->pval < 0) break;
					a_ptr->flags1 |= TR1_BLOWS;
					do_pval (a_ptr);
				}
				else if (r < 89)
				{
					a_ptr->to_d += 3 + rand_int (4);
					a_ptr->to_h += 3 + rand_int (4);
				}
				else if (r < 92) a_ptr->to_a += 3 + rand_int (3);
				else if (r < 98)
					a_ptr->weight = (a_ptr->weight * 9) / 10;
				else
					if (a_ptr->tval != TV_DIGGING)
					{
						a_ptr->flags1 |= TR1_TUNNEL;
						do_pval (a_ptr);
					}

				break;
			}
			case TV_BOOTS:
			{
				if (r < 10) a_ptr->flags3 |= TR3_FEATHER;
				else if (r < 50) a_ptr->to_a += 2 + rand_int (4);
				else if (r < 80)
				{
					a_ptr->flags1 |= TR1_STEALTH;
					do_pval (a_ptr);
				}
				else if (r < 90)
				{
					a_ptr->flags1 |= TR1_SPEED;
					if (a_ptr->pval < 0) break;
					if (a_ptr->pval == 0) a_ptr->pval = 3 + rand_int (8);
					else if (rand_int (2) == 0) a_ptr->pval++;
				}
				else a_ptr->weight = (a_ptr->weight * 9) / 10;
				break;
			}
			case TV_GLOVES:
			{
				if (r < 25) a_ptr->flags3 |= TR3_FREE_ACT;
				else if (r < 50)
				{
					a_ptr->flags1 |= TR1_DEX;
					do_pval (a_ptr);
				}
				else if (r < 75) a_ptr->to_a += 3 + rand_int (3);
				else
				{
					a_ptr->to_h += 2 + rand_int (3);
					a_ptr->to_d += 2 + rand_int (3);
					a_ptr->flags3 |= TR3_SHOW_MODS;
				}
				break;
			}
			case TV_HELM:
			case TV_CROWN:
			{
				if (r < 20) a_ptr->flags2 |= TR2_RES_BLIND;
				else if (r < 45) a_ptr->flags3 |= TR3_TELEPATHY;
				else if (r < 65) a_ptr->flags3 |= TR3_SEE_INVIS;
				else if (r < 75)
				{
					a_ptr->flags1 |= TR1_WIS;
					do_pval (a_ptr);
				}
				else if (r < 85)
				{
					a_ptr->flags1 |= TR1_INT;
					do_pval (a_ptr);
				}
				else a_ptr->to_a += 3 + rand_int (3);
				break;
			}
			case TV_SHIELD:
			{
				if (r < 20) a_ptr->flags2 |= TR2_RES_ACID;
				else if (r < 40) a_ptr->flags2 |= TR2_RES_ELEC;
				else if (r < 60) a_ptr->flags2 |= TR2_RES_FIRE;
				else if (r < 80) a_ptr->flags2 |= TR2_RES_COLD;
				else a_ptr->to_a += 3 + rand_int (3);
				break;
			}
			case TV_CLOAK:
			{
				if (r < 50)
				{
					a_ptr->flags1 |= TR1_STEALTH;
					do_pval (a_ptr);
				}
				else a_ptr->to_a += 3 + rand_int (3);
				break;
			}
			case TV_SOFT_ARMOR:
			case TV_HARD_ARMOR:
			{
				if (r < 8)
				{
					a_ptr->flags1 |= TR1_STEALTH;
					do_pval (a_ptr);
				}
				else if (r < 16) a_ptr->flags3 |= TR3_HOLD_LIFE;
				else if (r < 22)
				{
					a_ptr->flags1 |= TR1_CON;
					do_pval (a_ptr);
					if (rand_int (2) == 0)
						a_ptr->flags2 |= TR2_SUST_CON;
				}
				else if (r < 34) a_ptr->flags2 |= TR2_RES_ACID;
				else if (r < 46) a_ptr->flags2 |= TR2_RES_ELEC;
				else if (r < 58) a_ptr->flags2 |= TR2_RES_FIRE;
				else if (r < 70) a_ptr->flags2 |= TR2_RES_COLD;
				else if (r < 80)
					a_ptr->weight = (a_ptr->weight * 9) / 10;
				else a_ptr->to_a += 3 + rand_int (3);
				break;
			}
		}
	}
	else			/* Pick something universally useful. */
	{
		r = rand_int (47);
		switch (r)
		{
			case 0:
				a_ptr->flags1 |= TR1_STR;
				do_pval (a_ptr);
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_SUST_STR;
				break;
			case 1:
				a_ptr->flags1 |= TR1_INT;
				do_pval (a_ptr);
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_SUST_INT;
				break;
			case 2:
				a_ptr->flags1 |= TR1_WIS;
				do_pval (a_ptr);
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_SUST_WIS;
				if (a_ptr->tval == TV_SWORD || a_ptr->tval == TV_POLEARM)
					a_ptr->flags3 |= TR3_BLESSED;
				break;
			case 3:
				a_ptr->flags1 |= TR1_DEX;
				do_pval (a_ptr);
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_SUST_DEX;
				break;
			case 4:
				a_ptr->flags1 |= TR1_CON;
				do_pval (a_ptr);
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_SUST_CON;
				break;
			case 5:
				a_ptr->flags1 |= TR1_CHR;
				do_pval (a_ptr);
				if (rand_int (2) == 0) a_ptr->flags2 |= TR2_SUST_CHR;
				break;

			case 6:
				a_ptr->flags1 |= TR1_STEALTH;
				do_pval (a_ptr);
				break;
			case 7:
				a_ptr->flags1 |= TR1_SEARCH;
				do_pval (a_ptr);
				break;
			case 8:
				a_ptr->flags1 |= TR1_INFRA;
				do_pval (a_ptr);
				break;
			case 9:
				a_ptr->flags1 |= TR1_SPEED;
				if (a_ptr->pval == 0) a_ptr->pval = 3 + rand_int (3);
				else do_pval (a_ptr);
				break;

			case 10:
				a_ptr->flags2 |= TR2_SUST_STR;
				if (rand_int (2) == 0)
				{
					a_ptr->flags1 |= TR1_STR;
					do_pval (a_ptr);
				}
				break;
			case 11:
				a_ptr->flags2 |= TR2_SUST_INT;
				if (rand_int (2) == 0)
				{
					a_ptr->flags1 |= TR1_INT;
					do_pval (a_ptr);
				}
				break;
			case 12:
				a_ptr->flags2 |= TR2_SUST_WIS;
				if (rand_int (2) == 0)
				{
					a_ptr->flags1 |= TR1_WIS;
					do_pval (a_ptr);
					if (a_ptr->tval == TV_SWORD || a_ptr->tval == TV_POLEARM)
						a_ptr->flags3 |= TR3_BLESSED;
				}
				break;
			case 13:
				a_ptr->flags2 |= TR2_SUST_DEX;
				if (rand_int (2) == 0)
				{
					a_ptr->flags1 |= TR1_DEX;
					do_pval (a_ptr);
				}
				break;
			case 14:
				a_ptr->flags2 |= TR2_SUST_CON;
				if (rand_int (2) == 0)
				{
					a_ptr->flags1 |= TR1_CON;
					do_pval (a_ptr);
				}
				break;
			case 15:
				a_ptr->flags2 |= TR2_SUST_CHR;
				if (rand_int (2) == 0)
				{
					a_ptr->flags1 |= TR1_CHR;
					do_pval (a_ptr);
				}
				break;

			case 16:
			{
				if (rand_int (3) == 0) a_ptr->flags2 |= TR2_IM_ACID;
				break;
			}
			case 17:
			{
				if (rand_int (3) == 0) a_ptr->flags2 |= TR2_IM_ELEC;
				break;
			}
			case 18:
			{
				if (rand_int (4) == 0) a_ptr->flags2 |= TR2_IM_FIRE;
				break;
			}
			case 19:
			{
				if (rand_int (3) == 0) a_ptr->flags2 |= TR2_IM_COLD;
				break;
			}
			case 20: a_ptr->flags3 |= TR3_FREE_ACT; break;
			case 21: a_ptr->flags3 |= TR3_HOLD_LIFE; break;
			case 22: a_ptr->flags2 |= TR2_RES_ACID; break;
			case 23: a_ptr->flags2 |= TR2_RES_ELEC; break;
			case 24: a_ptr->flags2 |= TR2_RES_FIRE; break;
			case 25: a_ptr->flags2 |= TR2_RES_COLD; break;

			case 26: a_ptr->flags2 |= TR2_RES_POIS; break;
			case 27: a_ptr->flags2 |= TR2_RES_LITE; break;
			case 28: a_ptr->flags2 |= TR2_RES_DARK; break;
			case 29: a_ptr->flags2 |= TR2_RES_BLIND; break;
			case 30: a_ptr->flags2 |= TR2_RES_CONFU; break;
			case 31: a_ptr->flags2 |= TR2_RES_SOUND; break;
			case 32: a_ptr->flags2 |= TR2_RES_SHARD; break;
			case 33:
				if (rand_int (2) == 0)
					a_ptr->flags2 |= TR2_RES_NETHR;
				break;
			case 34: a_ptr->flags2 |= TR2_RES_NEXUS; break;
			case 35: a_ptr->flags2 |= TR2_RES_CHAOS; break;
			case 36:
				if (rand_int (2) == 0)
					a_ptr->flags2 |= TR2_RES_DISEN;
				break;
			case 37: a_ptr->flags3 |= TR3_FEATHER; break;
			case 38: a_ptr->flags3 |= TR3_LITE; break;
			case 39: a_ptr->flags3 |= TR3_SEE_INVIS; break;
			case 40:
				if (rand_int (3) == 0)
					a_ptr->flags3 |= TR3_TELEPATHY;
				break;
			case 41: a_ptr->flags3 |= TR3_SLOW_DIGEST; break;

			case 42:
			case 43:
				a_ptr->flags3 |= TR3_REGEN; break;

			case 44:
				if (rand_int (5) == 0)
				{
					a_ptr->flags3 |= TR3_DRAIN_EXP;
					do_curse (a_ptr);
				}
				break;
			case 45:
				if (rand_int (10) == 0)
				{
					a_ptr->flags3 |= TR3_TELEPORT;
					do_curse (a_ptr);
				}
				break;
			case 46:
				a_ptr->flags3 |= TR3_AGGRAVATE;
				if (rand_int (3) == 0) do_curse (a_ptr);
				break;
		}
	}

	/* Now remove contradictory or redundant powers. */
	remove_contradictory (a_ptr);
}

void remove_contradictory (artifact_type *a_ptr)
{
	if (a_ptr->flags3 & TR3_AGGRAVATE) a_ptr->flags1 &= ~(TR1_STEALTH);
	if (a_ptr->flags2 & TR2_IM_ACID) a_ptr->flags2 &= ~(TR2_RES_ACID);
	if (a_ptr->flags2 & TR2_IM_ELEC) a_ptr->flags2 &= ~(TR2_RES_ELEC);
	if (a_ptr->flags2 & TR2_IM_FIRE) a_ptr->flags2 &= ~(TR2_RES_FIRE);
	if (a_ptr->flags2 & TR2_IM_COLD) a_ptr->flags2 &= ~(TR2_RES_COLD);
	if (a_ptr->pval < 0)
	{
		if (a_ptr->flags1 & TR1_STR) a_ptr->flags2 &= ~(TR2_SUST_STR);
		if (a_ptr->flags1 & TR1_INT) a_ptr->flags2 &= ~(TR2_SUST_INT);
		if (a_ptr->flags1 & TR1_WIS) a_ptr->flags2 &= ~(TR2_SUST_WIS);
		if (a_ptr->flags1 & TR1_DEX) a_ptr->flags2 &= ~(TR2_SUST_DEX);
		if (a_ptr->flags1 & TR1_CON) a_ptr->flags2 &= ~(TR2_SUST_CON);
		if (a_ptr->flags1 & TR1_CHR) a_ptr->flags2 &= ~(TR2_SUST_CHR);
		a_ptr->flags1 &= ~(TR1_BLOWS);
	}
	if (a_ptr->flags3 & TR3_LIGHT_CURSE) a_ptr->flags3 &= ~(TR3_BLESSED);
	if (a_ptr->flags1 & TR1_KILL_DRAGON) a_ptr->flags1 &= ~(TR1_SLAY_DRAGON);
	if (a_ptr->flags3 & TR3_DRAIN_EXP) a_ptr->flags3 &= ~(TR3_HOLD_LIFE);
}

/* We've just added an ability which uses the pval bonus.  Make sure it's
   not zero.  If it's currently negative, leave it negative (heh heh). */
void do_pval (artifact_type *a_ptr)
{
	if (a_ptr->pval == 0) a_ptr->pval = 1 + rand_int (3);
	else if (a_ptr->pval < 0)
	{
		if (rand_int (2) == 0) a_ptr->pval--;
	}
	else if (rand_int (3) > 0) a_ptr->pval++;
}

/* Make it bad, or if it's already bad, make it worse! */
void do_curse (artifact_type *a_ptr)
{
	if ((a_ptr->pval > 0) && (rand_int (2) == 0))
		a_ptr->pval = -a_ptr->pval;
	if ((a_ptr->to_a > 0) && (rand_int (2) == 0))
		a_ptr->to_a = -a_ptr->to_a;
	if ((a_ptr->to_h > 0) && (rand_int (2) == 0))
		a_ptr->to_h = -a_ptr->to_h;
	if ((a_ptr->to_d > 0) && (rand_int (4) == 0))
		a_ptr->to_d = -a_ptr->to_d;

	if (a_ptr->flags3 & TR3_HEAVY_CURSE)
	{
		if (rand_int (3) == 0) a_ptr->flags3 |= TR3_PERMA_CURSE;
		return;
	}

	if (a_ptr->flags3 & TR3_LIGHT_CURSE)
	{
		if (rand_int (2) == 0) a_ptr->flags3 |= TR3_HEAVY_CURSE;
		return;
	}

	a_ptr->flags3 |= TR3_LIGHT_CURSE;
	if (rand_int (4) == 0) a_ptr->flags3 |= TR3_HEAVY_CURSE;
}

/* Evaluate the artifact's overall power level.  Perversely, curses and other
   bad things add to the artifact's "power" level; if that weren't the case,
   cursed artifacts would have too much extra power to compensate. */
int artifact_power (int a_idx, bool cannot_use_kind_cache)
{
	const artifact_type *a_ptr = &a_info[a_idx];
	int p = 0;
	s16b k_idx;
	object_kind *k_ptr = NULL;	/* silence the warning */

	/* Start with a "power" rating derived from the base item's level. */
	if (a_idx >= ART_MIN_NORMAL)
	{
		if (cannot_use_kind_cache)
			k_idx = lookup_kind (a_ptr->tval, a_ptr->sval);
		else
			k_idx = kinds [a_idx];

		if (k_idx)
		{
			k_ptr = &k_info[k_idx];
			p = k_ptr->level / 8;
		}
		/* Otherwise just forget it and use 0. ;-) */
	}

	/* Evaluate certain abilities based on type of object. */
	switch (a_ptr->tval)
	{
		case TV_BOW:
		{
			int mult;

			p += a_ptr->to_d / 2;
			mult = bow_multiplier (a_ptr->sval);
			if (a_ptr->flags1 & TR1_MIGHT) mult += a_ptr->pval;
			p += mult * 2;
			if (a_ptr->flags1 & TR1_SHOTS)
				if (a_ptr->pval > 0) p *= (2 * a_ptr->pval);
			p += (abs(a_ptr->to_h) + 3) / 4;	/* note */
			if (a_ptr->weight < k_ptr->weight) p++;
			break;
		}
		case TV_DIGGING:
		case TV_HAFTED:
		case TV_POLEARM:
		case TV_SWORD:
		{
			p += (a_ptr->dd * a_ptr->ds) / 2;
			if (a_ptr->flags1 & TR1_SLAY_EVIL) p = (p * 3) / 2;
			if (a_ptr->flags1 & TR1_KILL_DRAGON) p = (p * 3) / 2;
			if (a_ptr->flags1 & TR1_SLAY_ANIMAL) p = (p * 4) / 3;
			if (a_ptr->flags1 & TR1_SLAY_UNDEAD) p = (p * 4) / 3;
			if (a_ptr->flags1 & TR1_SLAY_DRAGON) p = (p * 4) / 3;
			if (a_ptr->flags1 & TR1_SLAY_DEMON) p = (p * 5) / 4;
			if (a_ptr->flags1 & TR1_SLAY_TROLL) p = (p * 5) / 4;
			if (a_ptr->flags1 & TR1_SLAY_ORC) p = (p * 5) / 4;
			if (a_ptr->flags1 & TR1_SLAY_GIANT) p = (p * 6) / 5;

			if (a_ptr->flags1 & TR1_BRAND_ACID) p = p * 2;
			if (a_ptr->flags1 & TR1_BRAND_ELEC) p = (p * 3) / 2;
			if (a_ptr->flags1 & TR1_BRAND_FIRE) p = (p * 4) / 3;
			if (a_ptr->flags1 & TR1_BRAND_COLD) p = (p * 4) / 3;

			p += a_ptr->to_d / 3;
			if (a_ptr->to_d > 15) p += (a_ptr->to_d - 14) / 2;

			if (a_ptr->flags1 & TR1_BLOWS)
			{
				if (a_ptr->pval > 3)
					p += 20000;	/* inhibit */
				else if (a_ptr->pval > 0)
					p = (p * 6) / (4 - a_ptr->pval);
			}

			if ((a_ptr->flags1 & TR1_TUNNEL) &&
			    (a_ptr->tval != TV_DIGGING))
				p += a_ptr->pval * 3;

			p += (abs(a_ptr->to_h) + 3) / 4;	/* note */

			/* Remember, weight is in 0.1 lb. units. */
			if (a_ptr->weight != k_ptr->weight)
				p += (k_ptr->weight - a_ptr->weight) / 20;

			break;
		}
		case TV_BOOTS:
		case TV_GLOVES:
		case TV_HELM:
		case TV_CROWN:
		case TV_SHIELD:
		case TV_CLOAK:
		case TV_SOFT_ARMOR:
		case TV_HARD_ARMOR:
		{
			p += (a_ptr->ac + 4) / 5;
			p += (a_ptr->to_h + 1) / 2;
			p += (a_ptr->to_d + 1) / 2;
			if (a_ptr->weight != k_ptr->weight)
				p += (k_ptr->weight - a_ptr->weight) / 30;
			break;
		}
		case TV_LITE:
		{
			p += 10;
			break;
		}
		case TV_RING:
		case TV_AMULET:
		{
			p += 20;
			break;
		}
	}

	/* Other abilities are evaluated independent of the object type. */
	p += (a_ptr->to_a + 3) / 4;
	if (a_ptr->to_a > 20) p += (a_ptr->to_a - 19) / 2;
	if (a_ptr->to_a > 30) p += (a_ptr->to_a - 29) / 2;
	if (a_ptr->to_a > 40) p += 20000;	/* inhibit */

	if (a_ptr->pval > 0)
	{
		if (a_ptr->flags1 & TR1_STR) p += a_ptr->pval * a_ptr->pval;
		if (a_ptr->flags1 & TR1_INT) p += a_ptr->pval * a_ptr->pval;
		if (a_ptr->flags1 & TR1_WIS) p += a_ptr->pval * a_ptr->pval;
		if (a_ptr->flags1 & TR1_DEX) p += a_ptr->pval * a_ptr->pval;
		if (a_ptr->flags1 & TR1_CON) p += a_ptr->pval * a_ptr->pval;
		if (a_ptr->flags1 & TR1_STEALTH) p += a_ptr->pval * a_ptr->pval;
	}
	else if (a_ptr->pval < 0)	/* hack: don't give large negatives */
	{
		if (a_ptr->flags1 & TR1_STR) p += a_ptr->pval;
		if (a_ptr->flags1 & TR1_INT) p += a_ptr->pval;
		if (a_ptr->flags1 & TR1_WIS) p += a_ptr->pval;
		if (a_ptr->flags1 & TR1_DEX) p += a_ptr->pval;
		if (a_ptr->flags1 & TR1_CON) p += a_ptr->pval;
		if (a_ptr->flags1 & TR1_STEALTH) p += a_ptr->pval;
	}
	if (a_ptr->flags1 & TR1_CHR) p += a_ptr->pval;
	if (a_ptr->flags1 & TR1_INFRA) p += a_ptr->pval / 2;
	if (a_ptr->flags1 & TR1_SPEED) p += (a_ptr->pval * 3) / 2;

	if (a_ptr->flags2 & TR2_SUST_STR) p += 6;
	if (a_ptr->flags2 & TR2_SUST_INT) p += 4;
	if (a_ptr->flags2 & TR2_SUST_WIS) p += 4;
	if (a_ptr->flags2 & TR2_SUST_DEX) p += 4;
	if (a_ptr->flags2 & TR2_SUST_CON) p += 4;
	if (a_ptr->flags2 & TR2_SUST_CHR) p += 1;
	if (a_ptr->flags2 & TR2_IM_ACID) p += 20;
	if (a_ptr->flags2 & TR2_IM_ELEC) p += 20;
	if (a_ptr->flags2 & TR2_IM_FIRE) p += 30;
	if (a_ptr->flags2 & TR2_IM_COLD) p += 20;
	if (a_ptr->flags3 & TR3_FREE_ACT) p += 8;
	if (a_ptr->flags3 & TR3_HOLD_LIFE) p += 10;
	if (a_ptr->flags2 & TR2_RES_ACID) p += 6;
	if (a_ptr->flags2 & TR2_RES_ELEC) p += 6;
	if (a_ptr->flags2 & TR2_RES_FIRE) p += 6;
	if (a_ptr->flags2 & TR2_RES_COLD) p += 6;
	if (a_ptr->flags2 & TR2_RES_POIS) p += 12;
	if (a_ptr->flags2 & TR2_RES_LITE) p += 8;
	if (a_ptr->flags2 & TR2_RES_DARK) p += 10;
	if (a_ptr->flags2 & TR2_RES_BLIND) p += 10;
	if (a_ptr->flags2 & TR2_RES_CONFU) p += 8;
	if (a_ptr->flags2 & TR2_RES_SOUND) p += 10;
	if (a_ptr->flags2 & TR2_RES_SHARD) p += 8;
	if (a_ptr->flags2 & TR2_RES_NETHR) p += 12;
	if (a_ptr->flags2 & TR2_RES_NEXUS) p += 10;
	if (a_ptr->flags2 & TR2_RES_CHAOS) p += 12;
	if (a_ptr->flags2 & TR2_RES_DISEN) p += 12;

	if (a_ptr->flags3 & TR3_FEATHER) p += 2;
	if (a_ptr->flags3 & TR3_LITE) p += 2;
	if (a_ptr->flags3 & TR3_SEE_INVIS) p += 8;
	if (a_ptr->flags3 & TR3_TELEPATHY) p += 20;
	if (a_ptr->flags3 & TR3_SLOW_DIGEST) p += 4;
	if (a_ptr->flags3 & TR3_REGEN) p += 8;
	if (a_ptr->flags3 & TR3_TELEPORT) p -= 20;	/* note */
	if (a_ptr->flags3 & TR3_DRAIN_EXP) p -= 16;	/* note */
	if (a_ptr->flags3 & TR3_AGGRAVATE) p -= 8;	/* note */
	if (a_ptr->flags3 & TR3_BLESSED) p += 4;
	if (a_ptr->flags3 & TR3_LIGHT_CURSE) p -= 2;		/* note */
	if (a_ptr->flags3 & TR3_HEAVY_CURSE) p -= 6;	/* note */
	if (a_ptr->flags3 & TR3_PERMA_CURSE) p -= 10;	/* note */

	return p;
}

/* Return nonzero if the whole set of random artifacts meets certain
   criteria.  Return 0 if we fail to meet those criteria (which will
   restart the whole process). */
int artifacts_acceptable (void)
{
	int swords = 5, polearms = 5, blunts = 5, bows = 3;
	int bodies = 5, shields = 3, cloaks = 3, hats = 4;
	int gloves = 4, boots = 4;
	int i;

	for (i = ART_MIN_NORMAL; i < MAX_A_IDX; i++)
	{
		switch (a_info[i].tval)
		{
			case TV_SWORD:
				swords--; break;
			case TV_POLEARM:
				polearms--; break;
			case TV_HAFTED:
				blunts--; break;
			case TV_BOW:
				bows--; break;
			case TV_SOFT_ARMOR:
			case TV_HARD_ARMOR:
				bodies--; break;
			case TV_SHIELD:
				shields--; break;
			case TV_CLOAK:
				cloaks--; break;
			case TV_HELM:
			case TV_CROWN:
				hats--; break;
			case TV_GLOVES:
				gloves--; break;
			case TV_BOOTS:
				boots--; break;
		}
	}

	if (swords > 0 || polearms > 0 || blunts > 0 || bows > 0 ||
	    bodies > 0 || shields > 0 || cloaks > 0 || hats > 0 ||
	    gloves > 0 || boots > 0)
	{
		if (randart_verbose)
		{
			char types [256];
			sprintf (types, "%s%s%s%s%s%s%s%s%s%s",
				swords > 0 ? " swords" : "",
				polearms > 0 ? " polearms" : "",
				blunts > 0 ? " blunts" : "",
				bows > 0 ? " bows" : "",
				bodies > 0 ? " body-armors" : "",
				shields > 0 ? " shields" : "",
				cloaks > 0 ? " cloaks" : "",
				hats > 0 ? " hats" : "",
				gloves > 0 ? " gloves" : "",
				boots > 0 ? " boots" : "");
			msg_format ("Restarting generation process: not enough%s\n",
				types);
		}
		return 0;
	}
	else
	{
		return 1;
	}
}

/* Calculate the multiplier we'll get with a given bow type.  This is done
   differently in 2.8.2 than it was in 2.8.1. */
int bow_multiplier (int sval)
{
	switch (sval)
	{
		case SV_SLING:
		case SV_SHORT_BOW: return 2;

		case SV_LONG_BOW:
		case SV_LIGHT_XBOW: return 3;

		case SV_HEAVY_XBOW: return 4;

		default: msg_format ("Illegal bow sval %s\n", sval);
	}
	return 0;
}

char *my_strdup (const char *s)
{
	char *t = malloc (strlen (s) + 1);
	if (t) strcpy (t, s);
	return t;
}

#endif /* GJW_RANDART */
