/* 
 This is displ.c, part of the source code for

 BMV version 1.0 alpha
 copyright by Jan Kybic, 26th July 1994

 Jan Kybic, Prosecka 681, Praha 9, Czech Republic, <kybic@earn.cvut.cz>
                                   or temporarily  <jkybic@vipunen.hut.fi>

 BMV is a very simple viewer of images in the pbm(5)-raw format
     and front end for GhostScript		
     based on Svgalib library for Linux

 BMV is distributed under GNU GPL (General Public License),
   you can obtain a copy from many FTP sites if you are interested for details
*/


#include "bmv.h"
#include <sys/time.h>       /* because of FD_SETSIZE */

/* the following variables local to this file contain the information about
   the videomode being used and what colours to use */
   
static int inkcol,papercol,bitsperpix,pixperbyte ;
vga_modeinfo *modeinfo ;


/* displaying is 1 if we are currently in the middle of displaying 
   if user presses a key, it is detected by backround routine which
   sets (through signal) displaying=0 and this causes displaying to stop */

volatile static sig_atomic_t displaying=0 ;


pid_t kbdpid=-1 ; /* pid of the keyboard monitoring process */

unsigned char *outbuff=NULL,*readbuff=NULL,*inbuff=NULL ;
                 /* buffers we need for showwindow */


void sighandler(int sig) ; /* receives SIGUSR1 and resets displaying */


/* opendisplay initializes graphic screen and sets the appropriate globals
   If the requested mode is not available, it tries to find other. */

void opendisplay()
{ 
  vga_init() ;

  /* try to find default mode, if user has not set it with the -v switch */
  if (vidmode==-1) vidmode=vga_getdefaultmode() ;
  if (vidmode==-1) vidmode=DVGAMODE ;

  /* If it is not available, find other */
  if (!vga_hasmode(vidmode)) 
    printf("Required videomode %d not available, trying others ...\n",vidmode) ;
  for(;vidmode>0 && !vga_hasmode(vidmode);vidmode--) ;
  
  vga_setmode(vidmode) ; vidmode=vga_getcurrentmode() ;
  if (vidmode==0) leave(1,"Failed to set any VGA mode",NULL) ;

  printf("Using VGA mode number %d.\n",vidmode) ;

  /* Get all the information */
  modeinfo=vga_getmodeinfo(vidmode) ;

  /* If we have too many colours, say sorry and exit. You can change it
     so that even in this case we would try to find an alternate mode,
     but I believe it is better to make user to do the choice */
     
  if (modeinfo->bytesperpixel>1)
   leave(1,"Videomodes with more than 1 byte per pixel not supported, sorry",
           "Try -v to set some other mode",NULL) ;
  if (modeinfo->colors==256)
      { inkcol=INKEIGHT ; papercol=PAPEREIGHT ; 
        bitsperpix=8 ; pixperbyte=1 ;}
   else if (modeinfo->colors==16)
      { inkcol=INKFOUR ; papercol=PAPERFOUR ; 
        bitsperpix=4 ; pixperbyte=2 ;}
   else if (modeinfo->colors==2)
      { inkcol=INKONE ; papercol=PAPERONE ; 
        bitsperpix=1 ; pixperbyte=8 ;}        
   else leave(1,"Unsupported number of colors, sorry",
                "Try -v to set some other mode",NULL) ; 
  
  screenw=modeinfo->width ; screenh=modeinfo->height ;
  vgaon=1 ;
}



/* closedisplay resets screen to text mode and kills the keyboard monitor
   if necessary */
   
void closedisplay()
{ if (kbdpid>0 && waitpid(kbdpid,NULL,WNOHANG)==0) kill(kbdpid,SIGTERM) ;
  if (!vgaon) return ;
  if (vga_getcurrentmode()) { puts("Returning to text mode") ;
                              vga_setmode(TEXT) ; }
  vgaon=0 ;
}  



/* showwindow assumes the screen is set, file is open and buffers are 
   allocated. It then displays a part of the PBM file with a given upper
   left corner and magnification */ 

void showwindow(int mag,int windowx,int windowy)
{ int scrline ;   /* current screen line for main loop */
  int line ;      /* current screen line for the output loop */
  int col ;       /* colour */
  int scrbits,imbits ;  /* a square of imbits x imbits in the image
                           corresponds to scrbits x scrbits in the screen */
  int windoww,windowh ; /* image window dimensions */
  int i,j,k ;           /* various use */
  unsigned char *p,*r ; /* pointers to buffers */
  long imbit,imbyte ;   /* position in bits and bytes from the beginning 
                           of the image */ 
  
  displaying=1 ;          /* set flag - we are displaying */
  kill(kbdpid,SIGUSR1) ;  /* start the keyboard monitor, if user presses a key
                             displaying will be set to 0 */
  
  /* compute size and scale factors */
  scrbits= (mag>1) ? mag : 1 ;
  imbits= (mag<-1) ? -mag : 1 ;
  windowx=(windowx/8)*8 ;
  windoww=screenw/scrbits*imbits ;
  windowh=screenh/scrbits*imbits ;
  if (windoww+windowx>imagew) windoww=imagew-windowx ;
  if (windowh+windowy>imageh) windowh=imageh-windowy ;

  scrline=0 ;
  while(scrline+scrbits<=screenh && windowy+imbits<=imageh && displaying)
   /* now for every line do: */
   { 
     /* initialize input buffer */
     memset(inbuff,0,screenw) ; 
     /* initialize output buffer - 0 here means that out of paper regions 
        will be black */
     memset(outbuff,0,modeinfo->linewidth) ;

     for(i=0;i<imbits && windowy<imageh;i++,windowy++)
       { 
         /* now for every line that should contribute to given screen line */
         imbit=windowy*imagew+windowx ;
         imbyte=imbit/8 ;
         
         /* read image line from file */
         if (fseek(file,offset+imbyte,SEEK_SET)==-1)
           leave(1,"Cannot seek PBM file",strerror(errno),NULL) ;

	 /* there should be a test for an error, but it is difficult to
	    know, how many bytes we should expect. I hope you forgive */
         fread(readbuff,windoww/8+1,1,file) ;
         
         /* now group pixels by imbits and add to inbuff   */
           for(k=128,p=readbuff;p<readbuff+windoww/8;p++)
             for(j=0;j<8;j++,k= (k==1) ? 128 : k>>1)
               if (*p & k) inbuff[((p-readbuff)*8+j)/imbits]++ ;
       }  

     /* now in inbuff we have counts, how many pixels from the image
        corresponding to particular screen pixels are black */


     /* we have to distinguish between monochrome modes and colour modes */ 
     if (pixperbyte!=8)

     /* for colour modes - each byte in outbuff represents one screen pixel */
       for(p=inbuff,r=outbuff;p<inbuff+windoww/imbits;p++)
         { col =(*p>PIXTRESH*(imbits*imbits)/100) ? inkcol : papercol ;
           for (k=0;k<scrbits;k++) *r++=col ; }
      else 

     /* monochrome modes -each byte in outbuff represens eight screen pixels */      
       for(p=inbuff,j=0;p<(unsigned char *)inbuff+windoww/imbits;p++)
         { col=(*p>PIXTRESH*(imbits*imbits)/100) ? inkcol : papercol ;  
            for (k=0;k<scrbits;k++,j++)
              *(outbuff+(j/8)) |=col ? (128>>(j%8)) : 0 ; }

      /* and now just draw the line scrbits-times */
      for(j=0;j<scrbits;j++)
         vga_drawscanline(scrline++,outbuff) ;                       
   }
   displaying=0 ;
}


#define PAGNOLEN 3   /* number of digits for a page number */

/* Display file gets a name of a rawPBM file as its argument and displays it 
   on the screen. Then it expects user to press a key and acts accordingly. 
   It is possible to change magnification, move a round and if we are 
   displaying PostScript, we can move to other pages */

void displayfile(char *filename) 
{ int windowx,windowy,step=STEP ;
      /* upper left corner of the viewing window, step size */
  char c ;   /* character read from the keyboard */
  int cont ; /* shall we continue with this file(flag) */
  int efstep ;  /* step size in image pixels */
  int numpage=1,nextpage=1 ;
                /* current page number, any more pages to display(flag) */
  
  while(nextpage) {
 
  windowx=0 ; windowy=0 ; /* set the upper left corner */

  /* if we are in PS mode, ask GS for the page */
  if (psview) makegsdisplay(numpage) ;

  openpbm(filename) ; /* open PBM file */
  printf("Image size: %u x %u\n",imagew,imageh) ;

  /* allocate buffers */
  if ((outbuff=malloc(screenw))==NULL)
    leave(1,"Cannot allocate memory for output buffer",NULL) ;
  if ((readbuff=malloc((screenw*MAXIMBITS)/8+1))==NULL)
    leave(1,"Cannot allocate memory for read buffer",NULL) ;
  if ((inbuff=malloc(screenw))==NULL)
    leave(1,"Cannot allocate memory for intermediate buffer",NULL) ;

  cont=1 ; 
  /* in this loop we process one page */
  while (cont)
  { /* clear the screen, so that the user knows, that something is happening */
    vga_clear() ;                

    /* display window */
    showwindow(mag,windowx,windowy) ; 

    
    readchar: /* that is the place to jump, if no redisply is needed */

    /* calculate effective step size in image pixels */
    efstep=(mag>0) ? step/mag : -step * mag ;

    /* read character and process it */	
    c=getchar() ; 
    switch (c) { 
          /* q - quit */
          case 'q' : cont=0 ; nextpage=0 ; break ;
    
	  /* f/r - increase/reduce step size */
          case 'f' : if (step<MAXSTEP) step*=2 ; goto readchar ;
          case 'r' : if (step>MINSTEP) step /=2 ; goto readchar ;

	  /* +/- - increas/reduce magnification ration
	           we try not to move the middle of the picture */
          case '+' : 
          case '-' : { int cx=windowx+(mag>0 ? screenw/mag : -screenw*mag )/2 ;
                       int cy=windowy+(mag>0 ? screenh/mag : -screenh*mag )/2 ;
                       if (c=='+' && mag<MAXIMBITS) 
                         if (mag==-1) mag=2 ; else mag++ ;
                       if (c=='-' && mag>-MAXIMBITS) 
                         if (mag==1) mag=-2 ; else mag-- ;
                       windowx=cx-(mag>0 ? screenw/mag : -screenw*mag )/2 ;
                       windowy=cy-(mag>0 ? screenh/mag : -screenh*mag )/2 ;
                       break ; }   

	  /* h,j,k,l - move left, down, up, right */                       
          case 'h' : windowx-=efstep ; 
                     break ;
          case 'j' : windowy+=efstep ; 
                     break ;
          case 'k' : windowy-=efstep ; 
                     break ;
          case 'l' : windowx+=efstep ; 
                     break ;
                     
          /* n,p - go to the next/previous page */           
          case 'n' : if (psview && usedsc && numpage<pages.numpages)
                                { numpage++ ; cont=0 ; break ; } 
                     goto readchar ;

          case 'p' : if (psview && usedsc && numpage>1)
                                { numpage-- ; cont=0 ; break ; } 
                     goto readchar ;              

          /* g reads PAGNOLEN digits and goes to that page */
          case 'g' : { int temp,i ; char c ;
                       for(temp=0,i=0;i<PAGNOLEN;i++)              
                        if (isdigit(c=getchar())) temp=temp*10+c-'0' ;
                           else goto readchar ;
		       if (temp>0 && temp<pages.numpages)
		                { numpage=temp ; cont=0 ; break ; }
		           else goto readchar ;     
		     }                            
                    } 
                    
    /* adjust window position if it exceeds limits */
    windowy=max(0,min(windowy,imageh-(mag>0 ? screenh/mag-mag*2 : -screenh*mag))) ;
    windowx=max(0,min(windowx,imagew-(mag>0 ? screenw/mag-mag*2 : -screenw*mag))) ;
                                       
  }          


/* Displaying is over, let us do the clean up */
free(readbuff) ; free(outbuff) ; free(inbuff) ;
if (fclose(file)!=0) 
  leave(1,"Cannot close PBM file",strerror(errno),NULL) ;
 }
}   
  
  
/* bkgdisplayfile should appears from outside to be the same as displayfile.
   However, it first of all starts a background process monitoring keyboard,
   the effect being that displaying is interrupted whenever user presses key
*/   
   
   
void bkgdisplayfile(char *filename)
{  
   
   if ((kbdpid=fork())<0)
     leave(1,"Cannot fork",NULL) ;
  
   /* we shall use SIGUSR1 for communication */
   signal(SIGUSR1,SIG_IGN) ;
   
   if (kbdpid)
     { /* Parent */
       /* enable running in background, though we are not really in background
          but the child needs access to the keyboard, so we lend it to it
          for a while */     
       vga_runinbackground(1) ;
       
       /* we install handler that resets displaying when invoked */
       signal(SIGUSR1,sighandler) ;

        /* do the display */
       displayfile(filename) ;

       /* if child is running, kill it */ 
       if (kbdpid>0) kill(kbdpid,SIGTERM) ;
       
       /* claim terminal back */
       tcsetpgrp(STDIN_FILENO,getpgrp()) ; 
       
       /* we do not need SIGUSR1 any more */
       signal(SIGUSR1,SIG_IGN) ;
     }
     
    else { /* This is the child */
      fd_set f ; /* f will contain a mask for stdin */

      /* if we receive SIGTERM, terminate silently */
      signal(SIGTERM,SIG_DFL) ;
      
      /* if we receive SIGINT, terminate and make parent terminate too */
      signal(SIGINT,siginthandler) ;
      
      /* SIGUSR1 will do nothing, but we use it to synchronize */
      signal(SIGUSR1,emptyhandler) ;
      
      /* we need an access to the terminal */
      tcsetpgrp(STDIN_FILENO,getpgrp()) ;
      
      /* this is an infinite loop, it ends when we receive SIGTERM */
      for(;;) { 
                /* If parent is not displaying, wait for SIGUSR1 that
                    it sends when it starts */
                if (!displaying) pause() ; 
                
                /* wait until character is available, or until parent
                   starts displaying again */
                FD_ZERO(&f) ; FD_SET(STDIN_FILENO,&f) ;
                select(FD_SETSIZE,&f,NULL,NULL,NULL) ; 

                /* if character is available, notify parent */
	        if (FD_ISSET(STDIN_FILENO,&f)) 
                   kill(getppid(),SIGUSR1) ;  
                }
         } 
   
   /* we do not need to run in background and as it is dangerous, switch it
      off */
      
   vga_runinbackground(0) ;
   kbdpid=-1 ;                                     
}


/* very simple signal handlers */

void sighandler(int sig)
{ displaying=0 ;
  signal(sig,sighandler) ;
}

void emptyhandler(int sig) 
{ signal(sig,emptyhandler) ; 
}
  
void siginthandler(int sig)
{ kill(getppid(),SIGTERM) ;
  kill(getpid(),SIGTERM) ;
}    

/* ********** end of displ.c ************ */
           