/*
 This is bmv.c, part of the source code for

 BMV version 1.0 alpha
 copyright by Jan Kybic, 26th July 1994

 Jan Kybic, Prosecka 681, Praha 9, Czech Republic, <kybic@earn.cvut.cz>
                                   or temporarily  <jkybic@vipunen.hut.fi>

 BMV is a very simple viewer of images in the pbm(5)-raw format
     and a front end for GhostScript		
     based on Svgalib library for Linux

 BMV is distributed under GNU GPL (General Public License),
   you can obtain a copy from many FTP sites if you are interested for details
*/

/* This file contains main() and functions for handling PBM files */
/* see also bmv.h for description of some globals */

#include "bmv.h"
#include <fcntl.h>
#include <stdarg.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>

int vgaon=0,vidmode=-1 ; /* we are in text mode originally */

FILE *file=NULL ;       /* we have not opened PBM file yet */

long offset ;
int screenw,screenh,imagew,imageh ;

int mag=1 ;             /* initial magnification */

int psview=1,testpbm=0 ; /* default is Postscript, no testing */

void sigtermhandler(int sig) ; /* prototypes of signal handlers */
void sigchldhandler(int sig) ;

/* initial message */
static char *initmsg=
"\nThis is BMV version " VER ", writen by Jan Kybic." ;

/* and some brief help */
static char *helpmsg=
"\nBMV is a front end for GhostScript intended for viewing PS files.\n"
"    You can also use it for viewing rawPBM image files.\n"
"Syntax: bmv [options] file\n"
"  options: -v<number>     VGA mode number for VGAlib, see <vga.h>\n"
"           -m<number>     initial magnification, can be negative\n" 
"           -b             display PBM file instead of PS file\n"
"           -r<string>     resolution- same as -r option for GhostScript\n" 
"           -h             displays this screen\n"
"While you are viewing picture you can use the following keys:\n"
"  h,j,k,l     to move left/down/up/right\n"
"  f,r         to increase/decrease step size\n"
"  +,-         to increase/decrease magnification\n"
"  n,p         to go to the next/previous page (only DSC documents)\n"
"  g<n><cr>    to go to page number <n>, <n> consists of exactly 3 digits\n"
"  q           to quit\n" ;




/* printerr is used for printing error messages */

void printerr(char *e)
{ fputs("!!! BMV: ",stdout) ;
  puts(e) ;
}  


/* leave prints all the strings it is passed via printerr, does all the
   cleanup and exits with status stat. It expects pointers to strings to be
   printed, the last one being NULL */

void leave(int stat,...)
{ va_list ptr ; char *e ;
  va_start(ptr,stat) ;
 
 for(va_start(ptr,stat);(e=va_arg(ptr,char *))!=NULL;) printerr(e) ;
     /* print all the error messages */
 va_end(ptr) ;
 
 /* now free the buffers */ 
 if (inbuff!=NULL) free(inbuff) ;
 if (outbuff!=NULL) free(outbuff) ;
 if (readbuff!=NULL) free(readbuff) ;

 /* call close... to do the rest of the cleaning */ 
 closepsfile() ;
 closedisplay() ;
 
 puts("Goodbye. Try it again !") ;
 exit(stat) ;
}


/* and here we have main */

void main(int argc,char *argv[])
{ 
  int curopt,temp ; char *tail ;

  /* install handlers - we want to terminate cleanly on SIGTERM,SIGINT
     do nothing for SIGUSR1 (we us it for communication) and
     on SIGCHLD we want to notice if GhostScript has had any troubles */
       
  signal(SIGTERM,sigtermhandler) ;
  signal(SIGINT,sigtermhandler) ;
  signal(SIGUSR1,emptyhandler) ;
  signal(SIGCHLD,sigchldhandler) ;
  
  puts(initmsg) ;                    /* greetings */
  if (argc<=1) puts(helpmsg) ;

  for(curopt=1;curopt<argc;curopt++) /* process options */
    if (*argv[curopt]=='-')
      switch (*(argv[curopt]+1))
        { case 'b': psview=0 ; break ;
        
          case 'r': strncpy(gsres,argv[curopt],GSRESLEN) ; break ;
        
          case 'v': temp=(int) strtol(argv[curopt]+2,&tail,0) ;
                    if (tail==argv[curopt]+2)
                       leave(1,"Wrong VGA mode - must be a number",NULL) ;
                    vidmode=temp ;
                    break ;
        
          case 'm': temp=(int) strtol(argv[curopt]+2,&tail,0) ;
                    if (tail==argv[curopt]+2)
                       leave(1,"Wrong magnification - must be a number",NULL) ;
                    if (temp==0 || temp>MAXIMBITS || temp<-MAXIMBITS)
                       leave(1,"Magnification must be within -" STRMAXIMBITS
                               "..." STRMAXIMBITS " excluding 0.",NULL) ;
                    mag=temp ;
                    break ;           
        
          case 'h': puts(helpmsg) ; exit(0) ; break ;                     
        
          default:  leave(1,"Wrong option -",argv[curopt],
                          "Try bmv -h.",NULL) ;
        }
    /* argv[curopt] is probably a filename */
    else { opendisplay() ; /* init display */

           /* call the appropriate display function */
           if (!psview) bkgdisplayfile(argv[curopt]) ; 
                   else displaypsfile(argv[curopt]) ;
                   
           closedisplay() ;  /* back to text */

           break ;   /* leave this out if you want to display more files 
                        in one run, but I do not think it is necessary */
         }                       
                           
  puts("Goodbye.") ; exit(0) ;
}  



/* openpbm expect filename of a rawPBM file to open. It opens the file
   reads the header and sets globals: file,imagew,imageh,offset
   if it encounters any error it exits, except when file is too short
   and testpbm=1, then it just silently resets testpbm=0. This is used
   by makegsdisply 
   See PBM(5) for description of a format */

void openpbm(char *filename)
{ char c ; 
  long length ; /* expected length of a file */
  int fno ;
  struct stat stbuf ;
  
  if (!testpbm) 
     printf("Opening PBM file %s for reading...\n",filename) ;
  
  if ((file=fopen(filename,"r"))==0) 
   leave(1,"Cannot open PBMfile for reading",strerror(errno),NULL) ;
  
  if ((fno=fileno(file))<0)
    leave(1,"Cannot get PBMfile descriptor",strerror(errno),NULL) ;

  if (fstat(fno,&stbuf))
     leave(1,"Cannot stat PBM file",strerror(errno),NULL) ;

  /* If file is shorter than PBMFILEMIN do not bother to read it 
     This is to prevent the test from fatal failure when GS has 
     not written the header yet in full */
     
  if ((long)stbuf.st_size<PBMFILEMIN)
     if (!testpbm) leave(1,"PBM file is very short",NULL) ;
              else { testpbm=0 ; fclose(file) ; return ; }

  /* Check magic numbers */
  if (fgetc(file)=='P')
   if (fgetc(file)=='4')
     if (isspace(fgetc(file)))
       { /* skip comments */
         while ((c=fgetc(file))=='#') 
            for (;c!=EOF && c!='\n';c=fgetc(file)) ;

         ungetc(c,file) ; /* we read too much */
         
	 /* read dimensions */
         if (fscanf(file,"%d %d",&imagew,&imageh)!=2)  
            leave(1,"Wrong format of PBM file",NULL) ;

         fgetc(file) ; /* skip separator */
         
         offset=ftell(file) ;  /* tell where data begin */

         imagew=(imagew+7)/8*8 ; /* round imagew up to multiple of 8 */

         /* try to guess the correct length 
            !!! This assumes ftell returns number of bytes from the
                beginnig of file */
         length=offset+imagew/8*imageh ;

         if (fstat(fno,&stbuf))
               leave(1,"Cannot stat PBM file",strerror(errno),NULL) ;

         /* Now compare the actual and expected length */
         if ((long)stbuf.st_size<length)
           if (!testpbm) leave(1,"PBM file is shorter than expected",NULL) ;
               else  { /* if this was just a test, fail silently */
                       testpbm=0 ; fclose(file) ; return ; }

         /* If testing, the file should not stay open */
         if (testpbm) fclose(file) ; 
         return ;      
       }  
  leave(1,"Invalid preamble in PBM file",NULL) ;
}            



/* sigtermhandler handles SIGTERM and SIGINT. It simply calls leave
   to do the cleaning. */		

void sigtermhandler(int sig) 
{ static volatile sig_atomic_t busy=0 ;

  /* the purpose of busy is to make sure we will not be called twice.
     It does not do any harm, but it does not like nice */
     
  if (busy>0) return ;
  busy=1 ;
  tcsetpgrp(STDOUT_FILENO,getpgrp()) ;
  leave(1,"I received a fatal signal",strsignal(sig),NULL) ;
}    
     
     
/* sigchldhandler checks if the child that exited was GhostScript.
   If it returned with nonzero status (I assume this means error),
   we leave, otherwise we would have to wait till timeout */
     
      
void sigchldhandler(int sig)
{ int status ;
  if (gspid>0)
    if (waitpid(gspid,&status,WNOHANG)==gspid)
      if (WIFEXITED(status) && WEXITSTATUS(status)!=0)
         leave(1,"GhostScript exited, returning nonzero",NULL) ;
  signal(SIGCHLD,sigchldhandler) ;
}     

/* ************** end of bmv.c *************** */          