/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: client.c
**
** Kommentar:
**  Das Hauptprogramm
**  und die Routinen, die Netzwerk, 3D-Berechnung und Grafik verbinden
*/


#include <stdio.h>
#include <signal.h>

#include "global.h"
#include "fehler.h"
#include "speicher.h"
#include "labyrinth.h"
#include "client.h"
#include "spieler.h"
#include "grafik.h"
#include "signale.h"
#include "ereignisse.h"
#include "einaus.h"
#include "rechne.h"
#include "netzwerk.h"
#include "spiel.h"

static char sccsid[] = "@(#)client.c	1.53 23 Nov 1994";


/* Breite und Laenge des aktuellen Spielfeldes: */
u_int feldbreite, feldlaenge;
block **spielfeld;        /* das Spielfeld */
block **feld_himricht[4]; /* das Spielfeld, in 4 Richtungen gedreht */

static int puffer_nr = 0;      /* Index auf Spieldaten-Puffer
                                  (welcher zuletzt gelesen wurde) */
static int puffer_gueltig = 0; /* Index auf Spieldaten-Puffer
                                  (welcher zuletzt beschrieben wurde) */

/* die Spieldaten werden in einem Doppelpuffer gespeichert;
   einer kann gelesen, der andere beschrieben werden */

static int spieler_aktiv[2]; /* der Spieler selbst ist im Spiel */
static int schuss_aktiv[2];  /* der Spieler selbst hat einen Schuss im
                                Spiel */
static int gegneranz[2];     /* Anzahl der aktuell sichtbaren Mitspieler */
static int schussanz[2];     /* Anzahl der sichtbaren Mitspieler-Schuesse */
static int abgeschossen_durch[2]; /* Farbe des Spielers, von dem man
                                     abgeschossen wurde */
static char *abschuss_spruch[2] = { NULL, NULL }; /* Spruch des Spielers, von
                                                     dem man abgeschossen
                                                     wurde */

/* Daten aller sichtbaren Spieler und Schuesse (Position, Farbe, ...): */
static struct spieler alle_spieler[2][SPIELERANZ];
static struct schuss alle_schuesse[2][SPIELERANZ];

static int alle_punkte[2][SPIELERANZ];  /* Punktestand aller Spieler; -1 fuer
                                           Spieler, die nicht existieren */
static char ereignisse[2][ERG_ANZ];     /* Ereignisse
                                           (siehe ereignisse.h) */
static char ereignisse_uebrig[ERG_ANZ]; /* noch nicht verarbeitete
                                           Ereignisse (temporaer) */

static int karteneu = 1; /* Flag, muss der Grundriss der Karte neu
                            gezeichnet werden? */

/* Fehlermeldung und -knopf von milder_fehler (fuer milden_fehler_abfragen) */
static char **fehlermeldung = NULL, *fehlerknopf = NULL;


/*
** drehlab
**  dreht das Spielfeld in 4 Richtungen (0, 90, 180, 270 Grad)
**  (damit die Projektionen nur fuer Richtung Norden implementiert werden
**  muessen)
**
** Seiteneffekte:
**  feld_himricht wird erzeugt
*/
static void drehlab()
{
	int i;    /* Index ueber die 4 Richtungen */
	int x, y; /* Index ueber Breite/Laenge des Spielfeldes */

	/* Speicher fuer 4 Spielfelder anlegen */
	/* bei ungeradem i Breite und Laenge tauschen */
	for (i = 1; i < 4; i++)
		speicher_belegen((void **)&feld_himricht[i],
			(i % 2 ? feldlaenge : feldbreite) * sizeof(block *));

	for (i = 1; i < 4; i++)
		for (y = 0; y < (i % 2 ? feldlaenge : feldbreite); y++)
			speicher_belegen((void **)&feld_himricht[i][y],
				(i % 2 ? feldbreite : feldlaenge) * sizeof(block));

	/* feld_himricht[0] wird nicht gedreht */
	feld_himricht[0] = spielfeld;

	/* feld_himricht[0] gedreht in feld_himricht[1,2,3] kopieren */
	for (y = 0; y < feldlaenge; y++)
		for (x = 0; x < feldbreite; x++)
			/* NORD = 0, WEST = 1, SUED = 2, OST = 3 */
			for (i = 0; i < 4; i++)
			{
				memcpy(&feld_himricht[1][feldlaenge - 1 - y]
					[x][(i + 3) % 4],
					&spielfeld[x][y][i], sizeof(struct wand));
				memcpy(&feld_himricht[2][feldbreite - 1 - x]
					[feldlaenge - 1 - y][(i + 2) % 4],
					&spielfeld[x][y][i], sizeof(struct wand));
				memcpy(&feld_himricht[3][y]
					[feldbreite - 1 - x][(i + 1) % 4],
					&spielfeld[x][y][i], sizeof(struct wand));
			}
}


/*
** objekt_listen_init
**  Listen anlegen fuer die Grafikobjekte, die rechne3d berechnet
**
** Parameter:
**  objekte: Zeiger auf die Struktur, die die Listen enthaelt
**
** Seiteneffekte:
**  *objekte wird veraendert
*/
static void objekt_listen_init(objekte)
struct objektdaten *objekte;
{
	/* Waende- und Kugelliste anlegen */
	liste_initialisieren(&objekte->anzwaende, (void **)&objekte->waende);
	liste_initialisieren(&objekte->anzkugeln, (void **)&objekte->kugeln);

	/* Spruch des Gegners loeschen */
	objekte->text = NULL;
}


/*
** objekt_listen_freigeben
**  gibt die Listen mit zu zeichnenden Grafikobjekten wieder frei
**
** Parameter:
**  objekte: Zeiger auf die Struktur, die die Listen enthaelt
*/
static void objekt_listen_freigeben(objekte)
struct objektdaten *objekte;
{
	int i; /* Index fuer die Kugelliste */

	liste_freigeben(&objekte->anzwaende, (void **)&objekte->waende);

	/* Unterliste (Sichbarkeitsbereiche der Kugeln) freigeben */
	for (i = 0; i < objekte->anzkugeln; i++)
		liste_freigeben(&objekte->kugeln[i].sichtanz,
			(void **)&objekte->kugeln[i].sichtbar);
	liste_freigeben(&objekte->anzkugeln, (void **)&objekte->kugeln);

	/* Speicher fuer Spruch des Gegners freigeben */
	if (objekte->text != NULL)
		speicher_freigeben((void **)&objekte->text);
}


/*
** punkte_vergleich
**  vergleicht zwei Punktestaende und gibt zurueck, in welcher Reihenfolge
**  sie angezeigt werden sollen
**
** Parameter:
**  punkte1: Zeiger auf einen Punktestand
**  punkte2: Zeiger auf anderen Punktestand
**
** Rueckgabewert:
**  < 0, falls Punktestand 1 vor Punktestand 2
**  = 0, falls egal
**  > 0, falls Punktestand 2 vor Punktestand 1
*/
static int punkte_vergleich(punkte1, punkte2)
struct punktestand *punkte1, *punkte2;
{
	/* verschieden viele Punkte? */
	if (punkte1->punkte != punkte2->punkte)
		return punkte2->punkte - punkte1->punkte;

	/* sonst nach Farbe sortieren */
	return (int)punkte1->farbe - (int)punkte2->farbe;
}


/*
** neuaufbau
**  erhaelt die neuen Spieldaten und startet die Grafikberechnung
**  und -ausgabe
**
** Parameter:
**  spieler_aktiv: Flag, ist der Spieler selbst im Spiel
**  gegneranz: Anzahl der aktuell sichtbaren Mitspieler
**  alle_spieler: Feld mit den Daten ueber alle sichtbaren Spieler
**  schuss_aktiv: Flag, hat der Spieler selbst einen Schuss im Spiel
**  schussanz: Anzahl der sichtbaren Schuesse der Mitspieler
**  alle_schuesse: Feld mit den Daten ueber alle sichtbaren Schuesse
**  abgeschossen_durch: Farbe des Spieler, von dem man abgeschossen wurde
**  abschuss_spruch: Spruch des Spieler, von dem man abgeschossen wurde
**  ereignisse: Feld mit den neuen Ereignissen
**  alle_punkte: Feld mit den Punktestaenden aller Spieler
**
** Seiteneffekte:
**  karteneu wird veraendert
*/
static void neuaufbau(spieler_aktiv, gegneranz, alle_spieler, schuss_aktiv,
	schussanz, alle_schuesse, abgeschossen_durch, abschuss_spruch, ereignisse,
	alle_punkte)
int spieler_aktiv, gegneranz, schuss_aktiv, schussanz;
struct spieler alle_spieler[SPIELERANZ];
struct schuss alle_schuesse[SPIELERANZ];
int abgeschossen_durch;
char *abschuss_spruch, ereignisse[ERG_ANZ];
int alle_punkte[SPIELERANZ];
{
	/* Struktur mit allen Grafik-Daten fuer die 3D-Sicht nach vorn und
	   Rueckspiegel */
	struct objektdaten blick_vor, blick_zurueck;
	int anzlinien;                /* Anzahl der Linien auf der Karte */
	int anzkreise;                /* Anzahl der Kreise auf der karte */
	struct kartenkreis *kreise;   /* Liste mit den Grafik-Daten der Linien
	                                 auf der Karte */
	struct kartenlinie *linien;   /* Liste mit den Grafik-Daten der Kreise
	                                 auf der Karte */
	struct spieler spieler_rueck; /* Daten des Spielers um 180 gedreht */
	struct punktedaten punkte;    /* Struktur mit den Daten der
	                                 Punktestaende */

	/* Listen anlegen, in den die Grafik-Daten abgelegt werden */
	objekt_listen_init(&blick_vor);
	objekt_listen_init(&blick_zurueck);
	liste_initialisieren(&anzkreise, (void **)&kreise);

	/* falls Karten-Grundriss gezeichnet werden muss, Grafik-Daten dafuer
	   berechnen */
	if (karteneu)
	{
		liste_initialisieren(&anzlinien, (void **)&linien);

		rechne_karte_linien(&anzlinien, &linien);
	}

	/* falls der Spieler lebt, muss der Horizont gezeichnet werden */
	blick_vor.hintergrund_zeichnen = blick_zurueck.hintergrund_zeichnen =
		spieler_aktiv;

	/* spieler_rueck entspricht dem Spieler mit Blick um 180 Grad gedreht */
	memcpy(&spieler_rueck, &alle_spieler[0], sizeof(struct spieler));
	spieler_rueck.blick = (spieler_rueck.blick + WINKANZ / 2) % WINKANZ;

	/* falls der Spieler lebt, werden die 3D-Berechungen fuer Front- und
	   Rueckansicht aufgerufen */
	if (spieler_aktiv)
	{
		rechne3d(&blick_vor, &alle_spieler[0], gegneranz,
			&alle_spieler[1], schussanz + schuss_aktiv,
			&alle_schuesse[1 - schuss_aktiv]);
		rechne3d(&blick_zurueck, &spieler_rueck, gegneranz,
			&alle_spieler[1], schussanz + schuss_aktiv,
			&alle_schuesse[1 - schuss_aktiv]);
	}
	else if (abgeschossen_durch >= 0)
	/* falls der Spieler nach einem Abschuss tot ist, wird das Gesicht des
	   Verursachers angezeigt */
	{
		struct kugel *kugel;           /* Daten des Gegnergesichts */
		struct ausschnitt *ausschnitt; /* Sichtbarkeitsbereich
		                                  des Gesichts */

		/* Listen mit Grafik-Daten erweitern */
		liste_verlaengern(&blick_vor.anzkugeln, (void **)&blick_vor.kugeln,
			sizeof(struct kugel));
		kugel = &blick_vor.kugeln[blick_vor.anzkugeln - 1];

		/* Grafik-Daten eines Gegnergesichts im Vollbild */
		kugel->mittelpunkt.x = FENSTER_BREITE / 2;
		kugel->mittelpunkt.y = FENSTER_HOEHE / 2;
		kugel->radiusx = FENSTER_BREITE / 4;
		kugel->radiusy = kugel->radiusx * 3 / 2;
		kugel->schatteny = kugel->schattenry = 0;
		kugel->blick = TRIGANZ / 2;
		kugel->farbe = abgeschossen_durch;

		/* Liste mit Sichtbarkeitsbereich (Gegner ist voll sichtbar) */
		liste_initialisieren(&kugel->sichtanz, (void **)&kugel->sichtbar);
		liste_verlaengern(&kugel->sichtanz, (void **)&kugel->sichtbar,
			sizeof(struct ausschnitt));
		ausschnitt = &kugel->sichtbar[kugel->sichtanz - 1];

		ausschnitt->x = 0;
		ausschnitt->breite = FENSTER_BREITE;

		/* Abschussspruch des Gegners bekannt? */
		if (abschuss_spruch != NULL)
		{
			/* dann Speicher belegen und kopieren */
			speicher_belegen((void **)&blick_vor.text,
				strlen(abschuss_spruch) + 1);
			strcpy(blick_vor.text, abschuss_spruch);
		}
	}

	/* falls Spieler an Leben, Karte mit ihm berechnen */
	if (spieler_aktiv)
		rechne_karte_kreise(&anzkreise, &kreise,
			gegneranz + 1, alle_spieler);
	else
	/* sonst nur die Mitspieler */
		rechne_karte_kreise(&anzkreise, &kreise,
			gegneranz, &alle_spieler[1]);

	{
		int i; /* Index fuer Spieler */

		/* Punktestaende in Liste kopieren */
		liste_initialisieren(&punkte.anzpunkte, (void **)&punkte.punkte);
		for (i = 0; i < SPIELERANZ; i++)
			if (alle_punkte[i] >= 0)
			{
				struct punktestand *stand; /* temporaerer Zeiger in Liste */

				/* Punktestand und Farbe des Spielers an Liste haengen */
				liste_verlaengern(&punkte.anzpunkte, (void **)&punkte.punkte,
					sizeof(struct punktestand));
				stand = &punkte.punkte[punkte.anzpunkte - 1];
				stand->punkte = alle_punkte[i]; /* Punkte */
				stand->farbe = i + 1; /* Farbe */
			}

		/* Punkte sortieren */
		if (punkte.anzpunkte)
			qsort(punkte.punkte, punkte.anzpunkte,
				sizeof(struct punktestand), punkte_vergleich);
	}

	/* Zeichenvorgang synchronisieren */
	zeichne_sync_anfang();

	/* Karten-Grundriss, falls notwendig, neuzeichnen */
	if (karteneu)
		zeichne_grundriss(anzlinien, linien);

	/* 3D-Frontansicht und Rueckspiegel zeichnen */
	zeichne_blickfeld(&blick_vor);
	zeichne_rueckspiegel(&blick_zurueck);

	/* Kreise auf der Karte zeichnen; falls Spieler tot, ihn selbst nicht */
	zeichne_karte(spieler_aktiv ? 0 : -1, anzkreise, kreise);

	/* falls Spieler tot, Kompass ohne Nadel zeichnen */
	if (spieler_aktiv)
		zeichne_kompass(alle_spieler[0].blick * (TRIGANZ / WINKANZ));
	else
		zeichne_kompass(-1);

	/* Punktestand zeichnen */
	zeichne_punkte(&punkte);

	/* audio-visuelle Darstellung der Ereignisse */
	ereignisse_darstellen(ereignisse);

	/* Zeichenvorgang synchronisieren */
	zeichne_sync_ende();

	/* alle Listen mit Grafik-Daten wieder freigeben */
	objekt_listen_freigeben(&blick_vor);
	objekt_listen_freigeben(&blick_zurueck);
	liste_freigeben(&anzkreise, (void **)&kreise);
	liste_freigeben(&punkte.anzpunkte, (void **)&punkte.punkte);

	if (karteneu)
	{
		liste_freigeben(&anzlinien, (void **)&linien);

		/* falls Karten-Grundriss neu gezeichnet wurde, muss dies naechstes
		   Mal vorerst nicht wiederholt werden */
		karteneu = 0;
	}
}


/*
** client_ende
**  beendet das Programm ohne Rueckfrage
*/
static void client_ende()
{
	/* lokale Datenstrukturen aufraeumen */
	grafik_ende();

	/* alle Netzverbindungen beenden */
	netzwerk_ende();

	exit(0);
}


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** spiel_puffer_anlegen
**  ermittelt die Zeiger auf die aktuellen Puffer, in die die neuen
**  Spieldaten abgelegt werden sollen
**
** Parameter:
**  spieler_aktiv_var: Zeiger auf Zeiger auf Flag,
**                     ist der Spieler selbst im Spiel?
**  spieler_var: Zeiger auf Zeiger auf Daten des Spielers selbst
**  gegneranz_var: Zeiger auf Zeiger auf Anzahl der aktuell
**                 sichtbaren Mitspieler
**  gegner_feld: Zeiger auf Liste mit den Daten der sichtbaren Mitspieler
**  schuss_aktiv_var: Zeiger auf Zeiger auf Flag, hat der Spieler
**                    selbst einen Schuss im Spiel?
**  schuss_var: Zeiger auf Zeiger auf Daten des eigenen Schusses
**  schussanz_var: Zeiger auf Zeiger auf Anzahl der sichtbaren
**                 Schuesse der Mitspieler
**  schuesse_feld: Zeiger auf Liste mit den Daten der sichtbaren
**                 Schuesse der Mitspieler
**  abgechossen_durch_var: Zeiger auf Zeiger auf Farbe des Spieler,
**                         von dem man abgeschossen wurde
**  abschluss_spruch_var: Zeiger auf Zeiger auf Spruch des Spieler,
**                        von dem man abgeschossen wurde
**  ereignisse_feld: Zeiger auf Liste mit den neuen Ereignissen
**  punkte_feld: Zeiger auf Liste mit den Punktestaenden
**
** Seiteneffekte:
**  ereignisse_uebrig wird gesetzt
*/
void spiel_puffer_anlegen(spieler_aktiv_var, spieler_var, gegneranz_var,
	gegner_feld, schuss_aktiv_var, schuss_var, schussanz_var,
	schuesse_feld, abgeschossen_durch_var, abschuss_spruch_var,
	ereignisse_feld, punkte_feld)
int **spieler_aktiv_var, **gegneranz_var;
int **schuss_aktiv_var, **schussanz_var;
struct spieler **spieler_var, **gegner_feld;
struct schuss **schuss_var, **schuesse_feld;
int **abgeschossen_durch_var;
char ***abschuss_spruch_var, **ereignisse_feld;
int **punkte_feld;
{
	/* Speicher fuer alten Spruch freigeben */
	if (abschuss_spruch[1 - puffer_nr] != NULL)
		speicher_freigeben((void **)&abschuss_spruch[1 - puffer_nr]);

	/* Zeiger auf die Puffer ermitteln; jeweils der andere als puffer_nr */
	*spieler_aktiv_var = &spieler_aktiv[1 - puffer_nr];
	*spieler_var = alle_spieler[1 - puffer_nr];
	*gegneranz_var = &gegneranz[1 - puffer_nr];
	*gegner_feld = &alle_spieler[1 - puffer_nr][1];

	*schuss_aktiv_var = &schuss_aktiv[1 - puffer_nr];
	*schuss_var = alle_schuesse[1 - puffer_nr];
	*schussanz_var = &schussanz[1 - puffer_nr];
	*schuesse_feld = &alle_schuesse[1 - puffer_nr][1];

	*abgeschossen_durch_var = &abgeschossen_durch[1 - puffer_nr];

	*abschuss_spruch_var = &abschuss_spruch[1 - puffer_nr];

	*ereignisse_feld = ereignisse[1 - puffer_nr];

	*punkte_feld = alle_punkte[1 - puffer_nr];

	/* falls die vorigen Pufferinhalte noch nicht abgearbeitet wurden,
	   alle Ereignisse retten */
	if (puffer_gueltig != puffer_nr)
	{
		int i;

		for (i = 0; i < ERG_ANZ; i++)
			ereignisse_uebrig[i] = (*ereignisse)[i];
	}
}


/*
** netzwerk_spielende
**  wird aufgerufen, wenn der Server den Client aus dem Spiel genommen hat;
**  es wird auf der Stelle eine Meldung ausgegben und das Programm beendet
*/
void netzwerk_spielende()
{
	static char *meldung[] = { "iMaze - Fatal Error", "",
		"Disconnected by server.", NULL };

	uebler_fehler(meldung, NULL);
}


/*
** signale_abfragen
**  wird aufgerufen, wenn die neuen Daten in die Puffer uebertragen wurden.
**  Liest die Eingabe-Signale aus
**
** Parameter:
**  signale: Feld mit allen Signalen
**
** Seiteneffekte:
**  puffer_gueltig wird gesetzt
*/
void signale_abfragen(signale)
char signale[SIGNALANZ];
{
	/* falls der vorige Pufferinhalt noch nicht abgearbeitet wurde,
	   werden die unbearbeiteten Ereignisse wieder uebernommen */
	if (puffer_gueltig != puffer_nr)
	{
		int i;

		for (i = 0; i < ERG_ANZ; i++)
			(*ereignisse)[i] |= ereignisse_uebrig[i];
	}

	/* der andere Puffer hat jetzt die aktuelle Daten */
	puffer_gueltig = 1 - puffer_nr;

	/* Abfrage der Eingabe-Signale */
	eingabe_abfragen(signale);
}


/*
** bewegung_deskriptor
**  liefert einen Deskriptor fuer die Verbindung zum Server im Spiel-Modus
**
** Rueckgabewert:
**  ein Zeiger auf den Deskriptor, NULL, falls es noch keine gibt
*/
void *bewegung_deskriptor()
{
	/* nur den Deskriptor vom Netzwerkteil abfragen und weiterreichen */
	return spiel_deskriptor();
}


/*
** bewegung
**  wartet auf Daten vom Server und initiiert evtl. die Neuberechnung und
**  -zeichnung aller Grafikausgaben
**
** Parameter:
**  ms: Zeit in Millisekunden, die maximal gewartet werden darf;
**      oder -1 fuer unbegrenztes Warten
**  zeichnen: Flag, ob die Grafik neu berechnet und gezeichnet werden soll
**
** Rueckgabewert:
**  1 fuer Fehler, sonst 0
**
** Seiteneffekt:
**  puffer_nr wird veraendert
*/
int bewegung(ms, zeichnen)
int ms, zeichnen;
{
	{
		static int rekursion = 0; /* Flag, ob spiel_paket_erwarten laeuft */

		/* Rekursion von spiel_paket_erwarten verhindern */
		if (rekursion)
			/* kein Fehler */
			return 0;

		rekursion = 1;

		/* Daten vom Server erwarten */
		if (spiel_paket_erwarten(ms))
		{
			rekursion = 0;

			/* Fehler aufgetreten */
			return 1;
		}

		rekursion = 0;
	}

	/* falls nicht gezeichnet werden soll, fertig */
	if (!zeichnen)
		/* kein Fehler */
		return 0;

	if (puffer_gueltig == puffer_nr)
		/* in diesem Fall liegen keine neuen Daten vor; kein Fehler */
		return 0;

	/* der andere Puffer enthaelt die neuen Daten */
	puffer_nr = puffer_gueltig;

	/* Neuberechnung und -zeichnung aller Grafikausgaben mit neuen Daten */
	neuaufbau(spieler_aktiv[puffer_nr], gegneranz[puffer_nr],
		alle_spieler[puffer_nr], schuss_aktiv[puffer_nr],
		schussanz[puffer_nr], alle_schuesse[puffer_nr],
		abgeschossen_durch[puffer_nr], abschuss_spruch[puffer_nr],
		ereignisse[puffer_nr], alle_punkte[puffer_nr]);

	/* kein Fehler */
	return 0;
}


/*
** client_spiel_starten
**  wird aufgerufen, um die Verbindung zum Server aufzubauen
**
** Parameter:
**  server_name: Name des Servers
**  spruch: Spruch des Spielers oder NULL
**
** Rueckgabewert:
**  1 fuer Fehler, sonst 0
**
** Seiteneffekte:
**  Daten werden vom Server empfangen; karteneu wird initialisiert
*/
int client_spiel_starten(server_name, spruch)
char *server_name, *spruch;
{
	/* Verbindung zum angegebenen Server aufbauen */
	if (verbindung_aufbauen(server_name))
		return 1;

	/* Spielparameter / Labyrinth empfangen */
	if (spielparameter_empfangen("iMaze xview client JC/HUK 1.2",
		"Anonymous User", strlen(spruch) ? spruch : NULL, &feldbreite,
		&feldlaenge, &spielfeld))
		return 1;

	/* Spielfeld drehen */
	drehlab();

	/* Karte neu aufbauen */
	karteneu = 1;

	/* jetzt kanns losgehen */
	if (verbindung_auf_spiel())
		return 1;

	/* kein Fehler */
	return 0;
}


/*
**  client_spiel_beenden
**   wird aufgerufen, wenn eine Spielrunde verlassen wird (disconnect)
*/
void client_spiel_beenden()
{
	/* Verbindung zum Server beenden */
	spiel_verlassen();
}


/*
** uebler_fehler
**  zeigt eine Fehlermeldung an und beendet nach Druecken eines Knopfes
**  das Programm
**
** Parameter:
**  meldung: Feld von Strings, die die Zeilen des auszugebenden Textes
**           beinhalten
**  knopf: Text der auf dem Knopf stehen soll;
**         falls NULL, so wird ein Standardtext verwendet
*/
void uebler_fehler(meldung, knopf)
char **meldung, *knopf;
{
	/* Fehler anzeigen */
	ueblen_fehler_anzeigen(meldung, knopf);

	/* Programm beenden */
	client_ende();
}


/*
** milder_fehler
**  merkt sich eine Fehlermeldung fuer die Routine milden_fehler_abfragen
**
** Parameter:
**  meldung: Feld von Strings, die die Zeilen des auszugebenden Textes
**           beinhalten
**  knopf: Text, der auf dem Knopf stehen soll;
**         falls NULL, so wird ein Standardtext verwendet
**
** Seiteneffekte:
**  fehlermeldung und fehlerknopf werden gesetzt
*/
void milder_fehler(meldung, knopf)
char **meldung, *knopf;
{
	int n; /* Anzahl der Fehlerstrings */
	int i; /* Index fuer Fehlerstrings */

	/* Speicher fuer vorige Fehlermeldung freigeben */
	if (fehlermeldung != NULL)
	{
		for (i = 0; meldung[i] != NULL; i++)
			speicher_freigeben((void **)&fehlermeldung[i]);
		speicher_freigeben((void **)&fehlermeldung);
	}

	/* Strings zaehlen */
	for (n = 0; meldung[n] != NULL; n++);

	/* Speicher fuer das Feld belegen und durch Nullzeiger begrenzen */
	speicher_belegen((void **)&fehlermeldung, (n + 1) * sizeof *fehlermeldung);
	fehlermeldung[n] = NULL;

	/* Fehlerstrings kopieren */
	for (i = 0; i < n; i++)
	{
		speicher_belegen((void **)&fehlermeldung[i], strlen(meldung[i]) + 1);
		strcpy(fehlermeldung[i], meldung[i]);
	}

	/* Speicher fuer vorige Knopfbeschriftung freigeben */
	if (fehlerknopf != NULL)
		speicher_freigeben((void **)&fehlerknopf);

	/* Knopfbeschriftung kopieren */
	if (knopf == NULL)
		fehlerknopf = NULL;
	else
	{
		speicher_belegen((void **)&fehlerknopf, strlen(knopf) + 1);
		strcpy(fehlerknopf, knopf);
	}
}


/*
** milden_fehler_abfragen
**  fragt den in milder_fehler gemerkten Fehlertext und -knopf ab
**
** Parameter:
**  meldung: Zeiger auf Feld von Strings, das die Zeilen des
**           auszugebenden Textes beinhalten soll
**  knopf: Zeiger auf Text, der auf dem Knopf stehen soll
**
** Seiteneffekte:
**  *meldung und *knopf werden gesetzt
*/
void milden_fehler_abfragen(meldung, knopf)
char ***meldung, **knopf;
{
	/* Rueckgabewerte setzen */
	*meldung = fehlermeldung;
	*knopf = fehlerknopf;
}


/*
** main
**  die Hauptroutine
**
** Parameter:
**  argc: Anzahl der Argumente inklusive Programmname
**  argv: Argumentliste
*/
void main(argc, argv)
int argc;
char **argv;
{
	/* Netzwerkroutinen initialisieren */
	if (netzwerk_init(&argc, argv))
		/* bei Fehler beenden */
		exit(1);

	/* Fenster oeffnen und lokale Datenstrukturen initialisieren */
	if (grafik_init(&argc, argv))
		/* bei Fehler beenden */
		exit(1);

	/* falls eine unbekannte Option angegeben wurde, Fehler melden */
	if (argc > 1)
	{
		fprintf(stderr, "%s: unknown option \"%s\"\n", argv[0], argv[1]);
		exit(1);
	}

	/* Hauptschleife */
	grafik_schleife();

	/* Programm beenden */
	client_ende();
}
