/*
   map.cc

   This file is part of LuxMan.
   
   Copyright (C) 1994,1995 Frank McIngvale (frankm@nuance.com)
   Portions Copyright (C) 1994 Bruce Tenison (rbtenis@ibm.net)
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <sys/mount.h>
#include <rawkey/rawkey.h>
#include <gtools/gtools.h>
#include <argv/argv.h>
#include "sigs.h"
#include "run.h"
#include "globals.h"
#include "score.h"
#include "sound.h"
#include "highscore.h"
#include "calib.h"
#include "play.h"
#include "error.h"

char cmd_help = ARGV_FALSE;

argv_t args[] = {
#ifdef DEBUG  
  { 't', "test", ARGV_BOOL, &gb_test_mode, "test", "enable test mode" },
  { 'l', "level", ARGV_INT, &gb_start_level, "level", "start level" },
#endif
  { 'r', "fps", ARGV_INT, &gb_frames_per_second, "fps", "set frames-per-second" },
  { 'd', "depth", ARGV_INT, &gb_ghost_search_depth, "depth", "set search depth" },
  { 'h', "help", ARGV_BOOL, &cmd_help, "help", "show this help screen" },
  { '8', "8k", ARGV_BOOL, &gb_use_8k_sound, "8k sound", "use 8kHz sound (11kHz default)" },
  { 'n', "nosnd", ARGV_BOOL, &gb_dont_use_sound, "no sound", "don't use sound" },
  { ARGV_LAST }
};

int init_sound( void )
{
  int rate;

  if ( snd_open_server( SND_LAST + 1, CHAN_LAST + 1 ) != 0 )
	{
	  gb_dont_use_sound = ARGV_TRUE;
	  return -1;
	}
  
  if ( gb_use_8k_sound == ARGV_TRUE )
	rate = 8000;
  else
	rate = 11025;

  load_sounds();

  if ( gb_dont_use_sound != ARGV_TRUE )
	{
	  if ( snd_open_sound( rate ) != 0 )
		gb_dont_use_sound = ARGV_TRUE;
	}
  
  return 0;
}

/*
   items		-- strings to put in menu
   num			-- number of strings in `items'
   x, y			-- upper left corner of menu
   default_item	-- default item [0..(num-1)]

   Returns [0..(num-1)] on success, -1 on error
*/   
int do_lux_menu( char **items, int num, int y, int default_item,
				int centered )
{
  Bitmap *lux, *lux_b;
  Font *font, *font_b;
  int w, h, i, j;
  int pos;
  int x, sc;
  int sc_p;

  sc_p = scancode_trans( 'p' );
  
  if ( default_item < 0 || default_item >= num )
	{
	  fatal( "Bad default (%s,%d)\n", __FILE__, __LINE__ );
	  return -1;	/* make gcc shut up */
	}
  else
	pos = default_item;
  
  lux = new Bitmap( "lux_r1.map" );
  lux_b = new Bitmap( "lux_r1.map" );
  lux_b->subst( BLACK, YELLOW );
  
  h = lux->h * 3/2;
  
  font = new Font( "small.font" );

  font_b = new Font( "small.font" );
  font_b->subst( BLACK, WHITE );
  
  /* Get max string width */
  w = 0;
  for( i=0; i<num; ++i )
	{
	  if ( (j=gr_textw( items[i], font )) > w )
		w = j;
	}

  w = w * 3/2;
  
  /* Center strings on screen */
  j = w + lux->w * 3/2;
  x = 320/2 - j/2;

  gr_frame( x - 5, y - 5, x + lux->w*3/2 + w + 5,
		   y + h*num + 5, 5, 5, LIGHTGRAY, WHITE, DARKGRAY, LIGHTGRAY );
		 
  /* Draw strings */
  for( i=0; i<num; ++i )
	{
	  if ( !centered )
		{
		  gr_textxy( items[i], 1 + x + lux->w * 3/2,
					1 + y + (i*h) + h/2 - gr_texth( items[i], font )/2,
					font_b );
		  gr_textxy( items[i], x + lux->w * 3/2,
					y + (i*h) + h/2 - gr_texth( items[i], font )/2,
					font_b );
		}
	  else
		{
		  gr_textxy( items[i], 1 + x + lux->w * 3/2 + w/2 -
					gr_textw( items[i],font )/2,
					1 + y + (i*h) + h/2 - gr_texth( items[i], font )/2,
					font_b );
		  gr_textxy( items[i], x + lux->w * 3/2 + w/2 -
					gr_textw( items[i],font )/2,
					y + (i*h) + h/2 - gr_texth( items[i], font )/2, font );
		}
	}	
  
  /* Draw pointer */
  put_bitmap_t( 1 + x + lux->w/4, 1 + y + (pos*h) + h/2 - lux->h/2, lux_b );
  put_bitmap_t( x + lux->w/4, y + (pos*h) + h/2 - lux->h/2, lux );
  gr_update();
  
  while( 1 )
	{
	  scan_keyboard();

	  if ( is_key_pressed( sc_p ) )
		{
		  shell_out(NULL);
		  gr_update();
		  continue;
		}

	  if ( is_key_pressed( ENTER_KEY ) )
		{
		  while( is_key_pressed( ENTER_KEY ) )
			scan_keyboard();
		  break;
		}
	  
	  if ( is_key_pressed( CURSOR_UP ) || is_key_pressed( CURSOR_DOWN ) )
		/* Erase current pointer */
		gr_fillbox( x, y + (pos*h) + h/2 - lux->h/2,
				   x + lux->w + 1, y + (pos)*h + h/2 + lux->h/2,
				   LIGHTGRAY );
	  else
		continue;

	  if ( is_key_pressed( CURSOR_UP ) )
		{
		  if ( pos == 0 )
			pos = num-1;
		  else
			--pos;

		  sc = CURSOR_UP;
		}
	  else
		{
		  if ( pos == num-1 )
			pos = 0;
		  else
			++pos;

		  sc = CURSOR_DOWN;
		}

	  put_bitmap_t( 1 + x + lux->w/4, 1 + y + (pos*h) + h/2 - lux->h/2,
				   lux_b );
	  put_bitmap_t( x + lux->w/4, y + (pos*h) + h/2 - lux->h/2, lux );
	  gr_update();

	  while( is_key_pressed( sc ) )
		scan_keyboard();
	}

  delete lux;
  delete lux_b;
  delete font;
  delete font_b;
  return pos;
}

int do_main_menu( int default_item )
{
  Font *big, *small;
  char *banner = "LuxMan";
  char *banner2 = "A maze game for Linux";
  char *banner3 = "(c) 1995 Frank McIngvale (frankm@nuance.com)";
 
  char *items[] = { "Start new game", "View highscores",
					"About LuxMan...", "Quit" };
  int r;
  
  big = new Font( "test.font" );
  small = new Font( "small.font" );
  
  gr_fillbox( 0,0,319,198,BLACK );

  gr_textxy( banner, 320/2 - gr_textw(banner,big)/2, 10, big );
  gr_textxy( banner2, 320/2 - gr_textw(banner2,small)/2, 35, small );
  gr_textxy( banner3, 320/2 - gr_textw(banner3,small)/2, 180, small );
  
  r = do_lux_menu( items, 4, 65, default_item, 1 );

  delete small;
  delete big;

  return r;
}

/* Returns 0 (easy) through 3 (hard), or -1 if user
   chooses `Back to Main' */
int ask_skill_level()
{
  char *labels[] = { "Hard", "Medium", "Easy", "Simple", "Back to Main" };
  char *banner = "Select Skill Level";
  int r;
  Font *font;

  font = new Font( "small.font" );
  
  gr_fillbox( 0,0,319,198, BLACK );
  gr_textxy( banner, 320/2 - gr_textw(banner,font)/2, 20, font );

  r = do_lux_menu( labels, 5, 50, 1, 1 );

  delete font;

  if ( r == 4 )
	return -1;
  else
	return (3-r);
}

void do_credits_screen()
{
  Font *small, *big;
  int x1=80, x2, y = 50;
  int spc=30;
  int sc_p;
  char buf[200];

  sc_p = scancode_trans( 'p' );
  
  big = new Font( "test.font" );
  small = new Font( "small.font" );

  x2 = x1 + gr_textw( "Graphics: ", small );
  
  gr_fillbox( 0,0,319,199,BLACK );

  sprintf( buf, "LuxMan v%s", gb_version_string );
  
  gr_textxy_cu( buf, 320/2, 15, big );
  
  gr_textxy_u( "Graphics", x1, y, small );
  gr_textxy( "Frank McIngvale", x2, y, small );

  gr_textxy_u( "Sound", x1, y+spc, small );
  gr_textxy( "Bruce Tenison", x2, y+spc, small );
  gr_textxy( "Frank McIngvale", x2, y+2*spc, small );
  
  gr_textxy_u( "Sampling", x1, y+3*spc, small );
  gr_textxy( "David Clem", x2, y+3*spc, small );
  gr_textxy( "Frank McIngvale", x2, y+4*spc, small );

  gr_update();
  
  while( !is_key_pressed( ENTER_KEY ) )
	{
	  scan_keyboard();

	  if ( is_key_pressed( sc_p ) )
		{
		  shell_out( NULL );
		  gr_update();
		}
	}
}

main( int argc, char *argv[] )
{
  int score, level, skill;
  int r=0, index=-1;
  
  printf("LuxMan v%s, Copyright (c) 1995 Frank McIngvale\n", gb_version_string );
  printf("LuxMan comes with ABSOLUTELY NO WARRANTY; see COPYING for details.\n\n");

  /* Init VGA and give up root privs */
  vga_init();

  argv_process( args, argc, argv );

  if ( cmd_help == ARGV_TRUE )
	{
	  argv_usage( args, ARGV_USAGE_LONG );
	  return 0;
	}

  if ( gb_test_mode == ARGV_TRUE )
	printf("Test mode -- Skipping calibration\n");
  else
	{
	  printf("The frame rate is now set to %d frames per second.\n",
			 gb_frames_per_second );
	  printf("If the game seems too fast, too slow, or too jerky,\n");
	  printf("you can adjust this value the `-r' option.\n\n");
	  printf("Calibrating delay...");
	  fflush( stdout );
	  calibrate_usleep( 3 );
	}
  
  gb_pid_sndserver = fork();

  if ( gb_pid_sndserver < 0 )
	{
	  printf("Error: Can't fork\n");
	  exit(1);
	}
  
  if ( gb_pid_sndserver == 0 )
	{
	  execlp( "luxman-snd", "luxman-snd", NULL );
	  
	  printf("Error: Unable to exec luxman-snd (sound server)\n");
	  exit(0);
	}
  
  gr_init( 1 );
  set_sig_handlers();
  rawmode_init();

#ifdef DEBUG
  printf("[Debugging version]\n");
#endif

  init_sound();
  
  do {
	r = do_main_menu( r );

	if ( r == 0 )
	  {
		skill = ask_skill_level();
		if ( skill == -1 )	/* Back to main */
		  continue;

		score = run_game( &level, skill );
		
		index = update_highscores( score, level, skill );
		display_highscores( index );
	  }
	else if ( r == 1 )
	  display_highscores( index );
	else if ( r == 2 )
	  do_credits_screen();

  } while( r != 3 );
  
  rawmode_exit();
  gr_close();

  snd_close_server();

  kill( gb_pid_sndserver, SIGTERM );
}



