/*
   map2c.cc

   Utility to convert libgtools bitmaps to C arrays.
   
   Copyright (C) 1994,1995 Frank McIngvale (frankm@nuance.com)
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 1, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <string.h>
#include <unistd.h>

/*
 * map2c
 *
 * Usage:
 *
 * 	map2c cfile mapfile1 ...
 *
 * Reads a bitmap from `mapfile' and outputs C code of the form:
 *
 *
 * unsigned char Bitmap_prefix_data[] = {
 * 	.
 * 	.
 * 	};
 *
 *
 * int Bitmap_prefix_width = W;
 *
 *
 * int Bitmap_prefix_height = H;
 *
 * .. which it write to file `cfile'.
 *
 * `prefix' is generated by removing any file extension from the mapfile
 * name.
 *
 * NOTE! If `cfile' exists, the old file is OVERWRITTEN.
 */
#include <gtools/bitmap.h>
#include <stdarg.h>
#include <stdlib.h>

char *gb_library_path = ".";

void fatal( char *fmt, ... )
{
  va_list ap;

  va_start( ap, fmt );
  vprintf( fmt, ap );
  exit(1);
}

int get_prefix( char *dest, char *src )
{
  char *p, *s;

  s = strrchr( src, '/' );
  if ( !s )
	s = src;
  else
	++s;
  
  p = strchr( s, '.' );

  if ( p )
	{
	  *p = 0;
	  strcpy( dest, s );
	  *p = '.';
	}
  else
	strcpy( dest, s );

  return 0;
}
	  
int do_file( FILE *fpout, char *prefix, char *mapname )
{
  Bitmap *map;
  int i, j;
  unsigned char c;
  unsigned char *m;

  map = new Bitmap( mapname );
  
  fprintf( fpout,"/* This code generated by map2c from file `%s' */\n\n",
		  mapname );
  
  fprintf( fpout, "/* Bitmap data */\n");
  fprintf( fpout, "unsigned char Bitmap_%s_data[] = {\n", prefix );

  for( i=0; i<map->h; ++i )
	{
	  m = map->map + i*map->w;
	  
	  for( j=0; j<map->w; ++j )
		{
		  c = m[j];

		  if ( j == (map->w-1) && i == (map->h-1) )
			fprintf( fpout, "%d ", (int)c );
		  else
			fprintf( fpout, "%d, ", (int)c );
		}

	  if ( i < (map->h-1) )
		fprintf( fpout, "\n" );
	  else
		fprintf( fpout, "};\n\n");
	}

  fprintf( fpout, "/* Bitmap width */\n");
  fprintf( fpout, "int Bitmap_%s_width = %d;\n\n", prefix, map->w );
  fprintf( fpout, "/* Bitmap height */\n");
  fprintf( fpout, "int Bitmap_%s_height = %d;\n", prefix, map->h );

  delete map;

  return 0;
}

main( int argc, char *argv[] )
{
  FILE *fp;
  int i;
  char prefix[200];
  
  if ( argc < 3 )
	{
	  printf("Usage: map2c cfile mapfile1 ...\n");
	  return -1;
	}

  fp = fopen( argv[1], "w" );

  for( i=2; i<argc; ++i )
	{
	  get_prefix( prefix, argv[i] );
	  do_file( fp, prefix, argv[i] );
	}

  fclose( fp );
}
