/* $Id: variable.c,v 30000.23 1993/05/25 01:08:50 kkeys Exp $ */
/******************************************************************
 * Copyright 1993 by Ken Keys.
 * Permission is granted to obtain and redistribute this code freely.
 * All redistributions must contain this header.  You may modify this
 * software, but any redistributions of modified code must be clearly
 * marked as having been modified.
 ******************************************************************/


/**************************************
 * Internal and environment variables *
 **************************************/

#include <ctype.h>
#include "port.h"
#include "tf.h"
#include "util.h"
#include "output.h"
#include "macro.h"
#include "socket.h"
#include "search.h"

char *FDECL(getvar,(char *name));
int   FDECL(setvar,(char *name, char *value, int exportflag));
void  FDECL(setivar,(char *name, int value, int exportflag));
void  FDECL(unsetvar,(char *name));
void  FDECL(listvar,(int exportflag));
int   FDECL(export,(char *name));

static Toggler *FDECL(set_tf_var,(Var *var, char *value));
static char    *FDECL(newstr,(char *name, char *value));
static Var     *FDECL(newvar,(char *name));
static void     FDECL(append_env,(char *name, char *value));
static char   **FDECL(find_env,(char *name));
static void     FDECL(remove_env,(char **envp));
static void     FDECL(replace_env,(char *name, char *value));

#define HASH_SIZE 197    /* prime number */

static HashTable var_table[1];
static int envsize;
static int envmax;

extern char **environ;

Var special_var[] = {
  { "MAIL"              , NULL, VAR_STR,  0    , change_mailfile  , NULL },
  { "background"        , NULL, VAR_FLAG, TRUE , tog_background   , NULL },
  { "backslash"         , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "bamf"              , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "barattr"           , NULL, VAR_STR , 0    , change_bar       , NULL },
  { "beep"              , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "borg"              , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "cleardone"         , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "clearfull"         , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "gag"               , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "gpri"              , NULL, VAR_INT , 0    , NULL             , NULL },
  { "hilite"            , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "hiliteattr"        , NULL, VAR_STR,  0    , change_hilite    , NULL },
  { "hook"              , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "hpri"              , NULL, VAR_INT , 0    , NULL             , NULL },
  { "insert"            , NULL, VAR_FLAG, TRUE , tog_insert       , NULL },
  { "isize"             , NULL, VAR_INT , 3    , change_isize     , NULL },
  { "kecho"             , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "kprefix"           , NULL, VAR_STR , 0    , NULL             , NULL },
  { "login"             , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "lp"                , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "lpquote"           , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "maildelay"         , NULL, VAR_INT,  60   , change_maildelay , NULL },
  { "max_recur"         , NULL, VAR_INT,  100  , NULL             , NULL },
  { "mecho"             , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "more"              , NULL, VAR_FLAG, FALSE, tog_more         , NULL },
  { "mprefix"           , NULL, VAR_STR , 0    , NULL             , NULL },
  { "oldslash"          , NULL, VAR_INT , 1    , NULL             , NULL },
  { "prompt_sec"        , NULL, VAR_INT , 0    , NULL             , NULL },
  { "prompt_usec"       , NULL, VAR_INT , 250000,NULL             , NULL },
  { "ptime"             , NULL, VAR_INT , 1    , NULL             , NULL },
  { "qecho"             , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "qprefix"           , NULL, VAR_STR , 0    , NULL             , NULL },
  { "quiet"             , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "quitdone"          , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "redef"             , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "refreshtime"       , NULL, VAR_INT,  250000,NULL             , NULL },
  { "shpause"           , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "snarf"             , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "sockmload"         , NULL, VAR_FLAG, FALSE, NULL             , NULL },
#if 0
  { "switch_on_connect" , NULL, VAR_FLAG, TRUE , NULL             , NULL },
#endif
  { "visual"            , NULL, VAR_FLAG, FALSE, tog_visual       , NULL },
  { "watchdog"          , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "watchname"         , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "wrap"              , NULL, VAR_FLAG, TRUE , NULL             , NULL },
  { "wraplog"           , NULL, VAR_FLAG, FALSE, NULL             , NULL },
  { "wrapsize"          , NULL, VAR_INT,  0,     NULL             , NULL },
  { "wrapspace"         , NULL, VAR_INT,  0,     NULL             , NULL }
};

/* initialize structures for variables */
void init_variables()
{
    char **oldenv, **p, *value, buf[20];
    int i;
    Var *var;

    init_hashtable(var_table, HASH_SIZE, strcmp);

    /* internal user variables */
    for (i = 0; i < NUM_VARS; i++) {
        var = &special_var[i];
        if (var->flags & VAR_STR) continue;
        sprintf(buf, "%d", var->ival);
        var->value = STRDUP(buf);
        var->node = hash_insert((GENERIC *)var, var_table);
    }

    /* environment variables */
    for (p = environ; *p; p++);
    envsize = 0;
    envmax = p - environ;
    oldenv = environ;
    environ = (char **)MALLOC((envmax + 1) * sizeof(char *));
    *environ = NULL;
    for (p = oldenv; *p; p++) {
        value = strchr(*p, '=');
        *value++ = '\0';
        i = binsearch(*p, (GENERIC*)special_var, NUM_VARS, sizeof(Var), strcmp);
        if (i < 0) {
            var = newvar(*p);
            var->value = STRDUP(value);
            var->node = hash_insert((GENERIC *)var, var_table);
            append_env(*p, value);
            var->flags |= VAR_EXPORT;
        } else {
            var = &special_var[i];
            /* if (var->value) */ FREE(var->value);
            set_tf_var(var, value);
        }
        *--value = '=';
    }
}

void init_values()
{
    int i;

    for (i = 0; i < NUM_VARS; i++) {
        if (special_var[i].ival && special_var[i].func)
            (*special_var[i].func)();
    }
}

/* get value for <name> */
char *getvar(name)
    char *name;
{
    Var *var;

    return (var = (Var *)hash_find(name, var_table)) ? var->value : NULL;
}

static char *newstr(name, value)
    char *name, *value;
{
    char *str;

    str = (char *)MALLOC(strlen(name) + 1 + strlen(value) + 1);
    return strcat(strcat(strcpy(str, name), "="), value);
}

static Var *newvar(name)
    char *name;
{
    Var *var;

    var = (Var *)MALLOC(sizeof(Var));
    var->name = STRDUP(name);
    var->value = NULL;
    var->flags = VAR_STR;
    var->ival = 0;
    var->func = NULL;
    var->node = NULL;
    return var;
}

/* Add "<name>=<value>" to environment.  Assumes name is not already defined. */
static void append_env(name, value)
    char *name, *value;
{
    if (envsize == envmax) {
        envmax = envsize + 5;
        environ = (char **)REALLOC((char*)environ, (envmax+1) * sizeof(char*));
    }
    environ[envsize] = newstr(name, value);
    environ[++envsize] = NULL;
}

/* Find the environment string for <name>. */
static char **find_env(name)
    char *name;
{
    char **envp;
    int len = strlen(name);

    for (envp = environ; *envp; envp++)
        if (strncmp(*envp, name, len) == 0 && (*envp)[len] == '=')
            return envp;
    return NULL;
}

/* Remove the environment string for <name>. */
static void remove_env(envp)
    char **envp;
{
    char *old;

    old = *envp;
    do *envp = *(envp + 1); while (*++envp);
    envsize--;
    FREE(old);
}

/* Replace the environment string for <name> with "<name>=<value>". */
static void replace_env(name, value)
    char *name, *value;
{
    char **envp;

    envp = find_env(name);
    FREE(*envp);
    *envp = newstr(name, value);
}

int setvar(name, value, exportflag)
    char *name, *value;
    int exportflag;
{
    Var *var;
    int i;
    Toggler *func = NULL;

    for (i = 0; name[i]; i++) {
        if (!isalnum(name[i]) && name[i] != '_') {
            tfputs("% illegal variable name", tferr);
            return 0;
        }
    }

    i = binsearch(name, (GENERIC *)special_var, NUM_VARS, sizeof(Var), strcmp);

    if (i >= 0) {
        var = &special_var[i];
        if (var->value) FREE(var->value);
        func = set_tf_var(&special_var[i], value);
    } else if ((var = (Var *)hash_find(name, var_table))) {
        FREE(var->value);
        var->value = STRDUP(value);
    } else {
        var = newvar(name);
        var->value = STRDUP(value);
    }

    if (var->node) {                     /* does it already exist? */
        if (var->flags & VAR_EXPORT) {
            replace_env(name, value);
        } else if (exportflag) {
            append_env(name, value);
            var->flags |= VAR_EXPORT;
        }
    } else {
        var->node = hash_insert((GENERIC *)var, var_table);
        if (exportflag) {
            append_env(name, value);
            var->flags |= VAR_EXPORT;
        }
    }

    if (func) (*func)();
    return 1;
}

void setivar(name, value, exportflag)
    char *name;
    int value, exportflag;
{
    static char buf[20];
    sprintf(buf, "%d", value);
    setvar(name, buf, exportflag);
}

int export(name)
    char *name;
{
    Var *var;

    if (!(var = (Var *)hash_find(name, var_table))) {
        tfprintf(tferr, "%% %s not defined.", name);
        return 0;
    }
    if (!(var->flags & VAR_EXPORT)) append_env(var->name, var->value);
    return 1;
}
    
void unsetvar(name)
    char *name;
{
    int oldval, i;
    Var *var;

    if (!(var = (Var *)hash_find(name, var_table))) return;

    hash_remove(var->node, var_table);

    i = binsearch(name, (GENERIC*)special_var, NUM_VARS, sizeof(Var), strcmp);
    if (i < 0) {
        if (var->flags & VAR_EXPORT) remove_env(find_env(name));
        FREE(var->name);
        FREE(var->value);
        FREE(var);
    } else {
        oldval = special_var[i].ival;
        special_var[i].ival = 0;
        if (special_var[i].flags & VAR_EXPORT) {
            remove_env(find_env(name));
            special_var[i].flags &= ~VAR_EXPORT;
        }
        FREE(special_var[i].value);
        if (oldval && special_var[i].func) (*special_var[i].func)();
    }
}

/* Set a special variable, with proper coersion of the value. */
/* FLAG vars are coerced to "0" or "1", INT vars to an integer. */
static Toggler *set_tf_var(var, value)
    Var *var;
    char *value;
{
    int oldival;
    Toggler *func;
    static char buffer[20];

    oldival = var->ival;

    if (var->flags & VAR_INT) {
        var->ival = atoi(value);
        sprintf(value = buffer, "%d", var->ival);
        func = (var->ival != oldival) ? var->func : NULL;

    } else if (var->flags & VAR_FLAG) {
        while (isspace(*value)) value++;
        if (isdigit(*value))
            var->ival = atoi(value) ? 1 : 0;
        else
            var->ival = (*value && cstrcmp(value, "off") != 0);
        strcpy(value = buffer, var->ival ? "1" : "0");
        func = (var->ival != oldival) ? var->func : NULL;

    } else /* if (var->flags & VAR_STR) */ {
        var->ival = (*value) ? 1 : 0;
        func = var->func;
    }

    var->value = STRDUP(value);
    return func;
}

void listvar(exportflag)
    int exportflag;
{
    int i;
    ListEntry *node;

    for (i = 0; i < var_table->size; i++) {
        if (var_table->bucket[i]) {
            for (node = var_table->bucket[i]->head; node; node = node->next) {
                if (!(((Var *)(node->data))->flags & VAR_EXPORT) == !exportflag)
                    oprintf("/%s %s=%s", exportflag ? "setenv" : "set",
                        ((Var *)(node->data))->name,
                        ((Var *)(node->data))->value);
            }
        }
    }
}

#ifdef DMALLOC
void free_vars()
{
    char **p;
    int i;
    ListEntry *node, *next;

    for (p = environ; *p; p++) FREE(*p);
    FREE(environ);

    for (i = 0; i < NUM_VARS; i++) {
        if (special_var[i].value) FREE(special_var[i].value);
        if (special_var[i].node) hash_remove(special_var[i].node, var_table);
    }

    for (i = 0; i < var_table->size; i++) {
        if (var_table->bucket[i]) {
            for (node = var_table->bucket[i]->head; node; node = next) {
                next = node->next;
                FREE(((Var *)(node->data))->name);
                FREE(((Var *)(node->data))->value);
                FREE(node->data);
                FREE(node);
            }
        }
    }
    free_hash(var_table);
}
#endif
