/*  Copyright 1992 John Bovey, University of Kent at Canterbury.
 *
 *  You can do what you like with this source code as long as
 *  you don't try to make money out of it and you include an
 *  unaltered copy of this message (including the copyright).
 */
/*
 * This module has been very heavily modified by R. Nation
 * (nation@rocket.sanders.lockheed.com).
 * No additional restrictions are applied
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 */

#include <stdarg.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <stdio.h>
#include <termios.h>
#include <unistd.h>
#include <signal.h>
#include <fcntl.h>
#include <grp.h>
#include <string.h>
#include <sys/time.h>
#include <stdlib.h>

#include "rxvt.h"
#include "command.h"
#include "screen.h"
#include "xsetup.h"
#include "sbar.h"

int size_set = 0;      /* flag set once the window size has been set */

#define NLMAX	15	/* max number of lines to scroll */

#define KBUFSIZE	256	/* size of keyboard mapping buffer */
#define COM_BUF_SIZE	512	/* size of buffer used to read from command */
#define COM_PUSH_MAX	20	/* max number of chars to push back in queue */
#define STRING_MAX      512     /* maximum size of strings to process */
#define ARGS_MAX         20      /* maximum # of args for esc sequences */

/*  Special character returned by get_com_char().
 */
#define GCC_NULL	0x100		/* Input buffer is empty */
#define ESC		033

/*  Flags used to control get_com_char();
 */
#define BUF_ONLY	1

/*  Global variables that are set up at the beginning and then not changed
 */
extern Display		*display;
extern Window		vt_win;
extern Window		main_win;
extern struct sbar_info sbar;

extern int login_shell;

int comm_fd = -1;/* file descriptor connected to the command */
static int comm_pid;	/* process id if child */
static int x_fd;	/* file descriptor of the X server connection */
static int fd_width;	/* width of file descriptors being used */
static int app_cur_keys = 0;/* flag to set cursor keys in application mode */
static int app_kp_keys = 0; /* flag to set application keypad keys */
static char ptynam[] = "/dev/ptyxx";
static char ttynam[] = "/dev/ttyxx";
static Atom wm_del_win;

extern char *console;
int console_fd;
/*  Terminal mode structures.
 */
static struct termios ttmode = 
{
  BRKINT | IGNPAR | ISTRIP | ICRNL | IXON | IMAXBEL,
  OPOST | ONLCR,
  B9600 | PARENB | CS8 | CREAD,
  ISIG | IEXTEN | ICANON | ECHO | ECHOE | ECHOK | ECHOCTL | ECHOKE,
  0,
  { 003, 034, 0177, 025,
      004, 000, 001, 000,
      021, 023, 032, 022,
      022, 017, 027, 026
      }
};

/*  Variables used for buffered command input.
 */
static unsigned char com_buf[COM_BUF_SIZE];
static unsigned char *com_buf_next, *com_buf_top;
static unsigned char com_stack[COM_PUSH_MAX];
static unsigned char *com_stack_top;
unsigned char mask = 0xff;

static void catch_child();
static void catch_sig(int);
static int run_command(char *,char **);
static char *lookup_key(XEvent *,int *);
static int get_com_char(int);
static void push_com_char(int);
void get_X_event();
void process_string(int);
void process_escape_sequence();
void process_csi_sequence();
void process_xterm_sequence();
void process_terminal_mode(int,int,int,int *);
void process_sgr_mode(int,int,int,int *);
void dummy();

/*  Catch a SIGCHLD signal and exit if the direct child has died.
 */
static void catch_child()
{
  if (wait((int *)NULL) == comm_pid)
    exit(0);
}

/*  Catch a fatal signal and tidy up before quitting
 */
static void catch_sig(int sig)
{
  signal(sig,SIG_DFL);
  setuid(getuid());
  kill(getpid(),sig);
}

/*  Run the command in a subprocess and return a file descriptor for the
 *  master end of the pseudo-teletype pair with the command talking to
 *  the slave.
 */
static int run_command(char *command,char **argv)
{
  int ptyfd, ttyfd;
  int uid, gid;
  char *s3, *s4;
  int i;
  int width, height;
  static char ptyc3[] = "pqrstuvwxyz";
  static char ptyc4[] = "0123456789abcdef";
  char argv0[256];

  /*  First find a master pty that we can open.
   */
  ptyfd = -1;
  for (s3 = ptyc3; *s3 != 0; s3++) 
    {
      for (s4 = ptyc4; *s4 != 0; s4++) 
	{
	  ptynam[8] = ttynam[8] = *s3;
	  ptynam[9] = ttynam[9] = *s4;
	  if ((ptyfd = open(ptynam,O_RDWR)) >= 0) 
	    {
	      if (geteuid() == 0 || access(ttynam,R_OK|W_OK) == 0)
		break;
	      else 
		{
		  close(ptyfd);
		  ptyfd = -1;
		}
	    }
	}
      if (ptyfd >= 0)
	break;
    }
  if (ptyfd < 0) 
    {
      error("Can't open a pseudo teletype");
      return(-1);
    }
  fcntl(ptyfd,F_SETFL,O_NDELAY);
  
  for (i = 1; i <= 15; i++)
    signal(i,catch_sig);
  signal(SIGCHLD,catch_child);
  comm_pid = fork();
  if (comm_pid < 0) 
    {
      error("Can't fork");
      return(-1);
    }
  if (comm_pid == 0) 
    {
      struct group *gr;
      
      if (setsid() < 0)
	error("failed to set process group");
      
      if ((ttyfd = open(ttynam,O_RDWR)) < 0) 
	{
	  error("could not open slave tty %s",ttynam);
	  exit(1);
	}
      uid = getuid();
      if ((gr = getgrnam("tty")) != NULL)
	gid = gr->gr_gid;
      else
	gid = -1;
      fchown(ttyfd,uid,gid);
      fchmod(ttyfd,0600);
      for (i = 0; i < getdtablesize(); i++)
	if (i != ttyfd)
	  close(i);
      dup(ttyfd);
      dup(ttyfd);
      dup(ttyfd);
      if (ttyfd > 2)
	close(ttyfd);
      if(mask == 0x7f)
	ttmode.c_cflag = B9600 | PARENB | CS7 | CREAD;
      ioctl(0,TCSETS,(char *)&ttmode);
      scr_get_size(&width,&height);
      tty_set_size(0,width,height);
      setgid(getgid());
      setuid(uid);
      if(login_shell)
	{
	  strcpy(argv0,"-");
	  strcat(argv0,argv[0]);
	  argv[0] = argv0;
	}
      execvp(command,argv);
      error("Couldn't execute %s",command);
      exit(1);
    }
  return(ptyfd);
}

/*  Tell the teletype handler what size the window is.  Called after a window
 *  size change.
 */
void tty_set_size(int fd,int width,int height)
{
  struct winsize wsize;
  
  if(fd < 0)
    return;

  wsize.ws_row = (unsigned short)height;
  wsize.ws_col = (unsigned short)width;
  ioctl(fd,TIOCSWINSZ,(char *)&wsize);
}

/*  Initialise the command connection.  This should be called after the X
 *  server connection is established.
 */
void init_command(char *command,char **argv)
{
  /*  Enable the delete window protocol.
   */
  wm_del_win = XInternAtom(display,"WM_DELETE_WINDOW",False);
  XSetWMProtocols(display,main_win,&wm_del_win,1);
  
  console_fd = -1;
  if(console != (char *)NULL)
    {
      console_fd = open(console,O_RDONLY);
      if(console_fd < 0)
	{
	  fprintf(stderr,"xvt: Can't open console %s for read\n",console);
	  console_fd = -1;
	}
      else
	{
	  fcntl(console_fd,F_SETFL,O_NDELAY);
	}
    }

  
  if ((comm_fd = run_command(command,argv)) < 0) 
    {
      error("Quitting");
      exit(1);
    }
  x_fd = XConnectionNumber(display);
  fd_width = getdtablesize();
  com_buf_next = com_buf_top = com_buf;
  com_stack_top = com_stack;
}

/*  Convert the keypress event into a string.
 */
static char *lookup_key(XEvent *ev,int *pcount)
{
  KeySym keysym;
  XComposeStatus compose;
  int count;
  static char kbuf[KBUFSIZE];
  char *s, *c;
  int meta;
  
  count = XLookupString(&ev->xkey,kbuf,KBUFSIZE,&keysym,&compose);
  meta = ev->xkey.state & Mod1Mask;
  s = NULL;
  switch (keysym) 
    {
    case XK_Up :
      s = app_cur_keys ? "\033OA" : "\033[A";
      break;
    case XK_Down :
      s = app_cur_keys ? "\033OB" : "\033[B";
      break;
    case XK_Right :
      s = app_cur_keys ? "\033OC" : "\033[C";
      break;
    case XK_Left :
      s = app_cur_keys ? "\033OD" : "\033[D";
      break;
    case XK_KP_F1 :
      s = "\033OP";
      break;
    case XK_KP_F2 :
      s = "\033OQ";
      break;
    case XK_KP_F3 :
      s = "\033OR";
      break;
    case XK_KP_F4 :
      s = "\033OS";
      break;
    case XK_KP_0 :
      s = app_kp_keys ? "\033Op" : "0";
      break;
    case XK_KP_1 :
      s = app_kp_keys ? "\033Oq" : "1";
      break;
    case XK_KP_2 :
      s = app_kp_keys ? "\033Or" : "2";
      break;
    case XK_KP_3 :
      s = app_kp_keys ? "\033Os" : "3";
      break;
    case XK_KP_4 :
      s = app_kp_keys ? "\033Ot" : "4";
      break;
    case XK_KP_5 :
      s = app_kp_keys ? "\033Ou" : "5";
      break;
    case XK_KP_6 :
      s = app_kp_keys ? "\033Ov" : "6";
      break;
    case XK_KP_7 :
      s = app_kp_keys ? "\033Ow" : "7";
      break;
    case XK_KP_8 :
      s = app_kp_keys ? "\033Ox" : "8";
      break;
    case XK_KP_9 :
      s = app_kp_keys ? "\033Oy" : "9";
      break;
    case XK_KP_Subtract :
      s = app_kp_keys ? "\033Om" : "-";
      break;
    case XK_KP_Separator :
      s = app_kp_keys ? "\033Ol" : ",";
      break;
    case XK_KP_Decimal :
      s = app_kp_keys ? "\033On" : ".";
      break;
    case XK_KP_Enter :
      s = app_kp_keys ? "\033OM" : "\r";
      break;
    case XK_Home :
      s = "\033Oq";
      break;
    case XK_End :
      s = "\033Ow";
      break;
    case XK_F1 :
      s = "\033[11~";
      break;
    case XK_F2 :
      s = "\033[12~";
      break;
    case XK_F3 :
      s = "\033[13~";
      break;
    case XK_F4 :
      s = "\033[14~";
      break;
    case XK_F5 :
      s = "\033[15~";
      break;
    case XK_F6 :
      s = "\033[17~";
      break;
    case XK_F7 :
      s = "\033[18~";
      break;
    case XK_F8 :
      s = "\033[19~";
      break;
    case XK_F9 :
      s = "\033[20~";
      break;
    case XK_F10 :
      s = "\033[21~";
      break;
    case XK_F11 :
      s = "\033[23~";
      break;
    case XK_F12 :
      s = "\033[24~";
      break;
    case XK_F13 :
      s = "\033[25~";
      break;
    case XK_F14 :
      s = "\033[26~";
      break;
    case XK_Help:
    case XK_F15 :
      s = "\033[28~";
	break;
    case XK_Menu:
    case XK_F16 :
      s = "\033[29~";
      break;
    case XK_F17 :
      s = "\033[31~";
      break;
    case XK_F18 :
      s = "\033[32~";
      break;
    case XK_F19 :
      s = "\033[33~";
      break;
    case XK_F20 :
      s = "\033[34~";
      break;
    case XK_Find :
      s = "\033[1~";
      break;
    case XK_Insert :
      s = "\033[2~";
      break;
    case XK_Select :
      s = "\033[4~";
      break;
    case XK_Prior :
      s = "\033[5~";
      break;
    case XK_Next :
      s = "\033[6~";
      break;
    }
  
  if (s != NULL)
    *pcount = strlen(s);
  else 
    {
      s = kbuf;
      *pcount = count;
    }
  if(meta && (mask == 0xff))
    for(c = s ; c < (s+(*pcount)) ; c++)
      *c |=0x80;
    
  return (s);
}

/*  Return the next input character after first passing any keyboard input
 *  to the command.  If flags & BUF_ONLY is true then only buffered 
 *  characters are returned and once the buffer is empty the special value 
 *  GCC_NULL is returned.  
 */
static int get_com_char(int flags)
{
  fd_set in_fdset;
  int count;
  char val;
  
  if (com_stack_top > com_stack)
    return(*--com_stack_top);
  
  if (com_buf_next < com_buf_top)
    return(*com_buf_next++ & mask);
  else if (flags & BUF_ONLY)
    return(GCC_NULL);
  
  for (;;) 
    {
      /* try reading from each descriptor before going into select(); */
      while(XPending(display))
	get_X_event();

      FD_ZERO(&in_fdset);
      FD_SET(comm_fd,&in_fdset);
      if(console_fd >=0)
	FD_SET(console_fd,&in_fdset);
      FD_SET(x_fd,&in_fdset);
      /* with Gcc -O2, Linux, the program fails unless you put something 
       * here */
      dummy();
      select(fd_width,&in_fdset,NULL,NULL,NULL);

      if((console_fd >=0) && (FD_ISSET(console_fd,&in_fdset)))
	{
	  count = read(console_fd,com_buf,1);
	  if (count > 0)
	    {
	      if(*com_buf=='\n')
		{
		  *(com_buf+1)='\r';
		  *(com_buf+2)=0;
		  count++;
		}
	      com_buf_next = com_buf;
	      com_buf_top = com_buf + count;
	      val = *com_buf_next;
	      com_buf_next++ ;
	      return(val & mask);	  
	    }
	}
      else if(FD_ISSET(comm_fd,&in_fdset))
	{
	  count = read(comm_fd,com_buf,COM_BUF_SIZE);
	  if (count > 0)
	    {
	      com_buf_next = com_buf;
	      com_buf_top = com_buf + count;
	      val = *com_buf_next;
	      com_buf_next++ ;
	      return(val & mask);	  
	    }
	}
    }
}
	
void dummy()
{
  int a;
  a=45;
}


void get_X_event()
{
  XEvent event;
  char *s;
  int count;
  Window root, child;
  int root_x, root_y, x, y;
  unsigned int mods;
  XEvent dummy;

  XNextEvent(display,&event);
  switch(event.type)
    {
    case KeyPress:
      s = lookup_key(&event,&count);
      send_string(s,count);
      return;
    case ClientMessage:
      if (event.xclient.format == 32 && event.xclient.data.l[0] == wm_del_win)
	exit(0);
      return;
    case MappingNotify:
      XRefreshKeyboardMapping(&event.xmapping);
      return;
    case GraphicsExpose:
    case Expose:
      while(XCheckTypedWindowEvent (display, event.xany.window, Expose, &dummy));
      while(XCheckTypedWindowEvent (display, event.xany.window, GraphicsExpose, &dummy));

      if (!size_set) 
	{	      
	  /*  Force a window resize if an exposure event
	   *  arrives before the first resize event.
	   */
	  resize_window();
	  size_set = 1;
	}
      if(event.xany.window==vt_win)
	{
	  scr_reset();
	  return;
	}
      else if(event.xany.window==sbar.sb_win)	  
	{
	  sbar_show(-1,-1,-1);
	  return;
	}
      else if(event.xany.window==sbar.sb_up_win)	  
	{
	  sbar_up_reset();
	  return;
	}
      else if(event.xany.window==sbar.sb_down_win)	  
	{
	  sbar_down_reset();
	  return;
	}
      return;
    case FocusIn:
      scr_focus(1);
      return;
    case FocusOut:
      scr_focus(0);
      return;
    case ConfigureNotify:
      resize_window();
      size_set = 1;
      return;
    case SelectionClear:
      scr_clear_selection();
      return;
    case SelectionNotify:
      scr_paste_primary(event.xselection.requestor,
			 event.xselection.property);
      return;
    case SelectionRequest:
      scr_send_selection(event.xselectionrequest.time,
			 event.xselectionrequest.requestor,
			 event.xselectionrequest.target,
			 event.xselectionrequest.property);
      return;
    case ButtonPress:
      if (event.xany.window == vt_win && event.xbutton.state == 0)
	{
	  switch (event.xbutton.button)
	    {
	    case Button1 :
	    case Button3 :
	      scr_start_selection(event.xbutton.x,event.xbutton.y);
	      return;
	    default:
	      return;
	    }
	}
      if (event.xany.window == sbar.sb_win)
	{
	  scr_move_to(event.xbutton.y);
	  return;
	}
      if (event.xany.window == sbar.sb_up_win)
	{
	  scr_move_up_down(UP);
	  return;
	}
      if (event.xany.window == sbar.sb_down_win)
	{
	  scr_move_up_down(DOWN);
	  return;
	}
      return;
    case ButtonRelease :
      if (event.xany.window == vt_win)
	{
	  switch (event.xbutton.button)
	    {
	    case Button1 :
	    case Button3 :
	      scr_make_selection(event.xbutton.time);
	      return;
	    case Button2 :
	      scr_request_selection(event.xbutton.time,event.xbutton.x,
				    event.xbutton.y);
	      return;
	    }
	}
      return;
    case MotionNotify :
      if ((event.xany.window == sbar.sb_win)
	  &&((event.xbutton.state&Button1Mask)||
	     (event.xbutton.state&Button2Mask)||
	     (event.xbutton.state&Button3Mask)))
	{
	  XQueryPointer(display,sbar.sb_win,&root,&child,
			&root_x,&root_y,&x,&y,&mods);
	  
	  scr_move_to(y);
	  return;
	}

      if (event.xany.window == vt_win && 
	  ((event.xbutton.state == Button1Mask)
	   ||(event.xbutton.state == Button3Mask)))
	{
	  scr_extend_selection(event.xbutton.x,event.xbutton.y);
	  return;
	}
      return;
    default:
      return;
    }
}


/*  Push an input character back into the input queue.
 */
static void push_com_char(int c)

{
  if (com_stack_top < com_stack + COM_PUSH_MAX)
    *com_stack_top++ = c;
}

/*  Send count characters directly to the command.
 */
void send_string(char *buf,int count)
{
  char s;
  register int i;
  
  for(i=0;i<count;i++)
    {
      s = (*buf == '\n')? '\r' : *buf ;
      if (write(comm_fd,&s,1) <= 0) 
	XBell(display,0);
      buf++;
    }
}


/*  Send printf formatted output to the command.  Only used for small ammounts
 *  of data.
 */
void cprintf(char *fmt,...)
{
  va_list args;
  static char buf[1024];
  
  va_start(args,fmt);
  
  vsprintf(buf,fmt,args);
  va_end(args);
  send_string(buf,strlen(buf));
}

/*  Return an input token
 */
void get_token()
{
  int c;
  
  while(1)
    {
      while((c = get_com_char(0)) == GCC_NULL);
      if (c >= ' ' || c == '\n' || c == '\r' || c == '\t') 
	process_string(c);
      else 
	switch(c)
	  {
	  case 0:         /* NUL does nothing */
	    break;
	  case EOF:
	    exit(0);
	    break;
	  case 5:
	    cprintf("\033[?6c");	/* I am a VT102 */
	    break;
	  case 0xb:
	    scr_index(1);
	    break;
	  case 0xc:
	    scr_index(1);
	    break;
	  case ESC:
	    process_escape_sequence();
	    break;
	  case '\b' :
	    scr_backspace();
	    break;
	  case '\007' :	/* bell */
	    XBell(display,0);
	    break;
	  case '\016':
	    scr_choose_charset(1);
	    break;
	  case '\017':
	    scr_choose_charset(0);
	    break;
	  }
    }
}


void process_string(int c)
{
  int i,nlcount;
  char string[STRING_MAX];

  i = 0;
  nlcount = 0;
  do 
    {
      string[i++] = c;
      c = get_com_char(BUF_ONLY);
      if (c == '\n' && ++nlcount >= NLMAX) 
	{
	  nlcount--;
	  break;
	}
    } while (!(c & ~0177) && (c >= ' ' || c == '\n' || c == '\r' 
			      || c == '\t') && i < STRING_MAX);
  if (c != GCC_NULL)
    push_com_char(c);
  string[i]=0;
  scr_string(string,i,nlcount);
} 

  
void process_escape_sequence()
{
  int c;

  c = get_com_char(0);
  switch(c)
    {
    case '[':
      process_csi_sequence();
      break;
    case ']':
      process_xterm_sequence();
      break;
    case '(':
      scr_set_charset(0,get_com_char(0));
      break;
    case ')':
      scr_set_charset(1,get_com_char(0));
      break;
    case '7':
      scr_save_cursor();
      break;
    case '8' :
      scr_restore_cursor();
      break;
    case '=' :
      app_kp_keys = 1;
      break;
    case '>' :
      app_kp_keys = 0;
      break;
    case 'D' :
      scr_index(1);
      break;
    case 'E':
      scr_string("\n\r",2,1);
      break;
    case 'M' :
      scr_index(-1);
      break;
    case 'Z' :
      cprintf("\033[?6c");	/* I am a VT102 */
      break;
    case '#':                /* set character size */
    case 'H' :               /* horizontal tab set */
    default:
      return;
    }
} 


void process_csi_sequence()
{
  int c,n,i,nargs,private=0;
  int arg[ARGS_MAX];

  arg[0]=0;
  arg[1]=0;

  c = get_com_char(0);
  if (c >= '<' && c <= '?') 
    {
      private = c;
      c = get_com_char(0);
    }

  /*  read any numerical arguments
   */
  i = 0;
  do 
    {
      n = 0;
      while (c >= '0' && c <= '9') 
	{
	  n = n * 10 + c - '0';
	  c = get_com_char(0);
	}
      if (i < ARGS_MAX)
	arg[i++] = n;
      if (c == ESC)
	push_com_char(c);
      if (c < ' ')
	return;
      if (c < '@')
	c = get_com_char(0);
    } while (c < '@' && c >= ' ');
  if (c == ESC)
    push_com_char(c);
  if (c < ' ')
    return;
  nargs = i;

  switch (c) 
    {
    case 'A':	/* cursor up */
      scr_move(0,((arg[0] == 0) ? -1 : -arg[0]),ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'B' :	/* cursor down */
      scr_move(0,((arg[0] == 0) ? 1 : arg[0]),ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'C' :	/* cursor forward */
      scr_move(((arg[0] == 0) ? 1 : arg[0]),0,ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'D' :	/* cursor back */
      scr_move(((arg[0] == 0) ? -1 : -arg[0]),0,ROW_RELATIVE | COL_RELATIVE);
      break;
    case 'f' :
    case 'H' :	/* position cursor */
      if(nargs==0)
	scr_move(0,0,0);
      else if (nargs == 1)
	scr_move(0,((arg[0]==0)? 0 : (arg[0]-1)),0);
      else 
	scr_move(arg[1] - 1,arg[0] - 1,0);
      break;
    case 'J' :
      scr_erase_screen(arg[0]);
      break;
    case 'K' :
      scr_erase_line(arg[0]);
      break;
    case 'L' :
      scr_insert_delete_lines( (arg[0]==0) ? 1 : arg[0],INSERT);
      break;
    case 'M' :
      scr_insert_delete_lines( (arg[0]==0) ? 1 : arg[0],DELETE);
      break;
    case 'P' :
      scr_insert_delete_characters( (arg[0]==0) ? 1 : arg[0],DELETE);
      break;
    case '@' :
      scr_insert_delete_characters( (arg[0]==0) ? 1 : arg[0],INSERT);
      break;
    case 'c' :
      cprintf("\033[?6c");	/* I am a VT102 */
      break;
    case 'h' :
    case 'l' :
      process_terminal_mode(c,private,nargs,arg);
      break;
    case 'm' :
      process_sgr_mode(c,private,nargs,arg);
      break;
    case 'n' :		/* request for information */
      switch (arg[0]) 
	{
	case 6 :
	  scr_report_position();
	  break;
	}
      break;
    case 'r' :		/* set top and bottom margins */
      if (nargs < 2 || arg[0] >= arg[1])
	scr_set_margins(0,10000);
      else
	scr_set_margins(arg[0] - 1,arg[1] - 1);
      break;
    case 'g' :                  /* tab clear */
      break;
    }
}



void process_xterm_sequence()
{
  int c,n,i,arg;
  char string[STRING_MAX];

  c = get_com_char(0);
  n = 0;
  while (c >= '0' && c <= '9') 
    {
      n = n * 10 + c - '0';
      c = get_com_char(0);
    }
  arg = n;

  c = get_com_char(0);
  i = 0;
  while (!(c & ~0177) && c != 7 && i < STRING_MAX) 
    {
      if (c >= ' ')
	string[i++] = c;
      c = get_com_char(0);
    }
  string[i] = 0;
  switch (arg) 
    {
    case 0 :
      change_window_name(string);
      change_icon_name(string);
      break;
    case 1 :
      change_icon_name(string);
      break;
    case 2 :
      change_window_name(string);
      break;
    }
}


void process_terminal_mode(int c,int private,int nargs,int *arg)
{
  int mode;

  mode = (c == 'h') ? HIGH : LOW;
  if (private == '?') 
    {
      switch (arg[0]) 
	{
	case 1 :
	  app_cur_keys = (mode == HIGH);
	  break;
	case 3:                 /* 132 columns mode is N/A */
	  break;
	case 4:                 /* Smooth scrolling is N/A */
	  break;
	case 6 :
	  scr_set_decom(mode);
	  break;
	case 7 :
	  scr_set_wrap(mode);
	  break;
	case 8:                 /* auto repeat is N/A */
	  break;
	case 47 :		/* switch to main screen */
	  scr_change_screen(mode);
	  break;
	}
    }
  else if (private == 0) 
    {
      if(arg[0]==4)
	scr_set_insert(mode);
    }
}


void process_sgr_mode(int c,int private,int nargs,int *arg)
{
  int i;

  if (nargs == 0)
    scr_change_rendition(1,~RS_NONE);
  else
    for (i = 0; i < nargs; i++) 
      switch (arg[i]) 
	{
	case 0 :
	  scr_change_rendition(1,~RS_NONE);
	  break;
	case 1 :
	  scr_change_rendition(0,RS_BOLD);
	  break;
	case 4 :
	  scr_change_rendition(0,RS_ULINE);
	  break;
	case 5 :
	  scr_change_rendition(0,RS_BLINK);
	  break;
	case 7 :
	  scr_change_rendition(0,RS_RVID);
	  break;
	case 22 :
	  scr_change_rendition(1,RS_BOLD);
	  break;
	case 24 :
	  scr_change_rendition(1,RS_ULINE);
	  break;
	case 25 :
	  scr_change_rendition(1,RS_BLINK);
	  break;
	case 27 :
	  scr_change_rendition(1,RS_RVID);
	  break;
    }
}
