/*  Copyright 1992 Robert Nation (nation@rocket.sanders.lockheed.com)
 *
 *  You can do what you like with this source code  as long as you include an
 *  unaltered copy of this message (including the copyright).
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 */
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <stdlib.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define CLASS	"Clock"

#define MW_EVENTS (ExposureMask | StructureNotifyMask)

/*  External global variables that are initialised at startup.
 */
Display		*display;
Window		main_win; /* parent window */
Colormap	colormap;
GC 		gc;	  /* GC for drawing text */
int    x_fd;              /* file descriptor of the X server connection */
char   *app_name = CLASS; /* the resource name */
int    screen;		  /* the X screen number */
XColor background_color;
XColor foreground_color;

static XSizeHints sizehints = {
	PMinSize | PResizeInc | PBaseSize,
	0, 0, 80, 80,	/* x, y, width and height */
	1, 1,		/* Min width and height */
	0, 0,		/* Max width and height */
	1, 1,		/* Width and height increments */
	1, 1,           /* x,y increments */
	1, 1,	        /* Aspect ratio */
	0, 0 ,		/* base size */
	0
};
#define BW 2
int sine[]={0,105,208,309,407,500,588,669,743,809,866,914,951,
	      978,995,999,995,978,951,914,866,809,743,669,588,500,
	      407,309,208,105,0,-105,-208,-309,-407,-500,-588,-669,
	      -743,-809,-866,-914,-951,-978,-995,-999,-994,-978,-951,
	      -914,-866,-809,-743,-669,-588,-500,-407,-309,-208,-105 };

void extract_resources(char *, char *, char *);
void create_window(int,char **);
void resize_window();
void get_event();
void dummy();
void error(char *);

void main(int argc,char **argv)
{
  int i;
  char *display_name = NULL;
  char *bg_color = "white";
  char *fg_color = "black";
  char *geom_string = "80x80";
  XGCValues gcv;
  
  display_name = getenv("DISPLAY");
  if(display_name == NULL)
    display_name = "unix:0.0";

  for(i=1;i<argc;i+=2)
    {
      if(strcmp(argv[i],"-display")==0)
	display_name = argv[i+1];
      else if(strcmp(argv[i],"-geometry")==0)
	geom_string = argv[i+1];
      else if(strcmp(argv[i],"-fg")==0)  
	fg_color = argv[i+1];
      else if(strcmp(argv[i],"-bg")==0)
	bg_color = argv[i+1];
      else 
	{
	  error("permitted arguments are:
-display <name>	        specify the display (server)
-geometry <spec>	the initial window geometry
-bg <colour>		background color
-fg <colour>		foreground color");
	}
    }

  if ((display = XOpenDisplay(display_name)) == NULL) 
      error("Can't open display");

  screen = DefaultScreen(display);
  colormap = DefaultColormap(display,screen);
  extract_resources(geom_string, fg_color, bg_color);
  
  create_window(argc,argv);
  
  /*  Create the graphics contexts.
   */
  gcv.foreground = foreground_color.pixel;
  gcv.background = background_color.pixel;
  gc = XCreateGC(display,main_win,GCForeground|GCBackground,&gcv);

  resize_window();

  get_event();
}

/*  Extract the resource fields that are needed to open the window.
 */
void extract_resources(char *geom_string, char *fg_color, char *bg_color)
{
  int x, y, width, height;
  int flags;
  
  flags = XParseGeometry(geom_string,&x,&y,&width,&height);
  
  if (flags & WidthValue) 
    {
      sizehints.width = width + sizehints.base_width;
      sizehints.flags |= USSize;
    }
  if (flags & HeightValue) 
    {
      sizehints.height = height + sizehints.base_height;
      sizehints.flags |= USSize;
    }
  if (flags & XValue) 
    {
      if (flags & XNegative)
	x = DisplayWidth(display,screen) + x - sizehints.width - 2*BW;
      sizehints.x = x;
      sizehints.flags |= USPosition;
    }
  if (flags & YValue) 
    {
      if (flags & YNegative)
	y = DisplayHeight(display,screen) + y - sizehints.height - 2*BW;
      sizehints.y = y;
      sizehints.flags |= USPosition;
    }
  
  /*  Do the foreground, and background colors.
   */
  if (XParseColor(display,colormap,fg_color,&foreground_color) == 0)
    fprintf(stderr,"invalid foreground color %s",fg_color);
  else if (XAllocColor(display,colormap,&foreground_color) == 0)
    fprintf(stderr,"can't allocate color %s",fg_color);

  if (XParseColor(display,colormap,bg_color,&background_color) == 0)
    fprintf(stderr,"invalid background color %s",bg_color);
  else if (XAllocColor(display,colormap,&background_color) == 0)
    fprintf(stderr,"can't allocate color %s",bg_color);
}

/*  Open and map the window.
 */
void create_window(int argc,char **argv)
{
  XTextProperty wname, iname;
  XClassHint class;
  XWMHints wmhints;
  
  main_win = XCreateSimpleWindow(display,DefaultRootWindow(display),
				 sizehints.x,sizehints.y,
				 sizehints.width,sizehints.height,
				 BW,foreground_color.pixel,
				 background_color.pixel);
  
  if (XStringListToTextProperty(&app_name,1,&wname) == 0) 
    error("cannot allocate window name");
  if (XStringListToTextProperty(&app_name,1,&iname) == 0) 
    error("cannot allocate icon name");
  class.res_name = app_name;
  class.res_class = CLASS;
  wmhints.input = True;
  wmhints.initial_state = NormalState;
  wmhints.flags = InputHint | StateHint;
  XSetWMProperties(display,main_win,&wname,&iname,argv,argc,
		   &sizehints,&wmhints,&class);
  XSelectInput(display,main_win,MW_EVENTS);
  XMapWindow(display,main_win);
}

/*  Redraw the whole window after an exposure or size change.
 */
void resize_window()
{
  Window root;
  time_t t;
  struct tm *tmval;
  int x, y,i,j;
  int width, height, border_width, depth;
  int angle1,angle2;
  int h_x,h_y,m_x,m_y,center_x,center_y;
  char time_string[100];
  char *str_ptr;
  XTextProperty name;

  XClearWindow(display,main_win);

  XGetGeometry(display,main_win,&root,&x,&y,&width,&height,&border_width,
	       &depth);

  center_x = width/2;
  center_y = height/2;

  t=time(0);
  tmval = localtime(&t);

  /* draw the hands */
  angle1 = (tmval->tm_hour%12)*5 + tmval->tm_min/12;
  h_x =   sine[angle1] * width  *60/200000 + center_x;
  h_y =   -(sine[(angle1+15)%60])* height *60/200000 + center_y;

  angle2 = tmval->tm_min;
  m_x =   sine[angle2] * width  *85/200000 + center_x;
  m_y = -(sine[(angle2+15)%60])* height *85/200000 + center_y;
  for(i=-1;i<2;i++)
    for(j=-1;j<2;j++)
      {
	XDrawLine(display,main_win,gc,center_x+i,center_y+j,h_x,h_y);
	XDrawLine(display,main_win,gc,center_x+i,center_y+j,m_x,m_y);
      }

  /* draw the clock face */
  for(i=0;i<60;i+=5)
    {
      angle1  = sine[i]*width;
      angle2  = -sine[(i+15)%60]*height;
      h_x = angle1 * 9 / 20000 + center_x;
      h_y = angle2 * 9 / 20000 + center_y;
      m_x = angle1 *10 / 20000 + center_x;
      m_y = angle2 *10 / 20000 + center_y;
      XDrawLine(display,main_win,gc,m_x,m_y,h_x,h_y);
    }

  strftime(time_string,100,"%a %h %d %I:%M %p",tmval);  
  str_ptr = time_string;
  if (XStringListToTextProperty(&str_ptr,1,&name) == 0) 
    {
      fprintf(stderr,"cannot allocate icon name");
      return;
    }
  XSetWMIconName(display,main_win,&name);
  XFree(name.value);
}

/*  Return the next input character after first passing any keyboard input
 *  to the command.  If flags & BUF_ONLY is true then only buffered 
 *  characters are returned and once the buffer is empty the special value 
 *  GCC_NULL is returned.  
 */
void get_event()
{
  fd_set in_fdset;
  XEvent event;
  int fd_width = 10;
  struct timeval tm;

  x_fd = XConnectionNumber(display);
  for (;;) 
    {
      /* try reading from each descriptor before going into select(); */
      while(XPending(display))
	XNextEvent(display,&event); 

      FD_ZERO(&in_fdset);
      FD_SET(x_fd,&in_fdset);
      tm.tv_sec = 60;
      tm.tv_usec = 0;
      /* with Gcc -O2, Linux, the program fails unless you put something 
       * here */
      dummy();
      select(fd_width,&in_fdset,NULL,NULL,&tm);
      resize_window();
    }
}
	
void dummy()
{
  int a;
  a=45;
}

void error(char *message)
{
  fprintf(stderr,"rclock: %s\n",message);
  exit(1);
}
