/****************************************************************************
 * Some parts of this module are based on Twm code, but have been 
 * siginificantly modified by Rob Nation (nation@rocket.sanders.lockheed.com)
 * Much of the code is new.
 ****************************************************************************/
/*****************************************************************************/
/**       Copyright 1988 by Evans & Sutherland Computer Corporation,        **/
/**                          Salt Lake City, Utah                           **/
/**  Portions Copyright 1989 by the Massachusetts Institute of Technology   **/
/**                        Cambridge, Massachusetts                         **/
/**                                                                         **/
/**                           All Rights Reserved                           **/
/**                                                                         **/
/**    Permission to use, copy, modify, and distribute this software and    **/
/**    its documentation  for  any  purpose  and  without  fee is hereby    **/
/**    granted, provided that the above copyright notice appear  in  all    **/
/**    copies and that both  that  copyright  notice  and  this  permis-    **/
/**    sion  notice appear in supporting  documentation,  and  that  the    **/
/**    names of Evans & Sutherland and M.I.T. not be used in advertising    **/
/**    in publicity pertaining to distribution of the  software  without    **/
/**    specific, written prior permission.                                  **/
/**                                                                         **/
/**    EVANS & SUTHERLAND AND M.I.T. DISCLAIM ALL WARRANTIES WITH REGARD    **/
/**    TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES  OF  MERCHANT-    **/
/**    ABILITY  AND  FITNESS,  IN  NO  EVENT SHALL EVANS & SUTHERLAND OR    **/
/**    M.I.T. BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL  DAM-    **/
/**    AGES OR  ANY DAMAGES WHATSOEVER  RESULTING FROM LOSS OF USE, DATA    **/
/**    OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER    **/
/**    TORTIOUS ACTION, ARISING OUT OF OR IN  CONNECTION  WITH  THE  USE    **/
/**    OR PERFORMANCE OF THIS SOFTWARE.                                     **/
/*****************************************************************************/


/***************************************************************************
 * 
 * Configure.c: reads the .fvwmrc or system.fvwmrc file, interprets it,
 * and sets up menus, bindings, colors, and fonts as specified
 *
 ***************************************************************************/

#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <X11/Xos.h>
#include "fvwm.h"
#include "menus.h"
#include "misc.h"
#include "parse.h"
#include "screen.h"

char *Stdback = "white";
char *Stdfore = "black";
char *Hiback = "white";
char *Hifore = "black";

void     GetColors();
Pixel    GetColor(char *);
void     ParseMouseEntry(char *,FILE *);
void     ParseKeyEntry(char *);
void     MakeMenu(MenuRoot *);
void     AddToMenu(MenuRoot *, char *, char *,int, int, int);
MenuRoot *NewMenuRoot(char *name);
void     AddFuncKey (char *, int, int, int, char *, int, int);
char     *safemalloc(int length);
char     *stripcpy(char *);
char     *stripcpy2(char *);
char     *stripcpy3(char *);
void     find_context(char *);
void     find_modifiers(char *);
void     find_func(char *);

int contexts;
int mods,func,func_val_1,func_val_2;


/*****************************************************************************
 * 
 * This routine is responsible for reading and parsing the config file
 *
 ****************************************************************************/
void MakeMenus()
{
  char *system_file = "/usr/lib/X11/fvwm/system.fvwmrc";
  char *home_file;
  MenuRoot *mr;
  FILE *fd = (FILE *)0;
  char line[256],*ptr,*tline;
  char *Home;			/* the HOME environment variable */
  int HomeLen;			/* length of Home */

  /* initialize some lists */
  Scr.MouseButtonRoot = NULL;
  Scr.FuncKeyRoot.next = NULL;
  Scr.NoTitle = NULL;

  /* find the home directory to look in */
  Home = getenv("HOME");
  if (Home == NULL)
    Home = "./";
  HomeLen = strlen(Home);

  home_file = safemalloc(HomeLen+8);
  strcpy(home_file,Home);
  strcat(home_file,"/.fvwmrc");

  fd = fopen(home_file,"r");
  if(fd == (FILE *)0)
    fd = fopen(system_file,"r");
  if(fd == (FILE *)0)
    fprintf(stderr,"fvwm: couldn't open file %s or %s\n",
	    system_file,home_file);

  if(fd != (FILE *)0)
    {
      tline = fgets(line,255,fd);
      while(tline != (char *)0)
	{
	  while(isspace(*tline))tline++;
	  if(strncasecmp(tline,"Font",4)==0)
	    {
	      ptr = stripcpy(&tline[4]);
	      Scr.StdFont.name = ptr;
	    }
	  else if(strncasecmp(tline,"StdForeColor",12)==0)
	    {
	      ptr = stripcpy(&tline[12]);
	      Stdfore = ptr;
	    }
	  else if(strncasecmp(tline,"StdBackColor",12)==0)
	    {
	      ptr = stripcpy(&tline[12]);
	      Stdback = ptr;
	    }
	  else if(strncasecmp(tline,"HiBackColor",11)==0)
	    {
	      ptr = stripcpy(&tline[11]);
	      Hiback = ptr;
	    }
	  else if(strncasecmp(tline,"HiForeColor",11)==0)
	    {
	      ptr = stripcpy(&tline[11]);
	      Hifore = ptr;
	    }
	  else if(strncasecmp(tline,"NoTitle",7)==0)
	    {
	      ptr = stripcpy(&tline[7]);
	      AddToList(&Scr.NoTitle, ptr);
	    }
	  else if(strncasecmp(tline,"Sticky",6)==0)
	    {
	      ptr = stripcpy(&tline[6]);
	      AddToList(&Scr.Sticky, ptr);
	    }
	  else if(strncasecmp(tline,"StaysOnTop",10)==0)
	    {
	      ptr = stripcpy(&tline[10]);
	      AddToList(&Scr.OnTop,ptr);
	    }
	  else if(strncasecmp(tline,"DeskTopSize",11)==0)
	    {
	      sscanf(&tline[11],"%dx%d",&Scr.VxMax,&Scr.VyMax);
	      Scr.VxMax = (Scr.VxMax)*Scr.MyDisplayWidth;
	      Scr.VxMax = (Scr.VxMax/32)*32 - Scr.MyDisplayWidth;
	      Scr.VyMax = (Scr.VyMax)*Scr.MyDisplayHeight;
	      Scr.VyMax = (Scr.VyMax/32)*32 - Scr.MyDisplayHeight;
	    }
	  else if(strncasecmp(tline,"Mouse",5)==0)
	    ParseMouseEntry(&tline[5],fd);
	  else if(strncasecmp(tline,"Key",3)==0)
	    ParseKeyEntry(&tline[3]);
	  else if((strlen(&tline[0])>1)&&(tline[0]!='#'))
	    fprintf(stderr,"fvwm: error in config file: %s\n",tline);
	  tline = fgets(line,255,fd);
	}
    }
  else
    {
      GetColors();
      mr = NewMenuRoot("Window Ops");

      AddToMenu(mr, "Window Ops", (char *)NULL, F_TITLE,0,0);
      AddToMenu(mr, "Xterm", "xterm &", F_EXEC,0,0);
      AddToMenu(mr, "Move Window", (char *)NULL, F_MOVE,0,0);
      AddToMenu(mr, "Resize Window", (char *)NULL, F_RESIZE,0,0);
      AddToMenu(mr, "Raise Window", (char *)NULL, F_RAISE,0,0);
      AddToMenu(mr, "Lower Window", (char *)NULL, F_LOWER,0,0);
      AddToMenu(mr, "(De)Iconify Window", (char *)NULL, F_ICONIFY,0,0);
      AddToMenu(mr, "Destroy Window", (char *)NULL, F_DESTROY,0,0);
      AddToMenu(mr, "Exit Fvwm", (char *)NULL, F_QUIT,0,0);

      MakeMenu(mr);
      Scr.MouseButtonRoot = (MouseButton *)safemalloc(sizeof(MouseButton));
      Scr.MouseButtonRoot->func = F_MENU;
      Scr.MouseButtonRoot->menu = mr;
      Scr.MouseButtonRoot->item = (MenuItem *)0;
      Scr.MouseButtonRoot->Button = 1;
      Scr.MouseButtonRoot->Context = C_ROOT;
      Scr.MouseButtonRoot->Modifier = 0;
      Scr.MouseButtonRoot->NextButton = (MouseButton *)0;
      Scr.MouseButtonRoot->val1 = 0;
      Scr.MouseButtonRoot->val2 = 0;

    }
  GetColors();
  XSync(dpy,0);
  return;
}

/****************************************************************************
 *
 * This routine loads all needed colors, and fonts,
 * and creates the GC's
 *
 ***************************************************************************/
void GetColors()
{
  static int have_em = 0;

  if(have_em) return;

  have_em = 1;
  /* setup default colors */
  if(Scr.d_depth < 2)
    {
      /* black and white - override user choices */
      Scr.StdColors.back = GetColor("White");
      Scr.StdColors.fore = GetColor("Black"); 
      Scr.HiColors.back =  GetColor("White");
      Scr.HiColors.fore =  GetColor("Black"); 
    }
  else
    {
      /* color - accept user choices */
      Scr.StdColors.back = GetColor(Stdback);
      Scr.StdColors.fore = GetColor(Stdfore); 
      Scr.HiColors.back =  GetColor(Hiback);
      Scr.HiColors.fore = GetColor(Hifore); 
    }

  /* always need black and white for the 3-D look */
  Scr.BWColors.fore = GetColor("Black");
  Scr.BWColors.back = GetColor("White");

  /* load the font */
  if ((Scr.StdFont.font = XLoadQueryFont(dpy, Scr.StdFont.name)) == NULL)
    {
      fprintf (stderr, "fvwm: unable to open font %s\n", Scr.StdFont.name);
      if ((Scr.StdFont.font = XLoadQueryFont(dpy, "fixed")) == NULL)
	{
	  fprintf (stderr, "fvwm: unable to open font %s\n", Scr.StdFont.name);
	  exit(1);
	}
    }
  Scr.StdFont.height = Scr.StdFont.font->ascent + Scr.StdFont.font->descent;
  Scr.StdFont.y = Scr.StdFont.font->ascent;
  Scr.EntryHeight = Scr.StdFont.height + 2;

  /* create graphics contexts */
  CreateGCs();

  FB(Scr.StdColors.fore,Scr.StdColors.back); 
  return;
}

/****************************************************************************
 * 
 *  Processes a line with a mouse binding
 *
 ****************************************************************************/ 
void ParseMouseEntry(char *tline,FILE *fd)
{
  char context[256],modifiers[256],function[256],newline[256], *pline,*ptr;
  char *ptr2;
  MenuRoot *mr=0;
  MenuItem *mi=0;
  MouseButton *temp;
  int button;

  sscanf(tline,"%d %s %s %s %d %d",&button,context,modifiers,function,
	 &func_val_1,&func_val_2);
  
  find_context(context);
  find_modifiers(modifiers);
  find_func(function);
  if(func == F_MENU)
    {
      GetColors();
      ptr = stripcpy2(tline);
      mr = NewMenuRoot(ptr);
      AddToMenu(mr, ptr, (char *)NULL, F_TITLE,0,0);
      pline = fgets(newline,255,fd);
      while(isspace(*pline))pline++;
      while((pline != (char *)0)&&(strncasecmp("EndMenu",pline,7)!=0))
	{
	  find_func(pline);
	  ptr2 = 0;
	  if(func == F_EXEC)
	    ptr2=stripcpy3(pline);
	  if(func != F_MENU)
	    AddToMenu(mr, stripcpy2(pline), ptr2, func,func_val_1,func_val_2);	  
	  else
	    {
	      fprintf(stderr,"fvwm: Submenus are not allowed\n");
	      func = 0;
	    }
	  pline = fgets(newline,255,fd);
	  while(isspace(*pline))pline++;
	}
      MakeMenu(mr);
      func = F_MENU;
    }
  else if(func == F_EXEC)
    {
      mi = (MenuItem *)safemalloc(sizeof(MenuItem));
      
      mi->root = (MenuRoot *)NULL;
      mi->next = (MenuItem *)NULL;
      mi->prev = (MenuItem *)NULL;
      mi->item_num = 0;
      mi->item = stripcpy2(tline);
      mi->action = stripcpy3(tline);
      mi->state = 0;
      mi->func = func;
      mi->strlen = strlen(mi->item);
      mi->val1 = 0;
      mi->val2 = 0;
    }
  
  temp = Scr.MouseButtonRoot;
  Scr.MouseButtonRoot = (MouseButton *)safemalloc(sizeof(MouseButton));
  Scr.MouseButtonRoot->func = func;
  Scr.MouseButtonRoot->menu = mr;
  Scr.MouseButtonRoot->item = mi;
  Scr.MouseButtonRoot->Button = button;
  Scr.MouseButtonRoot->Context = contexts;
  Scr.MouseButtonRoot->Modifier = mods;
  Scr.MouseButtonRoot->NextButton = temp;
  Scr.MouseButtonRoot->val1 = func_val_1;
  Scr.MouseButtonRoot->val2 = func_val_2;
  return;
}


/****************************************************************************
 * 
 *  Processes a line with a key binding
 *
 ****************************************************************************/ 
void ParseKeyEntry(char *tline)
{
  char context[256],modifiers[256],function[256],*ptr;
  char name[256];

  sscanf(tline,"%s %s %s %s %d %d",name,context,modifiers,function,
	 &func_val_1,&func_val_2);

  find_context(context);
  find_modifiers(modifiers);
  find_func(function);

  if(func == F_EXEC)
    ptr = stripcpy3(tline);
  
  AddFuncKey(name,contexts,mods,func,ptr,func_val_1,func_val_2);
}


/****************************************************************************
 * 
 * Turns the string context into an array of true/false values 
 *
 ****************************************************************************/ 
void find_context(char *context)
{
  int i=0;

  contexts=0;
  i=0;
  while(i<strlen(context))
    {
      switch(context[i])
	{
	case 'W':
	case 'w':
	  contexts |= C_WINDOW;
	  break;
	case 'T':
	case 't':
	  contexts |= C_TITLE;
	  break;
	case 'I':
	case 'i':
	  contexts |= C_ICON;
	  break;
	case 'R':
	case 'r':
	  contexts |= C_ROOT;
	  break;
	case 'F':
	case 'f':
	  contexts |= C_FRAME;
	  break;
	case 'S':
	case 's':
	  contexts |= C_SIDEBAR;
	  break;
        case 'A':
        case 'a':
	  contexts = C_WINDOW|C_TITLE|C_ICON|C_ROOT|C_FRAME|C_SIDEBAR;
	  break;
	default:
	  fprintf(stderr,"fvwm: Unrecognized context %c\n",context[i]);
	}
      i++;
    }
  return;
}

/****************************************************************************
 * 
 * Turns the string modifiers into a mask value
 *
 ****************************************************************************/ 
void find_modifiers(char *modifiers)
{
  int i=0;

  i=0;
  mods=0;
  while(i<strlen(modifiers))
    {
      switch(modifiers[i])
	{
	case 'S':
	case 's':
	  mods |= ShiftMask;
	  break;
	case 'C':
	case 'c':
	  mods |= ControlMask;
	  break;
	case 'M':
	case 'm':
	  mods |= Mod1Mask;
	  break;
	case 'N':
	case 'n':
	  mods=0;
	  break;
	default:
	  fprintf(stderr,"fvwm: Unrecognized modifier %c\n",modifiers[i]);
	}
      i++;
    }
}


/****************************************************************************
 * 
 * Turns the string function into an integer
 *
 ****************************************************************************/ 
void find_func(char *function)
{
  func_val_1 = 0;
  func_val_2 = 0;

  if(strncasecmp(function,"NOP",3)==0)
    func = F_NOP;
  else if(strncasecmp(function,"BEEP",4)==0)
    func = F_BEEP;
  else if(strncasecmp(function,"QUIT",4)==0)
    func = F_QUIT;
  else if(strncasecmp(function,"REFRESH",7)==0)
    func = F_REFRESH;
  else if(strncasecmp(function,"MOVE",4)==0)
    func = F_MOVE;
  else if(strncasecmp(function,"ICONIFY",7)==0)
    func = F_ICONIFY;
  else if(strncasecmp(function,"RESIZE",6)==0)
    func = F_RESIZE;
  else if(strncasecmp(function,"RAISE",5)==0)
    func = F_RAISE;
  else if(strncasecmp(function,"LOWER",5)==0)
    func = F_LOWER;
  else if(strncasecmp(function,"DELETE",6)==0)
    func = F_DELETE;
  else if(strncasecmp(function,"DESTROY",7)==0)
    func = F_DESTROY;
  else if(strncasecmp(function,"MENU",4)==0)
    func = F_MENU;
  else if(strncasecmp(function,"PANNER",6)==0)
    func = F_PANNER;
  else if(strncasecmp(function,"SCROLL",6)==0)
    {
      func = F_SCROLL;
    }
  else if(strncasecmp(function,"EXEC",4)==0)
    func = F_EXEC;
  else 
    {
      fprintf(stderr,"fvwm: Unrecognized key binding %s\n",function);
      func = 0;
    }
  return;
}

  

/****************************************************************************
 * 
 * Generates the window for a menu
 *
 ****************************************************************************/ 
void MakeMenu(MenuRoot *mr)
{
  MenuItem *cur;
  int width;
  unsigned long valuemask;
  XSetWindowAttributes attributes;
  
  /* lets first size the window accordingly */
  width = mr->width + 10;
  
  for (cur = mr->first; cur != NULL; cur = cur->next)
    cur->x = (width - XTextWidth(Scr.StdFont.font, cur->item,
				 cur->strlen))/2;
  mr->height = mr->items * Scr.EntryHeight;
  mr->width += 10;
  
  valuemask = (CWBackPixel | CWBorderPixel | CWEventMask);
  attributes.background_pixel = Scr.StdColors.back;
  attributes.border_pixel = Scr.StdColors.fore;
  attributes.event_mask = (ExposureMask | EnterWindowMask);
  mr->w = XCreateWindow (dpy, Scr.Root, 0, 0, (unsigned int) mr->width,
			 (unsigned int) mr->height, (unsigned int) 1,
			 CopyFromParent, (unsigned int) CopyFromParent,
			 (Visual *) CopyFromParent,
			 valuemask, &attributes);
  
  
  XSaveContext(dpy, mr->w, MenuContext, (caddr_t)mr);
  
  return;
}


/***********************************************************************
 *
 *  Procedure:
 *	AddToMenu - add an item to a root menu
 *
 *  Returned Value:
 *	(MenuItem *)
 *
 *  Inputs:
 *	menu	- pointer to the root menu to add the item
 *	item	- the text to appear in the menu
 *	action	- the string to possibly execute
 *	func	- the numeric function
 *	fore	- foreground color string
 *	back	- background color string
 *
 ***********************************************************************/
void AddToMenu(MenuRoot *menu, char *item, char *action,int func, 
	       int func_val_1,int func_val_2)
{
  MenuItem *tmp;
  int width;

  tmp = (MenuItem *)safemalloc(sizeof(MenuItem));
  tmp->root = menu;
  if (menu->first == NULL)
    {
      menu->first = tmp;
      tmp->prev = NULL;
    }
  else
    {
      menu->last->next = tmp;
      tmp->prev = menu->last;
    }
  menu->last = tmp;
  
  tmp->item = item;
  tmp->strlen = strlen(item);
  tmp->action = action;
  tmp->next = NULL;
  tmp->state = 0;
  tmp->func = func;
  tmp->val1 = func_val_1;
  tmp->val2 = func_val_2;

  width = XTextWidth(Scr.StdFont.font, item, tmp->strlen);
  if (width <= 0)
    width = 1;
  if (width > menu->width)
    menu->width = width;
  
  tmp->item_num = menu->items++;
}

/***********************************************************************
 *
 *  Procedure:
 *	NewMenuRoot - create a new menu root
 *
 *  Returned Value:
 *	(MenuRoot *)
 *
 *  Inputs:
 *	name	- the name of the menu root
 *
 ***********************************************************************/
MenuRoot *NewMenuRoot(char *name)
{
    MenuRoot *tmp;

    tmp = (MenuRoot *) safemalloc(sizeof(MenuRoot));
    tmp->name = name;
    tmp->first = NULL;
    tmp->last = NULL;
    tmp->items = 0;
    tmp->width = 0;
    tmp->w = None;
    tmp->real_menu = FALSE;
    return (tmp);
}



/***********************************************************************
 *
 *  Procedure:
 *	AddFuncKey - add a function key to the list
 *
 *  Inputs:
 *	name	- the name of the key
 *	cont	- the context to look for the key press in
 *	mods	- modifier keys that need to be pressed
 *	func	- the function to perform
 *	action	- the action string associated with the function (if any)
 *
 ***********************************************************************/
void AddFuncKey (char *name, int cont, int mods, int func,  char *action,
		 int val1, int val2)
{
  FuncKey *tmp;
  KeySym keysym;
  KeyCode keycode;

  /*
   * Don't let a 0 keycode go through, since that means AnyKey to the
   * XGrabKey call in GrabKeys().
   */
  if ((keysym = XStringToKeysym(name)) == NoSymbol ||
      (keycode = XKeysymToKeycode(dpy, keysym)) == 0)
    return;
  
  tmp = (FuncKey *) safemalloc(sizeof(FuncKey));
  tmp->next = Scr.FuncKeyRoot.next;
  Scr.FuncKeyRoot.next = tmp;
  
  tmp->name = name;
  tmp->keysym = keysym;
  tmp->keycode = keycode;
  tmp->cont = cont;
  tmp->mods = mods;
  tmp->func = func;
  tmp->action = action;
  tmp->val1 = val1;
  tmp->val2 = val2;
  
  return;
}

/****************************************************************************
 * 
 * Loads a single color
 *
 ****************************************************************************/ 
Pixel GetColor(char *name)
{
  XColor color, junkcolor;
  XWindowAttributes attributes;

  XGetWindowAttributes(dpy,Scr.Root,&attributes);
  if(!XAllocNamedColor (dpy, attributes.colormap, name, &color, &junkcolor))
    fprintf(stderr,"fvwm: couldn't allocate color %s\n",name);
  
  return color.pixel;
}



/****************************************************************************
 * 
 * Copies a string into a new, malloc'ed string
 * Strips leading spaces and trailing spaces and new lines
 *
 ****************************************************************************/ 
char *stripcpy(char *source)
{
  char *tmp,*ptr;

  while(isspace(*source))
    source++;
  tmp = source + strlen(source) -1;
  while(((isspace(*tmp))||(*tmp == '\n'))&&(tmp >=source))
    tmp--;
  *(tmp+1)=0;

  ptr = safemalloc(strlen(source)+1);
  strcpy(ptr,source);
  return ptr;
}
  


/****************************************************************************
 * 
 * Copies a string into a new, malloc'ed string
 * Strips all data before the first quote and after the second
 *
 ****************************************************************************/
char *stripcpy2(char *source)
{
  char *ptr;
  int count;

  while((*source != '"')&&(*source != 0))
    source++;
  if(*source == 0)
    {
      fprintf(stderr,"fvwm: Malformed mouse binding: %s\n",source);
      return 0;
    }
  source++;
  ptr = source;
  count = 0;
  while((*ptr!='"')&&(*ptr != 0))
    {
      ptr++;  
      count++;
    }
  ptr = safemalloc(count+1);
  strncpy(ptr,source,count);
  ptr[count]=0;
  return ptr;
}


/****************************************************************************
 * 
 * Copies a string into a new, malloc'ed string
 * Strips all data before the second quote. and strips trailing spaces and
 * new lines
 *
 ****************************************************************************/
char *stripcpy3(char *source)
{
  char *tmp,*ptr;
  int count;
  
  while((*source != '"')&&(*source != 0))
    source++;
  if(*source != 0)
    source++;
  while((*source != '"')&&(*source != 0))
    source++;
  if(*source == 0)
    {
      fprintf(stderr,"fvwm: Malformed mouse binding\n");
      return 0;
    }
  source++;
  while(isspace(*source))
    source++;
  tmp = source + strlen(source) -1;
  count = strlen(source);
  while(((isspace(*tmp))||(*tmp == '\n'))&&(tmp >=source))
    {
      tmp--;
      count--;
    }
  ptr = safemalloc(count+1);
  strncpy(ptr,source,count);
  ptr[count]=0;
  return ptr;
}
  


/***********************************************************************
 *
 *  Procedure:
 *	CreateGCs - open fonts and create all the needed GC's.  I only
 *		    want to do this once, hence the first_time flag.
 *
 ***********************************************************************/
void CreateGCs()
{
  XGCValues	    gcv;
  unsigned long   gcm;
  
  /* create GC's */
  gcm = GCFunction|GCLineWidth|GCForeground|GCSubwindowMode; 
  gcv.function = GXxor;
  gcv.line_width = 0;
  gcv.foreground = (((unsigned long) 1) << Scr.d_depth) - 1;
  gcv.subwindow_mode = IncludeInferiors;
  Scr.DrawGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);

  gcm = GCPlaneMask|GCGraphicsExposures|GCLineWidth|GCForeground|
    GCBackground|GCFont;
  gcv.plane_mask = AllPlanes;
  gcv.foreground = Scr.StdColors.fore;
  gcv.background = Scr.StdColors.back;
  gcv.font =  Scr.StdFont.font->fid;
  /*
   * Prevent GraphicsExpose and NoExpose events.  We'd only get NoExpose
   * events anyway;  they cause BadWindow errors from XGetWindowAttributes
   * call in FindScreenInfo (events.c) (since drawable is a pixmap).
   */
  gcv.graphics_exposures = False;
  gcv.line_width = 0;
  
  Scr.NormalGC = XCreateGC(dpy, Scr.Root, gcm, &gcv);
}

/***********************************************************************
 *
 *  Procedure:
 *	safemalloc - mallocs specified space or exits if there's a 
 *		     problem
 *
 ***********************************************************************/
char *safemalloc(int length)
{
  char *ptr;
  
  ptr = malloc(length);
  if(ptr == (char *)0)
    {
      fprintf(stderr,"fvwm: Couldn't malloc temporary space\n");
      exit(1);
    }
  return ptr;
}


/***********************************************************************
 *
 *  Procedure:
 *	AddToList - add a window name to the no title list
 *
 *  Inputs:
 *	name	- a pointer to the name of the window 
 *
 ***********************************************************************/
void AddToList(name_list **list, char *name)
{
  name_list *nptr;
  
  nptr = (name_list *)safemalloc(sizeof(name_list));
  
  nptr->next = *list;
  nptr->name = name;
  *list = nptr;
}    
