#!/bin/sh
#
# handleacct.sh
# Handles /var/log/acct file and old acct files
# Run this script periodically from root's crontab
# Sample line to add into /etc/crontab:
#00 * * * * root /usr/local/sbin/handleacct.sh -c -s 97
#
# For explanation of options see builtin help in this script
#
# Use this script at your own risk. No guarantee or warranty of
# any kind of is given against this script to destroy all files
# in your system or cause other damage. Read this script
# carefully before using it. Modify it to suite to your local
# environment.
#
# Apply GPL as copyright to this file. 
#
# Author: <jiivee@iki.fi> Juha Virtanen
# Last time modified on 12th March 1996

PATH=/sbin:/bin:/usr/sbin:/usr/bin:/usr/local/sbin:/usr/local/bin; export PATH

prog=`basename $0`
version=1.3.73

usage="Usage: ${prog}: [-v] [-h] [-c] [-s #] [-f file] [-d dir]"

# Log to syslog what we do.
# To not log, comment logging line out
slog ()
{
	logger -t "$prog" -i -p user.notice $*
}

# Where are old files saved
storedir=/var/log/accounting
storefile="${storedir}/`date +acct-%Y%m%d-%H%M`"
acctfile="/var/log/acct"
# Remove old accounting files after N hours, default is 25 hours.
rmtime=25
# By default, do not compress old files
compress=0

while [ $# -gt 0 ]; do
	case $1 in
		-c|--compress)
			compress=1
		;;
		-d|--storedir)
			storedir=$2
			shift
		;;
		-f|--acct_file)
			acctfile=$2
			shift
		;;
		-s|--save-time)
			rmtime=$2
			shift
		;;
		-v|--version)
			echo "${prog} version is $version"
			exit 0
		;;
		-h|--help)
			echo "$usage"
			echo ""
			echo "-h|--help           This help"
			echo "-c|--compress       Compress old files with gzip"
			echo "-d|--storedir dir   Directory to put old files to (${storedir})"
			echo "-f|--acct-file file Current accounting file (${acctfile})"
			echo "-s|--save-time #    How many hours to keep old accounting files (${rmtime} hours)"
			echo "-v|--version        Print version info"
			exit 0
		;;
		*)
			echo "Unknown option -- $1" 1>&2
			echo "$usage" 1>&2
			exit 2
		;;
	esac
	shift
done

if [ ! -f $acctfile ]; then
	slog "${acctfile}: Not found or not a regular file."
	echo "${prog}: ${acctfile}: Not found or not a regular file." 1>&2
	exit 1
fi
if [ ! -d $storedir ]; then
	mkdir -p $storedir
	if [ $? != 0 ]; then
		slog "mkdir: cannot make directory $storedir"
		echo "${prog}: mkdir: cannot make directory $storedir" 1>&2
		exit 2
	fi
	chown root.adm $storedir
	# Prevent curious people seeing old accounting data
	chmod 2750 $storedir
fi

# Move old file to new name and create new file
# before turning off and on accounting to minimalize the time
# we are running without accounting
slog "Handling $acctfile"
rm -f ${acctfile}.tmp
mv $acctfile ${acctfile}.tmp
touch $acctfile
accton
accton $acctfile
mv -f ${acctfile}.tmp $storefile
# Allow everybody to examine current accounting file
chown root.adm $acctfile
chmod 0755 $acctfile

slog "Handling $storefile"
tmpfile=/tmp/${prog}.$$
acctentries $storefile 2>&1 > $tmpfile
cat $tmpfile | slog
rm -f $tmpfile
# Compress old files to save space
if [ $compress = 1 ]; then
	rm -f ${storefile}.gz
	nice -n 14 gzip -9 $storefile
fi
sync

# Remove old accounting files after N hours
slog "Old accounting files removed after $rmtime hours."
time=`expr $rmtime \* 60`
storedir=`dirname $storefile`
for f in `find $storedir -maxdepth 1 -type f -mmin +$time -print`; do
	slog "removing $f"
	rm -f $f
done

#slog "Done."
#EOF
